/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CUndoDat - アンドゥ個々のデータ、書き込み/読み込み処理
*/


#include "CUndoDat.h"

#include "CLayerList.h"
#include "CLayerItem.h"
#include "CTileImage.h"

#include "AXString.h"
#include "AXMem.h"

#include "drawdat.h"
#include "struct.h"
#include "global.h"

#include "compressUndo.h"


//-------------------------

#define BUFMAXSIZE      (256 * 1024)

struct UNDOLAYERINFO
{
    BYTE    btOpacity,
            btColType,
            btAMaskType,
            btBlendMode;
    DWORD   dwCol,
            dwFlags;
};

AXMem   g_memWorkSrc(64 * 64 * sizeof(RGBAFIX15)),
        g_memWorkCmp(64 * 64 * sizeof(RGBAFIX15));

//-------------------------



//================================
// 書き込み
//================================


//! リンクされたレイヤの番号書き込み

BOOL CUndoDat::writeLinkLayersNO(CLayerItem *pTop,int cnt)
{
    CLayerItem *p;
    short no;

    if(!m_dat.alloc((cnt + 1) * 2)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    //レイヤ番号

    for(p = pTop; p; p = p->m_pLink)
    {
        no = (short)g_draw->player->getLayerItemNo(p);
        m_dat.write(&no, 2);
    }

    m_dat.closeWrite();

    return TRUE;
}

//! レイヤのイメージのみ書き込み

BOOL CUndoDat::writeLayerOnlyImage(CLayerItem *pItem)
{
    if(!pItem) return FALSE;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    writeLayerImage(pItem->m_pimg);

    m_dat.closeWrite();

    return TRUE;
}

//! pItem とその下、のイメージを２つを書き込み

BOOL CUndoDat::writeLayerTwoImage(CLayerItem *pItem)
{
    if(!pItem) return FALSE;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    writeLayerImage(pItem->m_pimg);
    writeLayerImage((pItem->next())->m_pimg);

    m_dat.closeWrite();

    return TRUE;
}

//! 下レイヤへ結合時の書き込み

BOOL CUndoDat::writeLayerCombine(BOOL bUndo)
{
    CLayerItem *p;

    p = _getLayer(m_nVal[1], m_nVal[2]);
    if(!p) return FALSE;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    writeLayerInfo(p);
    writeLayerImage(p->m_pimg);

    if(bUndo)
    {
        p = p->next();

        writeLayerInfo(p);
        writeLayerImage(p->m_pimg);
    }

    m_dat.closeWrite();

    return TRUE;
}

//! レイヤ単体書き込み(フォルダの場合もあり)

BOOL CUndoDat::writeLayerSingle(CLayerItem *pItem)
{
    if(!pItem) return FALSE;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    writeLayerInfo(pItem);
    writeLayerImage(pItem->m_pimg);

    m_dat.closeWrite();

    return TRUE;
}

//! フォルダ対応、レイヤ書き込み（フォルダの場合は複数）

BOOL CUndoDat::writeLayerFolder(CLayerItem *pItem)
{
    short wno[2];
    int no[2];
    CLayerItem *p;

    if(!pItem) return FALSE;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    for(p = pItem; p; p = (CLayerItem *)p->nextTreeItem(pItem))
    {
        //番号

        g_draw->player->getLayerItemNo_parent(p, no);

        wno[0] = no[0];
        wno[1] = no[1];

        m_dat.write(wno, sizeof(short) * 2);

        //

        writeLayerInfo(p);
        writeLayerImage(p->m_pimg);
    }

    //終了

    wno[0] = -1;
    wno[1] = -1;
    m_dat.write(wno, sizeof(short) * 2);

    m_dat.closeWrite();

    return TRUE;
}

//! 全レイヤを書き込み

BOOL CUndoDat::writeLayerAll()
{
    short wno[2];
    int no[2];
    CLayerItem *p;

    if(!m_dat.alloc(-1)) return FALSE;

    if(!m_dat.openWrite()) return FALSE;

    for(p = g_draw->player->getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        //番号

        g_draw->player->getLayerItemNo_parent(p, no);

        wno[0] = no[0];
        wno[1] = no[1];

        m_dat.write(wno, sizeof(short) * 2);

        //

        writeLayerInfo(p);
        writeLayerImage(p->m_pimg);
    }

    //終了

    wno[0] = -1;
    wno[1] = -1;
    m_dat.write(wno, sizeof(short) * 2);

    m_dat.closeWrite();

    return TRUE;
}


//---------------------


//! レイヤ情報書き込み
/*
    [名前] WORD:長さ, UNICODE:文字列(NULL含む)
    [UNDOLAYERINFO] (btColType が 255 でフォルダ)
*/

void CUndoDat::writeLayerInfo(CLayerItem *pItem)
{
    UNDOLAYERINFO info;
    WORD len;

    //レイヤ名

    len = pItem->m_strName.getLen() + 1;

    m_dat.write(&len, 2);
    m_dat.write((LPUSTR)pItem->m_strName, len << 1);

    //情報

    info.btColType      = (pItem->isFolder())? 255: pItem->m_nColType;
    info.btBlendMode    = pItem->m_nBlendMode;
    info.btOpacity      = pItem->m_nOpacity;
    info.btAMaskType    = pItem->m_nAmaskType;
    info.dwCol          = pItem->m_dwCol;
    info.dwFlags        = pItem->m_dwFlags;

    m_dat.write(&info, sizeof(UNDOLAYERINFO));
}

//! レイヤイメージ書き込み
/*
    [TILEIMGINFO]
    [タイルイメージ]
        DWORD : 位置 (-1 で終了)
        WORD  : 圧縮サイズ
        タイルデータ
*/

void CUndoDat::writeLayerImage(CTileImage *pimg)
{
    TILEIMGINFO info;
    DWORD i,cnt;
    WORD tsize;
    int tilesize;
    void **pp;
    LPBYTE pWorkSrc,pWorkDst;

    if(!pimg) return;

    //イメージ情報

    pimg->getInfo(&info);

    m_dat.write(&info, sizeof(TILEIMGINFO));

    //タイル

    pp  = pimg->getTileBuf();
    cnt = info.nArrayW * info.nArrayH;
    tilesize = pimg->getTileSize();

    pWorkSrc = g_memWorkSrc;
    pWorkDst = g_memWorkCmp;

    for(i = 0; i < cnt; i++, pp++)
    {
        if(*pp)
        {
            pimg->getSaveTileBuf(pWorkSrc, *pp);

            tsize = compressUndo(pWorkDst, pWorkSrc, tilesize);

            m_dat.write(&i, 4);
            m_dat.write(&tsize, 2);
            m_dat.write(pWorkDst, tsize);
        }
    }

    //終了

    i = (DWORD)-1;
    m_dat.write(&i, 4);
}


//=============================
// 復元
//=============================


//! レイヤのイメージのみ復元

BOOL CUndoDat::restoreLayerOnlyImage(CLayerItem *pItem)
{
    BOOL ret;

    if(!m_dat.openRead()) return FALSE;

    ret = restoreImage(pItem);

    m_dat.closeRead();

    return ret;
}

//! pItem とその下、の２つのイメージ復元

BOOL CUndoDat::restoreLayerTwoImage(CLayerItem *pItem)
{
    BOOL ret = FALSE;

    if(!m_dat.openRead()) return FALSE;

    if(!restoreImage(pItem)) goto END;
    if(!restoreImage(pItem->next())) goto END;

    ret = TRUE;

END:
    m_dat.closeRead();

    return ret;
}

//! レイヤ結合復元

BOOL CUndoDat::restoreLayerCombine(BOOL bUndo,FLAGRECT *prcf)
{
    FLAGRECT rcf;
    CLayerItem *p;
    BOOL ret = FALSE;

    p = _getLayer(m_nVal[1], m_nVal[2]);
    if(!p) return FALSE;

    rcf.clear();

    //

    if(!m_dat.openRead()) return FALSE;

    if(bUndo)
    {
        //上のレイヤを復元、下のレイヤのイメージを復元

        if(!restoreLayer(m_nVal[1], m_nVal[2], &rcf)) goto END;
        if(!restoreLayerInfoAndImage(p, &rcf)) goto END;
    }
    else
    {
        //結合後のイメージを復元、上のレイヤを削除

        p->m_pimg->getExistImgRectPx(&rcf);

        if(!restoreLayerInfoAndImage(p->next(), &rcf)) goto END;

        _deleteLayer(p);
    }

    ret = TRUE;

END:
    m_dat.closeRead();

    //

    g_draw->player->calcViewItemCnt();

    *prcf = rcf;

    return ret;
}

//! レイヤ単体復元
/*!
    @param prcf イメージの範囲がセットされる
*/

BOOL CUndoDat::restoreLayerSingle(int pno,int no,FLAGRECT *prcf)
{
    BOOL ret;

    prcf->clear();

    //

    if(!m_dat.openRead()) return FALSE;

    ret = restoreLayer(pno, no, prcf);

    m_dat.closeRead();

    //

    g_draw->player->calcViewItemCnt();

    return ret;
}

//! 複数レイヤ復元

BOOL CUndoDat::restoreLayerMul(FLAGRECT *prcf)
{
    short no[2];
    BOOL ret = FALSE;

    prcf->clear();

    //

    if(!m_dat.openRead()) return FALSE;

    while(1)
    {
        //番号

        m_dat.read(no, sizeof(short) * 2);

        if(no[1] == -1) break;

        //

        if(!restoreLayer(no[0], no[1], prcf)) goto END;
    }

    ret = TRUE;

END:
    m_dat.closeRead();

    g_draw->player->calcViewItemCnt();

    return ret;
}


//----------------------


//! レイヤ一つを復元
/*
    CLayerList::calcViewItemCnt() を最後に行うこと。
*/

BOOL CUndoDat::restoreLayer(int pno,int no,FLAGRECT *prcf)
{
    UNDOLAYERINFO info;
    WORD len;
    AXString name;
    CLayerItem *pitem;
    FLAGRECT rcf;

    //---------- レイヤ情報

    //名前

    m_dat.read(&len, 2);

    name.resize(len);

    m_dat.read((LPUSTR)name, len << 1);
    name.recalcLen();

    //情報

    m_dat.read(&info, sizeof(UNDOLAYERINFO));

    //---------- 作成

    pitem = g_draw->player->addLayerFromNo(pno, no);
    if(!pitem) return FALSE;

    //--------- 情報をセット

    pitem->m_strName    = name;
    pitem->m_nColType   = (info.btColType == 255)? 0: info.btColType;
    pitem->m_nBlendMode = info.btBlendMode;
    pitem->m_nOpacity   = info.btOpacity;
    pitem->m_nAmaskType = info.btAMaskType;
    pitem->m_dwFlags    = info.dwFlags;

    //-------- イメージ

    if(info.btColType == 255) return TRUE;

    if(!restoreImage(pitem))
        //エラー時、削除
        g_draw->player->deleteLayer(pitem);
    else
    {
        pitem->setLayerCol(info.dwCol);

        //範囲追加

        pitem->m_pimg->getExistImgRectPx(&rcf);
        prcf->combine(rcf);
    }

    return TRUE;
}

//! 既存のレイヤに情報とイメージを復元

BOOL CUndoDat::restoreLayerInfoAndImage(CLayerItem *pitem,FLAGRECT *prcf)
{
    UNDOLAYERINFO info;
    WORD len;
    AXString name;
    FLAGRECT rcf;

    //---------- レイヤ情報

    //名前

    m_dat.read(&len, 2);

    name.resize(len);

    m_dat.read((LPUSTR)name, len << 1);
    name.recalcLen();

    //情報

    m_dat.read(&info, sizeof(UNDOLAYERINFO));

    //--------- 情報をセット

    pitem->m_strName    = name;
    pitem->m_nColType   = info.btColType;
    pitem->m_nBlendMode = info.btBlendMode;
    pitem->m_nOpacity   = info.btOpacity;
    pitem->m_nAmaskType = info.btAMaskType;
    pitem->m_dwFlags    = info.dwFlags;

    //-------- イメージ

    if(!restoreImage(pitem)) return FALSE;

    //

    pitem->setLayerCol(info.dwCol);

    //範囲追加

    pitem->m_pimg->getExistImgRectPx(&rcf);
    prcf->combine(rcf);

    return TRUE;
}

//! イメージの復元

BOOL CUndoDat::restoreImage(CLayerItem *pItem)
{
    TILEIMGINFO info;
    DWORD pos;
    WORD tsize;
    int tilesize;
    CTileImage *pimg;
    void *pTile;
    LPBYTE pWorkSrc,pWorkDst;

    //イメージ情報

    m_dat.read(&info, sizeof(TILEIMGINFO));

    //イメージ確保 (イメージが存在する場合は再作成)

    if(!g_draw->player->allocImageUndo(pItem, info))
        return FALSE;

    //タイル

    pimg     = pItem->m_pimg;
    tilesize = pimg->getTileSize();

    pWorkSrc = g_memWorkSrc;
    pWorkDst = g_memWorkCmp;

    while(1)
    {
        m_dat.read(&pos, 4);
        if(pos == (DWORD)-1) break;

        m_dat.read(&tsize, 2);

        //タイル確保

        pTile = pimg->getTileAlloc(pos);
        if(!pTile) return FALSE;

        //タイルイメージ

        if(tsize == tilesize)
            m_dat.read(pWorkDst, tilesize);
        else
        {
            m_dat.read(pWorkSrc, tsize);

            uncompUndo((LPBYTE)pWorkDst, pWorkSrc, tilesize, tsize);
        }

        pimg->setSaveTileBuf(pTile, pWorkDst);
    }

    return TRUE;
}


//=============================
// アンドゥ用イメージ・64x64以内
//=============================


//! イメージ（少範囲）書き込み
/*
    bFirst : 最初のアンドゥ書き込み時
*/

BOOL CUndoDat::writeImageSmall(CTileImage *pimg,BOOL bFirst)
{
    int ix,iy,w,h,gpos,bpos,apos,size;
    RGBAFIX15 col;
    LPWORD pDst;

    w = m_nVal[3];
    h = m_nVal[4];

    //イメージをRGBAで取得

    pDst = g_memWorkSrc;

    gpos = w * h;
    bpos = gpos << 1;
    apos = bpos + gpos;

    for(iy = 0; iy < h; iy++)
    {
        for(ix = 0; ix < w; ix++, pDst++)
        {
            if(bFirst)
                pimg->getPixelUndoImg(&col, m_nVal[1] + ix, m_nVal[2] + iy);
            else
                pimg->getPixel(&col, m_nVal[1] + ix, m_nVal[2] + iy);

            *pDst      = col.r;
            pDst[gpos] = col.g;
            pDst[bpos] = col.b;
            pDst[apos] = col.a;
        }
    }

    //圧縮

    size = compressUndo(g_memWorkCmp, g_memWorkSrc, w * h * sizeof(RGBAFIX15));

    //書き込み

    if(!m_dat.alloc(sizeof(int) + size)) return FALSE;
    if(!m_dat.openWrite()) return FALSE;

    m_dat.write(&size, sizeof(int));
    m_dat.write(g_memWorkCmp, size);

    m_dat.closeWrite();

    return TRUE;
}

//! イメージ（少範囲）復元

BOOL CUndoDat::restoreImageSmall()
{
    CTileImage *pimg;
    int size,srcsize,w,h,ix,iy,gpos,bpos,apos;
    LPWORD pSrc;
    RGBAFIX15 col,colSrc;

    pimg = _getLayerImg(m_nVal[0]);
    if(!pimg) return FALSE;

    w = m_nVal[3];
    h = m_nVal[4];

    //読み込み・展開

    srcsize = w * h * sizeof(RGBAFIX15);

    if(!m_dat.openRead()) return FALSE;

    m_dat.read(&size, sizeof(int));

    if(size == srcsize)
        //無圧縮
        m_dat.read(g_memWorkSrc, srcsize);
    else
    {
        m_dat.read(g_memWorkCmp, size);
        uncompUndo(g_memWorkSrc, g_memWorkCmp, srcsize, size);
    }

    m_dat.closeRead();

    //イメージに点を打つ

    pSrc = g_memWorkSrc;

    gpos = w * h;
    bpos = gpos << 1;
    apos = bpos + gpos;

    for(iy = 0; iy < h; iy++)
    {
        for(ix = 0; ix < w; ix++, pSrc++)
        {
            /* R が 0xffff の場合は、変更なしの点。
               アンドゥイメージは、イメージに変更があったタイル単位で保存されているので、
               アンドゥの矩形範囲内であっても、アンドゥイメージ上に描画前のタイルイメージが保存されていない場合がある。
               そのため、アンドゥイメージ上でタイルが確保されていない部分は R = 0xffff としてある。
            */

            if(*pSrc == 0xffff) continue;

            col.r = *pSrc;
            col.g = pSrc[gpos];
            col.b = pSrc[bpos];
            col.a = pSrc[apos];

            pimg->getPixel(&colSrc, m_nVal[1] + ix, m_nVal[2] + iy);

            if(!(colSrc == col))
                pimg->setPixel_create(m_nVal[1] + ix, m_nVal[2] + iy, col);
        }
    }

    //範囲内の透明タイルを解放

    pimg->freeEmptyTileRect(m_nVal[1], m_nVal[2], w, h);

    return TRUE;
}


//=============================
// アンドゥ用イメージ・64x64以上
//=============================
/*
    アンドゥ書き込み時は、g_draw->pimgUndo のイメージを書き込む。

    [TILEIMGINFO] 描画前の画像情報
    [DWORD] 元が空でないタイル数
    [DWORD] 元が空のタイル数
    [タイル...]
        WORD x2 : 位置
        DWORD   : サイズ  (REDO時)最上位ビットONで元が空
        データ
*/


//! アンドゥイメージ書き込みサイズ＆タイル数計算 (アンドゥ時)
/*!
    @param pTileCnt  [0]通常タイル数 [1]UNDO空タイル数
    @return -1 でファイルに出力
*/

int CUndoDat::_getUndoImageSize(CTileImage *pimg,LPDWORD pTileCnt)
{
    void **ppTile;
    int cnt,tilesize,allsize;
    DWORD tcnt = 0,tempcnt = 0;
    LPBYTE pWork = g_memWorkSrc;

    allsize = sizeof(TILEIMGINFO) + 8;

    ppTile   = pimg->getTileBuf();
    cnt      = pimg->getArrayW() * pimg->getArrayH();
    tilesize = pimg->getTileSize();

    //

    for(; cnt > 0; cnt--, ppTile++)
    {
        if(*ppTile)
        {
            allsize += 8;

            if(*ppTile == (void *)CTileImage::TILEPT_EMPTY)
                tempcnt++;
            else
            {
                if(allsize < BUFMAXSIZE)
                {
                    pimg->getSaveTileBuf(pWork, *ppTile);

                    allsize += getCompressUndoSize(pWork, tilesize);
                }

                tcnt++;
            }
        }
    }

    pTileCnt[0] = tcnt;
    pTileCnt[1] = tempcnt;

    return (allsize >= BUFMAXSIZE)? -1: allsize;
}

//! アンドゥイメージ書き込み

BOOL CUndoDat::writeUndoImage(const TILEIMGINFO &info)
{
    CTileImage *pimg = g_draw->pimgUndo;
    int size,tilesize;
    DWORD tcnt[2],tsize;
    WORD x,y,w,h;
    void **ppTile;
    LPBYTE pWorkDst,pWorkSrc;

    //サイズ計算

    size = _getUndoImageSize(pimg, tcnt);

    //--------- 書き込み

    if(!m_dat.alloc(size)) return FALSE;
    if(!m_dat.openWrite()) return FALSE;

    //画像情報（描画前）

    m_dat.write(&info, sizeof(TILEIMGINFO));

    //タイル数

    m_dat.write(tcnt, 8);

    //タイル

    pWorkDst = g_memWorkCmp;
    pWorkSrc = g_memWorkSrc;

    ppTile   = pimg->getTileBuf();
    w        = pimg->getArrayW();
    h        = pimg->getArrayH();
    tilesize = pimg->getTileSize();

    for(y = 0; y < h; y++)
    {
        for(x = 0; x < w; x++, ppTile++)
        {
            if(!(*ppTile)) continue;

            m_dat.write(&x, 2);
            m_dat.write(&y, 2);

            if(*ppTile == (void *)CTileImage::TILEPT_EMPTY)
            {
                //ポインタが TILEPT_EMPTY は元データが空 -> サイズを0,最上位ビットをON

                tsize = 0x80000000;
                m_dat.write(&tsize, 4);
            }
            else
            {
                pimg->getSaveTileBuf(pWorkSrc, *ppTile);

                tsize = compressUndo(pWorkDst, pWorkSrc, tilesize);

                m_dat.write(&tsize, 4);
                m_dat.write(pWorkDst, tsize);
            }
        }
    }

    m_dat.closeWrite();

    return TRUE;
}

//! 反転書き込み時の書き込みサイズ取得 (this = src)
/*!
    @return -2 でエラー。-1 でファイルへ。
*/

int CUndoDat::_getUndoImageRevSize(CTileImage *pimg,BOOL bRedo)
{
    TILEIMGINFO info;
    int allsize,tilesize;
    DWORD tcnt[2],tsize,i,mx,my;
    WORD x,y;
    void *pTile;
    LPBYTE pWorkBuf = g_memWorkSrc;

    //

    if(!m_dat.openRead()) return -2;

    m_dat.read(&info, sizeof(TILEIMGINFO));
    m_dat.read(tcnt, 8);

    allsize  = sizeof(TILEIMGINFO) + 8;
    tilesize = pimg->getTileSize();
    mx       = (info.nOffX - pimg->getOffX()) >> 6;
    my       = (info.nOffY - pimg->getOffY()) >> 6;

    //

    for(i = tcnt[0] + tcnt[1]; i; i--)
    {
        m_dat.read(&x, 2);
        m_dat.read(&y, 2);
        m_dat.read(&tsize, 4);
        m_dat.seekRead(tsize & 0xffff);

        allsize += 8;

        //タイル

        if(bRedo)
            //UNDO -> REDO
            pTile = pimg->getTile(x, y);
        else
        {
            //REDO -> UNDO

            if(tsize & 0x80000000)
                pTile = NULL;
            else
                pTile = pimg->getTile(x + mx, y + my);
        }

        //圧縮サイズ

        if(pTile)
        {
            pimg->getSaveTileBuf(pWorkBuf, pTile);

            allsize += getCompressUndoSize(pWorkBuf, tilesize);
        }

        //指定サイズ超えたらファイルへ
        if(allsize >= BUFMAXSIZE) return -1;
    }

    m_dat.closeRead();

    return allsize;
}

//! アンドゥイメージ反転書き込み

BOOL CUndoDat::writeUndoImageRev(CUndoDat *pSrc,BOOL bRedo)
{
    CTileImage *pimg;
    TILEIMGINFO info;
    DWORD i,tcnt[2],tsize,size2;
    WORD x,y;
    int allsize,mx,my,tilesize;
    void *pTile;
    LPBYTE pWorkSrc,pWorkDst;
    BOOL ret = FALSE;

    //イメージ

    pimg = _getLayerImg(m_nVal[0]);
    if(!pimg) return FALSE;

    //サイズ計算

    allsize = pSrc->_getUndoImageRevSize(pimg, bRedo);
    if(allsize == -2) return FALSE;

    //------------- 書き込み

    if(!pSrc->m_dat.openRead()) return FALSE;

    pSrc->m_dat.read(&info, sizeof(TILEIMGINFO));
    pSrc->m_dat.read(tcnt, 8);

    mx = (info.nOffX - pimg->getOffX()) >> 6;
    my = (info.nOffY - pimg->getOffY()) >> 6;

    //

    if(!m_dat.alloc(allsize)) goto END;
    if(!m_dat.openWrite()) goto END;

    pimg->getInfo(&info);

    m_dat.write(&info, sizeof(TILEIMGINFO));
    m_dat.write(tcnt, 8);

    //タイル

    pWorkSrc = g_memWorkSrc;
    pWorkDst = g_memWorkCmp;

    tilesize = pimg->getTileSize();

    for(i = tcnt[0] + tcnt[1]; i; i--)
    {
        pSrc->m_dat.read(&x, 2);
        pSrc->m_dat.read(&y, 2);
        pSrc->m_dat.read(&tsize, 4);
        pSrc->m_dat.seekRead(tsize & 0xffff);

        m_dat.write(&x, 2);
        m_dat.write(&y, 2);

        if(bRedo)
            //UNDO -> REDO (イメージはまだUNDOされていない)
            pTile = pimg->getTile(x, y);
        else
        {
            //REDO -> UNDO (元が空なら空データ)

            if(tsize & 0x80000000)
                pTile = NULL;
            else
                pTile = pimg->getTile(x + mx, y + my);
        }

        //サイズ&データ

        tsize &= 0x80000000;

        if(!pTile)
            m_dat.write(&tsize, 4);
        else
        {
            pimg->getSaveTileBuf(pWorkSrc, pTile);

            size2 = compressUndo(pWorkDst, pWorkSrc, tilesize);

            tsize |= size2;

            m_dat.write(&tsize, 4);
            m_dat.write(pWorkDst, size2);
        }
    }

    m_dat.closeWrite();

    ret = TRUE;

END:
    pSrc->m_dat.closeRead();

    return ret;
}

//! アンドゥイメージから復元

BOOL CUndoDat::_run_undoImage(BOOL bRedo)
{
    CTileImage *pimg;
    TILEIMGINFO info;
    DWORD i,tcnt[2],tsize,tilesize;
    WORD x,y;
    LPBYTE pWorkRead,pWorkDst;
    void *pTile;
    BOOL ret = FALSE;

    //イメージ

    pimg = _getLayerImg(m_nVal[0]);
    if(!pimg) return FALSE;

    //-------------

    if(!m_dat.openRead()) return FALSE;

    //画像情報

    m_dat.read(&info, sizeof(TILEIMGINFO));

    //タイル数

    m_dat.read(tcnt, 8);

    //配列リサイズ（リドゥ時は先に行う）

    if(bRedo)
    {
        if(!pimg->resizeTileArray(info)) goto END;
    }

    //イメージ

    pWorkRead = g_memWorkSrc;
    pWorkDst  = g_memWorkCmp;

    tilesize = pimg->getTileSize();

    for(i = tcnt[0] + tcnt[1]; i; i--)
    {
        m_dat.read(&x, 2);
        m_dat.read(&y, 2);
        m_dat.read(&tsize, 4);

        tsize &= 0xffff;

        if(tsize == 0)
            pimg->freeTilePos(x, y);
        else
        {
            pTile = pimg->getTileAlloc(x, y);
            if(!pTile) goto END;

            if(tsize == tilesize)
                m_dat.read(pWorkDst, tsize);
            else
            {
                m_dat.read(pWorkRead, tsize);

                uncompUndo(pWorkDst, pWorkRead, tilesize, tsize);
            }

            pimg->setSaveTileBuf(pTile, pWorkDst);
        }
    }

    //配列リサイズ（アンドゥ時は読み込み後に）

    if(!bRedo)
    {
        if(!pimg->resizeTileArray(info)) goto END;
    }

    ret = TRUE;

END:
    m_dat.closeRead();

    return ret;
}
