/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CResizeCanvas - キャンバス拡大縮小処理

    CImageRGB16 のイメージを拡大縮小。

    中間ファイルに垂直リサイズしたイメージを出力 ->
     CImageRGB16 のサイズを変更 ->
      中間ファイルからイメージを読み込んで水平リサイズし、CImageRGB16 にセット。
*/


#include <stdlib.h>
#include <math.h>

#include "CResizeCanvas.h"

#include "CImageRGB16.h"
#include "CProgressDlg.h"
#include "CApp.h"

#include "AXFile.h"
#include "AXZlib.h"
#include "AXUtilFile.h"


//--------------------

enum
{
    TYPE_NEAREST,
    TYPE_MITCHELL,
    TYPE_LAGRANGE,
    TYPE_LANCZOS2,
    TYPE_LANCZOS3
};

//--------------------


CResizeCanvas::CResizeCanvas()
{
    m_pWeight = NULL;
    m_pIndex  = NULL;
}

CResizeCanvas::~CResizeCanvas()
{
    _freeParam();
}

//! 実行

BOOL CResizeCanvas::run(CImageRGB16 *pimg,const AXSize &sizeOld,const AXSize &sizeNew,
    int type,CProgressDlg *pProgDlg)
{
    AXMem memBuf;
    AXString str;

    m_pimg      = pimg;
    m_pProgDlg  = pProgDlg;
    m_szOld     = sizeOld;
    m_szNew     = sizeNew;
    m_nType	    = type;

    m_pProgDlg->setProgMax(100);

    //Y1列分バッファ

    if(!memBuf.alloc(m_szOld.w * sizeof(RGBFIX15)))
        return FALSE;

    //垂直リサイズ

    if(!_resizeV(memBuf)) return FALSE;

    _freeParam();

    //イメージ、新サイズで作成

    if(!pimg->create(m_szNew.w, m_szNew.h))
    {
        pimg->create(m_szOld.w, m_szOld.h);
        return FALSE;
    }

    //水平リサイズ

    if(!_resizeH(memBuf))
    {
        pimg->create(m_szOld.w, m_szOld.h);
        return FALSE;
    }

    _freeParam();

    //作業用ファイル削除

    str = CAPP->getTmpDir();
    str.path_add("resize");

    AXDeleteFile(str);

    return TRUE;
}

//! 垂直リサイズ

BOOL CResizeCanvas::_resizeV(void *pLineBuf)
{
    AXFile file;
    AXZlib zlib;
    AXString str;

    //作業用ファイル

    str = CAPP->getTmpDir();
    str.path_add("resize");

    if(!file.openWrite(str)) return FALSE;

    //zlib

    if(!zlib.initEncFile(&file, 6)) return FALSE;

    //oldw x newh のイメージを出力

    m_pProgDlg->beginProgSub(50, m_szNew.h);

    if(m_nType == TYPE_NEAREST)
        _resizeV_nearest(&zlib);
    else
    {
        if(!_setParam(m_szOld.h, m_szNew.h)) return FALSE;

        _resizeV_etc(&zlib, pLineBuf);
    }

    //

    zlib.endEncFile();
    file.close();

    return TRUE;
}

//! 水平リサイズ

BOOL CResizeCanvas::_resizeH(void *pLineBuf)
{
    AXFile file;
    AXZlib zlib;
    AXString str;
    BOOL ret;

    //作業用ファイル

    str = CAPP->getTmpDir();
    str.path_add("resize");

    if(!file.openRead(str)) return FALSE;

    //zlib

    if(!zlib.initDecFile(&file, file.getSize()))
        return FALSE;

    //

    m_pProgDlg->beginProgSub(50, m_szNew.w);

    if(m_nType == TYPE_NEAREST)
        ret =_resizeH_nearest(&zlib, pLineBuf);
    else
    {
        if(!_setParam(m_szOld.w, m_szNew.w)) return FALSE;

        ret = _resizeH_etc(&zlib, pLineBuf);
    }

    //

    zlib.end();
    file.close();

    return ret;
}


//==================================
// ニアレストネイバー
//==================================


//! 垂直リサイズ

void CResizeCanvas::_resizeV_nearest(AXZlib *pzlib)
{
    RGBFIX15 *pSrc;
    int i,sy,linesize;

    pSrc     = m_pimg->getBuf();
    linesize = sizeof(RGBFIX15) * m_szOld.w;

    for(i = 0; i < m_szNew.h; i++)
    {
        sy = i * m_szOld.h / m_szNew.h;

        pzlib->putEncFile(pSrc + sy * m_szOld.w, linesize);

        m_pProgDlg->incProgSub();
    }
}

//! 水平リサイズ

BOOL CResizeCanvas::_resizeH_nearest(AXZlib *pzlib,void *pLineBuf)
{
    RGBFIX15 *pSrc,*pDst;
    int iy,ix,sx,linesize;

    pSrc     = (RGBFIX15 *)pLineBuf;
    pDst     = m_pimg->getBuf();
    linesize = sizeof(RGBFIX15) * m_szOld.w;

    for(iy = m_szNew.h; iy > 0; iy--)
    {
        //Y1列読み込み

        if(!pzlib->getDecFile(pSrc, linesize)) return FALSE;

        //

        for(ix = 0; ix < m_szNew.w; ix++)
        {
            sx = ix * m_szOld.w / m_szNew.w;

            *(pDst++) = *(pSrc + sx);
        }

        m_pProgDlg->incProgSub();
    }

    return TRUE;
}


//==================================
// 他タイプ
//==================================


//! 垂直リサイズ

void CResizeCanvas::_resizeV_etc(AXZlib *pzlib,void *pLineBuf)
{
    RGBFIX15 *ps,*pd,*p;
    int iy,ix,i,pitch,n,linesize;
    double *pWeight,w,c[3];
    short *pIndex;

    pitch    = m_szOld.w;
    linesize = sizeof(RGBFIX15) * m_szOld.w;
    pWeight  = m_pWeight;
    pIndex   = m_pIndex;

    //

    for(iy = m_szNew.h; iy > 0; iy--)
    {
        ps = m_pimg->getBuf();
        pd = (RGBFIX15 *)pLineBuf;

        for(ix = m_szOld.w; ix > 0; ix--, pd++, ps++)
        {
            //RGB

            c[0] = c[1] = c[2] = 0;

            for(i = 0; i < m_nTap; i++)
            {
                p = ps + pIndex[i] * pitch;
                w = pWeight[i];

                c[0] += p->c[0] * w;
                c[1] += p->c[1] * w;
                c[2] += p->c[2] * w;
            }

            //セット

            for(i = 0; i < 3; i++)
            {
                n = (int)(c[i] + 0.5);
                if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

                pd->c[i] = n;
            }
        }

        //

        pzlib->putEncFile(pLineBuf, linesize);

        pWeight += m_nTap;
        pIndex  += m_nTap;

        m_pProgDlg->incProgSub();
    }
}

//! 水平リサイズ

BOOL CResizeCanvas::_resizeH_etc(AXZlib *pzlib,void *pLineBuf)
{
    RGBFIX15 *ps,*pDst,*pSrc;
    int ix,iy,linesize,i,n;
    double *pWeight,w,c[3];
    short *pIndex;

    pSrc     = (RGBFIX15 *)pLineBuf;
    pDst     = m_pimg->getBuf();
    linesize = sizeof(RGBFIX15) * m_szOld.w;

    for(iy = m_szNew.h; iy > 0; iy--)
    {
        //Y1列読み込み

        if(!pzlib->getDecFile(pLineBuf, linesize))
            return FALSE;

        //

        pWeight = m_pWeight;
        pIndex  = m_pIndex;

        for(ix = m_szNew.w; ix > 0; ix--, pDst++)
        {
            //RGB

            c[0] = c[1] = c[2] = 0;

            for(i = 0; i < m_nTap; i++)
            {
                ps = pSrc + *(pIndex++);
                w  = *(pWeight++);

                c[0] += ps->c[0] * w;
                c[1] += ps->c[1] * w;
                c[2] += ps->c[2] * w;
            }

            //

            for(i = 0; i < 3; i++)
            {
                n = (int)(c[i] + 0.5);
                if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

                pDst->c[i] = n;
            }
        }

        m_pProgDlg->incProgSub();
    }

    return TRUE;
}


//=============================
// パラメータ
//=============================


//! パラメータ解放

void CResizeCanvas::_freeParam()
{
    //重み

    if(m_pWeight)
    {
        ::free(m_pWeight);
        m_pWeight = NULL;
    }

    //インデックス

    if(m_pIndex)
    {
        ::free(m_pIndex);
        m_pIndex = NULL;
    }
}

//! パラメータセット

BOOL CResizeCanvas::_setParam(int srcl,int dstl)
{
    if(dstl < srcl)
        //縮小
        return _setParam_down(srcl, dstl);
    else
        //拡大
        return _setParam_up(srcl, dstl);
}

//! パラメータメモリ確保
/*!
    m_nTap はあらかじめセットしておく。
*/

BOOL CResizeCanvas::_allocParam(int len)
{
    m_pWeight = (double *)::malloc(sizeof(double) * len * m_nTap);
    if(!m_pWeight) return FALSE;

    m_pIndex = (short *)::malloc(sizeof(short) * len * m_nTap);
    if(!m_pIndex) return FALSE;

    return TRUE;
}

//! 縮小パラメータセット

BOOL CResizeCanvas::_setParam_down(int srcl,int dstl)
{
    int i,j,pos,range;
    double *pWork,sum,scale;
    double *pWeight;
    short *pIndex;

    range = (m_nType == TYPE_LANCZOS3)? 3: 2;

    m_nTap = ((range * 2) * srcl + (dstl - 1)) / dstl;

    //メモリ確保

    if(!_allocParam(dstl)) return FALSE;

    //

    pWork = (double *)::malloc(sizeof(double) * m_nTap);
    if(!pWork) return FALSE;

    //--------------

    pWeight = m_pWeight;
    pIndex  = m_pIndex;
    scale   = (double)dstl / srcl;

    for(i = 0; i < dstl; i++)
    {
        pos = (int)::floor((i - range + 0.5) * srcl / dstl + 0.5);
        sum = 0;

        for(j = 0; j < m_nTap; j++, pIndex++)
        {
            if(pos < 0)
                *pIndex = 0;
            else if(pos >= srcl)
                *pIndex = srcl - 1;
            else
                *pIndex = pos;

            pWork[j] = _getWeight((pos + 0.5) * scale - (i + 0.5));

            sum += pWork[j];
            pos++;
        }

        for(j = 0; j < m_nTap; j++)
            *(pWeight++) = pWork[j] / sum;
    }

    ::free(pWork);

    return TRUE;
}

//! 拡大パラメータセット

BOOL CResizeCanvas::_setParam_up(int srcl,int dstl)
{
    int i,j,n;
    double *pWork,pos,sum;
    double *pWeight;
    short *pIndex;

    m_nTap = 6;

    //メモリ確保

    if(!_allocParam(dstl)) return FALSE;

    //

    pWork = (double *)::malloc(sizeof(double) * m_nTap);
    if(!pWork) return FALSE;

    //---------

    pWeight = m_pWeight;
    pIndex  = m_pIndex;

    for(i = 0; i < dstl; i++)
    {
        pos = (i + 0.5) * srcl / dstl;
        n   = (int)::floor(pos - 2.5);
        pos = n + 0.5 - pos;
        sum = 0;

        for(j = 0; j < m_nTap; j++, pIndex++)
        {
            if(n < 0)
                *pIndex = 0;
            else if(n >= srcl)
                *pIndex = srcl - 1;
            else
                *pIndex = n;

            pWork[j] = _getWeight(pos);

            sum += pWork[j];
            pos += 1;
            n++;
        }

        for(j = 0; j < m_nTap; j++)
            *(pWeight++) = pWork[j] / sum;
    }

    ::free(pWork);

    return TRUE;
}

//! 重み計算

double CResizeCanvas::_getWeight(double d)
{
    d = ::fabs(d);

    switch(m_nType)
    {
        //mitchell
        case TYPE_MITCHELL:
            if (d < 1.0)
				return 7.0 * d * d * d / 6.0 - 2.0 * d * d + 8.0 / 9.0;
			else if(d >= 2.0)
				return 0.0;
            else
				return 2.0 * d * d - 10.0 * d / 3.0 - 7.0 * d * d * d / 18.0 + 16.0 / 9.0;
            break;

		//Lagrange
		case TYPE_LAGRANGE:
			if (d < 1.0)
				return 0.5 * (d - 2) * (d + 1) * (d - 1);
			else if(d >= 2.0)
				return 0.0;
			else
				return -(d - 3) * (d - 2) * (d - 1) / 6;
			break;

        //Lanczos2
        case TYPE_LANCZOS2:
            if(d < 2.2204460492503131e-016)
                return 1.0;
            else if(d >= 2.0)
                return 0.0;
            else
            {
                d *= M_PI;
                return ::sin(d) * ::sin(d / 2.0) / (d * d / 2.0);
            }
            break;

        //Lanczos3
        default:
            if(d < 2.2204460492503131e-016)
                return 1.0;
            else if(d >= 3.0)
                return 0.0;
            else
            {
                d *= M_PI;
                return ::sin(d) * ::sin(d / 3.0) / (d * d / 3.0);
            }
            break;
    }

    return 0;
}
