/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CPaletteList, CPalette - パレットデータ
*/

#include <stdlib.h>
#include <string.h>

#include "CPalette.h"

#include "CImage32.h"
#include "AXString.h"
#include "AXFile.h"
#include "AXFileWriteBuf.h"


//*************************************
// CPaletteList : パレットリスト
//*************************************
/*
    - パレットのタブ数を増やす時は CColorWin::m_nPalScrPos の数も変更する。
    - データが更新された時のみ終了時に保存。
      （パレットデータを変更したら PALETTE->updateON() を実行すること）
*/


CPaletteList *CPaletteList::m_pSelf = NULL;


CPaletteList::CPaletteList()
{
    m_pSelf   = this;
    m_bUpdate = FALSE;
}

//! 確保されていないパレットをデフォルト作成（ファイル読み込み後）

void CPaletteList::setDefault()
{
    int i;

    for(i = 0; i < CNT; i++)
    {
        if(!m_pal[i].getBuf())
        {
            m_pal[i].alloc(64);
            m_pal[i].clearWhite();
        }
    }
}

//! 全パレットをファイルから読み込み

void CPaletteList::loadFile(const AXString &filename)
{
    AXFile file;
    BYTE ver,cnt;
    WORD pcnt;
    int i;

    if(!file.openRead(filename)) return;

    //ヘッダ

    if(!file.readCompare("AZPLPLD")) return;

    //バージョン

    file.read(&ver, 1);
    if(ver != 0) return;

    //データ数

    file.read(&cnt, 1);

    if(cnt > CNT) cnt = CNT;

    //各パレット

    for(i = 0; i < cnt; i++)
    {
        //個数

        file.readWORD(&pcnt);

        if(pcnt > CPalette::MAXCNT) break;

        //確保

        if(!m_pal[i].alloc(pcnt)) break;

        //パレット

        file.read(m_pal[i].getBuf(), pcnt * sizeof(DWORD));
    }

    file.close();
}

//! 全パレットをファイルに保存

void CPaletteList::saveFile(const AXString &filename)
{
    AXFileWriteBuf file;
    int i;
    WORD pcnt;

    //データが更新されていなければそのまま

    if(!m_bUpdate) return;

    /* !エンディアンはシステムに依存 */

    if(!file.open(filename)) return;

    //ヘッダ
    file.putStr("AZPLPLD");
    //バージョン
    file.putBYTE(0);
    //データ数
    file.putBYTE(CNT);

    //各パレット

    for(i = 0; i < CNT; i++)
    {
        pcnt = (WORD)m_pal[i].getCnt();

        //個数

        file.put(&pcnt, 2);

        //パレット

        file.put(m_pal[i].getBuf(), pcnt * sizeof(DWORD));
    }

    file.close();
}



//*************************************
// CPalette : パレットデータクラス
//*************************************



CPalette::CPalette()
{
    m_pBuf = NULL;
    m_nCnt = 0;
}

CPalette::~CPalette()
{
    free();
}

//! 解放

void CPalette::free()
{
    if(m_pBuf)
    {
        ::free(m_pBuf);

        m_pBuf = NULL;
        m_nCnt = 0;
    }
}

//! バッファ確保

BOOL CPalette::alloc(int cnt)
{
    free();

    if(cnt < 1) cnt = 1;
    else if(cnt > MAXCNT) cnt = MAXCNT;

    m_pBuf = (LPDWORD)::malloc(cnt * sizeof(DWORD));
    if(!m_pBuf) return FALSE;

    m_nCnt = cnt;

    return TRUE;
}

//! 個数変更

void CPalette::resize(int cnt)
{
    int bkcnt = m_nCnt,i;
    LPDWORD p;

    if(!_resize(cnt)) return;

    //数が増えた時、増加分をクリア

    if(m_nCnt > bkcnt)
    {
        p = m_pBuf + bkcnt;

        for(i = m_nCnt - bkcnt; i > 0; i--)
            *(p++) = 0xffffff;
    }

    PALETTE->updateON();
}

//! バッファサイズ変更

BOOL CPalette::_resize(int cnt)
{
    LPDWORD pnew;

    if(!m_pBuf) return FALSE;

    //数調整

    if(cnt < 1) cnt = 1;
    else if(cnt > MAXCNT) cnt = MAXCNT;

    //

    pnew = (LPDWORD)::realloc(m_pBuf, sizeof(DWORD) * cnt);
    if(!pnew) return FALSE;

    m_pBuf = pnew;
    m_nCnt = cnt;

    return TRUE;
}

//! 白でクリア

void CPalette::clearWhite()
{
    LPDWORD p = m_pBuf;
    int i;

    for(i = m_nCnt; i > 0; i--)
        *(p++) = 0xffffff;

    PALETTE->updateON();
}

//! 色をセット

void CPalette::setCol(int no,DWORD col)
{
    m_pBuf[no] = col;

    PALETTE->updateON();
}

//! 色を挿入

void CPalette::insertCol(int no,DWORD col)
{
    int i;
    LPDWORD p;

    //リサイズ

    if(!_resize(m_nCnt + 1)) return;

    //ずらす

    p = m_pBuf + m_nCnt - 1;

    for(i = m_nCnt - no - 1; i > 0; i--, p--)
        *p = p[-1];

    //セット

    m_pBuf[no] = col;

    PALETTE->updateON();
}

//! 削除

void CPalette::deleteCol(int no)
{
    int i;
    LPDWORD p;

    //ずらす

    p = m_pBuf + no;

    for(i = m_nCnt - no - 1; i > 0; i--, p++)
        *p = p[1];

    //リサイズ

    _resize(m_nCnt - 1);

    PALETTE->updateON();
}

//! パレット移動

void CPalette::moveCol(int srcno,int dstno)
{
    DWORD col = m_pBuf[srcno];
    int i;
    LPDWORD p;

    if(srcno == dstno || dstno == srcno + 1) return;

    //

    p = m_pBuf + srcno;

    if(srcno < dstno)
    {
        for(i = dstno - srcno - 1; i > 0; i--, p++)
            *p = p[1];

        m_pBuf[dstno - 1] = col;
    }
    else
    {
        for(i = srcno - dstno; i > 0; i--, p--)
            *p = p[-1];

        m_pBuf[dstno] = col;
    }

    PALETTE->updateON();
}

//! ２点間グラデーション

void CPalette::gradient(int no1,int no2)
{
    DWORD col1,col2;
    int sr,sg,sb,er,eg,eb,i,r,g,b,len;
    LPDWORD p;

    if(no1 == no2) return;

    if(no1 > no2) i = no1, no1 = no2, no2 = i;

    col1 = m_pBuf[no1];
    col2 = m_pBuf[no2];

    sr = _GETR(col1);
    sg = _GETG(col1);
    sb = _GETB(col1);

    er = _GETR(col2);
    eg = _GETG(col2);
    eb = _GETB(col2);

    len = no2 - no1;
    p   = m_pBuf + no1;

    for(i = 0; i <= len; i++)
    {
        r = (er - sr) * i / len + sr;
        g = (eg - sg) * i / len + sg;
        b = (eb - sb) * i / len + sb;

        *(p++) = _RGB(r,g,b);
    }

    PALETTE->updateON();
}

//! 画像からパレット作成

BOOL CPalette::createFromImg(const AXString &filename)
{
    CImage32 img;
    LPDWORD pPal,pSrc,pp;
    int i,j,cnt,f;
    DWORD col;

    if(!img.loadImage(filename, NULL)) return FALSE;

    //最大数確保

    if(!alloc(MAXCNT)) return FALSE;

    //パレットセット

    pPal = m_pBuf;
    pSrc = img.getBuf();
    cnt  = 0;

    for(i = img.getWidth() * img.getHeight(); i > 0; i--)
    {
        col = *(pSrc++) & 0xffffff;

        //すでにパレットにあるか

        for(j = cnt, f = 0, pp = pPal; j; j--, pp++)
        {
            if(*pp == col) { f = 1; break; }
        }

        //パレット追加

        if(!f)
        {
            if(cnt == m_nCnt) break;

            pPal[cnt++] = col;
        }
    }

    //リサイズ

    _resize(cnt);

    PALETTE->updateON();

    return TRUE;
}


//==============================
// 読み込み
//==============================


//! パレットファイル読み込み(APL/ACO)

BOOL CPalette::loadFile(const AXString &filename)
{
    AXString ext;

    ext.path_ext(filename);

    if(ext.compareCase("apl") == 0)
        return loadAPL(filename);
    else if(ext.compareCase("aco") == 0)
        return loadACO(filename);
    else
        return FALSE;
}

//! APL 読み込み

BOOL CPalette::loadAPL(const AXString &filename)
{
    AXFile file;
    BYTE ver;
    WORD cnt;
    int i;
    LPDWORD pDst;

    if(!file.openRead(filename)) return FALSE;

    file.setEndian(AXFile::ENDIAN_LITTLE);

    //ヘッダ
    if(!file.readCompare("AZPPAL")) return FALSE;

    //バージョン
    file.read(&ver, 1);
    if(ver != 0) return FALSE;

    //個数
    file.readWORD(&cnt);

    //確保
    if(!alloc(cnt)) return FALSE;

    //パレット

    pDst = m_pBuf;

    for(i = m_nCnt; i > 0; i--)
        file.readDWORD(pDst++);

    file.close();

    return TRUE;
}

//! ACO 読み込み

BOOL CPalette::loadACO(const AXString &filename)
{
    AXFile file;
    WORD ver,cnt,len;
    int i;
    LPDWORD pDst;
    BYTE dat[10];

    if(!file.openRead(filename)) return FALSE;

    file.setEndian(AXFile::ENDIAN_BIG);

    //バージョン

    file.readWORD(&ver);
    if(ver != 1 && ver != 2) return FALSE;

    //個数

    file.readWORD(&cnt);

    //確保

    if(!alloc(cnt)) return FALSE;

    //パレット

    pDst = m_pBuf;

    for(i = m_nCnt; i > 0; i--)
    {
        /*
            color space (2)
            R,G,B,Z (2)
        */

        file.read(dat, 10);

        //色名(ver2)

        if(ver == 2)
        {
            file.seekCur(2);
            //文字数
            file.readWORD(&len);
            file.seekCur(len * 2);
        }

        //セット

        *(pDst++) = _RGB(dat[2],dat[4],dat[6]);
    }

    file.close();

    return TRUE;
}


//==============================
// 保存
//==============================


//! APL ファイルに保存

BOOL CPalette::saveAPL(const AXString &filename)
{
    AXFile file;
    int i;
    LPDWORD p;

    if(!file.openWrite(filename)) return FALSE;

    file.setEndian(AXFile::ENDIAN_LITTLE);

    //ヘッダ
    file.writeStr("AZPPAL");
    //バージョン
    file.writeBYTE(0);
    //個数
    file.writeWORD(m_nCnt);

    //パレット

    p = m_pBuf;

    for(i = m_nCnt; i > 0; i--)
        file.writeDWORD(*(p++));

    file.close();

    return TRUE;
}

//! ACO ファイルに保存

BOOL CPalette::saveACO(const AXString &filename)
{
    AXFile file;
    int i;
    LPDWORD p;
    DWORD col;
    BYTE dat[10];

    ::memset(dat, 0, 10);

    if(!file.openWrite(filename)) return FALSE;

    file.setEndian(AXFile::ENDIAN_BIG);

    //バージョン
    file.writeWORD(1);
    //個数
    file.writeWORD(m_nCnt);

    //パレット

    p = m_pBuf;

    for(i = m_nCnt; i > 0; i--)
    {
        col = *(p++);

        dat[2] = _GETR(col);
        dat[4] = _GETG(col);
        dat[6] = _GETB(col);

        file.write(dat, 10);
    }

    file.close();

    return TRUE;
}
