/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CLayerList - レイヤデータのリスト

    - 総数が変わった場合 m_nCnt を変更する。
    - レイヤウィンドウでの表示数が変わった場合、calcViewItemCnt() で m_nViewCnt を変更する。
*/


#include "CLayerList.h"

#include "CLayerItem.h"
#include "CTileImageRGBA.h"
#include "CTileImageGray.h"
#include "CTileImageA16.h"
#include "CTileImageA1.h"
#include "CProgressDlg.h"

#include "AXFile.h"
#include "AXZlib.h"
#include "AXMem.h"
#include "AXUtil.h"



CLayerList::CLayerList()
{
    m_nCnt = m_nViewCnt = 0;
}

//! すべて削除

void CLayerList::clearItem()
{
    AXTree::deleteAll();

    m_nCnt = m_nViewCnt = 0;
}

//! アイテム確保

CLayerItem *CLayerList::_allocItem()
{
    if(m_nCnt >= 400)
        return NULL;
    else
        return new CLayerItem;
}

//! イメージ確保

CTileImage *CLayerList::_allocImg(int coltype)
{
    switch(coltype)
    {
        case CLayerItem::COLTYPE_RGBA: return new CTileImageRGBA;
        case CLayerItem::COLTYPE_GRAY: return new CTileImageGray;
        case CLayerItem::COLTYPE_A16:  return new CTileImageA16;
        case CLayerItem::COLTYPE_A1:   return new CTileImageA1;
    }

    return NULL;
}


//===========================
// レイヤ追加
//===========================


//! レイヤ追加（イメージ作成しない）
/*
    pIns : NULL でトップの最後に追加。フォルダならフォルダの一番上、通常レイヤならその上。

    親フォルダが非展開の場合、展開させる。
*/

CLayerItem *CLayerList::addLayer(CLayerItem *pIns)
{
    CLayerItem *p = _allocItem();

    if(p)
    {
        //リストに追加

        if(!pIns)
            AXTree::add(NULL, p);
        else if(pIns->isFolder())
        {
            //フォルダの子の一番先頭
            AXTree::add(pIns, p);
            AXTree::move(p, pIns->m_pFirst);

            pIns->m_dwFlags |= CLayerItem::FLAG_EXPAND;
        }
        else
            AXTree::insert(pIns, p);

        //

        m_nCnt++;
        calcViewItemCnt();
    }

    return p;
}

//! 通常レイヤ追加（イメージ作成する）
/*
    pinfo : NULL で w,h を使う。
*/

CLayerItem *CLayerList::addLayer(CLayerItem *pIns,int coltype,const TILEIMGINFO *pinfo,int w,int h)
{
    CLayerItem *p;
    CTileImage *pimg;
    BOOL ret;

    //イメージ

    pimg = _allocImg(coltype);

    if(pinfo)
        ret = pimg->create(*pinfo);
    else
        ret = pimg->create(w, h);

    if(!ret) goto ERR;

    //レイヤ追加

    p = addLayer(pIns);
    if(!p) goto ERR;

    p->replaceImg(pimg);

    return p;

ERR:
    delete pimg;
    return NULL;
}

//! 番号指定でレイヤ追加
/*!
    calcViewCnt() を最後に行うこと。
*/

CLayerItem *CLayerList::addLayerFromNo(int pno,int no)
{
    CLayerItem *p,*pPos[2];

    //追加

    p = _allocItem();

    getLayerItemFromNo_parent(pPos, pno, no);

    if(pPos[1])
        AXTree::insert(pPos[1], p);
    else
        AXTree::add(pPos[0], p);

    //

    m_nCnt++;

    return p;
}

//! キャンバス新規作成時のデフォルトレイヤ作成

CLayerItem *CLayerList::addNewCanvasLayer(int w,int h)
{
    CLayerItem *p;

    p = addLayer(NULL, CLayerItem::COLTYPE_RGBA, NULL, w, h);

    if(p)
        p->m_strName = "layer0";

    return p;
}


//============================


//! 通常レイヤの複製を作成（フォルダは不可）

CLayerItem *CLayerList::copyLayer(CLayerItem *pItem)
{
    CLayerItem *p;
    CTileImage *pimg;

    //イメージ複製

    pimg = _allocImg(pItem->m_nColType);

    if(!pimg->copy(*(pItem->m_pimg))) goto ERR;

    //レイヤ追加

    p = addLayer(pItem);
    if(!p) goto ERR;

    p->replaceImg(pimg);
    p->copyInfo(*pItem);

    return p;

ERR:
    delete pimg;
    return NULL;
}

//! レイヤ削除

void CLayerList::deleteLayer(CLayerItem *p)
{
    //削除

    AXTree::deleteItem(p);

    //

    m_nCnt = AXTree::getItemCnt();

    calcViewItemCnt();
}

//! イメージを確保(アンドゥ用)

BOOL CLayerList::allocImageUndo(CLayerItem *p,const TILEIMGINFO &info)
{
    CTileImage *pimg = _allocImg(p->m_nColType);;

    if(!pimg->create(info)) return FALSE;

    p->replaceImg(pimg);

    return TRUE;
}


//===============================
//
//===============================


//! 一番最後の通常レイヤ取得

CLayerItem *CLayerList::getBottomNormalLayer()
{
    CLayerItem *p = (CLayerItem *)getTreeLastItem();

    for(; p && p->isFolder(); p = (CLayerItem *)p->prevTreeItem());

    return p;
}

//! 合成時用、一番下の表示イメージアイテム取得

CLayerItem *CLayerList::getBottomVisibleImg()
{
    if(m_pBottom)
        return getBottomItem()->prevThisVisibleImg();
    else
        return NULL;
}

//! 表示イメージ上で、指定位置のピクセルに一番最初に点があるレイヤを取得

CLayerItem *CLayerList::getTopPixLayer(int x,int y)
{
    CLayerItem *p,*pLast = NULL;

    //後ろから検索して、一番最後のアイテム

    for(p = getBottomVisibleImg(); p; p = p->prevVisibleImg())
    {
        if(!p->m_pimg->isPixelTransparent(x, y))
            pLast = p;
    }

    return pLast;
}

//! レイヤ一覧に表示されるアイテム数計算

void CLayerList::calcViewItemCnt()
{
    CLayerItem *p;

    m_nViewCnt = 0;

    for(p = getTopItem(); p; p = p->nextExpand(), m_nViewCnt++);
}

//! 通常レイヤの数取得

int CLayerList::getNormalLayerCnt()
{
    CLayerItem *p;
    int cnt = 0;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isNormal()) cnt++;
    }

    return cnt;
}


//=====================


//! アイテムのツリー上における位置番号取得

int CLayerList::getLayerItemNo(CLayerItem *pItem)
{
    CLayerItem *p;
    int no = 0;

    if(!pItem) return -1;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem(), no++)
    {
        if(p == pItem) return no;
    }

    return -1;
}

//! 位置番号からアイテム取得

CLayerItem *CLayerList::getLayerItemFromNo(int no)
{
    CLayerItem *p;
    int cnt = 0;

    if(no < 0) return NULL;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem(), cnt++)
    {
        if(no == cnt) return p;
    }

    return NULL;
}

//! 親の位置番号と、フォルダ内での位置番号取得

void CLayerList::getLayerItemNo_parent(CLayerItem *pItem,int *pdst)
{
    CLayerItem *p;
    int no = 0;

    for(p = pItem->prev(); p; p = p->prev(), no++);

    pdst[0] = getLayerItemNo(pItem->parent());
    pdst[1] = no;
}

//! 親の番号と位置情報からアイテム取得

void CLayerList::getLayerItemFromNo_parent(CLayerItem **ppDst,int parentno,int no)
{
    CLayerItem *pParent,*p;
    int cnt;

    pParent = getLayerItemFromNo(parentno);
    ppDst[0] = pParent;

    if(pParent)
        p = pParent->first();
    else
        p = getTopItem();

    for(cnt = 0; p && cnt != no; p = p->next(), cnt++);

    ppDst[1] = p;
}


//====================


//! 全レイヤのフラグをOFF

void CLayerList::flagOffAll(DWORD dwMask)
{
    CLayerItem *p;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
        p->m_dwFlags &= dwMask;
}

//! 塗りつぶし判定元のイメージ取得

CTileImage *CLayerList::getPaintRefImg()
{
    CLayerItem *p;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isPaint() && p->m_pimg)
            return p->m_pimg;
    }

    return NULL;
}

//! 移動ツール時の移動させるレイヤのリンクをセット
/*
    type : [0]カレント [1]つかんだレイヤ [2]すべてのレイヤ
    pItem: カレントまたはつかんだレイヤ
    * ロックされているレイヤは除く。
*/

CLayerItem *CLayerList::setLinkMove(int type,BOOL bDisableLink,CLayerItem *pItem)
{
    CLayerItem *p,*pFirst = NULL,*pBk;
    int flag,bLink;

    for(p = getTopItem(); p; )
    {
        bLink = (!bDisableLink && p->isLink());

        if(type == 2)
            flag = TRUE;
        else
            flag = (p == pItem || bLink);

        //フォルダの場合はフォルダ下すべて

        if(flag)
            p = p->_setLinkAndNext(&pFirst, &pBk, TRUE);
        else
            p = (CLayerItem *)p->nextTreeItem();
    }

    return pFirst;
}

//! チェックされているレイヤのリンクをセット

CLayerItem *CLayerList::setLinkCheck()
{
    CLayerItem *p,*pFirst = NULL,*pBk;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isNormal() && p->isCheck())
            p->_setLink(&pFirst, &pBk);
    }

    return pFirst;
}

//! リンクされているレイヤ数取得

int CLayerList::getLinkCnt(CLayerItem *pTop)
{
    CLayerItem *p;
    int cnt = 0;

    for(p = pTop; p; p = p->m_pLink, cnt++);

    return cnt;
}

//! リンクされているアイテムの表示イメージ範囲取得

void CLayerList::getLinkVisibleImgRect(CLayerItem *pTop,FLAGRECT *pDst)
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;

    rcf.clear();

    for(p = pTop; p; p = p->m_pLink)
    {
        if(p->getVisibleImgRect(&rcf2))
            rcf.combine(rcf2);
    }

    *pDst = rcf;
}

//! チェックフラグONのレイヤがあるか

BOOL CLayerList::isExistCheckImg()
{
    CLayerItem *p;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->m_pimg && p->isCheck())
            return TRUE;
    }

    return FALSE;
}

//! 複数レイヤ結合用のリンクセット
/*
    下位のレイヤから順に。
*/

CLayerItem *CLayerList::setLinkCombineSome(int type,CLayerItem *pCurrent)
{
    CLayerItem *p,*pBk,*pFirst=NULL;

    switch(type)
    {
        //フォルダ内の表示レイヤ
        case 0:
            for(p = (CLayerItem *)pCurrent->lastTreeItem(); p; p = (CLayerItem *)p->prevTreeItem(pCurrent))
            {
                if(p->m_pimg && p->isVisibleView())
                    p->_setLink(&pFirst, &pBk);
            }
            break;
        //すべての表示レイヤ
        case 1:
            for(p = getBottomVisibleImg(); p; p = p->prevVisibleImg())
                p->_setLink(&pFirst, &pBk);
            break;
        //チェックレイヤ（非表示も含む）
        case 2:
            for(p = (CLayerItem *)getTreeLastItem(); p; p = (CLayerItem *)p->prevTreeItem())
            {
                if(p->m_pimg && p->isCheck())
                    p->_setLink(&pFirst, &pBk);
            }
            break;
    }

    return pFirst;
}

//! レイヤマスクのイメージ取得
/*!
    @return カレントレイヤがマスクレイヤの場合は NULL
*/

CTileImage *CLayerList::getLayerMaskImg(CLayerItem *pCurrent)
{
    CLayerItem *p;

    if(pCurrent->isLayerMaskUnder())
    {
        //下レイヤをマスクに
        /*「下レイヤをマスクに」OFFのレイヤが出るまで下を検索。
           同フォルダのレイヤのみ。フォルダが来たら終了。 */

        for(p = pCurrent->next(); p && p->m_pimg; p = p->next())
        {
            if(!p->isLayerMaskUnder()) break;
        }
    }
    else
    {
        //マスクレイヤ検索

        for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
        {
            if(p->m_pimg && p->isLayerMask()) break;
        }
    }

    //マスクがカレントレイヤならなしに

    if(!p || p == pCurrent)
        return NULL;
    else
        return p->m_pimg;
}

//! キャンバスサイズ変更時

void CLayerList::resizeCanvas(int movx,int movy)
{
    CLayerItem *p;

    for(p = getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        if(p->isNormal())
            p->m_pimg->moveOffset(movx, movy);
    }
}

//! APLD から読み込んで追加

CLayerItem *CLayerList::addLayer_APLD(CLayerItem *pIns,const AXString &filename,CProgressDlg *pdlg)
{
    AXFile file;
    AXZlib zlib;
    BYTE ver,dat[8];
    CTileImage *pimg;
    CLayerItem *pItem;
    AXString strName;
    AXMem memBuf;

    //============== ファイルから読み込み（ビッグエンディアン）

    if(!file.openRead(filename)) return NULL;

    file.setEndian(AXFile::ENDIAN_BIG);

    //ヘッダ
    if(!file.readCompare("AZPLLYD")) return NULL;

    //バージョン
    file.read(&ver, 1);
    if(ver != 0) return NULL;

    //情報

    file.readStrLenAndUTF8(&strName);
    file.read(dat, 8);

    //----- イメージ

    pimg = _allocImg(dat[0]);

    if(!zlib.allocBuf(pimg->getTileSize())) goto ERR;
    if(!zlib.initDecBuf()) goto ERR;

    if(!memBuf.alloc(pimg->getTileSize())) goto ERR;

    //

    pdlg->setProgMax(30);

    if(!pimg->loadTiles(&file, &zlib, memBuf, pdlg, 30)) goto ERR;

    //

    file.close();

    //============== レイヤ追加

    pItem = addLayer(pIns);
    if(!pItem) goto ERR;

    pItem->m_strName    = strName;
    pItem->m_nOpacity   = dat[1];
    pItem->m_nBlendMode = dat[2];
    pItem->m_dwFlags    = (dat[3] & 1)? CLayerItem::FLAG_VISIBLE: 0;

    AXGetDWORDBE(dat + 4, &pItem->m_dwCol);

    pItem->replaceImg(pimg);
    pItem->setLayerCol(pItem->m_dwCol);

    return pItem;

ERR:
    delete pimg;
    return NULL;
}
