/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CLayerItem - レイヤの個々のデータ
*/

#include "CLayerItem.h"

#include "CTileImage.h"
#include "CProgressDlg.h"

#include "AXFileWriteBuf.h"
#include "AXZlib.h"
#include "AXMem.h"



CLayerItem::~CLayerItem()
{
    if(m_pimg) delete m_pimg;
}

CLayerItem::CLayerItem()
{
    m_pimg = NULL;

    m_nOpacity   = 128;
    m_nColType   = 0;
    m_nAmaskType = AMASK_OFF;
    m_nBlendMode = 0;
    m_dwFlags    = FLAG_VISIBLE;
    m_dwCol      = 0;
}

//! レイヤ色セット

void CLayerItem::setLayerCol(DWORD col)
{
    RGBFIX15 c;

    m_dwCol = col;

    //イメージにセット

    if(m_pimg)
    {
        c.set(col);
        m_pimg->setImageColor(c);
    }
}

//! イメージを置き換え

void CLayerItem::replaceImg(CTileImage *pNewImg)
{
    if(m_pimg) delete m_pimg;

    m_pimg = pNewImg;

    //カラータイプ

    m_nColType = pNewImg->getColType();
}

//! 情報コピー

void CLayerItem::copyInfo(const CLayerItem &src)
{
    m_strName       = src.m_strName;
    m_nOpacity      = src.m_nOpacity;
    m_nAmaskType    = src.m_nAmaskType;
    m_nBlendMode    = src.m_nBlendMode;
    m_dwFlags       = src.m_dwFlags & FLAG_VISIBLE;

    setLayerCol(src.m_dwCol);
}


//================================
// 前/次のアイテム取得
//================================


//! this も含む、前の表示イメージアイテム取得（合成時）

CLayerItem *CLayerItem::prevThisVisibleImg()
{
    CLayerItem *p = this;

    while(p)
    {
        if(!p->isVisibleFlag())
            //非表示
            p = (CLayerItem *)p->prevTreeItemStop();
        else
        {
            //表示

            if(!p->isFolder())  //通常レイヤなら終了
                break;
            else if(p->m_pLast) //フォルダで子がある場合、最後の子
                p = p->last();
            else                //フォルダで子がなければ、前のアイテム
                p = (CLayerItem *)p->prevTreeItemStop();
        }
    }

    return p;
}

//! 合成時用、前の表示イメージアイテム取得(thisは含まない)

CLayerItem *CLayerItem::prevVisibleImg()
{
    CLayerItem *p;

    p = (CLayerItem *)prevTreeItemStop();

    if(p) p = p->prevThisVisibleImg();

    return p;
}

//! 前の展開済みアイテム取得

CLayerItem *CLayerItem::prevExpand()
{
    CLayerItem *p;

    //前がなければ親へ

    if(!m_pPrev) return parent();

    //

    p = prev();

    while(p)
    {
        /* 子がない、または非展開フォルダの場合は終了。
           展開フォルダの場合は最後のアイテムで判定していく。 */

        if(!p->first() || !p->isExpand())
            break;
        else
            p = p->last();
    }

    return p;
}

//! 前の通常レイヤ取得

CLayerItem *CLayerItem::prevNormal()
{
    CLayerItem *p = (CLayerItem *)prevTreeItem();

    for(; p && p->isFolder(); p = (CLayerItem *)p->prevTreeItem());

    return p;
}

//! 次の展開済みアイテム取得（レイヤ一覧表示用）

CLayerItem *CLayerItem::nextExpand()
{
    if(m_pFirst && isExpand())
        //子があり展開されている場合は子へ
        return first();
    else
        //次へ
        return (CLayerItem *)nextTreeItemPass();
}

//! 次の表示イメージアイテム取得
/*!
    @param pParent この親の下位のみ取得
*/

CLayerItem *CLayerItem::nextVisibleImg(CLayerItem *pParent)
{
    CLayerItem *p = this;

    while(p)
    {
        if(p->isFolder())
        {
            //フォルダ

            if(p->m_pFirst && p->isVisibleFlag())
                p = p->first();
            else
                //非表示 or 子がない
                p = (CLayerItem *)p->nextTreeItemPass(pParent);
        }
        else
        {
            //通常レイヤ

            if(p != this && p->isVisibleFlag())
                break;
            else
                p = (CLayerItem *)p->nextTreeItemPass(pParent);
        }
    }

    return p;
}


//================================
//
//================================


//! フォルダの深さ取得

int CLayerItem::getDepth()
{
    CLayerItem *p;
    int depth = 0;

    for(p = parent(); p; p = p->parent(), depth++);

    return depth;
}

//! 描画時のロックチェック
/*
    親フォルダがロックされていた場合もロック。
*/

BOOL CLayerItem::isLockDraw()
{
    CLayerItem *p;

    for(p = this; p; p = p->parent())
    {
        if(p->isLock()) return TRUE;
    }

    return FALSE;
}

//! キャンバス上で表示がONかどうか（親適用）

BOOL CLayerItem::isVisibleView()
{
    CLayerItem *p;

    for(p = this; p; p = p->parent())
    {
        if(!p->isVisibleFlag()) return FALSE;
    }

    return TRUE;
}

//! 合成時の不透明度取得（親適用）

int CLayerItem::getViewOpacity()
{
    CLayerItem *p;
    int n;

    n = m_nOpacity;

    for(p = parent(); p; p = p->parent())
    {
        if(p->m_nOpacity != 128)
            n = (n * p->m_nOpacity + 64) >> 7;
    }

    return n;
}

//! 表示イメージがある部分の範囲取得（イメージ範囲外も含む）
/*
    フォルダの場合、フォルダ下の全ての表示レイヤの範囲
*/

BOOL CLayerItem::getVisibleImgRect(FLAGRECT *pDst)
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;

    rcf.clear();

    if(!isFolder())
    {
        //通常レイヤ

        if(isVisibleFlag())
            m_pimg->getExistImgRectPx(&rcf);
    }
    else
    {
        //フォルダ

        p = nextVisibleImg(this);

        for(; p; p = p->nextVisibleImg(this))
        {
            if(p->m_pimg->getExistImgRectPx(&rcf2))
                rcf.combine(rcf2);
        }
    }

    *pDst = rcf;

    return rcf.flag;
}


//================================
//
//================================


//! このレイヤが表示されるようにフラグをON
/*
    フォルダの場合は下位もすべてON。親もONにする。
*/

void CLayerItem::visibleOn()
{
    CLayerItem *p;

    //親

    for(p = parent(); p; p = p->parent())
        p->m_dwFlags |= FLAG_VISIBLE;

    //フォルダのすべてのレイヤ

    for(p = this; p; p = (CLayerItem *)p->nextTreeItem(this))
        p->m_dwFlags |= FLAG_VISIBLE;
}

//! イメージを左右/上下反転
/*
    フォルダの場合は下位も。ロックレイヤは除く。空イメージの場合も除く。
*/

BOOL CLayerItem::reverseHorzVert(BOOL bHorz,FLAGRECT *prcf)
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;
    BOOL ret = FALSE;

    rcf.clear();

    for(p = this; p; p = (CLayerItem *)p->nextTreeItem(this))
    {
        if(p->m_pimg && !p->isLockDraw())
        {
            p->m_pimg->getExistImgRectPx(&rcf2);
            if(!rcf2.flag) continue;        //空イメージは除く
            rcf.combine(rcf2);

            //

            if(bHorz)
                p->m_pimg->reverseHorz();
            else
                p->m_pimg->reverseVert();

            //

            p->m_pimg->getExistImgRectPx(&rcf2);
            rcf.combine(rcf2);

            ret = TRUE;
        }
    }

    *prcf = rcf;

    return ret;
}

//! イメージを90度回転（フォルダの場合は下位も）

BOOL CLayerItem::rotate90(BOOL bLeft,FLAGRECT *prcf)
{
    CLayerItem *p;
    FLAGRECT rcf,rcf2;
    BOOL ret = FALSE;

    rcf.clear();

    for(p = this; p; p = (CLayerItem *)p->nextTreeItem(this))
    {
        if(p->m_pimg && !p->isLockDraw())
        {
            p->m_pimg->getExistImgRectPx(&rcf2);
            if(!rcf2.flag) continue;        //空イメージは除く
            rcf.combine(rcf2);

            //

            if(bLeft)
                p->m_pimg->rotateLeft();
            else
                p->m_pimg->rotateRight();

            //

            p->m_pimg->getExistImgRectPx(&rcf2);
            rcf.combine(rcf2);

            ret = TRUE;
        }
    }

    *prcf = rcf;

    return ret;
}


//================================
// サブ
//================================


//! this にリンクをセット

void CLayerItem::_setLink(CLayerItem **ppFirst,CLayerItem **ppBk)
{
    if(!(*ppFirst))
        *ppFirst = this;
    else
        (*ppBk)->m_pLink = this;

    m_pLink = NULL;
    *ppBk   = this;
}

//! リンクをセット＆次のアイテム取得
/*
    - フォルダの場合は、フォルダ下すべて。
    - bEnableLock が TRUE の場合、ロックされているレイヤは除く。
*/

CLayerItem *CLayerItem::_setLinkAndNext(CLayerItem **ppFirst,CLayerItem **ppBk,BOOL bEnableLock)
{
    CLayerItem *p;

    for(p = this; p; p = (CLayerItem *)p->nextTreeItem(this))
    {
        if(!p->isFolder())
        {
            if(bEnableLock && p->isLockDraw()) continue;

            if(!(*ppFirst))
                *ppFirst = p;
            else
                (*ppBk)->m_pLink = p;

            p->m_pLink = NULL;
            *ppBk = p;
        }
    }

    return (CLayerItem *)nextTreeItemPass();
}


//===============================
// ファイル
//===============================


//! APLD(AzPainterレイヤファイル)に出力
/*
    ビッグエンディアン。
*/

BOOL CLayerItem::saveAPLD(const AXString &filename,CProgressDlg *pdlg)
{
    AXFileWriteBuf file;
    AXZlib zlib;
    AXMem memBuf;

    if(!zlib.allocBuf(m_pimg->getTileSize())) return FALSE;
    if(!zlib.initEncBuf(6)) return FALSE;

    if(!memBuf.alloc(m_pimg->getTileSize())) return FALSE;

    //---------

    if(!file.open(filename, 8192)) return FALSE;

    //ヘッダ
    file.putStr("AZPLLYD");
    //バージョン
    file.putBYTE(0);

    //名前
    file.putStrLenAndUTF8(m_strName);
    //カラータイプ
    file.putBYTE(m_nColType);
    //不透明度
    file.putBYTE(m_nOpacity);
    //合成モード
    file.putBYTE(m_nBlendMode);
    //フラグ
    file.putBYTE(isVisibleFlag()? 1: 0);
    //レイヤ色
    file.putDWORDBE(m_dwCol);

    //タイル

    pdlg->setProgMax(20);
    m_pimg->saveTiles(&file, &zlib, memBuf, pdlg, 20);

    file.close();

    return TRUE;
}

