/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CBrushList - ブラシリストデータ

    - m_pEditItem が常に編集用の現在データ。
    - 連動保存時は選択ブラシのデータも一緒に変更する。
    - ブラシ/テクスチャ画像は、ブラシが選択された時に読み込む。
    - 描画用パラメータは、ブラシの値が変更されたら、連動して変更する。
*/


#include "CBrushList.h"

#include "CBrushItem.h"
#include "CBrushItemImgList.h"
#include "CBrushSizeList.h"

#include "AXFileWriteBuf.h"
#include "AXMem.h"
#include "AXBuf.h"


//--------------------------

CBrushItemImgList g_brushimglist;

//--------------------------


//******************************************
// CBrushGroupItem - グループアイテム
//******************************************


CBrushGroupItem::CBrushGroupItem()
{
    m_bExpand = FALSE;
}

//! グループの縦表示アイテム数取得

int CBrushGroupItem::getViewAreaHCnt(int xcnt)
{
    if(m_bExpand)
        return (m_list.getCnt() + xcnt) / xcnt;
    else
        return 1;
}

//! ブラシ新規追加

CBrushItem *CBrushGroupItem::addNew()
{
    CBrushItem *p;

    if(m_list.getCnt() >= MAXITEMCNT) return NULL;

    m_list.add(p = new CBrushItem);

    p->setDefault();
    p->pGroup = this;

    return p;
}


//******************************************
// CBrushList
//******************************************


CBrushList *CBrushList::m_pSelf = NULL;


CBrushList::~CBrushList()
{
    delete m_pEditItem;
}

CBrushList::CBrushList()
{
    m_pSelf = this;

    //

    m_pEditItem = new CBrushItem;
    m_pEditItem->setDefault();

    m_pSelItem = NULL;
    m_pRegItem = NULL;
    m_pToggleItem = NULL;

    setDrawParam(FALSE);
}



//===========================
// コマンド
//===========================


//! 新規グループ作成

CBrushGroupItem *CBrushList::addNewGroup()
{
    CBrushGroupItem *p;

    if(AXList::getCnt() >= GROUP_MAXCNT) return NULL;

    AXList::add(p = new CBrushGroupItem);

    return p;
}

//! グループ削除
/*!
    @return 選択アイテムが削除したグループに含まれていた
*/

BOOL CBrushList::deleteGroup(CBrushGroupItem *pGroup)
{
    BOOL ret = FALSE;

    //選択アイテムがこのグループ内の場合、選択なしに

    if(m_pSelItem && m_pSelItem->pGroup == pGroup)
    {
        changeSelItem(NULL);
        ret = TRUE;
    }

    //登録ブラシがこのグループ内の場合

    if(m_pRegItem && m_pRegItem->pGroup == pGroup)
        changeRegItem(NULL);

    //切替用ブラシがこのグループ内の場合

    if(m_pToggleItem && m_pToggleItem->pGroup == pGroup)
        m_pToggleItem = NULL;

    //削除

    AXList::deleteItem(pGroup);

    return ret;
}

//! グループを上下に移動

void CBrushList::moveGroup(CBrushGroupItem *pGroup,BOOL bUp)
{
    if(bUp)
    {
        if(pGroup->prev())
            AXList::move(pGroup, pGroup->prev());
    }
    else
    {
        if(pGroup->next())
            AXList::move(pGroup, (pGroup->next())->next());
    }
}

//! 新規ブラシ作成

CBrushItem *CBrushList::addNewBrush(CBrushGroupItem *pGroup)
{
    CBrushItem *p;

    p = pGroup->addNew();
    if(!p) return NULL;

    changeSelItem(p);

    return p;
}

//! ブラシ削除
/*!
    @return 選択アイテムが変更された
*/

BOOL CBrushList::deleteBrush(CBrushGroupItem *pGroup,CBrushItem *pItem)
{
    CBrushItem *pSel;

    //削除後の選択

    if(pItem == m_pSelItem)
    {
        pSel = pItem->next();
        if(!pSel) pSel = pItem->prev();
    }

    //削除

    pGroup->m_list.deleteItem(pItem);

    //登録ブラシの場合

    if(pItem == m_pRegItem) changeRegItem(NULL);

    //切替用ブラシの場合

    if(pItem == m_pToggleItem) m_pToggleItem = NULL;

    //選択変更

    if(pItem == m_pSelItem)
    {
        changeSelItem(pSel);
        return TRUE;
    }
    else
        return FALSE;
}

//! ブラシ移動

void CBrushList::moveBrush(CBrushGroupItem *pGroupSrc,CBrushItem *pItemSrc,
        CBrushGroupItem *pGroupDst,CBrushItem *pItemDst)
{
    if(pItemSrc == pItemDst) return;

    //

    pGroupSrc->m_list.remove(pItemSrc);

    if(pItemDst)
        pGroupDst->m_list.insert(pItemSrc, pItemDst);
    else
        pGroupDst->m_list.add(pItemSrc);

    //

    pItemSrc->pGroup = pGroupDst;
}

//! テキストフォーマットから貼り付け
/*!
    @param pItem NULL で新規貼り付け
*/

BOOL CBrushList::pasteBrush(CBrushGroupItem *pGroup,CBrushItem *pItem,const AXString &str)
{
    BOOL bNew = (pItem == NULL);

    //グループに新規追加

    if(bNew)
    {
        pItem = pGroup->addNew();
        if(!pItem) return FALSE;
    }

    //貼り付け

    if(!pItem->paste(str)) return FALSE;

    //

    if(bNew)
        //新規時、選択
        changeSelItem(pItem);
    else
    {
        //上書き貼り付け時

        if(pItem == m_pSelItem)
            _updateSelItem();
        else if(pItem == m_pRegItem)
            setDrawParam(TRUE);
    }

    return TRUE;
}


//=============================
//
//=============================


//! 登録ブラシの画像を取得
/*
    画像リストは最近使われた数個をストックしている。
    登録ブラシの画像がストックから外れて削除されないように、
    CBrushItemImgList::getImage() を呼び出した後は必ずこの関数を実行すること。
    これで、常に登録ブラシの画像がリストの先頭に来る。
*/

void CBrushList::_getRegImage()
{
    if(m_pRegItem)
    {
        m_prreg.pimgBrush   = g_brushimglist.getImage(CBrushItemImgList::TYPE_BRUSH, m_pRegItem->strBrushImg);
        m_prreg.pimgTexture = g_brushimglist.getImage(CBrushItemImgList::TYPE_TEXTURE, m_pRegItem->strTexImg);
    }
}

//! 選択アイテムのデータを作業用にセット

void CBrushList::_updateSelItem()
{
    //編集用にコピー

    m_pEditItem->copyFrom(m_pSelItem);

    //描画用パラメータ

    setDrawParam(FALSE);

    //登録ブラシ用の画像が削除されないように、再取得

    _getRegImage();
}

//! 選択アイテム変更

BOOL CBrushList::changeSelItem(CBrushItem *p)
{
    if(p == m_pSelItem)
        return FALSE;
    else
    {
        //現在の選択が登録ブラシなら、編集結果で描画用パラメータをセット

        if(m_pRegItem && m_pSelItem == m_pRegItem)
            setDrawParam(TRUE);

        //

        m_pSelItem = p;

        //

        if(m_pSelItem) _updateSelItem();

        return TRUE;
    }
}

//! 登録ブラシ変更

void CBrushList::changeRegItem(CBrushItem *p)
{
    if(p != m_pRegItem)
    {
        m_pRegItem = p;

        if(m_pRegItem)
            setDrawParam(TRUE);
    }
}

//! 手動保存実行

void CBrushList::manualSave()
{
    //データコピー

    if(m_pSelItem)
        m_pSelItem->copyFrom(m_pEditItem);
}

//! 登録ブラシ/直前のブラシ切替

BOOL CBrushList::toggleRegItem()
{
    BOOL ret = FALSE;

    if(!m_pRegItem) return FALSE;

    if(m_pSelItem != m_pRegItem)
    {
        //登録ブラシに切替

        m_pToggleItem = m_pSelItem;

        ret = changeSelItem(m_pRegItem);
    }
    else if(m_pToggleItem)
    {
        //登録ブラシ -> 直前のブラシ

        ret = changeSelItem(m_pToggleItem);
    }

    return ret;
}


//=============================
// 描画用パラメータ
//=============================


//! ブラシアイテムデータから描画用パラメータセット

void CBrushList::setDrawParam(BOOL bReg)
{
    BRUSHDRAWPARAM *p;
    CBrushItem *pitem;

    if(bReg)
        p = &m_prreg, pitem = m_pRegItem;
    else
        p = &m_predit, pitem = m_pEditItem;

    //-------

    p->dRadius      = pitem->wRadius * 0.1;
    p->dOpacity     = pitem->btOpacity * 0.01;
    p->dMinSize     = pitem->wMinSize * 0.001;
    p->dMinOpacity  = pitem->wMinOpacity * 0.001;
    p->dIntervalSrc = pitem->wInterval * 0.01;

    p->nHoseiType   = pitem->btHoseiType;
    p->nHoseiStr    = pitem->btHoseiStr + 1;

    p->dwFlags = 0;

    if(!pitem->isAntiAlias()) p->dwFlags |= BRUSHDF_NONEAA;
    if(pitem->isCurve()) p->dwFlags |= BRUSHDF_CURVE;

    //ランダム

    if(pitem->wRan_sizeMin != 1000)
    {
        p->dwFlags |= BRUSHDF_RND_SIZE;
        p->dRandSizeMin = pitem->wRan_sizeMin * 0.001;
    }

    if(pitem->wRan_posLen != 0)
    {
        p->dwFlags |= BRUSHDF_RND_POS;
        p->dRandPosLen = pitem->wRan_posLen * 0.02;
    }

    //水彩

    if(pitem->isWaterOn()) p->dwFlags |= BRUSHDF_WATER;

    p->dWater[0] = pitem->wWater[0] * 0.001;
    p->dWater[1] = pitem->wWater[1] * 0.001;
    p->dWater[2] = pitem->wWater[2] * 0.001;

    //ブラシ形状

    if(pitem->btHardness <= 50)
        p->dShapeHard = (-0.4 - 0.7) * (pitem->btHardness * 0.02) + 0.7;
    else
        p->dShapeHard = (-5 + 0.4) * ((pitem->btHardness - 50) * 0.02) - 0.4;

    p->nRoughVal  = (pitem->wRoughness == 0)? 0: pitem->wRoughness + 1;
    p->nRotAngle  = (pitem->wRotAngle << 9) / 360;      //0-511
    p->nRotRandom = (pitem->wRotRandom << 8) / 180;

    if(pitem->isBrushRotDir()) p->dwFlags |= BRUSHDF_ROT_DIR;

    //筆圧（100=変化なし の場合は pow 計算をしないように）

    if(pitem->wPressSize != 100)
    {
        p->dwFlags |= BRUSHDF_GAMMA_SIZE;
        p->dGammaSize = pitem->wPressSize * 0.01;
    }

    if(pitem->wPressOpacity != 100)
    {
        p->dwFlags |= BRUSHDF_GAMMA_OPACITY;
        p->dGammaOpacity = pitem->wPressOpacity * 0.01;
    }

    //画像取得

    p->pimgBrush   = g_brushimglist.getImage(CBrushItemImgList::TYPE_BRUSH, pitem->strBrushImg);
    p->pimgTexture = g_brushimglist.getImage(CBrushItemImgList::TYPE_TEXTURE, pitem->strTexImg);
}


//=============================
// 値の変更時
//=============================


//! 選択アイテムが連動保存か

BOOL CBrushList::isSelLinkSave()
{
    return (m_pSelItem && m_pSelItem->isLinkSave());
}

//! 手動保存反転

void CBrushList::changeValManualSave()
{
    m_pEditItem->btFlags ^= CBrushItem::FLAG_MANUALSAVE;

    if(m_pSelItem)
        m_pSelItem->btFlags ^= CBrushItem::FLAG_MANUALSAVE;
}

//! 半径
/*!
    @param bDirect 編集上からではなく、直接値をセットする場合
*/

void CBrushList::changeValRadius(int val,BOOL bDirect)
{
    WORD max[8] = {100, 500, 1000, 2000, 3000, 4000, 5000, 6000};
    int i;

    //直接変更で、指定値が最大値を超える場合

    if(bDirect && val > m_pEditItem->wRadiusCtlMax)
    {
        for(i = 0; i < 8; i++)
        {
            if(val <= max[i])
            {
                m_pEditItem->wRadiusCtlMax = max[i];
                if(isSelLinkSave()) m_pSelItem->wRadiusCtlMax = max[i];

                break;
            }
        }
    }

    //

    m_pEditItem->wRadius = val;
    if(isSelLinkSave()) m_pSelItem->wRadius = val;

    m_predit.dRadius = val * 0.1;
}

//! 半径最大値（半径値も変化する場合あり）

void CBrushList::changeValRadiusCtlMax(int radius,int max)
{
    m_pEditItem->wRadius = radius;
    m_pEditItem->wRadiusCtlMax = max;

    if(isSelLinkSave())
    {
        m_pSelItem->wRadius = radius;
        m_pSelItem->wRadiusCtlMax = max;
    }

    m_predit.dRadius = radius * 0.1;
}

//! 濃度

void CBrushList::changeValOpacity(int val)
{
    m_pEditItem->btOpacity = val;
    if(isSelLinkSave()) m_pSelItem->btOpacity = val;

    m_predit.dOpacity = val * 0.01;
}

//! 塗りタイプ

void CBrushList::changeValPixType(int type)
{
    m_pEditItem->btPixType = type;
    if(isSelLinkSave()) m_pSelItem->btPixType = type;
}

//! 補正タイプ

void CBrushList::changeValHoseiType(int type)
{
    m_pEditItem->btHoseiType = type;
    if(isSelLinkSave()) m_pSelItem->btHoseiType = type;

    m_predit.nHoseiType = type;
}

//! 補正強さ

void CBrushList::changeValHoseiStr(int val)
{
    m_pEditItem->btHoseiStr = val;
    if(isSelLinkSave()) m_pSelItem->btHoseiStr = val;

    m_predit.nHoseiStr = val + 1;
}

//! サイズ最小

void CBrushList::changeValMinSize(int val)
{
    m_pEditItem->wMinSize = val;
    if(isSelLinkSave()) m_pSelItem->wMinSize = val;

    m_predit.dMinSize = val * 0.001;
}

//! 濃度最小

void CBrushList::changeValMinOpacity(int val)
{
    m_pEditItem->wMinOpacity = val;
    if(isSelLinkSave()) m_pSelItem->wMinOpacity = val;

    m_predit.dMinOpacity = val * 0.001;
}

//! 間隔 (1.0=半径)

void CBrushList::changeValInterval(int val)
{
    m_pEditItem->wInterval = val;
    if(isSelLinkSave()) m_pSelItem->wInterval = val;

    m_predit.dIntervalSrc = val * 0.01;
}

//! ランダムサイズ

void CBrushList::changeValRandSize(int val)
{
    m_pEditItem->wRan_sizeMin = val;
    if(isSelLinkSave()) m_pSelItem->wRan_sizeMin = val;

    if(val == 1000)
        m_predit.dwFlags &= ~BRUSHDF_RND_SIZE;
    else
    {
        m_predit.dwFlags |= BRUSHDF_RND_SIZE;
        m_predit.dRandSizeMin = val * 0.001;
    }
}

//! ランダム位置

void CBrushList::changeValRandPos(int val)
{
    m_pEditItem->wRan_posLen = val;
    if(isSelLinkSave()) m_pSelItem->wRan_posLen = val;

    if(val == 0)
        m_predit.dwFlags &= ~BRUSHDF_RND_POS;
    else
    {
        m_predit.dwFlags |= BRUSHDF_RND_POS;
        m_predit.dRandPosLen = val * 0.02;
    }
}

//! 水彩ON反転

void CBrushList::changeValWaterOn()
{
    m_pEditItem->btFlags ^= CBrushItem::FLAG_WATER;
    if(isSelLinkSave()) m_pSelItem->btFlags ^= CBrushItem::FLAG_WATER;

    m_predit.dwFlags ^= BRUSHDF_WATER;
}

//! 水彩・描画色の補充

void CBrushList::changeValWater1(int val)
{
    m_pEditItem->wWater[0] = val;
    if(isSelLinkSave()) m_pSelItem->wWater[0] = val;

    m_predit.dWater[0] = val * 0.001;
}

//! 水彩・キャンバス色の割合

void CBrushList::changeValWater2(int val)
{
    m_pEditItem->wWater[1]= val;
    if(isSelLinkSave()) m_pSelItem->wWater[1] = val;

    m_predit.dWater[1] = val * 0.001;
}

//! 水彩・描画色の割合

void CBrushList::changeValWater3(int val)
{
    m_pEditItem->wWater[2] = val;
    if(isSelLinkSave()) m_pSelItem->wWater[2] = val;

    m_predit.dWater[2] = val * 0.001;
}

//! 形状・硬さ

void CBrushList::changeValBrushHardness(int val)
{
    m_pEditItem->btHardness = val;
    if(isSelLinkSave()) m_pSelItem->btHardness = val;

    if(val <= 50)
        m_predit.dShapeHard = (-0.4 - 0.7) * (val * 0.02) + 0.7;
    else
        m_predit.dShapeHard = (-5 + 0.4) * ((val - 50) * 0.02) - 0.4;
}

//! 形状・荒さ

void CBrushList::changeValBrushRoughness(int val)
{
    m_pEditItem->wRoughness = val;
    if(isSelLinkSave()) m_pSelItem->wRoughness = val;

    m_predit.nRoughVal = (val == 0)? 0: val + 1;
}

//! 画像回転タイプ

void CBrushList::changeValBrushRotType(int type)
{
    if(type)
        m_pEditItem->btFlags |= CBrushItem::FLAG_IMGROT_DIR;
    else
        m_pEditItem->btFlags &= ~CBrushItem::FLAG_IMGROT_DIR;

    if(isSelLinkSave()) m_pSelItem->btFlags = m_pEditItem->btFlags;

    //

    if(type)
        m_predit.dwFlags |= BRUSHDF_ROT_DIR;
    else
        m_predit.dwFlags &= ~BRUSHDF_ROT_DIR;
}

//! 画像回転角度

void CBrushList::changeValBrushRotAngle(int val)
{
    m_pEditItem->wRotAngle = val;
    if(isSelLinkSave()) m_pSelItem->wRotAngle = val;

    m_predit.nRotAngle = (val << 9) / 360;
}

//! 画像回転ランダム

void CBrushList::changeValBrushRotRandom(int val)
{
    m_pEditItem->wRotRandom = val;
    if(isSelLinkSave()) m_pSelItem->wRotRandom = val;

    m_predit.nRotRandom = (val << 8) / 180;
}

//! 筆圧・サイズ

void CBrushList::changeValPressSize(int val)
{
    m_pEditItem->wPressSize = val;
    if(isSelLinkSave()) m_pSelItem->wPressSize = val;

    if(val == 100)
        m_predit.dwFlags &= ~BRUSHDF_GAMMA_SIZE;
    else
    {
        m_predit.dwFlags |= BRUSHDF_GAMMA_SIZE;
        m_predit.dGammaSize = val * 0.01;
    }
}

//! 筆圧・濃度

void CBrushList::changeValPressOpacity(int val)
{
    m_pEditItem->wPressOpacity = val;
    if(isSelLinkSave()) m_pSelItem->wPressOpacity = val;

    if(val == 100)
        m_predit.dwFlags &= ~BRUSHDF_GAMMA_OPACITY;
    else
    {
        m_predit.dwFlags |= BRUSHDF_GAMMA_OPACITY;
        m_predit.dGammaOpacity = val * 0.01;
    }
}

//! アンチエイリアス

void CBrushList::changeValAntiAlias()
{
    m_pEditItem->btFlags ^= CBrushItem::FLAG_ANTIALIAS;
    if(isSelLinkSave()) m_pSelItem->btFlags ^= CBrushItem::FLAG_ANTIALIAS;

    m_predit.dwFlags ^= BRUSHDF_NONEAA;
}

//! 曲線

void CBrushList::changeValCurve()
{
    m_pEditItem->btFlags ^= CBrushItem::FLAG_CURVE;
    if(isSelLinkSave()) m_pSelItem->btFlags ^= CBrushItem::FLAG_CURVE;

    m_predit.dwFlags ^= BRUSHDF_CURVE;
}

//! ブラシ画像

BOOL CBrushList::changeValBrushImg(const AXString &str)
{
    m_pEditItem->strBrushImg = str;
    if(isSelLinkSave()) m_pSelItem->strBrushImg = str;

    m_predit.pimgBrush = g_brushimglist.getImage(CBrushItemImgList::TYPE_BRUSH, str);

    _getRegImage();

    return str.isEmpty();
}

//! テクスチャ

void CBrushList::changeValTexture(const AXString &str)
{
    m_pEditItem->strTexImg = str;
    if(isSelLinkSave()) m_pSelItem->strTexImg = str;

    m_predit.pimgTexture = g_brushimglist.getImage(CBrushItemImgList::TYPE_TEXTURE, str);

    _getRegImage();
}


//=============================
// ファイル
//=============================


#define DATF_BRUSH          0
#define DATF_GROUP_START    1
#define DATF_END            2
#define DATF_BRUSH_SEL      0x40
#define DATF_GROUP_EXPAND   0x40
#define DATF_BRUSH_REG      0x80


//! ファイルに保存 (BigEndian)

void CBrushList::saveFile(const AXString &filename)
{
    AXFileWriteBuf file;
    int i,cnt;
    BYTE flag;
    LPWORD pwd;
    CBrushGroupItem *pGroup;
    CBrushItem *p;

    if(!file.open(filename)) return;

    //ヘッダ
    file.putStr("AZPLBRD");

    //バージョン
    file.putBYTE(0);
    //サブバージョン
    file.putBYTE(0);

    //--------- サイズリスト

    cnt = BRUSHSIZELIST->getCnt();

    //個数

    file.putWORDBE(cnt);

    //データ

    pwd = BRUSHSIZELIST->getBuf();

    for(i = cnt; i > 0; i--, pwd++)
        file.putWORDBE(*pwd);

    //--------- ブラシデータ

    for(pGroup = getTopItem(); pGroup; pGroup = pGroup->next())
    {
        //グループ開始

        file.putBYTE(DATF_GROUP_START | (pGroup->m_bExpand? DATF_GROUP_EXPAND: 0));

        //ブラシ

        for(p = pGroup->getTopItem(); p; p = p->next())
        {
            //フラグ

            flag = DATF_BRUSH;
            if(p == m_pSelItem) flag |= DATF_BRUSH_SEL;
            if(p == m_pRegItem) flag |= DATF_BRUSH_REG;

            file.putBYTE(flag);

            //名前、ブラシ画像、テクスチャ画像

            file.putStrLenAndUTF8(p->strName);
            file.putStrLenAndUTF8(p->strBrushImg);
            file.putStrLenAndUTF8(p->strTexImg);

            //WORDデータ

            pwd = &p->wRadius;

            for(i = CBrushItem::WORDBASE_NUM; i; i--, pwd++)
                file.putWORDBE(*pwd);

            //BYTEデータ

            file.put(&p->btOpacity, CBrushItem::BYTEBASE_NUM);
        }
    }

    //データ終了

    file.putBYTE(DATF_END);

    file.close();
}

//! ファイルから読み込み

void CBrushList::loadFile(const AXString &filename)
{
    AXMem mem;
    AXBuf buf;
    BYTE ver,subver,flag;
    int cnt,i;
    WORD wd;
    CBrushGroupItem *pGroup;
    CBrushItem *p,*pSel,*pReg;
    LPWORD pwd;

    //読み込み

    if(!AXFile::readFileFull(filename, &mem)) return;

    buf.init(mem, mem.getSize(), AXBuf::ENDIAN_BIG);

    //ヘッダ

    if(!buf.getStrCompare("AZPLBRD")) return;

    if(!buf.getBYTE(&ver)) return;
    if(!buf.getBYTE(&subver)) return;

    if(ver != 0 || subver != 0) return;

    //--------- サイズリスト

    cnt = buf.getWORD();

    BRUSHSIZELIST->alloc(cnt);

    for(i = cnt; i > 0; i--)
    {
        if(!buf.getWORD(&wd)) return;

        BRUSHSIZELIST->addSize(wd);
    }

    //--------- ブラシデータ

    pGroup = NULL;
    pSel = pReg = NULL;

    while(1)
    {
        if(!buf.getBYTE(&flag)) break;

        if(flag == DATF_END) break;

        if((flag & 15) == DATF_GROUP_START)
        {
            //グループ開始

            pGroup = addNewGroup();
            if(!pGroup) break;

            pGroup->m_bExpand = ((flag & DATF_GROUP_EXPAND) != 0);
        }
        else if((flag & 15) == DATF_BRUSH)
        {
            //ブラシ

            if(!pGroup) break;

            p = pGroup->addNew();
            if(!p) break;

            if(flag & DATF_BRUSH_SEL) pSel = p;
            if(flag & DATF_BRUSH_REG) pReg = p;

            //

            if(!buf.getStrLenAndUTF8(&p->strName)) break;
            if(!buf.getStrLenAndUTF8(&p->strBrushImg)) break;
            if(!buf.getStrLenAndUTF8(&p->strTexImg)) break;

            //

            pwd = &p->wRadius;

            for(i = CBrushItem::WORDBASE_NUM; i; i--, pwd++)
            {
                if(!buf.getWORD(pwd)) goto END;
            }

            //

            if(!buf.getDat(&p->btOpacity, CBrushItem::BYTEBASE_NUM))
                break;
        }
        else
            break;
    }

    //-------------

END:
    changeSelItem(pSel);
    changeRegItem(pReg);
}

//! 読み込み後、デフォルトセット

void CBrushList::setDefault()
{
    //グループが一つもない場合

    if(m_nCnt == 0)
        AXList::add(new CBrushGroupItem);
}
