/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include "drawdat.h"
#include "global.h"

#include "CCanvasWin.h"
#include "CPrevWin.h"
#include "CLayerWin.h"

#include "CXImage.h"
#include "CImageRGB16.h"
#include "CTileImageA1.h"

#include "CLayerList.h"
#include "CLayerItem.h"

#include "CConfig.h"

#include "drawOpDef.h"
#include "draw_update.h"
#include "draw_calc.h"


namespace draw
{


//=============================
// まとめて更新
//=============================


//! 全体更新（イメージ＆キャンバス＆プレビューウィンドウ）

void updateAll()
{
    updateImage();
    updateCanvas(TRUE);

    PREVWIN->draw();
}

//! 全体＋レイヤ一覧更新

void updateAllAndLayer()
{
    updateAll();

    LAYERWIN->updateLayerAll();
}

//! レイヤの、イメージがある範囲だけ更新
/*
    フォルダの場合、フォルダ下の全ての表示レイヤの範囲
*/

void updateLayerItemRect(CLayerItem *pItem)
{
    FLAGRECT rcf;

    if(pItem->getVisibleImgRect(&rcf))
        updateRectAndPrev(rcf);
}

//! 範囲更新＋プレビューウィンドウ (FLAGRECT)

void updateRectAndPrev(const FLAGRECT &rcf)
{
    AXRectSize rcs;

    if(getImgRect(&rcs, rcf))
    {
        updateRect(rcs);
        PREVWIN->updateDrawRect(rcs);
    }
}

//! 範囲更新＋プレビューウィンドウ (AXRectSize)

void updateRectAndPrev(const AXRectSize &rcs)
{
    if(rcs.x != -1)
    {
        updateRect(rcs);
        PREVWIN->updateDrawRect(rcs);
    }
}

//! 範囲更新 (FLAGRECT)

void updateRect(const FLAGRECT &rcf)
{
    AXRectSize rcs;

    if(getImgRect(&rcs, rcf))
        updateRect(rcs);
}

//! 範囲更新（イメージとキャンバス）
/*!
    @param rcs イメージ範囲（範囲内であること）
*/

void updateRect(const AXRectSize &rcs)
{
    AXRectSize rcsCanvas;

    //------- イメージ合成

    //背景

    if(g_conf->isBkCheck())
        g_draw->pimgBlend->fillCheck(rcs, g_draw->view.colBkCheck);
    else
        g_draw->pimgBlend->clear(rcs, g_draw->view.colImgBk);

    //各レイヤ合成

    update_blendImage(rcs);

    //特殊

    if(g_draw->work.nNowOpNo)
    {
        switch(g_draw->work.nNowOpNo)
        {
            //テキスト
            case OPNO_DRAWTEXT:
                g_draw->pimgTmp[0]->blendToRGB16(g_draw->pimgBlend, rcs,
                    g_draw->pcurlayer->getViewOpacity(), g_draw->funcBlendCol[g_draw->pcurlayer->m_nBlendMode]);
                break;
            //フィルタ・キャンバスプレビュー
            case OPNO_FILTERDLG:
                if(g_draw->work.funcUpdatePrev)
                    (*(g_draw->work.funcUpdatePrev))(rcs);
                break;
            //矩形編集・拡大縮小＆回転
            case OPNO_SCALEROTDLG:
                g_draw->pimgTmp[1]->blendToRGB16(g_draw->pimgBlend, rcs,
                    128, g_draw->funcBlendCol[g_draw->pcurlayer->m_nBlendMode]);
                break;
        }
    }

    //------- キャンバス更新

    if(g_draw->view.imgTowinRect(&rcsCanvas, rcs))
    {
        //キャンバス描画

        update_drawCanvas(rcsCanvas, TRUE);

        //[Debug]
        //g_draw->pimgCanvas->box(rcsCanvas.x, rcsCanvas.y, rcsCanvas.w, rcsCanvas.h, 0xff0000);

        //更新

        CANVASAREA->redraw(rcsCanvas);
    }
}


//=============================
// 各更新
//=============================


//! 描画後の範囲更新（プレビュー更新）
/*!
    @param rcs イメージ範囲（x == -1 で範囲なし）
*/

void updateAfterDraw(const AXRectSize &rcs)
{
    if(rcs.x != -1)
        PREVWIN->updateDrawRect(rcs);
}

//! プレビュー更新

void updatePrev(const FLAGRECT &rcf)
{
    AXRectSize rcs;

    if(getImgRect(&rcs, rcf))
        PREVWIN->updateDrawRect(rcs);
}


//=============================
// イメージ更新
//=============================


//! イメージ全体更新

void updateImage()
{
    AXRectSize rcs;

    rcs.set(0, 0, g_draw->nImgW, g_draw->nImgH);

    //背景

    if(g_conf->isBkCheck())
        g_draw->pimgBlend->fillCheck(rcs, g_draw->view.colBkCheck);
    else
        g_draw->pimgBlend->clear(g_draw->view.colImgBk);

    //イメージ合成

    update_blendImage(rcs);
}

//! レイヤイメージ合成処理

void update_blendImage(const AXRectSize &rcs)
{
    CLayerItem *p;
    int opacity;

    if(g_draw->work.nNowOpNo == OPNO_SELIMGMOVE && g_draw->pimgTmp[0])
    {
        //選択範囲イメージドラッグ移動時
        //（カレントレイヤの後にドラッグイメージを合成）

        for(p = g_draw->player->getBottomVisibleImg(); p; p = p->prevVisibleImg())
        {
            opacity = p->getViewOpacity();

            p->m_pimg->blendToRGB16(g_draw->pimgBlend, rcs, opacity, g_draw->funcBlendCol[p->m_nBlendMode]);

            if(p == g_draw->pcurlayer)
                g_draw->pimgTmp[0]->blendToRGB16(g_draw->pimgBlend, rcs, opacity, g_draw->funcBlendCol[p->m_nBlendMode]);
        }
    }
    else
    {
        //通常

        for(p = g_draw->player->getBottomVisibleImg(); p; p = p->prevVisibleImg())
        {
            p->m_pimg->blendToRGB16(g_draw->pimgBlend, rcs,
                p->getViewOpacity(), g_draw->funcBlendCol[p->m_nBlendMode]);
        }
    }
}


//=============================
//キャンバス更新
//=============================


//! キャンバス範囲指定更新
/*
    rcf : イメージ範囲
*/

void updateCanvas(const FLAGRECT &rcf)
{
    AXRectSize rcs,rcsCanvas;

    if(getImgRect(&rcs, rcf))
    {
        if(g_draw->view.imgTowinRect(&rcsCanvas, rcs))
        {
            update_drawCanvas(rcsCanvas, TRUE);

            CANVASAREA->redraw(rcsCanvas);
        }
    }
}

//! キャンバス全体更新

void updateCanvas(BOOL bReDraw,BOOL bHiQuality)
{
    AXRectSize rcs;

    //キャンバス描画

    rcs.set(0, 0, g_draw->view.szCanvas.w, g_draw->view.szCanvas.h);

    update_drawCanvas(rcs, bHiQuality);

    //更新

    if(bReDraw)
        CANVASAREA->redraw();
}

//! キャンバス描画
/*
    pimgBlend のイメージを元にウィンドウ表示用イメージを描画
*/

void update_drawCanvas(const AXRectSize &rcsDst,BOOL bHiQuality)
{
    DRAWCANVASINFO info;
    AXRectSize rcsImg;
    BOOL b1pxGrid,bImgRect;
    int w,h;

    if(!g_draw->pimgCanvas->isExist()) return;

    info.rcsDst     = rcsDst;
    info.nBaseX     = g_draw->view.ptBaseImg.x;
    info.nBaseY     = g_draw->view.ptBaseImg.y;
    info.nScrollX   = (g_draw->view.szCanvas.w >> 1) - g_draw->view.ptScr.x;
    info.nScrollY   = (g_draw->view.szCanvas.h >> 1) - g_draw->view.ptScr.y;
    info.bHRev      = g_draw->view.bHRev;
    info.dwExCol    = g_conf->dwCanvasBkCol;
    info.pParam     = &g_draw->view.param;

    //-------- イメージ

    if(g_draw->view.nAngle)
    {
        //回転あり

        if(bHiQuality)
            g_draw->pimgBlend->drawCanvasRotHiQuality(g_draw->pimgCanvas, info);
        else
            g_draw->pimgBlend->drawCanvasRotNormal(g_draw->pimgCanvas, info);
    }
    else
    {
        //回転なし（200%以下[100%は除く] は高品質で）

        if(bHiQuality && g_draw->view.nScale != 1000 && g_draw->view.nScale < 2000)
            g_draw->pimgBlend->drawCanvasScaleDown(g_draw->pimgCanvas, info);
        else
            g_draw->pimgBlend->drawCanvasNormal(g_draw->pimgCanvas, info);
    }

    //-------- 選択範囲・グリッド用、イメージ範囲取得

    bImgRect = draw::winToimgRect(&rcsImg, rcsDst);

    //-------- グリッド

    b1pxGrid = (g_conf->n1pxGridScale && g_draw->view.nScale >= g_conf->n1pxGridScale);

    if(bImgRect &&
      ( (g_conf->uEtcViewFlags & (CConfig::ETCVIEWF_GRID | CConfig::ETCVIEWF_GRIDSPLIT)) || b1pxGrid ) )
    {
        if(b1pxGrid)
        {
            //-------- 1pxグリッド

            //1pxグリッド

            g_draw->pimgCanvas->drawGrid(rcsDst, rcsImg, 1, 1, 0xffcccccc, info);

            //グリッド

            if(g_conf->isGrid())
            {
                g_draw->pimgCanvas->drawGrid(rcsDst, rcsImg,
                        g_conf->nGridW, g_conf->nGridH, g_conf->dwGridCol | 0xff000000, info);
            }

            //分割線

            if(g_conf->isGridSplit())
            {
                w = g_draw->nImgW / g_conf->nGridSplitX;
                h = g_draw->nImgH / g_conf->nGridSplitY;
                if(w < 2) w = 2;
                if(h < 2) h = 2;

                g_draw->pimgCanvas->drawGrid(rcsDst, rcsImg, w, h, g_conf->dwGridSplitCol | 0xff000000, info);
            }
        }
        else
        {
            //-------- 通常グリッド

            //グリッド

            if(g_conf->isGrid())
            {
                g_draw->pimgCanvas->drawGrid(rcsDst, rcsImg,
                        g_conf->nGridW, g_conf->nGridH, g_conf->dwGridCol, info);
            }

            //分割線

            if(g_conf->isGridSplit())
            {
                w = g_draw->nImgW / g_conf->nGridSplitX;
                h = g_draw->nImgH / g_conf->nGridSplitY;
                if(w < 2) w = 2;
                if(h < 2) h = 2;

                g_draw->pimgCanvas->drawGrid(rcsDst, rcsImg,
                        w, h, g_conf->dwGridSplitCol, info);
            }
        }
    }

    //-------- 選択範囲

    if(bImgRect && g_draw->work.rcfSel.flag && g_draw->pimgSel->isExist())
        g_draw->pimgSel->drawSelection(g_draw->pimgCanvas, info, rcsImg);
}

};
