/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    描画操作・サブ

    図形塗りつぶし、イメージ移動
    ルーペ、スポイト、キャンバス移動、キャンバス回転、倍率変更
    左右ドラッグでのブラシサイズ変更
*/


#include <math.h>

#include "drawdat.h"
#include "drawOpDef.h"

#include "draw_main.h"
#include "draw_opsub.h"
#include "draw_opxor.h"
#include "draw_opfunc.h"
#include "draw_update.h"
#include "draw_calc.h"

#include "CCanvasWin.h"
#include "CColorWin.h"
#include "CBrushWin.h"
#include "CLayerList.h"
#include "CLayerItem.h"
#include "CBrushList.h"
#include "CBrushItem.h"
#include "CConfig.h"

#include "CTileImageA1.h"
#include "CImageRGB16.h"
#include "CPolygonPaint.h"
#include "CDrawPaint.h"
#include "CUndo.h"
#include "cursor.h"

#include "global.h"

#include "AXUtilTime.h"


namespace draw
{

//====================================
// 共通処理
//====================================


//! 移動時、何もしない

void op_temp_move(UINT uModKey)
{

}

//! 離し時、何もしない（グラブ解放）

BOOL op_temp_up_release()
{
    return TRUE;
}

//! 離し時、何もしない（グラブ解放しない）

BOOL op_temp_up_norelease()
{
    return FALSE;
}

//! キャンバス移動時などの離し時

BOOL op_common_canvas_up()
{
    CANVASAREA->clearTimer_updateCanvas();
    CANVASAREA->restoreCursorTool();

    return TRUE;
}


//=============================
// 描画処理
//=============================


//! 多角形塗りつぶし 共通

void draw_fillPolygon_common()
{
    setBeforeDraw(g_draw->work.nOpToolNo);
    beginDraw();

    g_draw->pcurlayer->m_pimg->drawFillPolygon(g_draw->pPolyPaint,
        g_draw->work.optDrawTmp & 0xff, g_draw->work.optDrawTmp & (1<<11));

    endDraw();

    //解放

    g_draw->pPolyPaint->free();
}

//! 四角塗りつぶし描画

void draw_fillBox()
{
    DPOINT pt[4];
    AXRectSize rcs;
    int i;

    if(g_draw->view.nAngle == 0)
    {
        //------- 回転0の場合

        getDrawBox(&rcs);

        setBeforeDraw(g_draw->work.nOpToolNo);
        beginDraw();

        g_draw->pcurlayer->m_pimg->drawFillBox(rcs.x, rcs.y, rcs.w, rcs.h, g_draw->work.optDrawTmp & 0xff);

        endDraw();
    }
    else
    {
        //描画位置

        getDrawBoxPoint(pt);

        //点セット

        if(!g_draw->pPolyPaint->alloc(5)) return;

        for(i = 0; i < 4; i++)
            g_draw->pPolyPaint->add(pt[i].x, pt[i].y);

        g_draw->pPolyPaint->endPos();

        //描画

        draw_fillPolygon_common();
    }
}

//! 円塗りつぶし描画

void draw_fillCircle()
{
    double cx,cy,xr,yr;

    g_draw->view.winToimg(&cx, &cy, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);

    xr = g_draw->work.ptTmp[1].x * g_draw->view.param.dScaleDiv;
    yr = g_draw->work.ptTmp[1].y * g_draw->view.param.dScaleDiv;

    //描画
    /* optDrawTmp は setBeforeDraw() でセットされている */

    setBeforeDraw(g_draw->work.nOpToolNo);
    beginDraw();

    g_draw->pcurlayer->m_pimg->drawFillCircle(cx, cy, xr, yr,
        g_draw->view.param, g_draw->view.bHRev,
        g_draw->work.optDrawTmp & 0xff, g_draw->work.optDrawTmp & (1<<11));

    endDraw();
}


//=============================
// 多角形
//=============================
/*
    pimgXor に確定された直線を描画。
    現在の線は直接キャンバスに描画。
*/


//! 最初の押し時
/*!
    @return FALSE で初期化失敗
*/

BOOL op_polygon_down()
{
    double x,y;

    if(!g_draw->pimgXor->create(g_draw->view.szCanvas.w, g_draw->view.szCanvas.h))
        return FALSE;

    //

    if(!g_draw->pPolyPaint->alloc(50))
    {
        g_draw->pimgXor->free();
        return FALSE;
    }

    g_draw->view.winToimg(&x, &y, g_draw->work.ptWinNow.x, g_draw->work.ptWinNow.y);
    g_draw->pPolyPaint->add(x, y);

    //

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_create;

    g_draw->work.rcsTmp[0].set(0, 0, g_draw->view.szCanvas.w, g_draw->view.szCanvas.h);

    return TRUE;
}

//! 2度目以降の押し時

void op_polygon_down2()
{
    double x,y;
    RGBAFIX15 col;

    //点追加

    g_draw->view.winToimg(&x, &y, g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y);
    g_draw->pPolyPaint->add(x, y);

    //XOR

    col.set(0,0,0,0x8000);

    drawXorLine();
    g_draw->pimgXor->blendXor(g_draw->pimgCanvas, g_draw->work.rcsTmp[0]);

    g_draw->pimgXor->drawLineB(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y,
            g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y, col, FALSE);

    g_draw->pimgXor->blendXor(g_draw->pimgCanvas, g_draw->work.rcsTmp[0]);

    g_draw->work.ptTmp[0] = g_draw->work.ptTmp[1];

    drawXorLine();

    CANVASAREA->redraw();
}

//! 終了/キャンセル

BOOL opCancel_polygon(BOOL bDraw)
{
	//XOR消去

    drawXorLine();
    g_draw->pimgXor->blendXor(g_draw->pimgCanvas, g_draw->work.rcsTmp[0]);

    CANVASAREA->redraw();
    CANVASAREA->clearTimerUpdate(CCanvasWinArea::TIMERID_UPDATE);

	g_draw->pimgXor->free();

    //描画

    if(bDraw)
    {
        g_draw->pPolyPaint->endPos();

        //描画

        switch(g_draw->work.nOpAfter2)
        {
            //多角形塗りつぶし
            case OPAFTER2_DRAWFILLPOLYGON:
                draw_fillPolygon_common();
                break;
            //選択範囲
            case OPAFTER2_SELPOLYGON:
                draw_sel_polygonCommon();
                break;
        }
    }

	//削除

	g_draw->pPolyPaint->free();

	return TRUE;
}


//==================================
// 移動ツール
//==================================
/*
    dptTmp[0] : 総移動px数（ウィンドウ）
    ptTmp[0]  : 押し時の元オフセット位置（先頭レイヤ or 選択イメージ）
    ptTmp[1]  : オフセット位置総移動px数
    rcfDrawUp : 更新範囲（CCanvasWinArea）
    rcfTmp[0] : 全移動レイヤの更新範囲

    - フォルダ選択時はフォルダ下すべて移動。
    - 移動させるレイヤは work.pLayer を先頭に CLayerItem::m_pLink でリンクされている。
     （フォルダ・ロックレイヤは含まれない）
*/


//! 押し

BOOL op_imgmove_down()
{
    CLayerItem *pItem;
    int type,bDisableLink;
    AXPoint pt;

    type = g_draw->tool.optMove & 7;

    if(type == 3)
    {
        //-------- 選択範囲位置移動

        if(!isSelExist()) return FALSE;

        setOpInfo(OPNO_SELPOSMOVE, op_selposmove_move, op_selposmove_up);

        g_draw->pimgSel->getOffset(&g_draw->work.ptTmp[0]);
    }
    else
    {
        //-------- レイヤの位置移動

        //レイヤ

        if(type == 0)
            pItem = g_draw->pcurlayer;
        else if(type == 1)
        {
            getNowImgPoint(&pt);

            pItem = g_draw->player->getTopPixLayer(pt.x, pt.y);
            if(!pItem) return FALSE;
        }
        else
            pItem = NULL;

        //リンク無効（+Alt)

        if(g_draw->work.uModDown & MOD_ALT)
            bDisableLink = TRUE;
        else
            bDisableLink = g_draw->tool.optMove & (1<<3);

       //移動させるレイヤの先頭取得＆リンクをセット

        g_draw->work.pLayer = g_draw->player->setLinkMove(type, bDisableLink, pItem);

        if(!g_draw->work.pLayer) return FALSE;

        //

        setOpInfo(OPNO_IMGMOVE, op_imgmove_move, op_imgmove_up);

        g_draw->player->getLinkVisibleImgRect(g_draw->work.pLayer, &g_draw->work.rcfTmp[0]);

        g_draw->work.pLayer->m_pimg->getOffset(&g_draw->work.ptTmp[0]);
    }

    //

    g_draw->work.dptTmp[0].zero();
    g_draw->work.ptTmp[1].zero();
    g_draw->work.rcfDrawUp.clear();

    return TRUE;
}

//! 移動（イメージ位置）

void op_imgmove_move(UINT uModKey)
{
    AXPoint ptMov;
    CLayerItem *p;
    FLAGRECT rcf;

    if(calcMoveOffset(g_draw->work.pLayer->m_pimg,
            uModKey & MOD_SHIFT, uModKey & MOD_CTRL, &ptMov))
    {
        //オフセット位置変更

        for(p = g_draw->work.pLayer; p; p = p->m_pLink)
            p->m_pimg->moveOffset(ptMov.x, ptMov.y);

        //更新範囲

        rcf.combineMove(g_draw->work.rcfTmp[0], ptMov.x, ptMov.y);  //移動前+移動後

        g_draw->work.rcfTmp[0].move(ptMov.x, ptMov.y);

        //更新

        g_draw->work.rcfDrawUp.combine(rcf);

        CANVASAREA->setTimer_updateMove();
    }
}

//! 離し（イメージ位置）

BOOL op_imgmove_up()
{
    if(g_draw->work.ptTmp[1].x || g_draw->work.ptTmp[1].y)
    {
        FLAGRECT rcf;

        //アンドゥ

        g_draw->pundo->add_layerOffset(g_draw->work.ptTmp[1].x, g_draw->work.ptTmp[1].y,
                    g_draw->work.pLayer, g_draw->player->getLinkCnt(g_draw->work.pLayer));

        //更新

        CANVASAREA->delTimer(CCanvasWinArea::TIMERID_UPDATE_MOVE);

        updateRect(g_draw->work.rcfDrawUp); //残っている範囲

        //プレビュー（移動前の範囲+移動後の範囲）

        rcf.combineMove(g_draw->work.rcfTmp[0], -g_draw->work.ptTmp[1].x, -g_draw->work.ptTmp[1].y);

        updatePrev(rcf);
    }

    return TRUE;
}

//! 移動（選択範囲位置）

void op_selposmove_move(UINT uModKey)
{
    AXPoint ptMov;
    FLAGRECT rcf;

    if(calcMoveOffset(g_draw->pimgSel,
            uModKey & MOD_SHIFT, uModKey & MOD_CTRL, &ptMov))
    {
        //位置移動

        g_draw->pimgSel->moveOffset(ptMov.x, ptMov.y);

        //範囲

        rcf.combineMove(g_draw->work.rcfSel, ptMov.x, ptMov.y);  //移動前+移動後

        g_draw->work.rcfSel.move(ptMov.x, ptMov.y);

        //更新
        /* 選択範囲は輪郭を描画するので、範囲拡張 */

        rcf.inflate(2);
        g_draw->work.rcfDrawUp.combine(rcf);

        CANVASAREA->setTimer_updateMove();
    }
}

//! 離し（選択範囲イメージ位置）

BOOL op_selposmove_up()
{
    if(g_draw->work.ptTmp[1].x || g_draw->work.ptTmp[1].y)
    {
        CANVASAREA->delTimer(CCanvasWinArea::TIMERID_UPDATE_MOVE);

        updateCanvas(g_draw->work.rcfDrawUp); //残っている範囲
    }

    return TRUE;
}


//====================================
// 押し or 離し時のみの処理
//====================================


//! 塗りつぶし・離し

BOOL op_paint_up()
{
    CDrawPaint paint;
    CTileImage *pRef;
    AXPoint pt;
    int ret,type,diff,opacity;
    DWORD val;

    if(checkDrawLayer()) return TRUE;

    //判定元イメージ取得

    pRef = g_draw->player->getPaintRefImg();

    if(!pRef || g_draw->work.uModDown == MOD_SHIFT)
        pRef = g_draw->pcurlayer->m_pimg;

    //開始位置

    getNowImgPoint(&pt);

    //パラメータ

    if(g_draw->work.nOpToolNo == TOOL_PAINTERASE)
    {
        type = CDrawPaint::TYPE_ERASE;
        diff = opacity = 0;
    }
    else
    {
        val = g_draw->tool.optPaint;

        type    = val & 7;
        diff    = (val >> 13) & 1023;
        opacity = (val >> 6) & 127;
    }

    //描画

    ret = paint.init(type, diff, opacity, g_draw->colDraw,
                g_draw->pcurlayer->m_pimg, pRef, pt);

    if(ret == CDrawPaint::RET_SUCCESS)
    {
        setBeforeDraw(g_draw->work.nOpToolNo);

        beginDraw();

        paint.run();

        endDraw();
    }

    return TRUE;
}

//! 自動選択・離し

BOOL op_magicwand_up()
{
    CDrawPaint paint;
    AXPoint pt;
    int ret;
    DWORD val;

    if(g_draw->pcurlayer->isFolder()) return TRUE;

    val = g_draw->tool.optMagicWand;

    //開始位置

    getNowImgPoint(&pt);

    //初期化（カレントレイヤで判定）

    ret = paint.init(val & 7, (val >> 3) & 1023, 0x8000, g_draw->colDraw,
                g_draw->pimgSel, g_draw->pcurlayer->m_pimg, pt);

    if(ret != CDrawPaint::RET_SUCCESS) return TRUE;

    //描画（範囲追加）

    if(selAllocArray())
    {
        CANVASAREA->setCursorWait();

        setBeforeDraw(TOOL_MAGICWAND);
        paint.run();

        g_draw->work.rcfSel.combine(g_draw->drawinfo.rcfDraw);

        updateCanvas(g_draw->drawinfo.rcfDraw);

        CANVASAREA->restoreCursorTool();
    }

    return TRUE;
}

//! ルーペツール

BOOL op_zoomClick_down(BOOL bDown)
{
    setOpInfo(OPNO_TEMP, op_temp_move, op_temp_up_release);

    if(g_draw->work.uModDown == MOD_CTRL) bDown = !bDown;

    changeScaleToolUpDown(bDown);

    return TRUE;
}

//! スポイト

BOOL op_spoit_down()
{
    AXPoint pt;
    RGBFIX15 col;
    RGBAFIX15 colSrc;
    DWORD dwCol;

    setOpInfo(OPNO_TEMP, op_temp_move, op_temp_up_release);

    getNowImgPoint(&pt);

    //

    if(pt.x >= 0 && pt.x < g_draw->nImgW && pt.y >= 0 && pt.y < g_draw->nImgH)
    {
        //色取得

        if(g_draw->work.uModDown & MOD_CTRL)
        {
            //カレントレイヤ
            //(フォルダレイヤなら何もしない)

            if(g_draw->pcurlayer->isFolder())
				return TRUE;

            g_draw->pcurlayer->m_pimg->getPixel(&colSrc, pt.x, pt.y);

            col.set(colSrc);
        }
        else
            //合成後の色
            getPixelBlendColor(&col, pt.x, pt.y);

        //

        dwCol = col.toDWORD();

        //セット

        if(g_draw->work.uModDown & MOD_SHIFT)
        {
            //マスク色No.1にセット

            changeColMaskColor(0, dwCol | 0xff000000);
            COLORWIN->updateMaskColor();
        }
        else
        {
            //描画色
            changeDrawCol(dwCol);
            COLORWIN->setDrawColor();
        }
    }

    return TRUE;
}

//! 2点をクリックして中間色作成

BOOL op_midcolor_down()
{
    AXPoint pt;
    TIMENANO time;
    RGBFIX15 col;
    DWORD dwCol;
    int first,i;

    setOpInfo(OPNO_TEMP, op_temp_move, op_temp_up_release);

    getNowImgPoint(&pt);
    getPixelBlendColor(&col, pt.x, pt.y);

    AXGetTime(&time);

    //何回目か (前回から5秒以上経っていた場合、最初のクリックとする)

    if(g_draw->work.timeMidCol == 0)
        first = TRUE;
    else
        first = (time.sec > g_draw->work.timeMidCol + 5);

    //

    if(first)
    {
        //------- 最初の色

        g_draw->work.colMidCol  = col;
        g_draw->work.timeMidCol = time.sec;
    }
    else
    {
        //------- 中間色作成

        for(i = 0; i < 3; i++)
            col.c[i] = ((g_draw->work.colMidCol.c[i] - col.c[i]) >> 1) + col.c[i];

        dwCol = col.toDWORD();

        changeDrawCol(dwCol);
        COLORWIN->setDrawColor();

        g_draw->work.timeMidCol = 0;
    }

    return TRUE;
}

//! カレントレイヤ上で、クリックした色を描画色に置き換える

BOOL op_replaceColor_down()
{
    AXPoint pt;
    RGBAFIX15 col,colDraw;

    if(checkDrawLayer()) return FALSE;

    //

    setOpInfo(OPNO_TEMP, op_temp_move, op_temp_up_release);

    getNowImgPoint(&pt);

    getCurImg()->getPixel(&col, pt.x, pt.y);

    //

    setBeforeDraw_clearMask();
    g_draw->drawinfo.funcColor = &CTileImage::colfunc_overwrite;

    colDraw = g_draw->colDraw;
    colDraw.a = 0x8000;

    //

    beginDraw();

    if(col.a == 0)
        getCurImg()->replaceColor_tp(colDraw);
    else
        getCurImg()->replaceColor(col, colDraw);

    endDraw();

    return TRUE;
}



//====================================
// キャンバス移動
//====================================
/*
    ptTmp[0]  : 開始時のスクロール位置
*/


//! 押し時

BOOL op_canvasMove_down()
{
    setOpInfo(OPNO_CANVASMOVE, op_canvasMove_move, op_common_canvas_up);

    g_draw->work.ptTmp[0] = g_draw->view.ptScr;

    CANVASAREA->setCursorDrag(cursor::HANDDRAG);

    return TRUE;
}

//! 移動時

void op_canvasMove_move(UINT uModKey)
{
    AXPoint pt;

    pt.x = g_draw->work.ptTmp[0].x + (int)(g_draw->work.ptWinDown.x - g_draw->work.ptWinNow.x);
    pt.y = g_draw->work.ptTmp[0].y + (int)(g_draw->work.ptWinDown.y - g_draw->work.ptWinNow.y);

    if(pt != g_draw->view.ptScr)
    {
        g_draw->view.ptScr = pt;

        CANVASWIN->setScrollPos();
        CANVASAREA->setTimer_updateCanvas();
    }
}


//===============================
// キャンバス回転
//===============================
/*
    nTmp[0] : 現在の回転角度
    nTmp[1] : 前のキャンバス上の角度
*/


//! 押し

BOOL op_canvasRotate_down()
{
    double x,y;

    setOpInfo(OPNO_CANVASROTATE, op_canvasRotate_move, op_common_canvas_up);

    g_draw->work.nTmp[0] = g_draw->view.nAngle;

    //角度

    x = g_draw->work.ptWinNow.x - g_draw->view.szCanvas.w * 0.5;
    y = g_draw->work.ptWinNow.y - g_draw->view.szCanvas.h * 0.5;

    g_draw->work.nTmp[1] = (int)(::atan2(y, x) * 18000.0 / M_PI);

    //

    setScroll_reset();

    CANVASAREA->setCursorDrag(cursor::ROTATE);

    return TRUE;
}

//! 移動

void op_canvasRotate_move(UINT uModKey)
{
    double x,y;
    int angle,n;

    //キャンバス位置から角度計算

    x = g_draw->work.ptWinNow.x - g_draw->view.szCanvas.w * 0.5;
    y = g_draw->work.ptWinNow.y - g_draw->view.szCanvas.h * 0.5;

    angle = (int)(::atan2(y, x) * 18000.0 / M_PI);

    //前回の角度からの移動分を加算

    n = g_draw->work.nTmp[0] + angle - g_draw->work.nTmp[1];

    if(n < -18000) n += 36000;
    else if(n > 18000) n -= 36000;

    g_draw->work.nTmp[0] = n;

    //45度補正

    if(uModKey & MOD_SHIFT) n = n / 4500 * 4500;

    //更新

    if(n != g_draw->view.nAngle)
    {
        changeRotate(n, FALSE, FALSE, FALSE);

        CANVASAREA->setTimer_updateCanvas();
    }

    //

    g_draw->work.nTmp[1] = angle;
}


//===============================
// 上下ドラッグでの表示倍率変更
//===============================
/*
    nTmp[0] : 開始時の倍率
*/


//! 押し時

BOOL op_zoomDrag_down()
{
    setOpInfo(OPNO_ZOOMDRAG, op_zoomDrag_move, op_common_canvas_up);

    g_draw->work.nTmp[0] = g_draw->view.nScale;

    setScroll_reset();

    return TRUE;
}

//! 移動時

void op_zoomDrag_move(UINT uModKey)
{
    int scale;

    scale = g_draw->work.nTmp[0] + (int)(g_draw->work.ptWinDown.y - g_draw->work.ptWinNow.y) * 10;

    if(scale != g_draw->view.nScale)
    {
        changeScale(scale, FALSE, FALSE, FALSE);

        CANVASAREA->setTimer_updateCanvas();
    }
}

//====================================
// 左右ドラッグでのブラシサイズ変更
//====================================
/*
    ptTmp[0]   : 押し時の位置
    nTmp[0]    : 半径サイズ(px)
    nTmp[1]    : 押し時のブラシサイズ
    rcsTmp[0]  : XOR描画範囲
    pTmp       : 対象ブラシアイテム
*/


//! 押し時
/*!
    @param bRegBrush 登録ブラシのサイズを変更
*/

BOOL op_dragBrushSize_down(BOOL bRegBrush)
{
    CBrushItem *p;

    setOpInfo(OPNO_DRAGBRUSHSIZE, op_dragBrushSize_move, op_dragBrushSize_up);

    getNowPoint(&g_draw->work.ptTmp[0]);

    //対象ブラシ

    p = BRUSHLIST->getDrawBrush(bRegBrush);

    //

    g_draw->work.pTmp = p;

    g_draw->work.nTmp[0] = (int)(p->wRadius * 0.1 * g_draw->view.param.dScale + 0.5);
    g_draw->work.nTmp[1] = p->wRadius;

    drawXorBrushCircle(FALSE);

    return TRUE;
}

//! 移動時

void op_dragBrushSize_move(UINT uModKey)
{
    CBrushItem *p = (CBrushItem *)g_draw->work.pTmp;
    int n;

    //半径

    n = g_draw->work.nTmp[1] + (int)((g_draw->work.ptWinNow.x - g_draw->work.ptWinDown.x) * g_conf->sDragBrushSizeW);

    if(n < CBrushItem::RADIUS_MIN)
        n = CBrushItem::RADIUS_MIN;
    else if(n > p->wRadiusCtlMax)
        n = p->wRadiusCtlMax;

    //

    if(n != p->wRadius)
    {
        drawXorBrushCircle(TRUE);

        g_draw->work.nTmp[0] = (int)(n * 0.1 * g_draw->view.param.dScale + 0.5);

        drawXorBrushCircle(FALSE);

        //値変更(登録ブラシが対象で、現在選択されていない場合は直接値を変更)

        if(p == BRUSHLIST->getEditItem())
            BRUSHWIN->setBrushSize(n);
        else
            p->wRadius = n;
    }
}

//! 離し時

BOOL op_dragBrushSize_up()
{
    drawXorBrushCircle(TRUE);

    return TRUE;
}

};

