/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    ブラシ/ドット描画
*/

#include <stdlib.h>

#include "drawdat.h"
#include "drawOpDef.h"
#include "global.h"

#include "CTileImageA1.h"
#include "CPosBuf.h"

#include "draw_main.h"
#include "draw_opsub.h"
#include "draw_opxor.h"
#include "draw_opfunc.h"
#include "draw_calc.h"


namespace draw
{


//===========================
// ブラシ描画・自由線
//===========================


//! 押し時

void op_brush_free_down(BOOL bRegBrush,BOOL bPressMax)
{
    DRAWPOINT pt;

    setOpInfo(OPNO_BRUSH_FREE, op_brush_free_move, op_brush_free_up);

    //描画準備

    setBeforeDraw(TOOL_BRUSH, (bRegBrush)? 0: -1);

    //筆圧最大

    if(bPressMax)
        g_draw->work.uOpFlags |= OPFLAGS_BRUSH_PRESSMAX;

    //

    getDrawPoint(&pt, FALSE);
    if(bPressMax) pt.press = 1;

    g_draw->pPosBuf->init(CTileImage::m_pbrush->nHoseiType, CTileImage::m_pbrush->nHoseiStr,
            pt.x, pt.y, pt.press);

    getCurImg()->startDrawBrushFree(pt.x, pt.y, pt.press);

    //UNDO用準備

    beginDrawUndo();
}

//! 移動時

void op_brush_free_move(UINT uModKey)
{
    DRAWPOINT pt;
    CPosBuf::POSDAT *p;

    //位置

    getDrawPoint(&pt, TRUE);
    if(g_draw->work.uOpFlags & OPFLAGS_BRUSH_PRESSMAX) pt.press = 1;

    p = g_draw->pPosBuf->addPos(pt.x, pt.y, pt.press);

    //描画

    g_draw->drawinfo.rcfDraw.clear();

    getCurImg()->drawBrush_free(p->x, p->y, p->press);

    commonFreeDraw();
}

//! 離し時

BOOL op_brush_free_up()
{
    commonDrawEnd();

    return TRUE;
}


//=============================
// ドットペン・自由線
//=============================
/*
    +Shift : No.2のデータで描画

    nTmp[0]  : 細線か (setBeforeDraw でセット)
    ptTmp[0] : 前の位置
    ptTmp[1] : 2つ前の位置
*/


//! 押し

BOOL op_dotpen_free_down()
{
    AXPoint pt;
    int listno;

    setOpInfo(OPNO_DOT_FREE, op_dotpen_free_move, op_dotpen_free_up);

    //使用するリスト番号

    if(g_draw->work.uModDown == MOD_SHIFT)
        listno = 1;
    else
        listno = -1;

    //描画位置

    getDrawPointInt(&pt, FALSE);

    //セット

    setBeforeDraw(TOOL_DOTPEN, listno);

    g_draw->work.ptTmp[0] = g_draw->work.ptTmp[1] = pt;

    //UNDO用準備

    beginDrawUndo();

    //点を打つ（細線時も）

    getCurImg()->setPixelDotPen(pt.x, pt.y, g_draw->drawinfo.colDraw);

    //更新

    commonFreeDraw();

    return TRUE;
}

//! 移動時

void op_dotpen_free_move(UINT uModKey)
{
    AXPoint pt;
    BOOL bDraw = TRUE;

    getDrawPointInt(&pt, TRUE);

    //前回と同じ位置なら描画しない

    if(pt == g_draw->work.ptTmp[0]) return;

    //細線時、2つ前の位置と斜めの場合は描画しない

    if(g_draw->work.nTmp[0])
    {
        if(::abs(pt.x - g_draw->work.ptTmp[1].x) == 1 && ::abs(pt.y - g_draw->work.ptTmp[1].y) == 1 &&
           (pt.x == g_draw->work.ptTmp[0].x || pt.y == g_draw->work.ptTmp[0].y))
            bDraw = FALSE;
    }

    //描画

    if(bDraw)
    {
        g_draw->drawinfo.rcfDraw.clear();

        //描画

        if(g_draw->work.nTmp[0])
            getCurImg()->drawLineF(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y, pt.x, pt.y, g_draw->drawinfo.colDraw);
        else
            getCurImg()->drawLineB(g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y, pt.x, pt.y, g_draw->drawinfo.colDraw, TRUE);

        //更新

        commonFreeDraw();
    }

    //前の位置

    if(bDraw) g_draw->work.ptTmp[1] = g_draw->work.ptTmp[0];

    g_draw->work.ptTmp[0] = pt;
}

//! 離し時

BOOL op_dotpen_free_up()
{
    commonDrawEnd();

    return TRUE;
}


//=============================
// 描画処理
//=============================


//! 直線描画

void draw_line()
{
    DRECT rc;

    //描画位置取得

    getDrawLineRect(&rc);

    //描画

    setBeforeDraw(g_draw->work.nOpToolNo);
    beginDraw();
    
    if(g_draw->work.nOpToolNo == TOOL_BRUSH)
    {
		getCurImg()->startDrawBrushNoneFree(rc.x1, rc.y1);
        getCurImg()->drawBrush_lineHeadTail(rc.x1, rc.y1, rc.x2, rc.y2, g_draw->tool.headtail[0]);
    }
    else
        getCurImg()->drawLineB((int)rc.x1, (int)rc.y1, (int)rc.x2, (int)rc.y2, g_draw->drawinfo.colDraw, FALSE);

    endDraw();
}

//! 四角枠描画

void draw_box()
{
    DPOINT pt[4];
    CTileImage *pimg;

    //描画位置

    getDrawBoxPoint(pt);

    //描画

    setBeforeDraw(g_draw->work.nOpToolNo);
    beginDraw();

    pimg = getCurImg();

    if(g_draw->work.nOpToolNo == TOOL_BRUSH)
    {
        pimg->startDrawBrushNoneFree(pt[0].x, pt[0].y);

        pimg->drawBrush_line(pt[0].x, pt[0].y, pt[1].x, pt[1].y, 1, 1, 0);
        pimg->drawBrush_line(pt[1].x, pt[1].y, pt[2].x, pt[2].y, 1, 1, 0);
        pimg->drawBrush_line(pt[2].x, pt[2].y, pt[3].x, pt[3].y, 1, 1, 0);
        pimg->drawBrush_line(pt[3].x, pt[3].y, pt[0].x, pt[0].y, 1, 1, 0);
    }
    else
    {
        int i;
        AXPoint ptn[4];

        for(i = 0; i < 4; i++)
        {
            ptn[i].x = (int)pt[i].x;
            ptn[i].y = (int)pt[i].y;
        }

        pimg->drawLineB(ptn[0].x, ptn[0].y, ptn[1].x, ptn[1].y, g_draw->drawinfo.colDraw, FALSE);
        pimg->drawLineB(ptn[1].x, ptn[1].y, ptn[2].x, ptn[2].y, g_draw->drawinfo.colDraw, TRUE);
        pimg->drawLineB(ptn[2].x, ptn[2].y, ptn[3].x, ptn[3].y, g_draw->drawinfo.colDraw, TRUE);
        pimg->drawLineB(ptn[3].x, ptn[3].y, ptn[0].x, ptn[0].y, g_draw->drawinfo.colDraw, TRUE);
    }

    endDraw();
}

//! 円枠描画

void draw_circle()
{
    double cx,cy,xr,yr;

    //イメージ位置に変換

    g_draw->view.winToimg(&cx, &cy, g_draw->work.ptTmp[0].x, g_draw->work.ptTmp[0].y);

    xr = g_draw->work.ptTmp[1].x * g_draw->view.param.dScaleDiv;
    yr = g_draw->work.ptTmp[1].y * g_draw->view.param.dScaleDiv;

    //描画

    setBeforeDraw(g_draw->work.nOpToolNo);
    beginDraw();
    
    if(g_draw->work.nOpToolNo == TOOL_DOTPEN && g_draw->view.nAngle == 0)
        //ドットペンで回転表示 0 の場合、ドット用描画
        getCurImg()->drawEllipseDot((int)(cx - xr), (int)(cy - yr), (int)(cx + xr), (int)(cy + yr));
    else
    {
        getCurImg()->drawCircle(cx, cy, xr, yr, g_draw->view.param, g_draw->view.bHRev,
			(g_draw->work.nOpToolNo == TOOL_BRUSH));
    }

    endDraw();
}


//===========================
// 連続直線/集中線
//===========================


//! 連続直線/集中線、描画
/*
    op_xorline2_down2() 時
*/

void draw_lineSuccConc()
{
    DRECT rc;

    getDrawLineRect(&rc);

    g_draw->drawinfo.rcfDraw.clear();

    if(g_draw->work.nOpToolNo == TOOL_BRUSH)
    {
		getCurImg()->startDrawBrushNoneFree(rc.x1, rc.y1);
        getCurImg()->drawBrush_lineHeadTail(rc.x1, rc.y1, rc.x2, rc.y2, g_draw->tool.headtail[0]);
    }
    else
        getCurImg()->drawLineB((int)rc.x1, (int)rc.y1, (int)rc.x2, (int)rc.y2, g_draw->drawinfo.colDraw, FALSE);

    commonFreeDraw(FALSE);  //直接更新
}

//! 連続直線/集中線の終了処理
/*!
    @param bStartTo 連続直線時、最後に始点と結ぶ（BACKSPACEキー押し時）
*/

BOOL opCancel_lineSuccConc(BOOL bStartTo)
{
    drawXorLine();

    //始点と結ぶ

    if(bStartTo)
    {
        g_draw->work.ptTmp[1] = g_draw->work.ptTmp[2];

        draw_lineSuccConc();
    }

    commonDrawEnd();

    return TRUE;
}


//===========================
// ベジェ曲線
//===========================
/*
    ptTmp[0-3] : 各点
    rcsTmp[0]  : 更新範囲

    xorBox で直線指定 -> 制御点2つ指定(nOpAfter に操作進行)
*/


//! 直線指定後、制御点指定開始

BOOL op_bezier_up_first()
{
    setOpInfo(OPNO_BEZIER, op_bezier_move, op_bezier_up, 0);

    g_draw->work.ptTmp[2] = g_draw->work.ptTmp[1];
    g_draw->work.ptTmp[3] = g_draw->work.ptTmp[1];

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_create;

    drawXorBezier(FALSE);

    return FALSE;
}

//! 制御点操作時、移動

void op_bezier_move(UINT uModKey)
{
    AXPoint pt;

    drawXorBezier(TRUE);

    getNowPoint(&pt);

    g_draw->work.ptTmp[2 + g_draw->work.nOpAfter] = pt;

    if(g_draw->work.nOpAfter == 0)
        g_draw->work.ptTmp[3] = pt;

    drawXorBezier(FALSE);
}

//! 制御点操作時、離し
/*!
    @return 操作終了するか
*/

BOOL op_bezier_up()
{
    drawXorBezier(TRUE);

    if(g_draw->work.nOpAfter == 0)
    {
        //------ 次の制御点へ

        g_draw->work.ptTmp[3] = g_draw->work.ptTmp[2];
        g_draw->work.nOpAfter = 1;

        drawXorBezier(FALSE);

        return FALSE;
    }
    else
    {
        //------- 終了

        DPOINT pt[4];
        int i;

        g_draw->pimgXor->free();

        //イメージ位置

        for(i = 0; i < 4; i++)
            g_draw->view.winToimg(&pt[i].x, &pt[i].y, g_draw->work.ptTmp[i].x, g_draw->work.ptTmp[i].y);

        //描画

        setBeforeDraw(g_draw->work.nOpToolNo);
        beginDraw();

        getCurImg()->drawBezier(pt, g_draw->tool.headtail[1], (g_draw->work.nOpToolNo == TOOL_BRUSH));

        endDraw();

        return TRUE;
    }
}

//! キャンセル

BOOL opCancel_bezier()
{
    drawXorBezier(TRUE);

    g_draw->pimgXor->free();

    return TRUE;
}


};
