/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include "drawdat.h"

#include "CMainWin.h"
#include "CToolWin.h"
#include "CCanvasWin.h"
#include "CLayerWin.h"
#include "COptionWin.h"
#include "CStatusBar.h"
#include "CProgressDlg.h"

#include "CXImage.h"
#include "CImage8.h"
#include "CImage32.h"
#include "CImageRGB16.h"
#include "CTileImageRGBA.h"
#include "CTileImageGray.h"
#include "CTileImageA16.h"
#include "CTileImageA1.h"

#include "CLayerList.h"
#include "CLayerItem.h"
#include "CUndo.h"
#include "CResizeCanvas.h"

#include "CConfig.h"

#include "global.h"

#include "draw_main.h"
#include "draw_opfunc.h"
#include "draw_calc.h"
#include "draw_update.h"

#include "CApp.h"

#include "AXFile.h"
#include "AXUtilFile.h"


namespace draw
{


//==============================
// 初期化
//==============================


//! 設定ファイル読み込み後の初期化

void initAfterConfig()
{
    int i;

    //色

    g_draw->colDraw.set(g_conf->dwDrawCol);
    g_draw->colBack.set(g_conf->dwBackCol);

    g_draw->view.colImgBk.set(g_conf->dwImgBkCol);

    g_draw->view.colBkCheck[0].set(g_conf->dwBkCheckCol[0]);
    g_draw->view.colBkCheck[1].set(g_conf->dwBkCheckCol[1]);

    for(i = 0; i < CConfig::COLMASK_NUM; i++)
        g_draw->drawinfo.colMask[i].set(g_conf->dwColMask[i]);

    //オプションテクスチャ画像

    loadOptTexImg();

    //アンドゥ

    g_draw->pundo->setMaxUndoCnt(g_conf->nUndoCnt);
}

//! メインウィンドウ表示後の初期化

void initAfterShow()
{
    //イメージサイズから表示倍率計算

    g_draw->view.szCanvas.w = CANVASAREA->getWidth();
    g_draw->view.szCanvas.h = CANVASAREA->getHeight();

    g_draw->view.nScale = getScaleWithinCanvas();

    changeScaleAndRotate(TRUE);
}


//==========================
// イメージ
//==========================


//! 新規イメージ
/*!
    @param dpi       0 以下でデフォルト
    @param bAddLayer 新規レイヤを作成するか（APD読み込み時などはFALSE）
*/

BOOL newImage(int w,int h,int dpi,BOOL bAddLayer)
{
    CLayerItem *p;

    if(w > IMGSIZE_MAX || h > IMGSIZE_MAX) return FALSE;

    if(w < 1) w = 1;
    if(h < 1) h = 1;

    //UNDO・レイヤ削除

    g_draw->pundo->deleteAllDat();
    g_draw->pundo->setChangeNone();

    g_draw->player->clearItem();

    //情報

    g_draw->nImgW     = w;
    g_draw->nImgH     = h;
    g_draw->nImgDPI   = (dpi <= 0)? 96: dpi;
    g_draw->nSaveTPCol = -1;
    g_draw->pcurlayer  = NULL;

    //作業用

    if(!g_draw->pimgBlend->create(w, h)) return FALSE;

    changeImageSize();

    //レイヤ

    if(bAddLayer)
    {
        //デフォルトレイヤ追加

        p = g_draw->player->addNewCanvasLayer(w, h);
        if(!p) return FALSE;

        g_draw->pcurlayer = p;
    }

    return TRUE;
}

//! 画像読み込みエラー時の処理
/*!
    @return TRUEで更新、FALSEで更新なし
*/

BOOL errLoadImage()
{
    BOOL bLayerZero = !g_draw->player->getTop();

    if(bLayerZero || !g_draw->pcurlayer)
    {
        //新規イメージ

        if(bLayerZero)
            newImage(400, 400, -1, TRUE);

        //カレントレイヤ

        if(!g_draw->pcurlayer)
            g_draw->pcurlayer = g_draw->player->getTopItem();

        return TRUE;
    }

    return FALSE;
}

//! 画像から読み込んで作成

int loadImage(const AXString &filename,CProgressDlg *pdlg)
{
    CTileImage::IMGLOADINFO info;
    CTileImage *pimg;
    int ret = LOADERR_SUCCESS;

    //読み込み用確保

    pimg = new CTileImageRGBA;

    //読み込み

    if(!pimg->loadImage(filename, &info, pdlg))
    {
        ret = LOADERR_ETC;
        goto ERR;
    }

    //サイズ確認

    if(info.nWidth > IMGSIZE_MAX || info.nHeight > IMGSIZE_MAX)
    {
        ret = LOADERR_IMGSIZE;
        goto ERR;
    }

    //新規キャンバス

    if(!newImage(info.nWidth, info.nHeight, info.nDPI, TRUE))
    {
        ret = LOADERR_ETC;
        goto ERR;
    }

    //レイヤイメージ置き換え

    g_draw->pcurlayer->replaceImg(pimg);

    //透過色

    g_draw->nSaveTPCol = info.nTPCol;

    return ret;

ERR:
    delete pimg;
    return ret;
}


//==============================
// UNDO
//==============================


//! UNDOの更新フラグOFF

void setUndoChangeOff()
{
    g_draw->pundo->setChangeNone();
}

//! UNDO最大回数セット

void setUndoMaxCnt(int cnt)
{
    g_draw->pundo->setMaxUndoCnt(cnt);
}

//! イメージが変更されているか

BOOL isChangeImage()
{
    return g_draw->pundo->isChange();
}

//! アンドゥ・リドゥ実行

BOOL undoRedo(BOOL bRedo,LPINT pType,AXRectSize *prcs)
{
    AXUndo::RETURN ret;

    if(bRedo)
        ret = g_draw->pundo->redo();
    else
        ret = g_draw->pundo->undo();

    g_draw->pundo->getUpdateParam(pType, prcs);

    return (ret == AXUndo::RET_OK);
}


//==============================
//
//==============================


//! カレントレイヤのイメージ取得

CTileImage *getCurImg()
{
    return g_draw->pcurlayer->m_pimg;
}

//! 指定カラータイプの CTileImage を確保

CTileImage *allocTileImage(int coltype)
{
    switch(coltype)
    {
        case CTileImage::COLTYPE_RGBA:
            return new CTileImageRGBA;
        case CTileImage::COLTYPE_GRAY:
            return new CTileImageGray;
        case CTileImage::COLTYPE_A_16BIT:
            return new CTileImageA16;
        case CTileImage::COLTYPE_A_1BIT:
            return new CTileImageA1;
    }

    return NULL;
}

//! カレントレイヤのイメージの描画可能範囲を取得

void getCurImgDrawRect(AXRect *prc)
{
    if(g_draw->pcurlayer->isFolder())
        prc->set(0, 0, g_draw->nImgW - 1, g_draw->nImgH - 1);
    else
        g_draw->pcurlayer->m_pimg->getEnableDrawRectPixel(prc);
}

//! pimgBlend にイメージ合成（保存時など用）

void blendImage()
{
    AXRectSize rcs;

    rcs.set(0, 0, g_draw->nImgW, g_draw->nImgH);

    //背景

    g_draw->pimgBlend->clear(g_draw->view.colImgBk);

    //イメージ合成

    update_blendImage(rcs);
}

//! 指定位置の合成後の色取得
/*!
    @return FALSE でイメージ範囲外
*/

BOOL getPixelBlendColor(RGBFIX15 *pdst,int x,int y)
{
    CLayerItem *p;
    RGBAFIX15 colSrc;
    RGBFIX15 col;

    col = g_draw->view.colImgBk;

    for(p = g_draw->player->getBottomVisibleImg(); p; p = p->prevVisibleImg())
    {
        p->m_pimg->getPixel(&colSrc, x, y);

        //レイヤの不透明度適用

        colSrc.a = colSrc.a * p->getViewOpacity() >> 7;

        //合成

        if(colSrc.a)
        {
            g_draw->funcBlendCol[p->m_nBlendMode](&colSrc, col);

            col.r = ((colSrc.r - col.r) * colSrc.a >> 15) + col.r;
            col.g = ((colSrc.g - col.g) * colSrc.a >> 15) + col.g;
            col.b = ((colSrc.b - col.b) * colSrc.a >> 15) + col.b;
        }
    }

    *pdst = col;

    return (x >= 0 && x < g_draw->nImgW && y >= 0 && y < g_draw->nImgH);
}

//! オプションウィンドウのテクスチャ画像読み込み

BOOL loadOptTexImg()
{
    AXString str;
    CImage32 img;
    CImage32::LOADINFO info;

    g_draw->pimg8OptTex->free();

    if(g_conf->strOptTexPath.isEmpty()) return FALSE;

    //パス

    if((g_conf->strOptTexPath)[0] == '/')
    {
        axapp->getResourcePath(&str, "texture");
        str.path_add(g_conf->strOptTexPath.at(1));
    }
    else
    {
        str = g_conf->strUserTexDir;
        str.path_add(g_conf->strOptTexPath);
    }

    //読み込み(32bit)

    if(!img.loadImage(str, &info))
        return FALSE;

    //8bit変換

    return g_draw->pimg8OptTex->createFrom32bit(img, info.bAlpha);
}


//==============================
// 変更時
//==============================


//! イメージサイズ変更時（新規/開く/サイズ変更 時)

void changeImageSize()
{
    g_draw->drawinfo.nImgW = g_draw->nImgW;
    g_draw->drawinfo.nImgH = g_draw->nImgH;
}

//! 途中でのキャンバスイメージサイズ変更時

BOOL changeCanvasImgSize(int w,int h)
{
    //作業用サイズ変更

    if(!g_draw->pimgBlend->create(w, h))
    {
        g_draw->pimgBlend->create(g_draw->nImgW, g_draw->nImgH);
        return FALSE;
    }

    //

    g_draw->nImgW = w;
    g_draw->nImgH = h;

    return TRUE;
}

//! ファイル変更時（開く/新規作成 時）

void changeFile()
{
    //選択解除

    selRelease(FALSE);

    //表示倍率・回転

    g_draw->view.nScale = getScaleWithinCanvas();
    g_draw->view.nAngle = 0;
    g_draw->view.calcParam();

    TOOLWIN->changeScale();
    TOOLWIN->changeRotate();
}

//! ツール変更

void changeTool(int no)
{
    g_draw->tool.toolno = no;

    STATUSBAR->setLabel_help();

    TOOLWIN->changeTool();
    OPTIONWIN->changeTool();

    //ツールのカーソルセット

    CANVASAREA->setCursorTool();
}

//! ツールサブタイプ変更

void changeToolSubType(int no)
{
    g_draw->tool.toolSubNo[g_draw->tool.toolno] = no;

    STATUSBAR->setLabel_help();
}

//! キャンバスウィンドウのサイズが変わった時
/*!
    @param pct キャンバス中央のイメージ位置(NULLで現在の中央位置取得)
*/

void changeCanvasWinSize(BOOL bReDraw,AXPoint *pct)
{
    AXPoint pt;

    //中央のイメージ位置

    if(pct)
        pt = *pct;
    else
        g_draw->view.getImgPosCanvasMid(&pt);

    //サイズ取得

    g_draw->view.szCanvas.w = CANVASAREA->getWidth();
    g_draw->view.szCanvas.h = CANVASAREA->getHeight();

    //イメージ再作成

    g_draw->pimgCanvas->recreate(g_draw->view.szCanvas.w, g_draw->view.szCanvas.h, 32, 32);

    //スクロール位置

    g_draw->view.ptBaseImg = pt;
    g_draw->view.ptScr.zero();

    //

    CANVASWIN->setScroll();

    if(bReDraw) updateCanvas(TRUE);
}

//! 拡大率が変わった時
/*
    - キャンバスのスクロールバー範囲が変更される。
*/

void changeScale(int scale,BOOL bRedraw,BOOL bHiQuality,BOOL bScrReset)
{
    AXPoint pt;

    /* ツールウィンドウのバーなどで呼び出す場合、マウスを離した際に低品質表示->高品質表示にするため、
       常に更新を実行するようにすること。 */

    //中央位置

    if(bScrReset) g_draw->view.getImgPosCanvasMid(&pt);

    //セット

    if(scale < SCALE_MIN) scale = SCALE_MIN;
    else if(scale > SCALE_MAX) scale = SCALE_MAX;

    g_draw->view.nScale = scale;

    //

    if(bScrReset)
    {
        g_draw->view.ptBaseImg = pt;
        g_draw->view.ptScr.zero();
    }

    g_draw->view.calcParam();

    if(bRedraw) updateCanvas(TRUE, bHiQuality);

    CANVASWIN->setScroll();
    TOOLWIN->changeScale();
}

//! 虫眼鏡ツールでのキャンバス拡大縮小

void changeScaleToolUpDown(BOOL bDown)
{
    if((bDown && g_draw->view.nScale != SCALE_MIN) ||
       (!bDown && g_draw->view.nScale != SCALE_MAX))
    {
        g_draw->view.winToimg(&g_draw->view.ptBaseImg, g_draw->work.ptWinDown.x, g_draw->work.ptWinDown.y);

        g_draw->view.nScale  = getScaleUpDown(!bDown);

        g_draw->view.ptScr.x = g_draw->view.szCanvas.w / 2 - (int)g_draw->work.ptWinDown.x;
        g_draw->view.ptScr.y = g_draw->view.szCanvas.h / 2 - (int)g_draw->work.ptWinDown.y;

        g_draw->view.calcParam();

        CANVASWIN->setScroll();
        CANVASWIN->setScrollPos();

        updateCanvas(TRUE);
        TOOLWIN->changeScale();
    }
}

//! キャンバス回転が変わった時

void changeRotate(int angle,BOOL bRedraw,BOOL bHiQuality,BOOL bScrReset)
{
    //セット

    if(angle < -18000) angle += 36000;
    else if(angle > 18000) angle -= 36000;

    g_draw->view.nAngle = angle;

    //

    if(bScrReset) setScroll_reset();

    g_draw->view.calcParam();

    TOOLWIN->changeRotate();

    //更新

    if(bRedraw) updateCanvas(TRUE, bHiQuality);
}

//! 表示倍率と回転変更時
/*!
    初期表示時/キャンバスリセット時
*/

void changeScaleAndRotate(BOOL bRedraw)
{
    AXPoint pt;

    pt.x = g_draw->nImgW >> 1;
    pt.y = g_draw->nImgH >> 1;

    g_draw->view.calcParam();

    changeCanvasWinSize(FALSE, &pt);

    TOOLWIN->changeScale();
    TOOLWIN->changeRotate();

    if(bRedraw)
        updateCanvas(TRUE);
}

//! 他ウィンドウからのスクロール位置変更時
/*!
    キャンバス左右反転/プレビューからの位置変更 時

    @param pt イメージの基準位置
*/

void changeScrollPos(const AXPoint &pt,BOOL bReDraw)
{
    g_draw->view.ptBaseImg = pt;
    g_draw->view.ptScr.zero();

    CANVASWIN->setScroll();

    if(bReDraw) updateCanvas(TRUE);
}

//! 描画色変更

void changeDrawCol(DWORD col)
{
    g_conf->dwDrawCol = col & 0xffffff;

    g_draw->colDraw.set(col);
}

//! 色マスクの色変更
/*
    col : 上位8Bitが 0 で色なし、0以外で色あり
*/

void changeColMaskColor(int no,DWORD col)
{
    g_conf->dwColMask[no] = col;

    g_draw->drawinfo.colMask[no].set(col);
}


//================================
// コマンド
//================================


//! 表示倍率１段階上げる/下げる

void scaleUpDown(BOOL bUp)
{
    changeScale(getScaleUpDown(bUp));
}

//! キャンバス左右反転

void canvasHRev()
{
    AXPoint pt;

    g_draw->view.getImgPosCanvasMid(&pt);

    g_draw->view.bHRev ^= 1;

    changeScrollPos(pt, FALSE);      //反転後も現在のキャンバス中央になるように
    changeRotate(-g_draw->view.nAngle, TRUE, TRUE, FALSE); //回転も反転
}

//! イメージ全体が見えるように表示倍率変更・回転リセット

void resetCanvasView()
{
    g_draw->view.nScale = getScaleWithinCanvas();
    g_draw->view.nAngle = 0;

    setScroll_default();

    changeScaleAndRotate(TRUE);
}

//! 描画色/背景色入れ替え

void toggleDrawCol()
{
    DWORD dw;

    dw = g_conf->dwDrawCol;
    g_conf->dwDrawCol = g_conf->dwBackCol;
    g_conf->dwBackCol = dw;

    //

    g_draw->colDraw.set(g_conf->dwDrawCol);
    g_draw->colBack.set(g_conf->dwBackCol);
}

//! キャンバスサイズ変更

BOOL resizeCanvas(int w,int h,int topx,int topy)
{
    int bkw,bkh;

    bkw = g_draw->nImgW;
    bkh = g_draw->nImgH;

    //作業用変更

    if(!changeCanvasImgSize(w, h)) return FALSE;

    //UNDO

    g_draw->pundo->add_resizeCanvas(topx, topy, bkw, bkh);

    //処理（イメージのオフセット位置追加）

    g_draw->player->resizeCanvas(topx, topy);

    //

    selRelease(FALSE);

    return TRUE;
}

//(スレッド) キャンバス拡大縮小

void *thread_scaleCanvas(void *pParam)
{
    CProgressDlg *pDlg = (CProgressDlg *)pParam;
    int *pVal,ret;
    AXSize sizeOld,sizeNew;
    CResizeCanvas rescanv;

    pVal = (int *)pDlg->m_pParam1;

    sizeOld.w = g_draw->nImgW;
    sizeOld.h = g_draw->nImgH;
    sizeNew.w = pVal[0];
    sizeNew.h = pVal[1];

    //レイヤ合成

    draw::blendImage();

    //拡大縮小 (pimgBlend 自体がリサイズされる)

    ret = rescanv.run(g_draw->pimgBlend, sizeOld, sizeNew, pVal[2], pDlg);
    if(!ret) goto END;

    //pimgBlend からレイヤ用イメージ作成

    ret = ((CTileImage *)(pDlg->m_pParam2))->createFromImg(*g_draw->pimgBlend, pDlg);

END:
    pDlg->endThread(ret);

    return NULL;
}

//! キャンバス拡大縮小

BOOL scaleCanvas(int w,int h,int dpi,int type)
{
    CProgressDlg *pDlg;
    CTileImage *pimg;
    int val[3];

    //新レイヤ用イメージ作成

    pimg = allocTileImage(CLayerItem::COLTYPE_RGBA);

    //スレッド処理

    val[0] = w;
    val[1] = h;
    val[2] = type;

    pDlg = new CProgressDlg(MAINWIN, thread_scaleCanvas, val, pimg);

    if(!pDlg->run())
    {
        delete pimg;
        updateImage();
        return FALSE;
    }

    //UNDO

    g_draw->pundo->add_scaleCanvas();

    //イメージ情報変更

    g_draw->nImgW = w;
    g_draw->nImgH = h;

    if(dpi != -1)
        g_draw->nImgDPI = dpi;

    //レイヤ

    g_draw->player->clearItem();

    g_draw->pcurlayer = g_draw->player->addLayer(NULL);

    g_draw->pcurlayer->m_strName = "layer0";
    g_draw->pcurlayer->replaceImg(pimg);

    //

    selRelease(FALSE);

    return TRUE;
}


//================================
// スクロール
//================================


//! スクロール位置をデフォルトに（画像の中央を基準とする）

void setScroll_default()
{
    g_draw->view.ptBaseImg.x = g_draw->nImgW >> 1;
    g_draw->view.ptBaseImg.y = g_draw->nImgH >> 1;

    g_draw->view.ptScr.zero();
}

//! スクロール位置を現在のキャンバス中央にリセット

void setScroll_reset()
{
    g_draw->view.getImgPosCanvasMid(&g_draw->view.ptBaseImg);

    g_draw->view.ptScr.zero();

    CANVASWIN->setScroll();
}


//================================
// 選択範囲
//================================


//! 選択範囲・タイル配列確保

BOOL selAllocArray()
{
    AXRect rc;

    if(g_draw->pimgSel->isExist())
        return TRUE;
    else
    {
        getCurImgDrawRect(&rc);

        return g_draw->pimgSel->create(rc);
    }
}

//! 選択範囲・透明部分のタイルを解放（全て透明なら削除）

void selFreeEmpty()
{
    if(g_draw->pimgSel->freeEmptyTile())
    {
        g_draw->pimgSel->free();

        g_draw->work.rcfSel.clear();
    }
}

//! 選択範囲があるか

BOOL isSelExist()
{
    return (g_draw->work.rcfSel.flag && g_draw->pimgSel->isExist());
}

//! 選択解除

void selRelease(BOOL bUpdate)
{
    //イメージ解放

    g_draw->pimgSel->free();

    //キャンバス更新

    if(bUpdate)
        updateCanvas(g_draw->work.rcfSel);

    g_draw->work.rcfSel.clear();
}

//! すべて選択

void selAll()
{
    AXRect rc;

    g_draw->pimgSel->free();

    if(selAllocArray())
    {
        //描画

        getCurImgDrawRect(&rc);

        g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_create;

        g_draw->pimgSel->drawFillBox(rc.left, rc.top, rc.width(), rc.height(), 100);

        //

        g_draw->work.rcfSel.set(rc.left, rc.top, rc.right, rc.bottom);

        updateCanvas(g_draw->work.rcfSel);
    }
}

//! 選択反転

void selInverse()
{
    AXRect rc;

    if(selAllocArray())
    {
        //反転

        getCurImgDrawRect(&rc);

        g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_subdraw;
        g_draw->drawinfo.funcColor     = &CTileImage::colfunc_inverse_alpha;

        g_draw->drawinfo.rcfDraw.clear();

        g_draw->pimgSel->drawFillBox(rc.left, rc.top, rc.width(), rc.height(), 100);

        //透明部分を解放

        selFreeEmpty();

        //範囲

        g_draw->pimgSel->getExistImgRectPx(&g_draw->work.rcfSel);

        //更新

        updateCanvas(g_draw->drawinfo.rcfDraw);
    }
}

//! 塗りつぶし/消去

void selPaintErase(BOOL bErase)
{
    AXRect rc;

    if(checkDrawLayer()) return;

    //

    setBeforeDraw(-1);  //マスクなどだけセット

    g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixelDraw;
    g_draw->drawinfo.funcColor     = (bErase)? &CTileImage::colfunc_erase: &CTileImage::colfunc_overwrite;

    //--------

    beginDraw();

    //範囲なしの場合は全体

    if(isSelExist())
        g_draw->work.rcfSel.toRect(&rc);
    else
        getCurImgDrawRect(&rc);

    //描画

    g_draw->pcurlayer->m_pimg->drawFillBox(rc.left, rc.top, rc.width(), rc.height(), 100);

    endDraw();
}

//! 選択範囲コピー/切り取り

void selCopy(BOOL bCut)
{
    AXRect rc;
    AXString str;
    int check;

    //選択範囲なし

    if(!isSelExist()) return;

    //フォルダ時は除く。ロック時はコピーのみ

    check = checkDrawLayer();
    if(check == CHECKDRAW_FOLDER) return;

    if(check == CHECKDRAW_LOCK) bCut = FALSE;

    //コピー用イメージ作成

    g_draw->work.rcfSel.toRect(&rc);

    if(!allocTmpTileImg(0, rc)) return;

    //

    CANVASAREA->setCursorWait();

    //コピー/切り取り

    if(bCut)
    {
        beginDrawUndo();

        setBeforeDraw_clearMask();
        g_draw->drawinfo.funcColor = &CTileImage::colfunc_overwrite;
    }

    if(bCut)
        g_draw->pimgTmp[0]->copyCutSelectImage(g_draw->pcurlayer->m_pimg, g_draw->pimgSel);
    else
        g_draw->pimgTmp[0]->copySelectImage(g_draw->pcurlayer->m_pimg, g_draw->pimgSel);

    if(bCut)
        commonAfterDraw();

    //ファイルに出力

    str = CAPP->getTmpDir();
    str.path_add("copy");

    g_draw->pimgTmp[0]->saveCopyFile(str, g_draw->pcurlayer->m_dwCol);

    //

    freeTmpTileImg();

    CANVASAREA->restoreCursorTool();
}

//! 新規レイヤ貼り付け

void selPaste()
{
    AXString str;
    AXFile file;
    CTileImage *pimg;
    BYTE coltype;
    DWORD col;
    CLayerItem *p;

    str = CAPP->getTmpDir();
    str.path_add("copy");

    //ファイルが存在するか

    if(!AXIsExistFile(str)) return;

    //読み込み

    if(!file.openRead(str)) return;

    file.setEndian(AXFile::ENDIAN_BIG);

    file.read(&coltype, 1);
    file.readDWORD(&col);

    pimg = allocTileImage(coltype);

    if(!pimg->loadCopyFile(&file)) goto ERR;

    file.close();

    //新規レイヤ

    p = g_draw->player->addLayer(g_draw->pcurlayer);
    if(!p) goto ERR;

    p->replaceImg(pimg);
    p->setLayerCol(col);

    p->m_strName = "paste";

    //カレントレイヤ

    g_draw->pcurlayer = p;

    //更新

    g_draw->pundo->add_layerNew();

    updateLayerItemRect(p);
    LAYERWIN->updateLayerAll();

    return;

ERR:
    delete pimg;
}

//! 選択範囲1px拡張/縮小

void selInflateDeflate(BOOL bInflate)
{
    if(!isSelExist()) return;

    //

    CANVASAREA->setCursorWait();

    g_draw->drawinfo.funcColor = &CTileImage::colfunc_overwrite;
    g_draw->drawinfo.rcfDraw.clear();

    g_draw->pimgSel->edgeA1(bInflate);

    CANVASAREA->restoreCursorTool();

    //

    g_draw->work.rcfSel = g_draw->drawinfo.rcfDraw;

    if(!bInflate)
        selFreeEmpty();

    updateRect(g_draw->drawinfo.rcfDraw);
}

};
