/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    PSD 読み込み/保存
*/


#include "CProgressDlg.h"

#include "CImageRGB16.h"
#include "CTileImage.h"
#include "CLayerList.h"
#include "CLayerItem.h"

#include "AXByteString.h"
#include "AXPSDSave.h"
#include "AXPSDLoad.h"
#include "AXUtilFile.h"

#include "draw_main.h"
#include "draw_file.h"

#include "drawdat.h"
#include "global.h"


namespace draw
{

//---------------------

DWORD g_dwPSDBlendMode[] = {
    axpsd::BLEND_NORMAL, axpsd::BLEND_MULTIPLY, axpsd::BLEND_LINEAR_DODGE,
    axpsd::BLEND_SUBTRACT, axpsd::BLEND_SCREEN, axpsd::BLEND_OVERLAY,
    axpsd::BLEND_HARD_LIGHT, axpsd::BLEND_SOFT_LIGHT, axpsd::BLEND_DODGE,
    axpsd::BLEND_BURN, axpsd::BLEND_LINEAR_BURN, axpsd::BLEND_VIVID_LIGHT,
    axpsd::BLEND_LINEAR_LIGHT, axpsd::BLEND_PIN_LIGHT, axpsd::BLEND_DARKEN,
    axpsd::BLEND_LIGHTEN, axpsd::BLEND_DIFFERENCE
};

//---------------------

int _loadPSD_layer(AXPSDLoad &psd,CProgressDlg *pProgDlg);
int _loadPSD_alphach(AXPSDLoad &psd,CTileImage *pimg,const AXRectSize &rcs,CProgressDlg *pProgDlg);
int _loadPSD_oneimg(AXPSDLoad &psd,CProgressDlg *pProgDlg);

BOOL _savePSD_layer(AXPSDSave &psd,int nLayerCnt,CProgressDlg *pProgDlg);

//---------------------



//=================================
// 読み込み
//=================================


//! PSD読み込み
/*!
    @return AXPSDLoad::ERRCODE
*/

int loadPSD(const AXString &filename,CProgressDlg *pProgDlg)
{
    AXPSDLoad psd;
    int ret,dpih,dpiv;

    //開く

    ret = psd.openFile(filename);
    if(ret) return ret;

    //画像サイズ確認

    if(psd.getImgWidht() > IMGSIZE_MAX || psd.getImgHeight() > IMGSIZE_MAX)
        return 1000;

    //新規イメージ

    if(!newImage(psd.getImgWidht(), psd.getImgHeight(), -1, FALSE))
        return AXPSDLoad::ERR_MEMORY;

    //画像リソースデータ

    ret = psd.beginRes();
    if(ret) return ret;

    if(psd.readRes_resolution(&dpih, &dpiv))
        g_draw->nImgDPI = dpih;

    psd.endRes();

    //レイヤ

    ret = psd.beginLayer();
    if(ret) return ret;

    if(psd.getLayerCnt() == 0 || psd.getBits() == 1)
    {
        //レイヤなしの場合は一枚絵から読み込み

        psd.endLayer();

        ret = _loadPSD_oneimg(psd, pProgDlg);
    }
    else
        //レイヤ読み込み
        ret = _loadPSD_layer(psd, pProgDlg);

    if(ret) return ret;

    //

    psd.close();

    //----------

    g_draw->pcurlayer = g_draw->player->getTopItem();

    return AXPSDLoad::ERR_SUCCESS;
}

//! (sub) レイヤ読み込み

int _loadPSD_layer(AXPSDLoad &psd,CProgressDlg *pProgDlg)
{
    AXPSDLoad::LAYERINFO info;
    TILEIMGINFO imginfo;
    AXRectSize rcs;
    CLayerItem *p;
    int i,ret,ix,iy,coltype,bGrayScale;
    LPBYTE pSrc;
    RGBAFIX15 col;

    bGrayScale = (psd.getColMode() == axpsd::COLMODE_GRAYSCALE);

    //レイヤカラータイプ

    coltype = (bGrayScale)? CLayerItem::COLTYPE_GRAY: CLayerItem::COLTYPE_RGBA;

    //-------- レイヤ情報

    p = NULL;

    for(i = psd.getLayerCnt(); i > 0; i--)
    {
        ret = psd.readLayerInfo(&info);
        if(ret) return ret;

        //タイルイメージ情報

        if(info.nLeft >= info.nRight || info.nTop >= info.nBottom)
        {
            //全体が透明の状態

            imginfo.nOffX = imginfo.nOffY = 0;
            imginfo.nArrayW = imginfo.nArrayH = 1;
        }
        else
        {
            imginfo.nOffX   = info.nLeft;
            imginfo.nOffY   = info.nTop;
            imginfo.nArrayW = (info.nRight - info.nLeft + 63) / 64;
            imginfo.nArrayH = (info.nBottom - info.nTop + 63) / 64;
        }

        //レイヤ作成

        p = g_draw->player->addLayer(p, coltype, &imginfo, 0, 0);
        if(!p) return AXPSDLoad::ERR_MEMORY;

        //名前

        p->m_strName.setLocal(info.szName);

        //不透明度

        p->m_nOpacity = (int)(info.btOpacity * 128.0 / 255.0 + 0.5);

        //非表示

        if(info.bHide) p->m_dwFlags &= ~CLayerItem::FLAG_VISIBLE;

        //合成モード

        for(ix = 0; ix < BLENDMODE_NUM; ix++)
        {
            if(g_dwPSDBlendMode[ix] == info.dwBlend) break;
        }

        p->m_nBlendMode = (ix == BLENDMODE_NUM)? 0: ix;
    }

    //---------- レイヤイメージ

    ret = psd.beginLayerImageTop();
    if(ret) return ret;

    pProgDlg->setProgMax(psd.getLayerCnt() * 16);

    for(p = g_draw->player->getBottomItem(); p; p = p->prev())
    {
        //空イメージの場合

        if(!psd.beginLayerImageEach(&rcs))
        {
            pProgDlg->addProgPos(16);
            continue;
        }

        //アルファチャンネルを先に読み込み

        ret = _loadPSD_alphach(psd, p->m_pimg, rcs, pProgDlg);
        if(ret) return ret;

        //RGBチャンネル読み込み (グレイスケールの場合は 1ch のみ)

        for(i = 0; i < 3; i++)
        {
            //チャンネル

            ret = psd.beginLayerImageCh(i);

            if(ret == AXPSDLoad::ERR_NO_CHANNEL)
            {
                pProgDlg->addProgPos(4);
                continue;
            }
            else if(ret)
                return ret;

            //イメージ

            pProgDlg->beginProgSub((bGrayScale)? 12: 4, rcs.h);

            for(iy = 0; iy < rcs.h; iy++)
            {
                ret = psd.readLayerImageChLine();
                if(ret) return ret;

                pSrc = psd.getLineBuf();

                for(ix = 0; ix < rcs.w; ix++, pSrc++)
                {
                    p->m_pimg->getPixel(&col, rcs.x + ix, rcs.y + iy);

                    if(col.a)
                    {
                        col.c[i] = _8TO16COL(*pSrc);

                        if(i == 0)
                            col.c[1] = col.c[2] = col.c[0];

                        p->m_pimg->setPixel_create(rcs.x + ix, rcs.y + iy, col);
                    }
                }

                pProgDlg->incProgSub();
            }

            //

            if(bGrayScale) break;
        }
    }

    psd.endLayer();

    return AXPSDLoad::ERR_SUCCESS;
}

//! イメージ、アルファチャンネル読み込み

int _loadPSD_alphach(AXPSDLoad &psd,CTileImage *pimg,const AXRectSize &rcs,CProgressDlg *pProgDlg)
{
    int ix,iy,ret;
    LPBYTE pSrc;
    RGBAFIX15 col;

    col.r = col.g = col.b = 0;

    pProgDlg->beginProgSub(4, rcs.h);

    //Aチャンネル取得

    ret = psd.beginLayerImageCh(AXPSDLoad::CHANNELID_ALPHA);

    //

    if(ret == AXPSDLoad::ERR_SUCCESS)
    {
        //Aチャンネルがある場合

        for(iy = 0; iy < rcs.h; iy++)
        {
            ret = psd.readLayerImageChLine();
            if(ret) return ret;

            pSrc = psd.getLineBuf();

            for(ix = 0; ix < rcs.w; ix++, pSrc++)
            {
                if(*pSrc)
                {
                    col.a = _8TO16COL(*pSrc);

                    pimg->setPixel_create(rcs.x + ix, rcs.y + iy, col);
                }
            }

            pProgDlg->incProgSub();
        }
    }
    else if(ret == AXPSDLoad::ERR_NO_CHANNEL)
    {
        //Aチャンネルがない場合、すべて不透明

        col.a = 0x8000;

        for(iy = 0; iy < rcs.h; iy++)
        {
            for(ix = 0; ix < rcs.w; ix++)
                pimg->setPixel_create(rcs.x + ix, rcs.y + iy, col);

            pProgDlg->incProgSub();
       }
    }
    else
        //他エラー
        return ret;

    return AXPSDLoad::ERR_SUCCESS;
}

//! 一枚絵イメージから読み込み

int _loadPSD_oneimg(AXPSDLoad &psd,CProgressDlg *pProgDlg)
{
    int ret,type,i,ix,iy,cnt,w,h;
    LPBYTE pDst,pSrc,pTmpBuf;
    BYTE c,f;
    CLayerItem *p;

    w = g_draw->nImgW;
    h = g_draw->nImgH;

    //

    if(psd.getBits() == 1)
        type = 0, i = CLayerItem::COLTYPE_GRAY;
    else if(psd.getBits() == 8 && psd.getColMode() == axpsd::COLMODE_GRAYSCALE)
        type = 1, i = CLayerItem::COLTYPE_GRAY;
    else if(psd.getBits() == 8 && psd.getColMode() == axpsd::COLMODE_RGB && psd.getImgChCnt() >= 3)
        type = 2, i = CLayerItem::COLTYPE_RGBA;
    else
        return AXPSDLoad::ERR_FORMAT;

    //レイヤ作成

    p = g_draw->player->addLayer(NULL, i, NULL, w, h);
    if(!p) return AXPSDLoad::ERR_MEMORY;

    p->m_strName = "layer0";

    //--------- 各チャンネル
    /* pimgBlend のバッファに RGBA(BYTE) を書き込んでから変換  */

    ret = psd.beginImage();
    if(ret) return ret;

    //

    pTmpBuf = (LPBYTE)g_draw->pimgBlend->getBuf();

    //1bit or グレイスケール

    if(type < 2)
    {
        pProgDlg->setProgMax(100);
        pProgDlg->beginProgSub(50, h);

        ret = psd.beginImageCh();
        if(ret) return ret;

        pDst = pTmpBuf;
    }

    //

    switch(type)
    {
        //1bit
        case 0:
            for(iy = h; iy; iy--)
            {
                ret = psd.readImageChLine();
                if(ret) return ret;

                pSrc = psd.getLineBuf();

                for(ix = w, f = 0x80; ix; ix--, pDst += 4)
                {
                    c = (*pSrc & f)? 0: 255;

                    pDst[0] = pDst[1] = pDst[2] = c;
                    pDst[3] = 255;

                    f >>= 1;
                    if(f == 0) f = 0x80, pSrc++;
                }

                pProgDlg->incProgSub();
            }
            break;
        //グレイスケール
        case 1:
            for(iy = h; iy; iy--)
            {
                ret = psd.readImageChLine();
                if(ret) return ret;

                pSrc = psd.getLineBuf();

                for(ix = w; ix; ix--, pDst += 4)
                {
                    c = *(pSrc++);

                    pDst[0] = pDst[1] = pDst[2] = c;
                    pDst[3] = 255;
                }

                pProgDlg->incProgSub();
            }
            break;
        //RGB or RGBA
        default:
            cnt = (psd.getImgChCnt() >= 4)? 4: 3;

            pProgDlg->setProgMax(cnt * 10 + 50);

            for(i = 0; i < cnt; i++)
            {
                ret = psd.beginImageCh();
                if(ret) return ret;

                pDst = pTmpBuf + i;

                pProgDlg->beginProgSub(10, h);

                for(iy = h; iy; iy--)
                {
                    ret = psd.readImageChLine();
                    if(ret) return ret;

                    pSrc = psd.getLineBuf();

                    for(ix = w; ix; ix--, pDst += 4)
                        *pDst = *(pSrc++);

                    pProgDlg->incProgSub();
                }
            }

            //Aチャンネルがない場合すべて不透明に

            if(cnt == 3)
            {
                pDst = pTmpBuf + 3;

                for(i = w * h; i; i--, pDst += 4)
                    *pDst = 255;
            }
            break;
    }

    //RGBA8bit から変換

    if(!p->m_pimg->convertFromRGBA8bit(pTmpBuf, w, h, pProgDlg, 50))
        return AXPSDLoad::ERR_MEMORY;

    return AXPSDLoad::ERR_SUCCESS;
}

//! PSD 読み込みエラーコードを draw::LOADERR_* に

int errPSDtoLOAD(int err)
{
    switch(err)
    {
        case AXPSDLoad::ERR_SUCCESS:
            return draw::LOADERR_SUCCESS;
        case AXPSDLoad::ERR_OPENFILE:
            return draw::LOADERR_OPENFILE;
        case AXPSDLoad::ERR_FORMAT:
            return draw::LOADERR_FORMAT;
        case 1000:
            return draw::LOADERR_IMGSIZE;
        default:
            return draw::LOADERR_ETC;
    }
}


//==================================
// 保存
//==================================


//! PSD保存 (レイヤあり)
/*
    RGBA8bit。レイヤフォルダは含まない。
*/

int savePSD(const AXString &filename,CProgressDlg *pProgDlg)
{
    AXPSDSave psd;
    AXPSDSave::INFO info;
    int nLayerCnt,ch,ix,iy;
    LPBYTE pSrc,pDst;

    nLayerCnt = g_draw->player->getNormalLayerCnt();

    pProgDlg->setProgMax(nLayerCnt * 4 + 3);

    //開く

    info.nWidth     = g_draw->nImgW;
    info.nHeight    = g_draw->nImgH;
    info.nImgCh     = 3;
    info.nBits      = 8;
    info.nColMode   = axpsd::COLMODE_RGB;

    if(!psd.openFile(filename, &info)) return SAVERET_ERR;

    //画像リソース

    psd.beginRes();
    psd.writeRes_resolution(g_draw->nImgDPI, g_draw->nImgDPI);
    psd.writeRes_curlayer(0);
    psd.endRes();

    //レイヤ (フォルダしかない場合はレイヤ数 0)

    if(nLayerCnt == 0)
        psd.beginLayer(0);
    else
    {
        if(!_savePSD_layer(psd, nLayerCnt, pProgDlg)) goto ERR;
    }

    //一枚絵

    if(!psd.beginImage()) goto ERR;

    for(ch = 0; ch < 3; ch++)
    {
        psd.beginImageCh();

        pSrc = (LPBYTE)g_draw->pimgBlend->getBuf() + ch;

        for(iy = g_draw->nImgH; iy > 0; iy--)
        {
            pDst = psd.getLineBuf();

            for(ix = g_draw->nImgW; ix > 0; ix--, pSrc += 3)
                *(pDst++) = *pSrc;

            psd.writeImageChLine();
        }

        psd.endImageCh();

        //

        pProgDlg->incProg();
    }

    psd.close();

    return SAVERET_OK;

ERR:
    psd.close();
    AXDeleteFile(filename);
    return SAVERET_ERR;
}

//! (sub) PSD レイヤ出力

BOOL _savePSD_layer(AXPSDSave &psd,int nLayerCnt,CProgressDlg *pProgDlg)
{
    CLayerItem *p,*pBottom;
    AXPSDSave::LAYERINFO info;
    FLAGRECT rcf;
    AXRectSize rcs;
    AXByteString strb;
    int ch,ix,iy;
    RGBAFIX15 col;
    LPBYTE pDst;

    pBottom = g_draw->player->getBottomNormalLayer();

    if(!psd.beginLayer(nLayerCnt)) return FALSE;

    //各レイヤ情報

    for(p = pBottom; p; p = p->prevNormal())
    {
        p->m_pimg->getExistImgRectPx(&rcf);

        p->m_strName.toLocal(&strb, 255);

        info.nLeft     = rcf.x1;
        info.nTop      = rcf.y1;
        info.nRight    = rcf.x2;
        info.nBottom   = rcf.y2;
        info.dwBlend   = g_dwPSDBlendMode[p->m_nBlendMode];
        info.btOpacity = (int)(p->m_nOpacity * 255.0 / 128.0 + 0.5);
        info.bHide     = !p->isVisibleView();
        info.szName    = strb;

        psd.writeLayerInfo(&info);
    }

    //イメージ

    if(!psd.beginLayerImageTop()) return FALSE;

    for(p = pBottom; p; p = p->prevNormal())
    {
        if(!psd.beginLayerImageEach(&rcs))
            //空イメージの場合
            pProgDlg->addProgPos(4);
        else
        {
            for(ch = 0; ch < 4; ch++)
            {
                psd.beginLayerImageCh();

                for(iy = 0; iy < rcs.h; iy++)
                {
                    pDst = psd.getLineBuf();

                    for(ix = 0; ix < rcs.w; ix++)
                    {
                        p->m_pimg->getPixel(&col, rcs.x + ix, rcs.y + iy);

                        *(pDst++) = _16TO8COL(col.c[ch]);
                    }

                    psd.writeLayerImageChLine();
                }

                psd.endLayerImageCh();

                pProgDlg->incProg();
            }
        }
    }

    psd.endLayer();

    return TRUE;
}

//! PSD保存（レイヤなし）
/*
    [type] 0:8bitRGB, 1:Grayscale, 2:monochrome
*/

int savePSD_nolayer(const AXString &filename,int type,CProgressDlg *pProgDlg)
{
    AXPSDSave psd;
    AXPSDSave::INFO info;
    int iy,ix,i,j,w,h;
    BYTE val,f;
    LPBYTE pDst,pSrc;

    w = g_draw->nImgW;
    h = g_draw->nImgH;

    //情報

    info.nWidth  = w;
    info.nHeight = h;
    info.nImgCh  = (type == 0)? 3: 1;
    info.nBits   = (type == 2)? 1: 8;

    if(type == 0)
        info.nColMode = axpsd::COLMODE_RGB;
    else if(type == 1)
        info.nColMode = axpsd::COLMODE_GRAYSCALE;
    else
        info.nColMode = axpsd::COLMODE_BITMAP;

    //開く

    if(!psd.openFile(filename, &info)) return SAVERET_ERR;

    //画像リソース

    psd.beginRes();
    psd.writeRes_resolution(g_draw->nImgDPI, g_draw->nImgDPI);
    psd.endRes();

    //レイヤ(0)

    psd.beginLayer(0);

    //イメージ

    if(!psd.beginImage()) return SAVERET_ERR;

    pSrc = (LPBYTE)g_draw->pimgBlend->getBuf();

    switch(type)
    {
        //8bitRGB
        case 0:
            pProgDlg->beginProgSub(30, h * 3, TRUE);

            for(i = 0; i < 3; i++)
            {
                psd.beginImageCh();

                pSrc = (LPBYTE)g_draw->pimgBlend->getBuf() + i;

                for(iy = h; iy > 0; iy--)
                {
                    pDst = psd.getLineBuf();

                    for(ix = w; ix > 0; ix--, pSrc += 3)
                        *(pDst++) = *pSrc;

                    psd.writeImageChLine();

                    pProgDlg->incProgSub();
                }

                psd.endImageCh();
            }
            break;
        //grayscale
        case 1:
            pProgDlg->beginProgSub(30, h, TRUE);

            psd.beginImageCh();

            for(iy = h; iy > 0; iy--)
            {
                pDst = psd.getLineBuf();

                for(ix = w; ix > 0; ix--, pSrc += 3)
                    *(pDst++) = (pSrc[0] + pSrc[1] + pSrc[2]) / 3;

                psd.writeImageChLine();

                pProgDlg->incProgSub();
            }

            psd.endImageCh();
            break;
        //monochrome (白を0, 白以外を1)
        case 2:
            pProgDlg->beginProgSub(30, h, TRUE);

            psd.beginImageCh();

            for(iy = h; iy > 0; iy--)
            {
                pDst = psd.getLineBuf();
                ix   = w;

                for(i = (w + 7) / 8; i > 0; i--)
                {
                    val = 0;
                    f   = 0x80;

                    for(j = 8; j && ix; j--, ix--, pSrc += 3, f >>= 1)
                    {
                        if(pSrc[0] != 0xff || pSrc[1] != 0xff || pSrc[2] != 0xff)
                            val |= f;
                    }

                    *(pDst++) = val;
                }

                psd.writeImageChLine();

                pProgDlg->incProgSub();
            }

            psd.endImageCh();
            break;
    }

    //

    psd.close();

    return SAVERET_OK;
}

};
