/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    APD(AzPainter) 読み込み/保存
*/


#include "CProgressDlg.h"

#include "CImageRGB16.h"
#include "CTileImage.h"
#include "CLayerList.h"
#include "CLayerItem.h"

#include "AXFileWriteBuf.h"
#include "AXZlib.h"

#include "draw_main.h"
#include "draw_file.h"

#include "drawdat.h"
#include "global.h"


namespace draw
{

//-----------------------

int _loadAPD_ver1(AXFile &file,CProgressDlg *pProgDlg);
int _loadAPD_ver2(AXFile &file,CProgressDlg *pProgDlg);

//-----------------------


//=====================================
// 保存
//=====================================


//! APD保存 (ver2)

int saveAPD(const AXString &filename,CProgressDlg *pProgDlg)
{
    AXFileWriteBuf file;
    AXZlib zlib;
    AXMem memBuf;
    int tilesize,n;
    CLayerItem *p;

    tilesize = sizeof(RGBAFIX15) * 64 * 64;

    if(!zlib.allocBuf(tilesize)) return SAVERET_ERR;
    if(!zlib.initEncBuf(6)) return SAVERET_ERR;

    if(!memBuf.alloc(tilesize)) return SAVERET_ERR;

    //---------

    if(!file.open(filename, 8192)) return SAVERET_ERR;

    //ヘッダ・バージョン

    file.putStr("AZPDATA");
    file.putBYTE(1);

    //全体情報

    file.putBYTE('I');
    file.putDWORDBE(6);

    file.putWORDBE(g_draw->nImgW);
    file.putWORDBE(g_draw->nImgH);
    file.putWORDBE(g_draw->nImgDPI);

    //レイヤ

    file.putBYTE('L');
    file.putWORDBE(g_draw->player->getCnt());

    pProgDlg->setProgMax(g_draw->player->getCnt() * 10);

    for(p = g_draw->player->getTopItem(); p; p = (CLayerItem *)p->nextTreeItem())
    {
        //親のレイヤ番号

        n = g_draw->player->getLayerItemNo(p->parent());

        file.putWORDBE((n == -1)? 0xffff: n);

        //フラグ

        n = 0;
        if(p->isFolder()) n |= 1;
        if(p == g_draw->pcurlayer) n |= 2;

        file.putBYTE(n);

        //情報

        file.putStrLenAndUTF8(p->m_strName);

        file.putBYTE(p->m_nColType);
        file.putBYTE(p->m_nOpacity);
        file.putBYTE(p->m_nBlendMode);
        file.putBYTE(p->m_nAmaskType);
        file.putDWORDBE(p->m_dwCol);
        file.putDWORDBE(p->m_dwFlags);

        //イメージ

        if(p->isNormal())
            p->m_pimg->saveTiles(&file, &zlib, memBuf, pProgDlg, 10);
    }

    file.putWORDBE(0xfffe);

    file.close();

    return SAVERET_OK;
}


//=====================================
// 読み込み
//=====================================


//! APD読み込み

int loadAPD(const AXString &filename,CProgressDlg *pProgDlg)
{
    AXFile file;
    BYTE ver;
    int ret;

    //開く

    if(!file.openRead(filename)) return LOADERR_OPENFILE;

    //ヘッダ

    if(!file.readCompare("AZPDATA")) return LOADERR_FORMAT;

    //バージョン

    file.read(&ver, 1);

    if(ver == 0)
        ret = _loadAPD_ver1(file, pProgDlg);
    else if(ver == 1)
    {
        ret = _loadAPD_ver2(file, pProgDlg);

        g_draw->player->calcViewItemCnt();
    }
    else
        ret = LOADERR_FORMAT;

    //

    file.close();

    return ret;
}

//! APD ver2 読み込み (Big Endian)

int _loadAPD_ver2(AXFile &file,CProgressDlg *pProgDlg)
{
    BYTE btSig,btFlag,btVal[4];
    DWORD dwSize;
    WORD wd[3];
    LONGLONG pos;
    int i,tilesize,bInfo = FALSE;
    AXZlib zlib;
    AXMem memBuf;
    CLayerItem *p;

    file.setEndian(AXFile::ENDIAN_BIG);

    //

    tilesize = sizeof(RGBAFIX15) * 64 * 64;

    if(!zlib.allocBuf(tilesize)) return LOADERR_ETC;
    if(!zlib.initDecBuf()) return LOADERR_ETC;

    if(!memBuf.alloc(tilesize)) return LOADERR_ETC;

    //

    while(1)
    {
        if(file.read(&btSig, 1) == 0) break;

        if(btSig == 'L')
        {
            //------ レイヤ

            if(!bInfo) return LOADERR_ETC;

            //レイヤ数

            file.readWORD(wd);

            pProgDlg->setProgMax(wd[0] * 10);

            //

            while(1)
            {
                //親の番号

                if(!file.readWORD(wd)) return LOADERR_SUCCESS;

                if(wd[0] == 0xfffe) return LOADERR_SUCCESS;

                //フラグ

                file.read(&btFlag, 1);

                //レイヤ作成

                p = g_draw->player->addLayerFromNo((wd[0] == 0xffff)? -1: wd[0], -1);
                if(!p) return LOADERR_ETC;

                if(btFlag & 2) g_draw->pcurlayer = p;

                //情報

                file.readStrLenAndUTF8(&p->m_strName);
                file.read(btVal, 4);
                file.readDWORD(&p->m_dwCol);
                file.readDWORD(&p->m_dwFlags);

                p->m_nColType   = (btVal[0] >= 4)? 0: btVal[0];
                p->m_nOpacity   = btVal[1];
                p->m_nBlendMode = (btVal[2] >= BLENDMODE_NUM)? 0: btVal[2];
                p->m_nAmaskType = (btVal[3] >= 4)? 0: btVal[3];

                //フォルダ時

                if(btFlag & 1) continue;

                //イメージ

                p->m_pimg = allocTileImage(p->m_nColType);

                p->setLayerCol(p->m_dwCol);

                if(!p->m_pimg->loadTiles(&file, &zlib, memBuf, pProgDlg, 10))
                    return LOADERR_ETC;
            }
        }
        else
        {
            //識別子+データサイズ(4)

            file.readDWORD(&dwSize);

            pos = file.getPositionLong();

            if(btSig == 'I')
            {
                //全体情報

                for(i = 0; i < 3; i++)
                    file.readWORD(wd + i);

                if(!newImage(wd[0], wd[1], wd[2], FALSE))
                    return LOADERR_ETC;

                bInfo = TRUE;
            }

            file.seekTop(pos + dwSize);
        }
    }

    return LOADERR_ETC;
}

//! APD ver1 読み込み (Little Endian)

int _loadAPD_ver1(AXFile &file,CProgressDlg *pProgDlg)
{
    AXZlib zlib;
    DWORD dwSize,dwLayerInfoSize;
    WORD wd,wLayerCnt,wLayerSel;
    BYTE bt;
    CLayerItem *p;
    char name[32];
    int i,j,iy,w,h,pitch;
    LPBYTE pTmpBuf,pDst,pd;
    BYTE blendmode[21] = {
        0, 1, 2, 3, 4, 5, 7, 6, 8, 9, 10, 11, 12, 13, 16, 15, 14, 0,0,0,0
    };

    //

    file.setEndian(AXFile::ENDIAN_LITTLE);

    if(!zlib.allocBuf(16 * 1024)) return LOADERR_ETC;

    //------- ヘッダデータ

    //ヘッダサイズ
    file.readDWORD(&dwSize);

    //幅・高さ

    file.readWORD(&wd);
    w = wd;

    file.readWORD(&wd);
    h = wd;

    //レイヤ数
    file.readWORD(&wLayerCnt);
    //レイヤ通算カウント
    file.seekCur(2);
    //カレントレイヤNo
    file.readWORD(&wLayerSel);

    file.seekCur(dwSize - 10);

    //-------- プレビューイメージ

    file.seekCur(4);          //幅・高さ
    file.readDWORD(&dwSize);  //圧縮サイズ
    file.seekCur(dwSize);

    //-------- レイヤ

    //レイヤ情報データサイズ

    file.readDWORD(&dwLayerInfoSize);

    //新規イメージ

    if(!newImage(w, h, -1, FALSE)) return LOADERR_ETC;

    //各レイヤ

    p       = NULL;
    pTmpBuf = (LPBYTE)g_draw->pimgBlend->getBuf();
    pitch   = w << 2;

    pProgDlg->setProgMax(wLayerCnt * 10);

    for(i = wLayerCnt; i; i--)
    {
        //レイヤ追加

        p = g_draw->player->addLayer(p, CLayerItem::COLTYPE_RGBA, NULL, w, h);
        if(!p) return LOADERR_ETC;

        //情報

        file.read(name, 32);
        p->m_strName.setUTF8(name, 31);

        file.read(&bt, 1);

        if(bt >= 21) bt = 0;
        p->m_nBlendMode = blendmode[bt];

        file.read(&bt, 1);
        p->m_nOpacity = bt;

        file.read(&bt, 1);
        p->m_dwFlags = 0;
        if(bt & 1) p->m_dwFlags |= CLayerItem::FLAG_VISIBLE;

        file.seekCur(dwLayerInfoSize - 35);

        //-------- イメージ

        //圧縮サイズ

        file.readDWORD(&dwSize);

        //イメージ読み込み (pimgBlend のバッファに)

        if(!zlib.initDecFile(&file, dwSize)) return LOADERR_ETC;

        pProgDlg->beginProgSub(5, h);

        pDst = pTmpBuf + pitch * (h - 1);

        for(iy = h; iy; iy--, pDst -= pitch)
        {
            if(!zlib.getDecFile(pDst, pitch)) return LOADERR_ETC;

            //BGRA -> RGBA

            pd = pDst;

            for(j = w; j; j--, pd += 4)
            {
                bt = pd[0], pd[0] = pd[2], pd[2] = bt;
            }

            pProgDlg->incProgSub();
        }

        zlib.end();

        //変換

        if(!p->m_pimg->convertFromRGBA8bit(pTmpBuf, w, h, pProgDlg, 5))
            return LOADERR_ETC;
    }

    //カレントレイヤ

    g_draw->pcurlayer = g_draw->player->getLayerItemFromNo(wLayerSel);

    return LOADERR_SUCCESS;
}

};
