/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include <math.h>

#include "drawVIEW.h"


namespace draw
{

VIEW::VIEW()
{
    nScale = 1000;
    nAngle = 0;
    bHRev  = FALSE;

    ptScr.zero();
    ptBaseImg.zero();
    szCanvas.w = szCanvas.h = 100;

    calcParam();
}

//! パラメータ計算

void VIEW::calcParam()
{
    double d;

    d = nScale * 0.001;

    param.dScale    = d;
    param.dScaleDiv = 1.0 / d;

    d = nAngle * M_PI / 18000.0;

    param.dCos    = ::cos(d);
    param.dSin    = ::sin(d);
    param.dCosRev = ::cos(-d);
    param.dSinRev = ::sin(-d);
}

//! ウィンドウ位置 -> イメージ位置

void VIEW::winToimg(double *pX,double *pY,double x,double y)
{
    double xx,yy;

    x += ptScr.x - (szCanvas.w >> 1);
    y += ptScr.y - (szCanvas.h >> 1);

    xx = x * param.dCosRev - y * param.dSinRev;
    yy = x * param.dSinRev + y * param.dCosRev;

    if(bHRev) xx = -xx;

    *pX = xx * param.dScaleDiv + ptBaseImg.x;
    *pY = yy * param.dScaleDiv + ptBaseImg.y;
}

//! ウィンドウ位置 -> イメージ位置(int)

void VIEW::winToimg(AXPoint *pdst,double x,double y)
{
    winToimg(&x, &y, x, y);

    pdst->x = (int)x;
    pdst->y = (int)y;
}

//! イメージ位置 -> ウィンドウ位置

void VIEW::imgTowin(AXPoint *pdst,int x,int y)
{
    double dx,dy;

    dx = (x - ptBaseImg.x) * param.dScale;
    dy = (y - ptBaseImg.y) * param.dScale;

    if(bHRev) dx = -dx;

    pdst->x = (int)(dx * param.dCos - dy * param.dSin) + (szCanvas.w >> 1) - ptScr.x;
    pdst->y = (int)(dx * param.dSin + dy * param.dCos) + (szCanvas.h >> 1) - ptScr.y;
}

//! キャンバスの中央のイメージ位置取得

void VIEW::getImgPosCanvasMid(AXPoint *pdst)
{
    double x,y;

    winToimg(&x, &y, szCanvas.w * 0.5, szCanvas.h * 0.5);

    pdst->x = (int)x;
    pdst->y = (int)y;
}

//! イメージ範囲 -> キャンバスウィンドウ範囲(描画時の更新範囲)
/*!
    @return 範囲外でFALSE
*/

BOOL VIEW::imgTowinRect(AXRectSize *pdst,const AXRectSize &src)
{
    AXPoint pt[4];
    int x1,y1,x2,y2,i;

    //右・下端は+1

    x1 = src.x, y1 = src.y;
    x2 = src.x + src.w, y2 = src.y + src.h;

    //最小・最大

    if(nAngle == 0)
    {
        imgTowin(pt    , x1, y1);
        imgTowin(pt + 1, x2, y2);

        if(pt[0].x < pt[1].x)
            x1 = pt[0].x, x2 = pt[1].x;
        else
            x1 = pt[1].x, x2 = pt[0].x;

        if(pt[0].y < pt[1].y)
            y1 = pt[0].y, y2 = pt[1].y;
        else
            y1 = pt[1].y, y2 = pt[0].y;
    }
    else
    {
        imgTowin(pt    , x1, y1);
        imgTowin(pt + 1, x2, y1);
        imgTowin(pt + 2, x1, y2);
        imgTowin(pt + 3, x2, y2);

        x1 = x2 = pt[0].x;
        y1 = y2 = pt[0].y;

        for(i = 1; i < 4; i++)
        {
            if(x1 > pt[i].x) x1 = pt[i].x;
            if(y1 > pt[i].y) y1 = pt[i].y;
            if(x2 < pt[i].x) x2 = pt[i].x;
            if(y2 < pt[i].y) y2 = pt[i].y;
        }
    }

    //念のため範囲拡張

    x1--, y1--;
    x2++, y2++;

    //範囲外判定

    if(x2 < 0 || y2 < 0 || x1 >= szCanvas.w || y1 >= szCanvas.h)
        return FALSE;

    //調整

    if(x1 < 0) x1 = 0;
    if(y1 < 0) y1 = 0;
    if(x2 >= szCanvas.w) x2 = szCanvas.w - 1;
    if(y2 >= szCanvas.h) y2 = szCanvas.h - 1;

    //

    pdst->set(x1, y1, x2 - x1 + 1, y2 - y1 + 1);

    return TRUE;
}

//! AXRect をキャンバスの範囲内に

void VIEW::rectInCanvas(AXRect *prc)
{
    if(prc->left < 0) prc->left = 0;
    if(prc->top < 0) prc->top = 0;
    if(prc->right >= szCanvas.w) prc->right = szCanvas.w - 1;
    if(prc->bottom >= szCanvas.h) prc->bottom = szCanvas.h - 1;
}

//! AXRect をキャンバスの範囲内にして AXRectSize へ

BOOL VIEW::rectInCanvas(AXRectSize *pdst,const AXRect &rcSrc)
{
    AXRect rc = rcSrc;

    //範囲外

    if(rc.right < 0 || rc.bottom < 0) return FALSE;
    if(rc.left >= szCanvas.w || rc.top >= szCanvas.h) return FALSE;

    //調整

    rectInCanvas(&rc);

    //セット

    pdst->set(rc);

    return TRUE;
}

//! ウィンドウでの移動総数をイメージのpxに変換

void VIEW::getMoveDst(AXPoint *pDst,const AXPoint &src)
{
    pDst->x = (int)((src.x * param.dCosRev - src.y * param.dSinRev) * param.dScaleDiv);
    pDst->y = (int)((src.x * param.dSinRev + src.y * param.dCosRev) * param.dScaleDiv);

    if(bHRev) pDst->x = -(pDst->x);
}


};
