/** 
 * @file SFMT-alti.h 
 *
 * @brief SIMD oriented Fast Mersenne Twister(SFMT)
 * pseudorandom number generator
 *
 * @author Mutsuo Saito (Hiroshima University)
 * @author Makoto Matsumoto (Hiroshima University)
 *
 * Copyright (C) 2007 Mutsuo Saito, Makoto Matsumoto and Hiroshima
 * University. All rights reserved.
 *
 * The new BSD License is applied to this software.
 * see LICENSE.txt
 */

#ifndef LIBSAD_SFMT_ALTI_H
#define LIBSAD_SFMT_ALTI_H

inline static vector guint vec_recursion(vector guint a, vector guint b, vector guint c, vector guint d) ALWAYSINLINE;

/**
 * This function represents the recursion formula in AltiVec and BIG ENDIAN.
 * @param a a 128-bit part of the interal state array
 * @param b a 128-bit part of the interal state array
 * @param c a 128-bit part of the interal state array
 * @param d a 128-bit part of the interal state array
 * @return output
 */
     inline static vector guint vec_recursion(vector guint a, vector guint b, vector guint c, vector guint d)
{

    const vector guint sl1 = ALTI_SL1;
    const vector guint sr1 = ALTI_SR1;
#ifdef ONLY64
    const vector guint mask = ALTI_MSK64;
    const vector unsigned char perm_sl = ALTI_SL2_PERM64;
    const vector unsigned char perm_sr = ALTI_SR2_PERM64;
#else
    const vector guint mask = ALTI_MSK;
    const vector unsigned char perm_sl = ALTI_SL2_PERM;
    const vector unsigned char perm_sr = ALTI_SR2_PERM;
#endif
    vector guint v, w, x, y, z;
    x = vec_perm(a, (vector unsigned int)perm_sl, perm_sl);
    v = a;
    y = vec_sr(b, sr1);
    z = vec_perm(c, (vector unsigned int)perm_sr, perm_sr);
    w = vec_sl(d, sl1);
    z = vec_xor(z, w);
    y = vec_and(y, mask);
    v = vec_xor(v, x);
    z = vec_xor(z, y);
    z = vec_xor(z, v);
    return z;
}

/**
 * This function fills the internal state array with pseudorandom
 * integers.
 */
inline static void gen_rand_all(void)
{
    gint i;
    vector guint r, r1, r2;

    r1 = sfmt[N - 2].s;
    r2 = sfmt[N - 1].s;
    for (i = 0; i < N - POS1; i++)
    {
        r = vec_recursion(sfmt[i].s, sfmt[i + POS1].s, r1, r2);
        sfmt[i].s = r;
        r1 = r2;
        r2 = r;
    }
    for (; i < N; i++)
    {
        r = vec_recursion(sfmt[i].s, sfmt[i + POS1 - N].s, r1, r2);
        sfmt[i].s = r;
        r1 = r2;
        r2 = r;
    }
}

/**
 * This function fills the user-specified array with pseudorandom
 * integers.
 *
 * @param array an 128-bit array to be filled by pseudorandom numbers.  
 * @param size number of 128-bit pesudorandom numbers to be generated.
 */
inline static void gen_rand_array(w128_t * array, gint size)
{
    gint i, j;
    vector guint r, r1, r2;

    r1 = sfmt[N - 2].s;
    r2 = sfmt[N - 1].s;
    for (i = 0; i < N - POS1; i++)
    {
        r = vec_recursion(sfmt[i].s, sfmt[i + POS1].s, r1, r2);
        array[i].s = r;
        r1 = r2;
        r2 = r;
    }
    for (; i < N; i++)
    {
        r = vec_recursion(sfmt[i].s, array[i + POS1 - N].s, r1, r2);
        array[i].s = r;
        r1 = r2;
        r2 = r;
    }
    /* main loop */
    for (; i < size - N; i++)
    {
        r = vec_recursion(array[i - N].s, array[i + POS1 - N].s, r1, r2);
        array[i].s = r;
        r1 = r2;
        r2 = r;
    }
    for (j = 0; j < 2 * N - size; j++)
    {
        sfmt[j].s = array[j + size - N].s;
    }
    for (; i < size; i++)
    {
        r = vec_recursion(array[i - N].s, array[i + POS1 - N].s, r1, r2);
        array[i].s = r;
        sfmt[j++].s = r;
        r1 = r2;
        r2 = r;
    }
}

#ifndef ONLY64
#  if defined(__APPLE__)
#    define ALTI_SWAP (vector guchar) (4, 5, 6, 7, 0, 1, 2, 3, 12, 13, 14, 15, 8, 9, 10, 11)
#  else
#    define ALTI_SWAP {4, 5, 6, 7, 0, 1, 2, 3, 12, 13, 14, 15, 8, 9, 10, 11}
#  endif
/**
 * This function swaps high and low 32-bit of 64-bit integers in user
 * specified array.
 *
 * @param array an 128-bit array to be swaped.
 * @param size size of 128-bit array.
 */
inline static void swap(w128_t * array, gint size)
{
    gint i;
    const vector unsigned char perm = ALTI_SWAP;

    for (i = 0; i < size; i++)
    {
        array[i].s = vec_perm(array[i].s, (vector unsigned int)perm, perm);
    }
}
#endif

#endif /* LIBSAD_SFMT_ALTI_H */
