/* Copyright (C) 2007 One Laptop Per Child
 * Author: Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef ABI_PLUGIN_BUILTIN
#define abi_plugin_register abipgn_develop_register
#define abi_plugin_unregister abipgn_develop_unregister
#define abi_plugin_supports_version abipgn_develop_supports_version
#endif

#include "xap_Module.h"
#include "xap_App.h"
#include "xap_Frame.h"

#include "xap_Menu_Layouts.h"
#include "ev_Menu_Actions.h"
#include "ev_EditMethod.h"
#include "ap_Menu_Id.h"

#include "HighlightManager.h"

// -----------------------------------------------------------------------
//
//      Abiword Plugin Interface 
//
// -----------------------------------------------------------------------

ABI_PLUGIN_DECLARE(Develop)

static void s_register_menus(HighlightManager* pHighlightManager);
static bool s_set_language(AV_View* v, EV_EditMethodCallData *d);

static HighlightManager* pHighlightManager = NULL;

ABI_FAR_CALL
int abi_plugin_register (XAP_ModuleInfo * mi)
{
	mi->name = "AbiWord Develop";
	mi->desc = "This plugin enables various tools for source code development, such as syntax highlighting";
	mi->version = ABI_VERSION_STRING;
	mi->author = "Marc Maurer <uwog@uwog.net>";
	mi->usage = "";

	pHighlightManager = new HighlightManager();
	s_register_menus(pHighlightManager);
	
	return 1;
}

ABI_FAR_CALL
int abi_plugin_unregister (XAP_ModuleInfo * mi)
{
	mi->name = 0;
	mi->desc = 0;
	mi->version = 0;
	mi->author = 0;
	mi->usage = 0;
	
	// TODO: unregister menus

	DELETEP(pHighlightManager);

	return 1;
}

ABI_FAR_CALL
int abi_plugin_supports_version (UT_uint32 major, UT_uint32 minor, UT_uint32 release)
{
	return 1;
}

void s_register_menus(HighlightManager* pHighlightManager)
{
	UT_return_if_fail(pHighlightManager);

    // First we need to get a pointer to the application itself.
    XAP_App *pApp = XAP_App::getApp();
    EV_EditMethodContainer* pEMC = pApp->getEditMethodContainer();
    int frameCount = pApp->getFrameCount();
    XAP_Menu_Factory * pFact = pApp->getMenuFactory();    
    EV_Menu_ActionSet* pActionSet = pApp->getMenuActionSet();
	
	// TODO: make this a translatable set of strings
	// const XAP_StringSet * pSS = pApp->getStringSet();
    
	// The Highlight menu items
	XAP_Menu_Id separatorId = pFact->addNewMenuAfter("Main", NULL, AP_MENU_ID_VIEW_SHOWPARA, EV_MLF_Separator);
    pFact->addNewLabel(NULL, separatorId, NULL, NULL);

	XAP_Menu_Id highlightId = pFact->addNewMenuAfter("Main", NULL, separatorId, EV_MLF_BeginSubMenu);
    pFact->addNewLabel(NULL, highlightId, "Highlight", "Highlight Tip");
    EV_Menu_Action* myHighlightAction = new EV_Menu_Action (
		highlightId,    			 // id that the layout said we could use
		1,                      // yes, we have a sub menu.
		0,                      // no, we don't raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		NULL,                   // no callback function to call.
		NULL,                   // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
	pActionSet->addAction(myHighlightAction);

	// Add our supported languages
	// TODO: organize these into sections
	XAP_Menu_Id lastId = highlightId;
	const map<string, LanguageDefinition*>& vLangs = pHighlightManager->getAllLanguages();
	for (map<string, LanguageDefinition*>::const_iterator cit = vLangs.begin(); cit != vLangs.end(); cit++)
	{
		LanguageDefinition* pLangDef = (*cit).second;
		UT_continue_if_fail(pLangDef);

		XAP_Menu_Id langId = pFact->addNewMenuAfter("Main", NULL, lastId, EV_MLF_Normal);
	    pFact->addNewLabel(NULL, langId, pLangDef->getLanguage().c_str(), "lang tip");
		EV_Menu_Action* myLangAction = new EV_Menu_Action (
			langId,   	 			 // id that the layout said we could use
			0,                      // no, we don't have a sub menu.
			0,                      // yes, we raise a dialog.
			0,                      // no, we don't have a checkbox.
			0,                      // no radio buttons for me, thank you
			"s_set_language",   	 // name of callback function to call.
			NULL, // Function for whether not label is enabled/disabled checked/unchecked
			NULL,                    // Function to compute Menu Label "Dynamic Label"
			pLangDef->getLanguage().c_str() // HACK: abuse the script name field to pass the selected language in
		);
		pActionSet->addAction(myLangAction);
		EV_EditMethod *myLangEditMethod = new EV_EditMethod (
			"s_set_language",    // name of callback function
			s_set_language,      // callback function itself.
			EV_EMT_REQUIREDATA,     // require a "script name" (which we abuse for the selected language
			""                      // description -- allegedly never used for anything
		);
		pEMC->addEditMethod(myLangEditMethod);
		lastId = langId;
	}

	// end of the highlight menu
	XAP_Menu_Id endHighlightId = pFact->addNewMenuAfter("Main", NULL, lastId, EV_MLF_EndSubMenu);
	pFact->addNewLabel(NULL, endHighlightId, "End Highlight", NULL);
	EV_Menu_Action* myEndHighlightAction = new EV_Menu_Action (
		endHighlightId,     // id that the layout said we could use
		0,                      // no, we don't have a sub menu.
		0,                      // no, we raise a dialog.
		0,                      // no, we don't have a checkbox.
		0,                      // no radio buttons for me, thank you
		NULL,                   // name of callback function to call.
		NULL,                   // Function for whether not label is enabled/disabled checked/unchecked
		NULL                    // Function to compute Menu Label "Dynamic Label"
	);
    pActionSet->addAction(myEndHighlightAction);
    

    // We need to go through and add the menu element to each "frame" 
    // of the application.  We can iterate through the frames by doing
    // XAP_App::getFrameCount() to tell us how many frames there are,
    // then calling XAP_App::getFrame(i) to get the i-th frame.
    for(int i = 0; i < frameCount;++i)
    {
        // Get the current frame that we're iterating through.
		XAP_Frame* pFrame = pApp->getFrame(i);
		pFrame->rebuildMenus();
    }
}

bool s_set_language(AV_View* v, EV_EditMethodCallData *d)
{
	UT_return_val_if_fail(pHighlightManager, false);
	UT_return_val_if_fail(v, false);
	UT_return_val_if_fail(d, false);

	UT_DEBUGMSG(("s_set_language()\n"));
	UT_DEBUGMSG(("Language selected: %s\n", d->getScriptName().c_str()));

	FV_View* pView = reinterpret_cast<FV_View*>(v);
	FL_DocLayout* pDocLayout = pView->getLayout();
	UT_return_val_if_fail(pDocLayout, false);

	// add a highlighter for this document layout (this will immediately start to populate
	// the highlighter, and thus it will highlight this layout on the fly
	pHighlightManager->addHighlighter(pDocLayout, d->getScriptName().c_str());

	// update the screen to make sure all dirty runs are redrawn
	pView->updateScreen();

	return true;
}
