/*
 * AbiScan - Abiword plugin for Gnome Scan and Ocropus
 * Copyright (C) 2002 by Martin Sevior
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef ABI_PLUGIN_BUILTIN
#define abi_plugin_register abipgn_abiscan_register
#define abi_plugin_unregister abipgn_abiscan_unregister
#define abi_plugin_supports_version abipgn_abiscan_supports_version
#endif

#include <gnome-scan.h>
#include "abiscan-sink.h"
#include "ut_assert.h"
#include "ut_debugmsg.h"
#include "xap_Module.h"
#include "xap_App.h"
#include "xap_Frame.h"
#include "fv_View.h"
#include "ap_Menu_Id.h"
#include "ev_Menu_Actions.h"
#include "ev_Menu.h"
#include "ev_Menu_Layouts.h"
#include "ev_Menu_Labels.h"
#include "ev_EditMethod.h"
#include "xap_Menu_Layouts.h"
#include "fg_Graphic.h"
#include "ie_imp.h"
#include "ie_impGraphic.h"
#include "ie_exp.h"
#include "ie_types.h"
#include "ut_sleep.h"


ABI_PLUGIN_DECLARE(AbiScan)
static bool AbiScan_invoke(AV_View* v, EV_EditMethodCallData *d);

static const char* AbiScan_MenuLabel = "Import from &scanner";
static const char* AbiScan_MenuTooltip = "Scan and recognize documents.";

//
// AbiScan_addToMenus
// -----------------------
//   Adds "Edit Image" option to AbiWord's Context Menu.
//
static void
AbiScan_addToMenus()
{
  // First we need to get a pointer to the application itself.
  XAP_App *pApp = XAP_App::getApp();

    
  // Create an EditMethod that will link our method's name with
  // it's callback function.  This is used to link the name to 
  // the callback.
  EV_EditMethod *myEditMethod = new EV_EditMethod("AbiScan_invoke",  // name of callback function
						  AbiScan_invoke,    // callback function itself.
						  0,                 // no additional data required.
						  ""                 // description -- allegedly never used for anything
						  );
   
  // Now we need to get the EditMethod container for the application.
  // This holds a series of Edit Methods and links names to callbacks.
  EV_EditMethodContainer* pEMC = pApp->getEditMethodContainer();
    
  // We have to add our EditMethod to the application's EditMethodList
  // so that the application will know what callback to call when a call
  // to "AiksaurusABI_invoke" is received.
  pEMC->addEditMethod(myEditMethod);
  

  // Now we need to grab an ActionSet.  This is going to be used later
  // on in our for loop.  Take a look near the bottom.
  EV_Menu_ActionSet* pActionSet = pApp->getMenuActionSet();

    
  // We need to go through and add the menu element to each "frame" 
  // of the application.  We can iterate through the frames by doing
  // XAP_App::getFrameCount() to tell us how many frames there are,
  // then calling XAP_App::getFrame(i) to get the i-th frame.

  int frameCount = pApp->getFrameCount();
  XAP_Menu_Factory * pFact = pApp->getMenuFactory();
  //
  // Put it in the context menu.
  //
  XAP_Menu_Id newID = pFact->addNewMenuAfter("Main",NULL,"&Picture",EV_MLF_Normal);
  pFact->addNewLabel(NULL,newID,AbiScan_MenuLabel, AbiScan_MenuTooltip);

  // Create the Action that will be called.
  EV_Menu_Action* myAction = new EV_Menu_Action(
						newID,                     // id that the layout said we could use
						0,                      // no, we don't have a sub menu.
						1,                      // yes, we raise a dialog.
						0,                      // no, we don't have a checkbox.
						0,                      // no radio buttons for me, thank you
						"AbiScan_invoke",  // name of callback function to call.
						NULL,                   // don't know/care what this is for
						NULL                    // don't know/care what this is for
						);

  // Now what we need to do is add this particular action to the ActionSet
  // of the application.  This forms the link between our new ID that we 
  // got for this particular frame with the EditMethod that knows how to 
  // call our callback function.  

  pActionSet->addAction(myAction);
    
  for(int i = 0;i < frameCount;++i)
    {
      // Get the current frame that we're iterating through.
      XAP_Frame* pFrame = pApp->getFrame(i);
      pFrame->rebuildMenus();
    }
}

static void
AbiScan_RemoveFromMenus ()
{
  // First we need to get a pointer to the application itself.
  XAP_App *pApp = XAP_App::getApp();

  // remove the edit method
  EV_EditMethodContainer* pEMC = pApp->getEditMethodContainer() ;
  EV_EditMethod * pEM = ev_EditMethod_lookup ( "AbiScan_invoke" ) ;
  pEMC->removeEditMethod ( pEM ) ;
  DELETEP( pEM ) ;

  // now remove crap from the menus
  int frameCount = pApp->getFrameCount();
  XAP_Menu_Factory * pFact = pApp->getMenuFactory();

  pFact->removeMenuItem("Main",NULL,AbiScan_MenuLabel);
  for(int i = 0;i < frameCount;++i)
    {
      // Get the current frame that we're iterating through.
      XAP_Frame* pFrame = pApp->getFrame(i);
      pFrame->rebuildMenus();
    }
}
    
// -----------------------------------------------------------------------
//
//      Abiword Plugin Interface 
//
// -----------------------------------------------------------------------

  
GtkWidget *dialog = NULL;
  
ABI_FAR_CALL
int abi_plugin_register (XAP_ModuleInfo * mi)
{
  mi->name = "AbiScan";
  mi->desc = "Use this to import document from your scanner through Gnome Scan and Ocropus within AbiWord";
  mi->version = ABI_VERSION_STRING;
  mi->author = "Étienne Bersac <bersace03@laposte.net>";
  mi->usage = "No Usage";
    
  gint argc = 0;
  gchar **argv = NULL;
  gnome_scan_init(&argc, &argv);
  GnomeScanJob*job = gnome_scan_job_new();
  GnomeScanSettings*settings = gnome_scan_settings_new();
  gnome_scan_job_set_settings(job, settings);
  GnomeScanSink*sink = abiscan_sink_new();
  gnome_scan_job_set_sink (job, sink);
  dialog = gnome_scan_dialog_new(NULL, job);

  // Add to AbiWord's menus.
  AbiScan_addToMenus();
    
  return 1;
}


ABI_FAR_CALL
int abi_plugin_unregister (XAP_ModuleInfo * mi)
{
  mi->name = 0;
  mi->desc = 0;
  mi->version = 0;
  mi->author = 0;
  mi->usage = 0;

  AbiScan_RemoveFromMenus ();
  gnome_scan_exit();

  return 1;
}


ABI_FAR_CALL
int abi_plugin_supports_version (UT_uint32 major, UT_uint32 minor, UT_uint32 release)
{
  return 1; 
}

// -----------------------------------------------------------------------
//
//     AbiScan Invocation Code
//
// -----------------------------------------------------------------------


//
// AbiScan_invoke
// -------------------
//   This is the function that we actually call to invoke the Gimp.
//
bool 
AbiScan_invoke(AV_View* v, EV_EditMethodCallData *d)
{
  gnome_scan_dialog_run (GNOME_SCAN_DIALOG(dialog));

  return false;
}
