/****************************************************************************************
*  YAROCK                                                                               *
*  Copyright (c) 2010-2014 Sebastien amardeilh <sebastien.amardeilh+yarock@gmail.com>   *
*                                                                                       *
*  This program is free software; you can redistribute it and/or modify it under        *
*  the terms of the GNU General Public License as published by the Free Software        *
*  Foundation; either version 2 of the License, or (at your option) any later           *
*  version.                                                                             *
*                                                                                       *
*  This program is distributed in the hope that it will be useful, but WITHOUT ANY      *
*  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A      *
*  PARTICULAR PURPOSE. See the GNU General Public License for more details.             *
*                                                                                       *
*  You should have received a copy of the GNU General Public License along with         *
*  this program.  If not, see <http://www.gnu.org/licenses/>.                           *
*****************************************************************************************/

#include "playlisteditor.h"
#include "playqueue_model.h"
#include "task_manager.h"
#include "playlistview.h"

#include "threadmanager.h"
#include "widgets/statuswidget.h"
#include "widgets/dialogs/addstreamdialog.h"
#include "widgets/dialogs/filedialog.h"
#include "debug.h"

#include <QtGui>
#include <QtCore>

/*
********************************************************************************
*                                                                              *
*    Class PlaylistEditor                                                      *
*                                                                              *
********************************************************************************
*/
PlaylistEditor::PlaylistEditor(QWidget *parent) : QDialog(parent)
{
    /* init */
    m_model         = new PlayqueueModel();
    m_task_manager  = m_model->manager();

    m_view  = new PlaylistView(this, m_model);
    m_view->setMode((int) PLAYQUEUE::MODE_ARTIST);

    /* playlist item */
    m_playlist = MEDIA::PlaylistPtr( new MEDIA::Playlist() );
    m_playlist->id       =  -1;
    m_playlist->url      =  QString();
    m_playlist->name     =  QString("untitled playlist");
    m_playlist->p_type   =  T_DATABASE;

    /* ui */    
    create_ui();
    slot_on_type_change(0);

    QObject::connect(m_task_manager, SIGNAL(playlistSaved()), ThreadManager::instance(), SLOT(populateLocalPlaylistModel()));
}

PlaylistEditor::~PlaylistEditor()
{
    delete m_model;
}


void PlaylistEditor::closeEvent ( QCloseEvent * event )
{
    QDialog::closeEvent(event);
}

void PlaylistEditor::hideEvent ( QHideEvent * )
{
    //! force to quit if "esc" is pressed instead of hiding dialog window
    this->setResult(QDialog::Rejected);
    QDialog::reject();
    this->close();
}

/*******************************************************************************
 ui part
*******************************************************************************/
void PlaylistEditor::create_ui()
{
    //! ui part
    this->setModal(false);
    this->setWindowTitle(tr("Playlist Editor"));
    this->resize(500,620);
    this->setAttribute( Qt::WA_DeleteOnClose, true );
   

    //In Linux with KDE this code make a window without an close and minimize
    // and maximize buttons in title bar.
    this->setWindowFlags( Qt::Dialog | Qt::WindowTitleHint );

    // set toolbar & action
    QMenu *menu = new QMenu();
    menu->addAction( QIcon(":/images/track-48x48.png"), tr("&Add media to playlist"), this, SLOT(slot_add_file()));
    menu->addAction( QIcon(":/images/folder.png"), tr("&Add directory to playlist"), this, SLOT(slot_add_dir()));
    menu->addAction( QIcon(":/images/media-url-48x48.png"),tr("&Add Url..."), this, SLOT(slot_add_url()));
    
    QToolButton* add_button = new QToolButton(this);
    add_button->setIcon(QIcon(":/images/add_32x32.png"));
    add_button->setToolTip(tr("Add"));
    add_button->setMenu(menu);
    add_button->setAutoRaise(true);
    add_button->setPopupMode(QToolButton::InstantPopup);
    
    QToolBar* toolbar = new QToolBar(this);
    toolbar->setOrientation(Qt::Vertical);
    toolbar->addWidget( add_button );
    toolbar->addAction( QIcon::fromTheme("edit-clear-list"), tr("Clear playlist"), m_model, SLOT(clear()));

    
    /* ui items */    
    ui_edit_name = new ExLineEdit(this);
    ui_edit_name->clearFocus();
    ui_edit_name->setInactiveText(tr("playlist name"));

    ui_buttonBox = new QDialogButtonBox();
    ui_buttonBox->setOrientation(Qt::Horizontal);
    ui_buttonBox->setStandardButtons(QDialogButtonBox::Apply|QDialogButtonBox::Close);
    connect(ui_buttonBox, SIGNAL(clicked ( QAbstractButton *)), this, SLOT(slot_on_buttonbox_clicked(QAbstractButton *)));

    

    QFrame* frame = new QFrame();
    frame->setFrameShape(QFrame::StyledPanel);
    frame->setFrameShadow(QFrame::Sunken);

    
    ui_combo_type = new QComboBox();
    ui_combo_type->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    ui_combo_type->addItem(tr("database"));
    ui_combo_type->addItem(tr("file"));  
    connect(ui_combo_type, SIGNAL(currentIndexChanged ( int )), this, SLOT(slot_on_type_change(int)));

    ui_file_path = new ExLineEdit(this);
    ui_file_path->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    ui_file_path->setInactiveText(tr("file path"));

    
    ui_save_path = new QToolButton(this);
    ui_save_path->setIcon(QIcon(":/images/folder.png"));
    ui_save_path->setToolTip(tr("Open"));
    connect(ui_save_path, SIGNAL(clicked()), this, SLOT(slot_dialog_file_to_save()));
    
    
    /* ---- Layouts ---- */
    QVBoxLayout *vl0 = new QVBoxLayout(frame);
    vl0->setSpacing(0);
    vl0->setContentsMargins(0, 0, 0, 0);
    vl0->addWidget(m_view);

    QVBoxLayout *vl1 = new QVBoxLayout();
    vl1->setSpacing(0);
    vl1->setContentsMargins(0, 0, 0, 0);
    vl1->addItem(new QSpacerItem(20, ui_edit_name->height(), QSizePolicy::Minimum, QSizePolicy::Minimum));
    vl1->addWidget(toolbar);
    vl1->addItem(new QSpacerItem(20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding));

    QHBoxLayout *hl0 = new QHBoxLayout();
    hl0->setSpacing(0);
    hl0->setContentsMargins(0, 0, 0, 0);
    hl0->addWidget(ui_combo_type);
    hl0->addWidget(ui_file_path);
    hl0->addWidget(ui_save_path);


    QVBoxLayout *vl2 = new QVBoxLayout();
    vl2->setSpacing(0);
    vl2->setContentsMargins(0, 0, 0, 0);
//     vl2->addLayout(hl0);    
    vl2->addWidget(ui_edit_name);
    vl2->addWidget(frame);    
    vl2->addLayout(hl0);
    vl2->addWidget(ui_buttonBox);
    
    QHBoxLayout *hl1 = new QHBoxLayout(this);
    hl1->setSpacing(0);
    hl1->setContentsMargins(0, 0, 0, 0);
    hl1->addLayout(vl1);
    hl1->addLayout(vl2);
    
    /* focus : avoid focus to be on line edit */
    this->setFocus(Qt::OtherFocusReason);
}


/*******************************************************************************
    slot_on_buttonbox_clicked
*******************************************************************************/
void PlaylistEditor::slot_on_buttonbox_clicked(QAbstractButton * button)
{
    QDialogButtonBox::ButtonRole role = ui_buttonBox->buttonRole(button);
    
    if(role == QDialogButtonBox::RejectRole) 
    {
      this->setResult(QDialog::Rejected);
      QDialog::reject();
      this->close();
    }
    else if (role == QDialogButtonBox::ApplyRole)
    {
      /* save current database */
      if(m_playlist->p_type == T_DATABASE)
      {   
         int database_id = m_playlist->id;
         const QString name = ui_edit_name->text();

         m_task_manager->playlistSaveToDb( name , database_id);
      }
      else if (m_playlist->p_type == T_FILE)
      {
          //Debug::debug() << "PlaylistEditor::slot_on_buttonbox_clicked => SAVE TO FILE";
          m_task_manager->playlistSaveToFile( m_playlist->url );
      }
    }
}

/*******************************************************************************
    slot_on_type_change
*******************************************************************************/
void PlaylistEditor::slot_on_type_change(int type)
{
   Debug::debug() << "### slot_on_type_change"; 
   ui_file_path->setEnabled(type != 0);
   ui_save_path->setEnabled(type != 0);

   m_playlist->p_type   = type == 0 ? T_DATABASE : T_FILE;
}


/*******************************************************************************
 public interface
*******************************************************************************/
void PlaylistEditor::setPlaylist(MEDIA::PlaylistPtr playlist)
{
    m_playlist = playlist;
    
    /* update ui  */
    ui_combo_type->setCurrentIndex(m_playlist->p_type == T_DATABASE ? 0 : 1);
    ui_combo_type->setEnabled(false);
    ui_edit_name->setText( playlist->name );
    
    if(m_playlist->p_type == T_FILE) 
    {
      ui_file_path->setText( m_playlist->url);
    }

    /* fill playlist content */
    m_model->clear();
    foreach(MEDIA::MediaPtr media, playlist->children())   {
      m_model->addMediaItem( MEDIA::TrackPtr::staticCast(media) );
    }
}


/*******************************************************************************
 Playlist Population
*******************************************************************************/
void PlaylistEditor::slot_add_file()
{
    FileDialog fd(this, FileDialog::AddFiles, tr("Add music files or playlist"));

    if(fd.exec() == QDialog::Accepted) {
       QStringList files  = fd.addFiles();
       
       m_task_manager->playlistAddFiles(files);
    }      
}


void PlaylistEditor::slot_add_dir()
{
    /*--------------------------------------------------*/
    /* add dir to playqueue                             */
    /* -------------------------------------------------*/
    FileDialog fd(this, FileDialog::AddDirs, tr("Add music directories"));

    if(fd.exec() == QDialog::Accepted) {
      QStringList dirs  = fd.addDirectories();
      m_task_manager->playlistAddFiles(dirs);
    }    
}


void PlaylistEditor::slot_add_url()
{
    AddStreamDialog stream_dialog(this,false);

    if(stream_dialog.exec() == QDialog::Accepted)
    {
      const QString url   = stream_dialog.url();

      if(!QUrl(url).isEmpty() && QUrl(url).isValid()) {
        const QString name  = stream_dialog.name();

        MEDIA::TrackPtr media = MEDIA::TrackPtr(new MEDIA::Track());
        media->setType(TYPE_STREAM);
        media->id          = -1;
        media->url         = url;
        media->name        = !name.isEmpty() ? name : url ;
        media->isFavorite  = false;
        media->isPlaying   = false;
        media->isBroken    = false;
        media->isPlayed    = false;
        media->isStopAfter = false;

        m_model->addMediaItem(media);
        media.reset();
      }
      else {
        StatusWidget::instance()->startShortMessage("invalid url can not be added !!", STATUS::TYPE_WARNING, 5000);
      }
    }
}


/*******************************************************************************
    slot_dialog_file_to_save
*******************************************************************************/
void PlaylistEditor::slot_dialog_file_to_save()
{
    FileDialog fd(this, FileDialog::SaveFile, tr("Save playlist to file"));
  
    if(fd.exec() == QDialog::Accepted) {
      ui_file_path->setText( fd.saveFile() );
      m_playlist->url = ui_file_path->text();
    }
}


