/****************************************************************************************
*  YAROCK                                                                               *
*  Copyright (c) 2010-2014 Sebastien amardeilh <sebastien.amardeilh+yarock@gmail.com>   *
*                                                                                       *
*  This program is free software; you can redistribute it and/or modify it under        *
*  the terms of the GNU General Public License as published by the Free Software        *
*  Foundation; either version 2 of the License, or (at your option) any later           *
*  version.                                                                             *
*                                                                                       *
*  This program is distributed in the hope that it will be useful, but WITHOUT ANY      *
*  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A      *
*  PARTICULAR PURPOSE. See the GNU General Public License for more details.             *
*                                                                                       *
*  You should have received a copy of the GNU General Public License along with         *
*  this program.  If not, see <http://www.gnu.org/licenses/>.                           *
*****************************************************************************************/

// local
#include "core/database/databasebuilder.h"
#include "core/database/database.h"
#include "core/database/databasemanager.h"

#include "core/mediaitem/mediaitem.h"
#include "core/mediaitem/playlist_parser.h"

#include "utilities.h"
#include "debug.h"


// Qt
#include <QtCore>
#include <QImage>
#include <QtSql/QSqlDatabase>
#include <QtSql/QSqlQuery>

/*
********************************************************************************
*                                                                              *
*    Class DataBaseBuilder                                                     *
*                                                                              *
********************************************************************************
*/
DataBaseBuilder::DataBaseBuilder()
{
    m_exit            = false;
}

/*******************************************************************************
   DataBaseBuilder::filesFromFilesystem
*******************************************************************************/
QStringList DataBaseBuilder::filesFromFilesystem(const QString& directory)
{
  QStringList files;
  const QStringList filters = QStringList()
  /* Audio */    << "*.mp3"  << "*.ogg" << "*.wav" << "*.flac" << "*.m4a" << "*.aac"
  /* Playlist */ << "*.m3u" << "*.m3u8" << "*.pls" << "*.xspf";

  QDir dir(directory);
  dir.setNameFilters(filters);
  dir.setFilter(QDir::Files | QDir::NoDotAndDotDot | QDir::NoSymLinks);

  QDirIterator it(dir, QDirIterator::NoIteratorFlags);
  while(it.hasNext())
  {
      it.next();
      files << it.fileInfo().absoluteFilePath().toUtf8();
  }

  return files;
}


/*******************************************************************************
   DataBaseBuilder::filesFromFilesystem
*******************************************************************************/
QHash<QString,uint> DataBaseBuilder::filesFromDatabase(const QString& directory)
{  
    QHash<QString,uint> files;
  
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `directories` WHERE `path`=:val;");
    q.bindValue(":val", directory );
    q.exec();

    if ( q.next() ) {
      int dir_id = q.value(0).toString().toInt();

      q.prepare("SELECT `id`,`filename`,`mtime` FROM `tracks` WHERE `dir_id`=:dirId;");
      q.bindValue(":dirId", dir_id );
      Debug::debug() << q.exec();
      while (q.next())
        files.insert(q.value(1).toString(),q.value(2).toUInt());
      
      q.prepare("SELECT `id`,`filename`,`mtime` FROM `playlists` WHERE `dir_id`=:dirId;");
      q.bindValue(":dirId", dir_id );
      Debug::debug() << q.exec();
      while (q.next())
        files.insert(q.value(1).toString(),q.value(2).toUInt());      
    }
    
    //Debug::debug() << "DataBaseBuilder::filesFromDatabase files:" << files;
    return files;
}

/*******************************************************************************
   DataBaseBuilder::rebuildFolder
     -> User entry point : add folder to parse
*******************************************************************************/
void DataBaseBuilder::rebuildFolder(QStringList folder)
{
    m_db_dirs.clear();
    m_fs_dirs.clear();

    m_folders.clear();
    m_folders.append(folder);
}

/*******************************************************************************
   DataBaseBuilder::run
*******************************************************************************/
void DataBaseBuilder::run()
{
    if (m_folders.isEmpty()) return;
    int idxCount   = 0;
    int fileCount  = 0;

    Database db;
    if (!db.connect()) return;
    m_sqlDb = db.sqlDb();

    Debug::debug() << "- DataBaseBuilder -> starting Database update";

    
    /*-----------------------------------------------------------*/
    /* Get directories from filesystem                           */
    /* ----------------------------------------------------------*/    
    foreach(const QString& root_dir, m_folders)
    {
      Debug::debug() << "- DataBaseBuilder -> ROOT DIR :" << root_dir;
      m_fs_dirs << root_dir;
      
      QDir dir(root_dir);
      dir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot | QDir::NoSymLinks);

      QDirIterator it(dir,QDirIterator::Subdirectories);
      
      while(it.hasNext())
      {
        it.next();
        if (it.fileInfo().isDir())
        {
          QString dir_path = it.fileInfo().canonicalFilePath().toUtf8();
          //Debug::debug() << "#### DataBaseBuilder -> canonicalFilePath: " << dir_path;
          if(!m_fs_dirs.contains(dir_path))
            m_fs_dirs << dir_path;
        }
      }
    }

    fileCount = m_fs_dirs.count();
    
    /*-----------------------------------------------------------*/
    /* Get directories from database                             */
    /* ----------------------------------------------------------*/    
    QSqlQuery dirQuery("SELECT path, mtime FROM directories;",*m_sqlDb);
    while (dirQuery.next())
      m_db_dirs.insert(dirQuery.value(0).toString(),dirQuery.value(1).toUInt());
    
    /*-----------------------------------------------------------*/
    /* Update database                                           */
    /* ----------------------------------------------------------*/
    //On SQLite --> it's MUCH faster to have everything in one transaction
    //with only one disk write than to commit every each insert individually
    QSqlQuery("BEGIN TRANSACTION;",*m_sqlDb);    
    
    foreach(const QString& dir_path, m_fs_dirs)
    {
      if(m_exit)
        break;

      //! If the directory is NOT in database then insert
      if (!m_db_dirs.contains(dir_path) )
      {
        addDirectory(dir_path);
      }
      //! If the file is in database but has another mtime then update it
      else if ( m_db_dirs[dir_path] != QFileInfo(dir_path).lastModified().toTime_t() )
      {
        updateDirectory(dir_path);
      }

      m_db_dirs.remove(dir_path);

       //! signal progress
       if(fileCount > 0) {
         int percent = 100 - ((fileCount - ++idxCount) * 100 / fileCount);
         emit buildingProgress(percent);
       }
    } // end foreach file in filesystem    
    
    
    
    //! Get files that are in DB but not on filesystem
    QHashIterator<QString, uint> i(m_db_dirs);
    while (i.hasNext()) {
        i.next();
        removeDirectory(i.key());
    }

    m_db_dirs.clear();

    // Check for interprets/albums/genres... that are not used anymore
    cleanUpDatabase();

    // Store last update time
    QSqlQuery q("UPDATE `db_attribute` SET `value`=:date WHERE `name`=lastUpdate;",*m_sqlDb);
    q.bindValue(":date", QDateTime::currentDateTime().toTime_t());
    q.exec();

    // Now write all data to the disk
    QSqlQuery("COMMIT TRANSACTION;",*m_sqlDb);

    Debug::debug() << "- DataBaseBuilder -> end Database update";
    if(!m_exit)
      emit buildingFinished();    

}


void DataBaseBuilder::addDirectory(const QString& path)
{
    Debug::debug() << "- DataBaseBuilder -> addDirectory :" << path;
    foreach (const QString& file, filesFromFilesystem(path) )
    {
        if (MEDIA::isAudioFile(file) )
           insertTrack(file);
        else
           insertPlaylist(file);
    }
    
    insertDirectory(path);
}

void DataBaseBuilder::updateDirectory(const QString& path)
{
    Debug::debug() << "- DataBaseBuilder -> updateDirectory :" << path;
    
    QStringList files_fs = filesFromFilesystem(path);
    QHash<QString,uint> files_db = filesFromDatabase(path);  

    foreach(const QString& filepath, files_fs)
    {
      //! If the file is NOT in database then insert
      if (!files_db.contains(filepath) )
      {
        if (MEDIA::isAudioFile(filepath) )
          insertTrack(filepath);
        else
          insertPlaylist(filepath);
      }
      //! If the file is in database but has another mtime then update it
      else if (files_db[filepath] != QFileInfo(filepath).lastModified().toTime_t())
      {
        if (MEDIA::isAudioFile(filepath) )
          updateTrack(filepath);
        else
          updatePlaylist(filepath);
      }

       files_db.remove(filepath);
    } // end foreach file in filesystem
    
    
    //! Get files that are in DB but not on filesystem
    QHashIterator<QString, uint> i(files_db);
    while (i.hasNext()) {
        i.next();
        if( MEDIA::isAudioFile(i.key()) )
          removeTrack(i.key());
        else
          removePlaylist(i.key());
    }
    
    QSqlQuery q("", *m_sqlDb);
    q.prepare("UPDATE `directories` SET `mtime`=? WHERE `path`=?;");
    q.addBindValue(QFileInfo(path).lastModified().toTime_t());
    q.addBindValue(path);
    q.exec();
}

void DataBaseBuilder::removeDirectory(const QString& path)
{
    Debug::debug() << "- DataBaseBuilder -> removeDirectory :" << path;

    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `directories` WHERE `path`=:val;");
    q.bindValue(":val", path );
    q.exec();

    if ( q.next() ) {
      int dir_id = q.value(0).toString().toInt();
      
      q.prepare("DELETE FROM `tracks` WHERE `dir_id`=?;");
      q.addBindValue(dir_id);
      q.exec();

      q.prepare("DELETE FROM `playlists` WHERE `dir_id`=?;");
      q.addBindValue(dir_id);
      q.exec();
      
      q.prepare("DELETE FROM `directories` WHERE `id`=?;");
      q.addBindValue(dir_id);
      q.exec();
    }
}

/*******************************************************************************
   DataBaseBuilder::insertDirectory
*******************************************************************************/
int DataBaseBuilder::insertDirectory(const QString & path)
{
    Debug::debug() << "DataBaseBuilder -> insertDirectory " << path;

    uint mtime   = QFileInfo(path).lastModified().toTime_t();
    
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `directories` WHERE `path`=:val;");
    q.bindValue(":val", path );
    Debug::debug() << q.exec();

    if ( !q.next() ) {
      q.prepare("INSERT INTO `directories`(`path`,`mtime`) VALUES (?,?);");
      q.addBindValue(path);
      q.addBindValue(mtime);
      Debug::debug() << q.exec();

      if(q.numRowsAffected() < 1) return -1;
      q.prepare("SELECT `id` FROM `directories` WHERE `path`=:val;");
      q.bindValue(":val", path);
      q.exec();
      q.next();
    }
    return q.value(0).toString().toInt();
}

/*******************************************************************************
   DataBaseBuilder::insertTrack
     -> MEDIA::FromLocalFile(fname) to get track metadata
     -> MEDIA::coverName(media) to get hash of covername
*******************************************************************************/
void DataBaseBuilder::insertTrack(const QString& filename)
{
    QFileInfo fileInfo(filename);

    QString fname = fileInfo.filePath().toUtf8();

    uint mtime = QFileInfo(filename).lastModified().toTime_t();

    Debug::debug() << "- DataBaseBuilder -> insert track :" << filename;

    //! storage localtion
    QString storageLocation = UTIL::CONFIGDIR + "/albums/";

    //! Read tag from URL file (with taglib)
    int disc_number = 0;
    MEDIA::TrackPtr track = MEDIA::FromLocalFile(fname, &disc_number);
    QString  cover_name   = track->coverName();

    //! DIRECTORIES part in database
    int id_dir = insertDirectory( QFileInfo(filename).canonicalPath() );
    
    //! GENRE part in database
    int id_genre = insertGenre( track->genre );

    //! YEAR part in database
    int id_year = insertYear( track->year );

    //! ARTIST part in database
    int id_artist = insertArtist( track->artist );

    //! ALBUM part in database
    int id_album = insertAlbum(
        track->album,
        id_artist,
        cover_name,
        track->year,
        disc_number
        );

    //! mise à jour du cover
    storeCoverArt(storageLocation + cover_name, track->url);

    if( DatabaseManager::instance()->DB_PARAM().checkCover )
      recupCoverArtFromDir(storageLocation + cover_name, track->url);

    //! TRACK part in database
    QSqlQuery query(*m_sqlDb);
    query.prepare("INSERT INTO `tracks`(`filename`,`trackname`,`number`,`length`," \
                  "`artist_id`,`album_id`,`year_id`,`genre_id`,`dir_id`,`mtime`," \
                  "`playcount`,`rating`,`albumgain`,`albumpeakgain`,`trackgain`,`trackpeakgain`) " \
                  "VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);");
    query.addBindValue(fname);
    query.addBindValue(track->title);
    query.addBindValue(track->num);
    query.addBindValue(track->duration);

    query.addBindValue(id_artist);
    query.addBindValue(id_album);
    query.addBindValue(id_year);
    query.addBindValue(id_genre);
    query.addBindValue(id_dir);

    query.addBindValue(mtime);
    query.addBindValue(track->playcount);
    query.addBindValue(track->rating);
    query.addBindValue(track->albumGain);
    query.addBindValue(track->albumPeak);
    query.addBindValue(track->trackGain);
    query.addBindValue(track->trackPeak);
    query.exec();

    // delete media from memory
    if(track) {
      track.reset();
      delete track.data();
    }
}


/*******************************************************************************
   DataBaseBuilder::insertGenre
*******************************************************************************/
int DataBaseBuilder::insertGenre(const QString & genre)
{
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `genres` WHERE `genre`=:val;");
    q.bindValue(":val", genre );
    q.exec();

    if ( !q.next() ) {
      q.prepare("INSERT INTO `genres`(`genre`) VALUES (:val);");
      q.bindValue(":val", genre);
      q.exec();

      if(q.numRowsAffected() < 1) return -1;
      q.prepare("SELECT `id` FROM `genres` WHERE `genre`=:val;");
      q.bindValue(":val", genre);
      q.exec();
      q.next();
    }
    return q.value(0).toString().toInt();
}

/*******************************************************************************
   DataBaseBuilder::insertYear
*******************************************************************************/
int DataBaseBuilder::insertYear(int year)
{
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `years` WHERE `year`=:val;");
    q.bindValue(":val", year );
    q.exec();

    if ( !q.next() ) {
      q.prepare("INSERT INTO `years`(`year`) VALUES (:val);");
      q.bindValue(":val", year);
      q.exec();

      if(q.numRowsAffected() < 1) return -1;
      q.prepare("SELECT `id` FROM `years` WHERE `year`=:val;");
      q.bindValue(":val", year);
      q.exec();
      q.next();
    }
    return q.value(0).toString().toInt();
}

/*******************************************************************************
   DataBaseBuilder::insertArtist
*******************************************************************************/
int DataBaseBuilder::insertArtist(const QString & artist)
{
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `artists` WHERE `name`=:val;");
    q.bindValue(":val", artist );
    q.exec();

    if ( !q.next() ) {
      q.prepare("INSERT INTO `artists`(`name`,`favorite`,`playcount`,`rating`) VALUES (:val,0,0,-1);");
      q.bindValue(":val", artist);
      q.exec();

      if(q.numRowsAffected() < 1) return -1;
      q.prepare("SELECT `id` FROM `artists` WHERE `name`=:val;");
      q.bindValue(":val", artist);
      q.exec();
      q.next();
    }
    return q.value(0).toString().toInt();
}

/*******************************************************************************
   DataBaseBuilder::insertAlbum
*******************************************************************************/
int DataBaseBuilder::insertAlbum(const QString & album, int artist_id,const QString & cover,int year,int disc)
{
    QSqlQuery q("", *m_sqlDb);
    q.prepare("SELECT `id` FROM `albums` WHERE `name`=:val AND `artist_id`=:id AND `disc`=:dn;");
    q.bindValue(":val", album );
    q.bindValue(":id", artist_id );
    q.bindValue(":dn", disc );
    q.exec();

    if ( !q.next() ) {
      q.prepare("INSERT INTO `albums`(`name`,`artist_id`,`cover`,`year`,`favorite`,`playcount`,`rating`,`disc`) VALUES (:val,:id,:cov,:y,0,0,-1,:dn);");
      q.bindValue(":val", album);
      q.bindValue(":id", artist_id );
      q.bindValue(":cov", cover );
      q.bindValue(":y", year );
      q.bindValue(":dn", disc );
      q.exec();

      if(q.numRowsAffected() < 1) return -1;
      q.prepare("SELECT `id` FROM `albums` WHERE `name`=:val AND `artist_id`=:id AND `disc`=:dn;");
      q.bindValue(":val", album );
      q.bindValue(":id", artist_id );
      q.bindValue(":dn", disc );
      q.exec();
      q.next();
    }
    return q.value(0).toString().toInt();
}


/*******************************************************************************
   DataBaseBuilder::updateTrack
*******************************************************************************/
void DataBaseBuilder::updateTrack(const QString& filename)
{
    removeTrack(filename);
    insertTrack(filename);
}

/*******************************************************************************
   DataBaseBuilder::removeTrack
*******************************************************************************/
void DataBaseBuilder::removeTrack(const QString& filename)
{
    Debug::debug() << "- DataBaseBuilder -> Deleting track :" << filename;
    QFileInfo fileInfo(filename);
    QString fname = fileInfo.filePath().toUtf8();

    QSqlQuery query(*m_sqlDb);
    query.prepare("DELETE FROM `tracks` WHERE `filename`=?;");
    query.addBindValue(fname);
    query.exec();
}


/*******************************************************************************
   DataBaseBuilder::insertPlaylist
*******************************************************************************/
void DataBaseBuilder::insertPlaylist(const QString& filename)
{
    QFileInfo fileInfo(filename);
    QString fname = fileInfo.filePath().toUtf8();
    QString pname = fileInfo.baseName();
    uint mtime    = fileInfo.lastModified().toTime_t();

    Debug::debug() << "- DataBaseBuilder -> insert playlist :" << filename;

    int favorite = 0;

    
    
    //! DIRECTORIES part in database
    int id_dir = insertDirectory( QFileInfo(filename).canonicalPath() );
    
    
    //! PLAYLIST part in database
    QSqlQuery query(*m_sqlDb);
    query.prepare("INSERT INTO `playlists`(`filename`,`name`,`type`,`favorite`,`dir_id`,`mtime`)" \
                  "VALUES(?,?,?,?,?,?);");

    query.addBindValue(fname);
    query.addBindValue(pname);
    query.addBindValue((int) T_FILE);
    query.addBindValue(favorite);
    query.addBindValue(id_dir);
    query.addBindValue(mtime);
    query.exec();


    //! PLAYLIST ITEM part in database
    QList<MEDIA::TrackPtr> list =  MEDIA::PlaylistFromFile(filename);
    foreach (MEDIA::TrackPtr mi, list)
    {
      QString url           = mi->url;
      QString name          = QFileInfo(url).baseName();

      //! Playlist Item part in database
      Debug::debug() << "- DataBaseBuilder -> insert playlistitem url: " << url;

      query.prepare("INSERT INTO `playlist_items`(`url`,`name`,`playlist_id`)" \
                    "VALUES(?," \
                    "       ?," \
                    "       (SELECT `id` FROM `playlists` WHERE `filename`=?));");
      query.addBindValue(url);
      query.addBindValue(name);
      query.addBindValue(fname);
      query.exec();
    } // end foreach url into playlist
}

/*******************************************************************************
   DataBaseBuilder::updatePlaylist
*******************************************************************************/
void DataBaseBuilder::updatePlaylist(const QString& filename)
{
    removePlaylist(filename);
    insertPlaylist(filename);
}

/*******************************************************************************
   DataBaseBuilder::removePlaylist
*******************************************************************************/
void DataBaseBuilder::removePlaylist(const QString& filename)
{
    Debug::debug() << "- DataBaseBuilder -> removePlaylist :" << filename;
    QFileInfo fileInfo(filename);
    QString fname = fileInfo.filePath().toUtf8();

    QSqlQuery query(*m_sqlDb);
    query.prepare("DELETE FROM `playlists` WHERE `filename`=?;");
    query.addBindValue(fname);
    query.exec();
}


/*******************************************************************************
   DataBaseBuilder::cleanUpDatabase
*******************************************************************************/
void DataBaseBuilder::cleanUpDatabase()
{
    {
      QSqlQuery query("DELETE FROM `albums` WHERE `id` NOT IN (SELECT `album_id` FROM `tracks` GROUP BY `album_id`);", *m_sqlDb);
    }
    {
      QSqlQuery query("DELETE FROM `genres` WHERE `id` NOT IN (SELECT `genre_id` FROM `tracks` GROUP BY `genre_id`);", *m_sqlDb);
    }
    {
      QSqlQuery query("DELETE FROM `artists` WHERE `id` NOT IN (SELECT `artist_id` FROM `tracks` GROUP BY `artist_id`);", *m_sqlDb);
    }
    {
      QSqlQuery query("DELETE FROM `years` WHERE `id` NOT IN (SELECT `year_id` FROM `tracks` GROUP BY `year_id`);", *m_sqlDb);
    }
    {
      QSqlQuery query("DELETE FROM `playlist_items` WHERE `playlist_id` NOT IN (SELECT `id` FROM `playlists`);", *m_sqlDb);
    }
}


/*******************************************************************************
   DataBaseBuilder::storeCoverArt
*******************************************************************************/
void DataBaseBuilder::storeCoverArt(const QString& coverFilePath, const QString& trackFilename)
{
    //Debug::debug() << "- DataBaseBuilder -> storeCoverArt " << coverFilePath;

    //! check if cover art already exist
    QFile file(coverFilePath);
    if(file.exists()) return;

    //! get cover image from file
    QImage image = MEDIA::LoadImageFromFile(trackFilename);
    if( !image.isNull() )
      image.save(coverFilePath, "png", -1);
}

/*******************************************************************************
   DataBaseBuilder::recupCoverArtFromDir
*******************************************************************************/
void DataBaseBuilder::recupCoverArtFromDir(const QString& coverFilePath, const QString& trackFilename)
{
    //Debug::debug() << "- DataBaseBuilder -> recupCoverArtFromDir " << coverFilePath;

    //! check if coverArt already exist
    QFile file(coverFilePath);
    if(file.exists()) return ;

    //! search album art into file source directory
    const QStringList imageFilters = QStringList() << "*.jpg" << "*.png";
    QDir sourceDir(QFileInfo(trackFilename).absolutePath());

    sourceDir.setNameFilters(imageFilters);

    QStringList entryList = sourceDir.entryList(imageFilters, QDir::Files, QDir::Size);

    while(!entryList.isEmpty()) {
      //! I take the first one (the biggest one)
      //!WARNING simplification WARNING
      QString file = QFileInfo(trackFilename).absolutePath() + "/" + entryList.takeFirst();
      QImage image = QImage(file);
      //! check if not null image (occur when file is KO)
      if(!image.isNull()) {
        image = image.scaled(QSize(110, 110), Qt::KeepAspectRatio, Qt::SmoothTransformation);
        //! check if save is OK
        if(image.save(coverFilePath, "png", -1))
          break;
      }
    }
}

