/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>

#include <directfb.h>

#include "dfbsee.h"
#include "rotate.h"
#include "video-draw.h"


typedef struct 
{
  IDirectFBSurface  *dest;
  IDirectFBSurface  *surface;
  IDirectFBSurface  *extra_surface;
  DFBRectangle       rect;
  RotationType       rot;
} VideoContext;

static VideoContext video_ctx = { 0, };


static int
video_stretch_callback (void *ctx)
{
  VideoContext *video = (VideoContext *) ctx;
  
  video->dest->StretchBlit (video->dest, video->surface, NULL, &video->rect);

  return 0;
}

static int
video_rotate_callback (void *ctx)
{
  VideoContext *video = (VideoContext *) ctx;

  if (hw_stretchblit)
    {
      rotate_blit (video->extra_surface, video->surface, video->rot, 0, 0);
      video->dest->StretchBlit (video->dest, 
                                video->extra_surface, NULL, &video->rect);
    }
  else
    {
      rotate_blit (video->dest, video->surface, 
                   video->rot, video->rect.x, video->rect.y);
    }

  return 0;
}

static void
check_surface (IDirectFBSurface **surface,
               int                width,
               int                height,
               int                needed)
{
  IDirectFBSurface *s = *surface;

  if (s)
    {
      int w, h;
      
      s->GetSize (s, &w, &h);
      if (!needed || w != width || h != height)
        {
          s->Release (s);
          s = *surface = NULL;
        }
    }

  if (!s && needed)
    {
      DFBSurfaceDescription dsc;

      dsc.flags = DSDESC_WIDTH | DSDESC_HEIGHT | DSDESC_PIXELFORMAT;
      dsc.width  = width;
      dsc.height = height;
      primary->GetPixelFormat (primary, &dsc.pixelformat);      

      dfb->CreateSurface (dfb, &dsc, surface);
    }
}

void
video_draw (IDirectFBSurface       *dest,
            IDirectFBVideoProvider *video,
            int                     width,
            int                     height,
            DFBRectangle           *rect,
            RotationType            rot)
{
  DFBSurfaceDescription dsc;    
  int video_width;
  int video_height;
  int stretch;

  video->GetSurfaceDescription (video, &dsc);
  video_width  = dsc.width;
  video_height = dsc.height;

  video_ctx.dest =  dest;
  video_ctx.rect = *rect;
  video_ctx.rot  =  rot;

  stretch = rot || rect->w != video_width || rect->h != video_height;

  if (hw_stretchblit)
    {
      check_surface (&video_ctx.surface, video_width, video_height, stretch);
    }
  else
    {
      switch (rot)
        {
        case ROT_LEFT:
        case ROT_RIGHT:
          check_surface (&video_ctx.surface, rect->h, rect->w, stretch);
          break;
        default:
          check_surface (&video_ctx.surface, rect->w, rect->h, stretch);
          break;
        }
    }

  check_surface (&video_ctx.extra_surface, width, height,
                 rot && stretch && hw_stretchblit);

  if (rot)
    video->PlayTo (video, video_ctx.surface, NULL, 
                   video_rotate_callback, &video_ctx);
  else if (stretch && hw_stretchblit)
    video->PlayTo (video, video_ctx.surface, NULL, 
                   video_stretch_callback, &video_ctx);
  else
    video->PlayTo (video, dest, rect, NULL, NULL);
}

void
video_draw_deinit (void)
{
  if (video_ctx.surface)
    {
      video_ctx.surface->Release (video_ctx.surface);
      video_ctx.surface = NULL;
    }
  if (video_ctx.extra_surface)
    {
      video_ctx.extra_surface->Release (video_ctx.extra_surface);
      video_ctx.extra_surface = NULL;
    }
}
