/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>

#include <directfb.h>

#include "dfbsee.h"
#include "rotate.h"


#define SWAP(a,b,tmp) { (tmp) = *(b); *(b) = *(a); *(a) = (tmp); }

#define ROTATE_RIGHT(type)\
{\
  type d;\
  type s;\
  while (--width)\
    {\
      h = height;\
      d = (type) dest + width;\
      s = src;\
      while (--h)\
        {\
          *d = *s;\
          s++;\
          (__u8*)d += dest_pitch;\
        }\
      (__u8*)src += src_pitch;\
    }\
}\

#define ROTATE_LEFT(type)\
{\
  type d;\
  type s;\
  (__u8 *) dest += (height - 1) * dest_pitch;\
  for (w = 0; w < width; w++)\
    {\
      h = height;\
      d = (type) dest + w;\
      s = src;\
      while (h--)\
        {\
          *d = *s;\
          s++;\
          (__u8*)d -= dest_pitch;\
        }\
      (__u8*)src += src_pitch;\
    }\
}\

#define ROTATE_HALF(type)\
{\
  type d;\
  type s;\
  (__u8 *) dest += (height - 1) * dest_pitch;\
  while (--height)\
    {\
      d = (type) dest + width - 1;\
      s = src;\
      w = width;\
      while (--w)\
        {\
          *d = *s;\
          s++;\
          d--;\
        }\
      (__u8*)src += src_pitch;\
      (__u8*)dest -= dest_pitch;\
    }\
}\


static void
rotate (void         *dest,
        int           dest_pitch,
        void         *src,
        int           src_pitch,
        int           width,
        int           height,
        int           bpp,
        RotationType  rot)
{
  int w, h, tmp;

  switch (rot)
    {
    case ROT_LEFT:
      SWAP (&width, &height, tmp);
      switch (bpp)
        {
        case 2:
          ROTATE_LEFT (__u16 *)
          break;
        case 4:
          ROTATE_LEFT (__u32 *)
          break;
        default:
          break;
        }
      break;

    case ROT_RIGHT:
      SWAP (&width, &height, tmp);
      switch (bpp)
        {        
        case 2:
          ROTATE_RIGHT (__u16 *)
          break;
        case 4:
          ROTATE_RIGHT (__u32 *)
          break;
        default:
          break;
        }
      break;

    case ROT_HALF:
      switch (bpp)
        {
        case 2:
          ROTATE_HALF (__u16 *)
          break;
        case 4:
           ROTATE_HALF (__u32 *)
          break;
        default:
          break;
        }
      
    default:
      break;
    }
}

void
rotate_blit (IDirectFBSurface *destination,
             IDirectFBSurface *source,
             RotationType      rot,
             int               x,
             int               y)
{
  DFBSurfacePixelFormat pixelformat;
  void *src;
  void *dest;
  int src_pitch, dest_pitch;
  int width, height;

  source->GetSize (source, &width, &height);
  source->GetPixelFormat (source, &pixelformat);
  
  source->Lock (source, DSLF_READ, &src, &src_pitch);
  destination->Lock (destination, DSLF_READ, &dest, &dest_pitch);

  (__u8 *) dest += dest_pitch * y + DFB_BYTES_PER_PIXEL (pixelformat) * x;

  rotate (dest, dest_pitch,
          src, src_pitch, width, height, DFB_BYTES_PER_PIXEL (pixelformat),
          rot); 

  source->Unlock (source);
  destination->Unlock (destination);
}

IDirectFBSurface *
rotate_surface_new (IDirectFBSurface *source,
                    RotationType      rot)
{
  IDirectFBSurface      *destination;
  DFBSurfaceDescription  dsc;
  void *src;
  void *dest;
  int src_pitch, dest_pitch;
  int width, height;

  if (!source)
    return NULL;

  source->GetSize (source, &width, &height);
  source->GetPixelFormat (source, &dsc.pixelformat);

  switch (DFB_BYTES_PER_PIXEL (dsc.pixelformat))
    {
    case 2:
    case 4:
      break;
    default:
      printf ("rotate: unsupported pixelformat (%dbit)\n", 
              DFB_BITS_PER_PIXEL (dsc.pixelformat));
      source->AddRef (source);
      return source;
    }

  dsc.flags = DSDESC_WIDTH | DSDESC_HEIGHT | DSDESC_PIXELFORMAT;

  switch (rot)
    {
    case ROT_LEFT:
    case ROT_RIGHT:
      dsc.width  = height;
      dsc.height = width;
      break;
    default:
      dsc.width  = width;
      dsc.height = height;
      break;
    }

  if (dfb->CreateSurface (dfb, &dsc, &destination) != DFB_OK)
    {
      source->AddRef (source);
      return source;
    }

  source->Lock (source, DSLF_READ, &src, &src_pitch);
  destination->Lock (destination, DSLF_READ, &dest, &dest_pitch);

  rotate (dest, dest_pitch, 
          src, src_pitch, width, height, DFB_BYTES_PER_PIXEL (dsc.pixelformat),
          rot); 

  source->Unlock (source);
  destination->Unlock (destination);
  
  return destination;
}

RotationType
rotate_type (RotationType  orig,
             RotationType  rot,
             int          *width,
             int          *height)
{
  int tmp;

  switch (rot)
    {
    case ROT_NONE:
      return orig;

    case ROT_LEFT:
      SWAP (width, height, tmp);
      switch (orig)
        {
        case ROT_NONE:
          return ROT_LEFT;
        case ROT_LEFT:
          return ROT_HALF;
        case ROT_RIGHT:
          return ROT_NONE;
        case ROT_HALF:
          return ROT_RIGHT;
        }
      break;

    case ROT_RIGHT:
      SWAP (width, height, tmp);
      switch (orig)
        {
        case ROT_NONE:
          return ROT_RIGHT;
        case ROT_LEFT:
          return ROT_NONE;
        case ROT_RIGHT:
          return ROT_HALF;
        case ROT_HALF:
          return ROT_LEFT;
        }
      break;

    case ROT_HALF:
      switch (orig)
        {
        case ROT_NONE:
          return ROT_HALF;
        case ROT_LEFT:
          return ROT_RIGHT;
        case ROT_RIGHT:
          return ROT_LEFT;
        case ROT_HALF:
          return ROT_NONE;
        }
      break;
    }

  return orig;
}
