/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <directfb.h>

#include "dfbsee.h"
#include "image-load.h"
#include "interface.h"
#include "media.h"
#include "rotate.h"
#include "video-draw.h"
#include "video-load.h"


static Media *head = NULL;
static Media *tail = NULL;

static IDirectFBSurface  *dest = NULL;


static Media *
media_new (const char *filename)
{
  Media *media;

  media = calloc (1, sizeof (Media));
  if (!media)
    {
      perror(__FILE__ ": media_new");
      return NULL;
    }

  media->filename = (char*) filename;
  media->zoom = 1000.0;
  
  return media;
}


static void
media_load (Media *media)
{
  interface_set_media (media);

  if (!media->loaded)
    {
      interface_update (CLEAR);
      interface_set_message ("Loading...", 1);

      media->can_seek = 0;
      media->color_adjust_caps = DCAF_NONE;

      if (access (media->filename, R_OK))
        {
          media->error = strdup (strerror (errno));
        }
      else
        {
          media->image = image_load (media->filename, 
                                     &media->width, 
                                     &media->height, 
                                     &media->caps);

          if (!media->image)
            media->video = video_load (media->filename, 
                                       &media->width, 
                                       &media->height,
                                       &media->length,
                                       &media->can_seek, 
                                       &media->color_adjust_caps);
        
          if (!media->image && !media->video)
            media->error =  strdup ("No media provider for this file");
        }

      media->loaded = 1;
      interface_set_message (NULL, 0);
    }
   interface_update (SIZE | PROGRESS); 
}

Media * 
media_add (const char *filename)
{
  Media *media;

  if (!filename)
    return head;

  media = media_new (filename);
  if (!media)
    return head;

  if (tail)
    {
      tail->next = media;
      media->prev = tail;
    }
  
  tail = media;

  if (!head)
    {
      head = tail;
    }
  else
    {
      head->prev = media;
      media->next = head;
    }

  return head;
}

Media *
media_previous (Media *media)
{
  if (media && media->prev)
    {
      if (media->video)
        media->video->Stop (media->video);

      media = media->prev;
      media_load (media);
      media_redraw (media);
    }

  return media;
}

Media *
media_next (Media *media)
{
  if (media && media->next)
    {
      if (media->video)
        {
          media->video->Stop (media->video);
          video_draw_deinit ();
        }

      media = media->next;
      media_load (media);
      media_redraw (media);
    }
  
  return media;
}

double
media_calc_size (Media            *media,
                 double            zoom,
                 DFBRectangle     *rect)
{
  int src_width  = media->width;
  int src_height = media->height;
  int dest_width;
  int dest_height;

  dest->GetSize (dest, &dest_width, &dest_height);

  /*  consider aspect ratio  */
  if (override_ratio && aspect > 0.0)
    {
      double src_aspect = (double) src_width / (double) src_height;

      if (aspect > src_aspect)
        src_width = (double) src_height * aspect + 0.5;
      else
        src_height = (double) src_width / aspect + 0.5;
    }
 
  /*  apply zoom factor  */
  src_width  = zoom * (double) src_width  + 0.5;
  src_height = zoom * (double) src_height + 0.5;

  /*  clamp to sreen  */
  if (src_width > dest_width)
    {
      src_height = (double) src_height * 
        (double) dest_width / (double) src_width + 0.5;
      src_width = dest_width;
    }
  if (src_height > dest_height)
    {
      src_width = (double) src_width * 
        (double) dest_height / (double) src_height + 0.5;
      src_height = dest_height;
    }

  if (rect)
    {
      rect->x = corner ? 0 : (dest_width  - src_width)  / 2;
      rect->y = corner ? 0 : (dest_height - src_height) / 2;
      rect->w = src_width;
      rect->h = src_height;
    }

  return MIN ((double) src_width / (double) media->width, 
              (double) src_height / (double) media->height);
}

void
media_set_zoom (Media  *media,
                double  zoom)
{
  if (zoom < 0.01)
    zoom = 0.01;

  media->zoom = media_calc_size (media, zoom, NULL);
  media_redraw (media);
}

void
media_set_dest (Media            *media,
                IDirectFBSurface *new_dest)
{
  if (new_dest != dest)
    {
      dest = new_dest;
      if (media->video)
        media->video->Stop (media->video);
      interface_update (SCREEN | FRAME);
      media_load (media);
      media_redraw (media);
    }
}

void
media_redraw (Media *media)
{
  DFBRectangle rect;
  double scale;

  if (!media)
    return;

  if (media->video)
    media->video->Stop (media->video);

  interface_update (CLEAR);
  if (media->error || (!media->video && !media->image))
    {
      interface_set_message (media->error ? media->error : "ERROR", 0);
      return;
    }

  if (zoomed)
    scale = media_calc_size (media, media->zoom, &rect);
  else
    scale = media_calc_size (media, 1.0, &rect);
  
  interface_set_zoom (scale);
  interface_set_rect (&rect);
}

void
media_rotate (Media *media,
              int    negative)
{
  IDirectFBSurface *src;
  IDirectFBSurface *dest;

  if (media->image)
    {
      src = media->image;
      dest = rotate_surface_new (src, negative ? ROT_RIGHT : ROT_LEFT);

      interface_lock ();
      media->image = dest;
      dest->GetSize (dest, &media->width, &media->height);
      interface_unlock ();
      
      src->Release (src);

      interface_update (SIZE);
      media_redraw (media);
    }
  else if (media->video)
    {
      media->rot = rotate_type (media->rot, 
                                negative ? ROT_RIGHT : ROT_LEFT,
                                &media->width, &media->height);
      interface_update (SIZE);
      media_redraw (media);
    }
}

void
media_toggle_play (Media *media)
{
  if (!media->video)
    return;

  if (media->running)
    {
      media->video->Stop (media->video);
      media->running = 0;
    }
  else
    media_redraw (media);
}

void
media_home (Media *media)
{
  if (media->video && media->can_seek)
    {
      media->video->SeekTo (media->video, 0);
      interface_update (PROGRESS);
    }
}

void
media_end (Media *media)
{
  double len;

  if (media->video && media->can_seek)
    {
      if (media->video->GetLength (media->video, &len) == DFB_OK
          && len > 10.0) 
        {
          media->video->SeekTo (media->video, len - 10.0);
          interface_update (PROGRESS);
        }
    }
}

void
media_skip (Media  *media,
            double  skip)
{
  double pos;

  if (media->video && media->can_seek)
    {
      if (media->video->GetPos (media->video, &pos) != DFB_OK)
        return;
      
      pos += skip;

      if (pos < 0.0)
        pos = 0.0;

      media->video->SeekTo (media->video, pos);
      interface_update (PROGRESS);
    }
}

void
media_color_adjust (Media *media,
                    int    value)
{
  DFBColorAdjustment adj;

  if (media->video)
    {
      value *= 0x800;
      media->video->GetColorAdjustment (media->video, &adj);

      switch (media->color_adjustment)
        {
        case DCAF_BRIGHTNESS:
          adj.brightness = (value ? 
                            CLAMP ((int) adj.brightness + value, 0x0, 0xFFFF) :
                            0x8000);
          break;
        case DCAF_CONTRAST:
          adj.contrast = (value ?
                          CLAMP ((int) adj.contrast + value, 0x0, 0xFFFF) :
                          0x8000);
          break;
        case DCAF_HUE:
          adj.hue = (value ?
                     CLAMP ((int) adj.hue + value, 0x0, 0xFFFF) :
                     0x8000);
          break;
        case DCAF_SATURATION:
          adj.saturation = (value ?
                            CLAMP ((int) adj.saturation + value, 0x0, 0xFFFF) :
                            0x8000);
          break;
        default:
          break;
        }

      media->video->SetColorAdjustment (media->video, &adj);
      interface_update (COLOR);
    }
}

void
media_set_color_adjustment (Media                   *media,
                            DFBColorAdjustmentFlags  flags)
{
  media->color_adjustment = media->color_adjust_caps & flags;

  if (flags == DCAF_NONE || media->color_adjustment == flags)
    interface_update (COLOR);
}

void
media_deinit (void)
{
  Media *media;
  Media *next;

  interface_set_media (NULL);

  for (media = head; media; media = next)
    {
      if (media->image)
	media->image->Release (media->image);
      if (media->video)
	media->video->Release (media->video);
      if (media->error)
        free (media->error);
      next = media->next;
      free (media);
      if (next == tail)
        break;
    }
}
