/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h> /* free */

#include <directfb.h>

#include "dfbsee.h"
#include "options.h"
#include "interface.h"
#include "media.h"
#include "volume.h"


/*  global variables  */
IDirectFB        *dfb     = NULL;
IDirectFBSurface *primary = NULL;
DFBResult         err     = DFB_OK;
int        screen_width   = 0;
int        screen_height  = 0;
int        screen_depth   = 0;
int        fullscreen     = 0;
int        override_ratio = 0;
int        corner         = 0;
int        set_mode       = 0;
int        show_time      = 0;
int        zoomed         = 0;
int        slide_show     = -1;
int        hw_stretchblit = 0;
double     aspect         = 0.0;


int
main (int    argc, 
      char **argv)
{
  DFBCardCapabilities    caps;
  DFBSurfaceDescription  dsc;
  IDirectFBInputBuffer  *keybuffer;
  IDirectFBSurface      *image_area;
  Media *media = NULL;
  double zoom;
  double adjust;
  int    help   = 0;
  int    paused = 0;
  int    quit   = 0;
  int    volume_adjustment = 0;

  DFBCHECK (DirectFBInit (&argc, &argv));

  media = options_parse (argc, argv);

  if (!media)
    return 1;

  /* create the super interface */
  DFBCHECK (DirectFBCreate (&dfb));

  dfb->GetCardCapabilities (dfb, &caps);
  hw_stretchblit = caps.acceleration_mask & DFXL_STRETCHBLIT;
  
  DFBCHECK (dfb->SetCooperativeLevel (dfb, DFSCL_FULLSCREEN));
  DFBCHECK (dfb->CreateInputBuffer (dfb, DICAPS_KEYS, &keybuffer));

  if (set_mode)
    {      
      printf ("trying to set video mode to %dx%d @ %dbpp ... ",
	      screen_width, screen_height, screen_depth);
      if (dfb->SetVideoMode (dfb, 
                             screen_width, screen_height, screen_depth) 
          == DFB_OK)
	printf ("OK\n");
      else
	printf ("failed\n");
    }

  /* create the primary surface */
  dsc.flags = DSDESC_CAPS;
  dsc.caps = DSCAPS_PRIMARY;

  DFBCHECK (dfb->CreateSurface (dfb, &dsc, &primary));
  primary->GetSize (primary, &screen_width, &screen_height);

  if (aspect == 0.0)
    aspect = (double) screen_width / (double) screen_height;

  image_area = interface_init ();

  media_set_dest (media, fullscreen ? primary : image_area);

  while (!quit)
    {
      DFBInputEvent evt;

      if (volume_adjustment)
        {
          if (keybuffer->WaitForEventWithTimeout (keybuffer, 2, 0) == DFB_TIMEOUT)
            {
              volume_close();
              volume_adjustment = 0;
            }
        }
      else if (!paused && slide_show >= 0)
        {
          if (slide_show == 0 ||
              keybuffer->WaitForEventWithTimeout (keybuffer, slide_show, 0) == DFB_TIMEOUT)
            media = media_next (media);
        }
      else
        {
          keybuffer->WaitForEvent (keybuffer);
        }

      while (keybuffer->GetEvent (keybuffer, &evt) == DFB_OK)
	{
          if (help)
            {
              if (evt.type == DIET_KEYRELEASE && evt.keycode == DIKC_F1)
                {
                  media_redraw (media);
                  help = 0;
                }
            }
          else if (media->color_adjustment)
            {
              switch (evt.type)
                {
                case DIET_KEYPRESS:
                  switch (evt.keycode)
                    {
                    case DIKC_LEFT:
                    case DIKC_DOWN:
                      media_color_adjust (media, -1);
                      break;

                    case DIKC_RIGHT:
                    case DIKC_UP:
                      media_color_adjust (media, 1);
                      break;

                    case DIKC_SPACE:
                      media_color_adjust (media, 0);
                      break;
                      
                    default:
                      break;
                    }
                  break;

                case DIET_KEYRELEASE:
                  switch (evt.keycode)
                    {
                    case DIKC_B:
                    case DIKC_C:
                    case DIKC_H:
                    case DIKC_S:
                      media_set_color_adjustment (media, DCAF_NONE);
                      break;
                    default:
                      break;
                    }
                  break;

                default:
                  break;
                }
            }
          else if (volume_adjustment)
            {
              switch (evt.type)
                {
                case DIET_KEYPRESS:
                  switch (evt.keycode)
                    {
                    case DIKC_LEFT:
                    case DIKC_DOWN:
                      volume_adjust (-1);
                      break;

                    case DIKC_RIGHT:
                    case DIKC_UP:
                      volume_adjust (1);
                      break;

                    case DIKC_VOLUMEUP:
                      volume_adjust (10);
                      break;

                    case DIKC_VOLUMEDOWN:
                      volume_adjust (-10);
                      break;

                    default:
                      break;
                    }
                  break;

                case DIET_KEYRELEASE:
                  switch (evt.keycode)
                    {
                    case DIKC_V:
                      volume_close ();
                      volume_adjustment = 0;
                      break;
                    default:
                      break;
                    }
                  break;

                default:
                  break;
                }
            }
          else /* !color_adjustment */
            {
              switch (evt.type) 
                {
                case DIET_KEYPRESS:
                  switch (evt.keycode)
                    {
                    case DIKC_SPACE:
                    case DIKC_PAGEDOWN:
                      media = media_next (media);
                      break;
                      
                    case DIKC_BACKSPACE:
                    case DIKC_PAGEUP:
                      media = media_previous (media);
                      break;
                      
                    case DIKC_HOME:
                      media_home (media);
                      break;
                      
                    case DIKC_END:
                      media_end (media);
                      break;
                      
                    case DIKC_PLAYPAUSE:
                    case DIKC_ENTER:
                      media_toggle_play (media);
                      break;
                      
                    case DIKC_LEFT:
                    case DIKC_RIGHT:
                      if (evt.modifiers & DIMK_CTRL)
                        adjust = 60.0;
                      else if (evt.modifiers & DIMK_SHIFT)
                        adjust = 1.0;
                      else
                        adjust = 10.0;
                      
                      if (evt.keycode == DIKC_LEFT)
                        media_skip (media, - adjust);
                      else
                        media_skip (media, adjust);
                      break;
                      
                    case DIKC_I:
                    case DIKC_F:
                      fullscreen = !fullscreen;
                      media_set_dest (media, 
                                      fullscreen ? primary : image_area);
                      break;
                      
                    case DIKC_Z:
                      zoomed = !zoomed;
                      media_redraw (media);
                      break;
                      
                    case DIKC_R:
                      override_ratio = !override_ratio;
                      media_redraw (media);
                      break;
                      
                    case DIKC_V:
                      if (!volume_open ())
                        volume_adjustment = 1;
                      break;
                      
                    case DIKC_VOLUMEUP:
                      if (!volume_open ())
                        {
                          volume_adjustment = 1;
                          volume_adjust (10);
                        }
                      break;
                      
                    case DIKC_VOLUMEDOWN:
                      if (!volume_open ())
                        {
                          volume_adjust (-10);
                          volume_adjustment = 1;
                        }
                      break;
                      
                    case DIKC_B:
                      media_set_color_adjustment (media, DCAF_BRIGHTNESS);
                      break;
                    case DIKC_C:
                      media_set_color_adjustment (media, DCAF_CONTRAST);
                      break;
                    case DIKC_H:
                      media_set_color_adjustment (media, DCAF_HUE);
                      break;  
                    case DIKC_S:
                      media_set_color_adjustment (media, DCAF_SATURATION);
                      break;
                      
                    case DIKC_KP_PLUS:
                    case DIKC_KP_MINUS:
                    case DIKC_UP:
                    case DIKC_DOWN:
                      if (!zoomed)
                        {
                          zoomed = 1;
                          zoom = media_calc_size (media, 1.0, NULL);
                        }
                      else
                        zoom = media->zoom;
                      
                      if (evt.modifiers & DIMK_CTRL)
                        adjust = 1.0;
                      else if (evt.modifiers & DIMK_SHIFT)
                        adjust = 0.01;
                      else
                        adjust = 0.1;
                      
                      if (evt.keycode == DIKC_KP_MINUS || 
                          evt.keycode == DIKC_DOWN)
                        media_set_zoom (media, zoom - adjust);
                      else
                        media_set_zoom (media, zoom + adjust);
                      break;
                      
                    case DIKC_ZOOM:
                    case DIKC_M:
                      zoomed = 1;
                      media_set_zoom (media, 1000.0);
                      break;
                      
                    case DIKC_9:
                      media_rotate (media, 0);
                      break;
                      
                    case DIKC_0:
                      media_rotate (media, 1);
                      break;
                      
                    case DIKC_T:
                      show_time = !show_time;
                      interface_update (TIME);
                      break;
                      
                    case DIKC_F1:
                      if (media->video)
                        media->video->Stop (media->video);
                      interface_update (CLEAR | HELP);
                      help = 1;
                      break;
                      
                    case DIKC_ESCAPE:
                    case DIKC_Q:
                    case DIKC_POWER:
                      quit = 1;
                      break;
                      
                    default:
                      break;
                    }
                  break;
                  
                case DIET_KEYRELEASE:
                  if (evt.keycode == DIKC_ENTER)
                    paused = !paused;
                  break;
                  
                default:
                  break;
                }
            }
        }
    }

  media_deinit ();
  interface_deinit ();

  primary->Release (primary);
  keybuffer->Release (keybuffer);
  dfb->Release (dfb);

  return 0;
}
