/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>

#include <pthread.h>

#include <directfb.h>

#include "dfbsee.h"
#include "interface.h"
#include "interface-draw.h"
#include "media.h"


typedef struct _InterfaceState InterfaceState;
struct _InterfaceState {
  InterfaceUpdateFlags  update;

  Media                *media;
  int                   volume;
  double                zoom;
  DFBRectangle          rect;
  const char           *message;
  int                   message_frame;
  int                   message_anim;
};


static pthread_mutex_t  interface_mutex  = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t   interface_cond   = PTHREAD_COND_INITIALIZER;
static pthread_t        interface_thread = 0;
static InterfaceState   state;

static void * interface_thread_func (void *data);


IDirectFBSurface *
interface_init (void)
{
  IDirectFBSurface *sub;
  int               ready = 0;

  sub = interface_draw_init ();

  pthread_create (&interface_thread, NULL, interface_thread_func, &ready);

  while (!ready)
    sched_yield();

  return sub;
}

void
interface_deinit (void)
{
  if (interface_thread)
    {
      pthread_cancel (interface_thread);
      pthread_join (interface_thread, NULL);
      interface_thread = 0;
    }

  interface_draw_deinit ();
}

void 
interface_set_message (const char *message,
                       int         animate)
{
  pthread_mutex_lock (&interface_mutex);

  if (state.message)
    free ((void *) state.message);

  state.message = NULL;
  state.message_frame = 0;
  state.message_anim = animate;

  if (message)
    state.message = strdup (message);
  
  state.update |= MESSAGE;

  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void 
interface_set_media (Media *media)
{
  pthread_mutex_lock (&interface_mutex);

  state.media = media;
  state.zoom  = 0.0;
  state.update |= (TITLE | SIZE | ZOOM | PROGRESS);

  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void
interface_set_volume (int volume)
{
  pthread_mutex_lock (&interface_mutex);

  state.volume = volume;
  state.update |= VOLUME;

  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void
interface_set_zoom (double zoom)
{
  pthread_mutex_lock (&interface_mutex);

  state.zoom = zoom;
  state.update |= ZOOM;

  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void
interface_set_rect (DFBRectangle *rect)
{
  pthread_mutex_lock (&interface_mutex);

  state.rect = *rect;
  state.update |= IMAGE;

  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void
interface_update (InterfaceUpdateFlags update)
{
  pthread_mutex_lock (&interface_mutex);
  state.update |= update;
  pthread_cond_signal (&interface_cond);
  pthread_mutex_unlock (&interface_mutex);
}

void
interface_lock (void)
{
  pthread_mutex_lock (&interface_mutex);
}

void
interface_unlock (void)
{
  pthread_mutex_unlock (&interface_mutex);
}

static void *
interface_thread_func (void *data)
{
  struct timeval  now;
  struct timespec timeout;
  DFBColorAdjustment adj;
  double pos;
  unsigned int lasttime = 0;
  unsigned int thistime = 0;

  memset (&state, 0, sizeof (InterfaceState));
  state.volume = -1;
  
  pthread_mutex_lock (&interface_mutex);

  gettimeofday (&now, NULL);
  timeout.tv_sec  = now.tv_sec + 1;
  timeout.tv_nsec = 0;

  *((int*)data) = 1;

  while (1)
    {
      if (pthread_cond_timedwait (&interface_cond, &interface_mutex, &timeout)
          == ETIMEDOUT)
        {
          if (state.media && state.media->can_seek && 
              state.media->color_adjustment == DCAF_NONE &&
              state.volume < 0)
            state.update |= PROGRESS;
          if (show_time)
            state.update |= TIME;
        }
        
      gettimeofday (&now, NULL);
      if (state.message_anim)
        {
          timeout.tv_sec  = now.tv_sec;
          timeout.tv_nsec = (now.tv_usec + 10000) * 1000;
          thistime = now.tv_sec * 1000 + now.tv_usec / 1000;
          if (lasttime)
            state.message_frame += (thistime - lasttime) / 8;
          lasttime = thistime;
          state.update |= MESSAGE;
        }
      else
        {
          timeout.tv_sec  = now.tv_sec + 1;
          timeout.tv_nsec = 0;
          lasttime = 0;
        }

      if (state.update == NONE)
        continue;

      pos = 0.0;
      adj.flags = DCAF_NONE;

      if (state.update & (SCREEN | CLEAR))
        interface_draw_clear (state.update & SCREEN);
      
      if (state.update & HELP)
        interface_draw_help ();

      if (state.update & MESSAGE)
        state.message_frame = interface_draw_message (state.message,
                                                      state.message_frame);

      if (!fullscreen)
        {
          if (state.update & FRAME)
            interface_draw_frame ();

          if (state.update & TITLE)
            interface_draw_title (state.media ? state.media->filename : NULL);

          if (state.update & SIZE)
            {
              if (state.media)
                interface_draw_size (state.media->width, state.media->height);
              else
                interface_draw_size (0, 0);
            }

          if (state.update & ZOOM)
            interface_draw_zoom (state.zoom);

          if (state.update & TIME)
            interface_draw_time (&now.tv_sec);

          if (state.update & VOLUME)
            interface_draw_volume (state.volume);
        }

      if (state.media)
        {
          Media *media = state.media;
          
          if (media->image)
            {
              if (state.update & IMAGE)
                interface_draw_image (media->image, 
                                      media->width, media->height,
                                      media->caps,
                                      &state.rect);
            }
          else if (media->video)
            {
              if (state.update & IMAGE)
                {
                  interface_draw_video (media->video, 
                                        media->width, media->height,
                                        &state.rect, media->rot);
                  state.media->running = 1;
                }
              
              if (state.update & COLOR)
                {
                  media->video->GetColorAdjustment (media->video, &adj);
                  adj.flags &= state.media->color_adjustment;
                }
              else if (state.update & PROGRESS)
                {
                  media->video->GetPos (media->video, &pos);
                }
            }

          if (!fullscreen && !(state.update & VOLUME))
            {
              if (state.update & COLOR)
                interface_draw_color_adjust (&adj);
              else if (state.update & PROGRESS)
                interface_draw_progress (pos, media->length);
            }
        }

      state.update = NONE;
    }

  pthread_mutex_unlock (&interface_mutex);

  return NULL;
}
