/* DirectFB Image Viewer
 *
 * Copyright (C) 2001  convergence integrated media
 * Authors: Sven Neumann <sven@convergence.de>
 *          Andreas Hundt <andi@convergence.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <directfb.h>

#include "dfbsee.h"
#include "interface-draw.h"
#include "video-draw.h"


#define SET_COLOR_LIGHT(s) ((s)->SetColor ((s), 116, 154, 224, 255))
#define SET_COLOR_DARK(s)  ((s)->SetColor ((s),  53,  66, 234, 255))
#define SET_COLOR_BLACK(s) ((s)->SetColor ((s),   0,   0,   0, 255))

#define HEAD_HEIGHT        20
#define FRAME_WIDTH        79
#define FOOT_HEIGHT        15
#define FONT_HEIGHT        28
#define SMALL_FONT_HEIGHT  18
#define ANIM_WIDTH         32


static IDirectFBSurface *top_left_surface    = NULL;
static IDirectFBSurface *bottom_left_surface = NULL;
static IDirectFBSurface *top_right_surface   = NULL;
static IDirectFBFont    *font                = NULL;
static IDirectFBFont    *small_font          = NULL;

static IDirectFBSurface *image_area          = NULL;
static IDirectFBSurface *title_surface       = NULL;
static IDirectFBSurface *size_surface        = NULL;
static IDirectFBSurface *zoom_surface        = NULL;
static IDirectFBSurface *progress_surface    = NULL;
static IDirectFBSurface *time_surface        = NULL;


struct Help
{
  const char *key;
  const char *str;
};

static struct Help help[] =
{
  { "Z",                  "Toggle Zoom"            },
  { "Cursor Up",          "Increase Zoom"          },
  { "Cursor Down",        "Decrease Zoom"          },
  { "M",                  "Maximize Zoom"          },
  { "F",                  "Toggle Fullscreen"      }, 
  { "R",                  "Toggle Override Ratio"  },
  { "Space | PageDown",   "Next file"              },
  { "Backspace | PageUp", "Previous file"          },
  { "Cursor Left",        "Rewind Video"           },
  { "Cursor Down",        "Forward Video"          },
  { "Home",               "Jump to Start of Video" },
  { "End",                "Jump to End of Video"   },
  { "Enter",              "Pause/Restart Video"    },
  { "V",                  "Adjust Volume"          },
  { "B/C/S/H",            "Adjust Video Colors"    },
  { "T",                  "Show/Hide current time" },
  { "Esc | Q",            "Quit"                   }
};
static int num_help = sizeof (help) / sizeof (help[0]);


IDirectFBSurface *
interface_draw_init (void)
{
  IDirectFBImageProvider *provider;
  DFBSurfaceDescription   dsc;
  DFBFontDescription      font_dsc;
  DFBRectangle            rect;
  int width, height;

  /*  create images  */
  DFBCHECK (dfb->CreateImageProvider (dfb, DATADIR "top_left.png",
				      &provider));
  provider->GetSurfaceDescription (provider, &dsc);
  dsc.flags |= DSDESC_PIXELFORMAT;
  dsc.pixelformat = DSPF_A8;
  DFBCHECK (dfb->CreateSurface (dfb, &dsc, &top_left_surface));
  provider->RenderTo (provider, top_left_surface);
  provider->Release (provider);

  DFBCHECK (dfb->CreateImageProvider (dfb, DATADIR "bottom_left.png",
				      &provider));
  provider->GetSurfaceDescription (provider, &dsc);
  dsc.flags |= DSDESC_PIXELFORMAT;
  dsc.pixelformat = DSPF_A8;
  DFBCHECK (dfb->CreateSurface (dfb, &dsc, &bottom_left_surface));
  provider->RenderTo (provider, bottom_left_surface);
  provider->Release (provider);

  DFBCHECK (dfb->CreateImageProvider (dfb, DATADIR "top_right.png",
				      &provider));
  provider->GetSurfaceDescription (provider, &dsc);
  dsc.flags |= DSDESC_PIXELFORMAT;
  dsc.pixelformat = DSPF_A8;
  DFBCHECK (dfb->CreateSurface (dfb, &dsc, &top_right_surface));
  provider->RenderTo (provider, top_right_surface);
  provider->Release (provider);

  /*  create fonts  */
  font_dsc.flags = DFDESC_HEIGHT;
  font_dsc.height = FONT_HEIGHT;

  DFBCHECK (dfb->CreateFont (dfb, DATADIR "decker.ttf", &font_dsc, &font));

  font_dsc.flags = DFDESC_HEIGHT;
  font_dsc.height = SMALL_FONT_HEIGHT;

  DFBCHECK (dfb->CreateFont (dfb, DATADIR "decker.ttf",
			     &font_dsc, &small_font));

  /*  title surface  */
  top_left_surface->GetSize (top_left_surface, &width, &height);
  rect.y = 5;
  rect.h = HEAD_HEIGHT;
  rect.x = 130 + width;
  top_right_surface->GetSize (top_right_surface, &width, NULL); 
  rect.w = screen_width - width - 8 - rect.x;
  primary->GetSubSurface (primary, &rect, &title_surface);
  title_surface->SetFont (title_surface, font);

  /*  time surface  */
  top_left_surface->GetSize (top_left_surface, &width, &height);
  rect.y = 8;
  rect.h = HEAD_HEIGHT - 6;
  rect.x = 40 + width;
  rect.w = 80;
  primary->GetSubSurface (primary, &rect, &time_surface);
  time_surface->SetFont (time_surface, small_font);

  /*  size surface  */
  rect.x = 5;
  rect.w = FRAME_WIDTH;
  rect.y = 5 + height;
  rect.h = 200;
  primary->GetSubSurface (primary, &rect, &size_surface);
  size_surface->SetFont (size_surface, small_font);

  /*  zoom surface  */
  bottom_left_surface->GetSize (bottom_left_surface, &width, &height);
  rect.y += rect.h + 5;
  rect.h = screen_height - height - rect.y;
  primary->GetSubSurface (primary, &rect, &zoom_surface);
  zoom_surface->SetFont (zoom_surface, small_font);

  /*  progress surface  */
  rect.x = width + 50;
  rect.y = screen_height - 5 - FOOT_HEIGHT;
  rect.w = screen_width - 5 - rect.x;
  rect.h = FOOT_HEIGHT;
  primary->GetSubSurface (primary, &rect, &progress_surface);
  progress_surface->SetFont (progress_surface, small_font);

  /*  image surface  */
  rect.x = FRAME_WIDTH + 15;
  rect.y = HEAD_HEIGHT + 15;
  rect.w = screen_width - FRAME_WIDTH - 20;
  rect.h = screen_height - HEAD_HEIGHT - FOOT_HEIGHT - 30;
  primary->GetSubSurface (primary, &rect, &image_area);

  primary->SetFont (primary, font);
  image_area->SetFont (image_area, font);

  return image_area;
}

void
interface_draw_deinit (void)
{
  video_draw_deinit ();
  zoom_surface->Release (zoom_surface);
  top_left_surface->Release (top_left_surface);
  bottom_left_surface->Release (bottom_left_surface);
  top_right_surface->Release (top_right_surface);
}

void
interface_draw_clear (int screen)
{
  IDirectFBSurface *dest;
  int w, h;

  dest = (screen || fullscreen) ? primary : image_area;

  dest->GetSize (dest, &w, &h);

  SET_COLOR_BLACK (dest);
  primary->FillRectangle (dest, 0, 0, w, h);
}

void
interface_draw_frame (void)
{
  int width, height;

  /*  topleft corner image  */
  SET_COLOR_LIGHT (primary);
  top_left_surface->GetSize (top_left_surface, &width, &height);
  primary->SetBlittingFlags (primary, (DSBLIT_BLEND_ALPHACHANNEL |
				       DSBLIT_COLORIZE));
  primary->Blit (primary, top_left_surface, NULL, 5, 5);

  /*  the top left rectangle  */
  primary->FillRectangle (primary, 5 + width, 5, 120, HEAD_HEIGHT);

  /*  bottom left corner image  */
  bottom_left_surface->GetSize (bottom_left_surface, &width, &height);
  SET_COLOR_DARK (primary);
  primary->Blit (primary, bottom_left_surface, NULL,
		 5, screen_height - 5 - height);

  /*  the bottom left rectangle  */
  primary->FillRectangle (primary,
			  5 + width, screen_height - 5 - FOOT_HEIGHT,
			  screen_width - 10 - width, FOOT_HEIGHT);

  /*  top right corner image  */
  top_right_surface->GetSize (top_right_surface, &width, &height);
  primary->Blit (primary, top_right_surface, NULL,
		 screen_width - 5 - width, 5);
  primary->FillRectangle (primary,
			  screen_width - 5 - width - 3, 5, 3, height);

  primary->SetBlittingFlags (primary, DSBLIT_NOFX);

  SET_COLOR_LIGHT (size_surface);
  size_surface->FillRectangle (size_surface, 0, 0, screen_width, screen_height);
  SET_COLOR_DARK (zoom_surface);
  zoom_surface->FillRectangle (zoom_surface, 0, 0, screen_width, screen_height);
  SET_COLOR_DARK (title_surface);
  title_surface->FillRectangle (title_surface, 0, 0, screen_width, screen_height);
}

int
interface_draw_message (const char *message,
                        int         frame)
{
  static DFBRectangle ink = { 0, 0, 0, 0 };
  static DFBRectangle log = { 0, 0, 0, 0 };
  static IDirectFBSurface *surface = NULL;
  static const char *last_message = NULL;
  static int last = 0;
  IDirectFBSurface *dest;
  DFBRectangle rect;
  int w, h;

  dest = fullscreen ? primary : image_area;
  dest->GetSize (dest, &w, &h);

  if (surface)
    {
      if (message && strcmp (message, last_message))
        {
          surface->Release (surface);
          surface = NULL;
          SET_COLOR_BLACK (dest);
          dest->FillRectangle (dest, ink.x + w/2, ink.y + h/2, ink.w, ink.h);
          ink.x = ink.y = ink.w = ink.h = 0;
          last = 0;
          free ((void *) last_message);
        }
    }

  if (!message)
    return -1;

  if (!surface)
    {
      DFBSurfaceDescription dsc;
      
      font->GetStringExtents (font, message, -1, &log, &ink);
      
      dsc.flags = DSDESC_WIDTH | DSDESC_HEIGHT | DSDESC_PIXELFORMAT;
      dsc.width  = ink.w;
      dsc.height = 2 * ink.h;
      primary->GetPixelFormat (primary, &dsc.pixelformat);

      if (dfb->CreateSurface (dfb, &dsc, &surface) == DFB_OK)
        {
          SET_COLOR_BLACK (surface);
          surface->FillRectangle (surface, 0, 0, ink.w, 2 * ink.h);
          surface->SetFont (surface, font);
          SET_COLOR_DARK (surface);
          surface->DrawString (surface, message, -1, 
                               log.x - ink.x, log.y - ink.y, DSTF_TOPLEFT);
          SET_COLOR_LIGHT (surface);
          surface->DrawString (surface, message, -1, 
                               log.x - ink.x, log.y - ink.y + ink.h, 
                               DSTF_TOPLEFT);
          
          ink.x -= log.w / 2;
          ink.y += log.h / 2;                
          
          last_message = strdup (message);
        }
    }
  
  if (surface)
    {
      if (frame == 0)
        {
          rect.y = 0; rect.x = 0; rect.w = ink.w; rect.h = ink.h;
          dest->SetBlittingFlags (dest, DSBLIT_NOFX);
          dest->Blit (dest, surface, &rect, ink.x + w/2, ink.y + h/2);
        }
      else
        {
          while (frame > ink.w + ANIM_WIDTH)
            frame -= ink.w + ANIM_WIDTH;
      
          dest->SetBlittingFlags (dest, DSBLIT_NOFX);
          
          rect.y = 0;
          rect.x = last - ANIM_WIDTH;
          rect.w = 
            (frame >= last) ? MIN (ANIM_WIDTH, frame - last) : ANIM_WIDTH; 
          rect.h = ink.h;
          dest->Blit (dest, surface, &rect, rect.x + ink.x + w/2, ink.y + h/2);

          rect.y = ink.h;
          if (frame >= last)
            {
              rect.x = MAX (last, frame - ANIM_WIDTH);
              rect.w = frame - last;
            }
          else
            {
              rect.x = frame - ANIM_WIDTH;
              rect.w = ANIM_WIDTH;
            }
          rect.h = ink.h;
          dest->Blit (dest, surface, &rect, rect.x + ink.x + w/2, ink.y + h/2);

          last = frame;
        }
    }

  return frame;
}

void
interface_draw_title (const char *title)
{ 
  static int str_width = 0;
  int width, height;

  if (fullscreen)
    return;

  title_surface->GetSize (title_surface, &width, &height);

  if (str_width)
    {
      SET_COLOR_DARK (title_surface);
      title_surface->FillRectangle (title_surface, 
                                    width - str_width - 8, 0, 
                                    str_width + 8, height);
      str_width = 0;
    }

  if (title)
    {
      int i;
      char *dup;

      dup = strdup (title);

      for (i = 0; i < strlen (dup); i++)
	dup[i] = toupper (dup[i]);

      font->GetStringWidth (font, dup, -1, &str_width);

      SET_COLOR_BLACK (title_surface);
      title_surface->FillRectangle (title_surface,
                                    width - str_width - 8, 0, 
                                    str_width + 8, height);
      SET_COLOR_DARK (title_surface);
      title_surface->DrawString (title_surface, dup, -1,
                                 width - str_width - 5, height, DSTF_LEFT);

      free (dup);
    }
}
  
void
interface_draw_size (int width, 
                     int height)
{
  static DFBRectangle ink = { 0, 0, 0, 0 };
  DFBRectangle log;
  int w, h;
  char buffer[16];

  if (fullscreen)
    return;

  if (ink.w && ink.h)
    {
      SET_COLOR_LIGHT (size_surface);
      size_surface->FillRectangle (size_surface, ink.x, ink.y, ink.w, ink.h);
      ink.w = ink.h = 0;
    }

  if (width > 0 && height > 0)
    {
      size_surface->GetSize (size_surface, &w, &h);

      snprintf (buffer, sizeof (buffer), "%dx%d", width, height);
      
      small_font->GetStringExtents (small_font, buffer, -1, &log, &ink);

      SET_COLOR_BLACK (size_surface);
      if (log.w > w - 2)
	{
	  snprintf (buffer, sizeof (buffer), "%d", width);
	  size_surface->DrawString (size_surface, buffer, -1,
                                    w / 2, h - 16 - 2 * SMALL_FONT_HEIGHT, 
                                    DSTF_CENTER);
	  size_surface->DrawString (size_surface, "x", -1,
                                    w / 2, h - 16 - SMALL_FONT_HEIGHT,
                                    DSTF_CENTER);
	  snprintf (buffer, sizeof (buffer), "%d", height);
	  size_surface->DrawString (size_surface, buffer, -1,
                                    w / 2, h - 16, 
                                    DSTF_CENTER);

          /* this is not the real rectangle, but a cheap approximation */
          ink.x += (w - log.w) / 2;
          ink.y += h - 16 - 2 * SMALL_FONT_HEIGHT;
          ink.h += 2 * SMALL_FONT_HEIGHT;
	}
      else
	{
	  size_surface->DrawString (size_surface, buffer, -1,
                                    (w - log.w) / 2, h - 16, DSTF_LEFT);
          ink.x += (w - log.w) / 2;
          ink.y += h - 16;
	}
    }
}

void
interface_draw_zoom (double zoom)
{
  static DFBRectangle ink = { 0, 0, 0, 0 };
  DFBRectangle log;
  char buffer[16];
  int  w, h;

  if (fullscreen)
    return;

  if (ink.w && ink.h)
    {
      SET_COLOR_DARK (zoom_surface);
      zoom_surface->FillRectangle (zoom_surface, ink.x, ink.y, ink.w, ink.h);
      ink.w = ink.h = 0;
    }

  if (zoom > 0.0)
    {
      zoom_surface->GetSize (zoom_surface, &w, &h);

      SET_COLOR_BLACK (zoom_surface);
      
      snprintf (buffer, sizeof (buffer), "%d%%", (int) (zoom * 100.0 + 0.5));
      small_font->GetStringExtents (small_font, buffer, -1, &log, &ink);
      zoom_surface->DrawString (zoom_surface, 
                                buffer, -1, 
                                w / 2 - log.w / 2, 8 - log.y, DSTF_LEFT);  
      ink.x += w / 2 - log.w / 2;
      ink.y += 8 - log.y;
    }
}

static void
interface_make_timestring (char   *buffer, 
                           size_t  len, 
                           double  seconds)
{
  int h, m, s;

  if (seconds > 0)
    s = (int) seconds;
  else
    s = 0;

  h = s / 3600;
  s -= 3600 * h;
  m = s / 60;
  s -= 60 * m;
  if (s < 0)
    s = 0;
  snprintf (buffer, len, "%d:%02d:%02d", h, m, s);
}

void
interface_draw_progress (double pos, 
                         double length)
{
  char buffer[16];
  int  w, h;
  int  start;

  if (fullscreen)
    return;

  progress_surface->GetSize (progress_surface, &w, &h);

  if (length <= 0.0)
    {
      SET_COLOR_DARK (progress_surface);
      progress_surface->FillRectangle (progress_surface, 0, 0, w, h); 
      return;
    }

  start = (w * pos) / length + 1;

  SET_COLOR_LIGHT (progress_surface);
  progress_surface->FillRectangle (progress_surface, 0, 0, start, h);

  SET_COLOR_DARK (progress_surface);
  progress_surface->FillRectangle (progress_surface, start, 0, w - start, h); 

  SET_COLOR_BLACK (progress_surface);
  interface_make_timestring (buffer, sizeof (buffer), length);
  progress_surface->DrawString (progress_surface, 
                                buffer, -1, w - 5, h - 1, DSTF_RIGHT);

  interface_make_timestring (buffer, sizeof (buffer), pos);
  progress_surface->DrawString (progress_surface, 
                                buffer, -1, 5, h - 1, DSTF_LEFT);
}

void
interface_draw_volume (int volume)
{
  char buffer[16];
  int  w, h;
  int  start;

  if (fullscreen)
    return;

  progress_surface->GetSize (progress_surface, &w, &h);

  if (volume < 0)
    {
      SET_COLOR_DARK (progress_surface);
      progress_surface->FillRectangle (progress_surface, 0, 0, w, h);
      return;
    }

  start = (w * volume) / 100 + 1;

  SET_COLOR_LIGHT (progress_surface);
  progress_surface->FillRectangle (progress_surface, 0, 0, start, h);

  SET_COLOR_DARK (progress_surface);
  progress_surface->FillRectangle (progress_surface, start, 0, w - start, h);


  sprintf (buffer, "%d %%", volume);

  SET_COLOR_BLACK (progress_surface);
  progress_surface->DrawString (progress_surface, 
                                buffer, -1, w/2, h - 1, DSTF_CENTER);
}

void
interface_draw_time (const time_t *now)
{
  char buffer[10];
  struct tm *local;
  int w, h;

  local = localtime (now);
  sprintf (buffer, "%2d:%02d:%02d", 
           local->tm_hour, local->tm_min, local->tm_sec);

  time_surface->GetSize (time_surface, &w, &h);
  SET_COLOR_LIGHT (time_surface);
  time_surface->FillRectangle (time_surface, 0, 0, w, h);

  
  if (show_time)
    {
      SET_COLOR_BLACK (time_surface);
      time_surface->DrawString (time_surface, buffer, -1, w, h, DSTF_RIGHT);
    }
}

void
interface_draw_color_adjust (DFBColorAdjustment *adj)
{
  int w, h, value;
  char *name;

  if (fullscreen)
    return;

  progress_surface->GetSize (progress_surface, &w, &h);

  switch (adj->flags)
    {
    case DCAF_BRIGHTNESS:
      name = "BRIGHTNESS";
      value = adj->brightness;
      break;
    case DCAF_CONTRAST:
      name = "CONTRAST";
      value = adj->contrast;
      break;
    case DCAF_HUE:
      name = "HUE";
      value = adj->hue;
      break;
    case DCAF_SATURATION:
      name = "SATURATION";
      value = adj->saturation;
      break;
    default:
      SET_COLOR_DARK (progress_surface);
      progress_surface->FillRectangle (progress_surface, 0, 0, w, h); 
      return;
    }

  SET_COLOR_DARK (progress_surface);
  progress_surface->FillRectangle (progress_surface, 0, 0, 125, h);

  SET_COLOR_BLACK (progress_surface);
  progress_surface->DrawString (progress_surface, 
                                name, -1, 120, h - 1, DSTF_RIGHT);

  value = (value * (w - 125)) / 0xFFFF + 1;
  
  SET_COLOR_LIGHT (progress_surface);
  progress_surface->FillRectangle (progress_surface, 125, 0, value, h);
  
  SET_COLOR_DARK (progress_surface);
  progress_surface->FillRectangle (progress_surface, 
                                   125 + value, 0, w - 125 - value, h);
}

void
interface_draw_image (IDirectFBSurface     *image,
                      int                   width,
                      int                   height,
                      DFBImageCapabilities  caps,
                      DFBRectangle         *rect)
{
  IDirectFBSurface *dest = fullscreen ? primary : image_area;

  if (caps & DICAPS_COLORKEY)
    dest->SetBlittingFlags (dest, DSBLIT_SRC_COLORKEY);

  if (rect->w == width && rect->h == height)
    dest->Blit (dest, image, NULL, rect->x, rect->y);
  else
    dest->StretchBlit (dest, image, NULL, rect);

  if (caps & DICAPS_COLORKEY)
    dest->SetBlittingFlags (dest, DSBLIT_NOFX);  
}

void
interface_draw_video (IDirectFBVideoProvider *video,
                      int                     width,
                      int                     height,
                      DFBRectangle           *rect,
                      RotationType            rot)
{
  video_draw (fullscreen ? primary : image_area, 
              video, width, height, rect, rot);
}

void
interface_draw_help (void)
{
  IDirectFBSurface *dest;
  int x, y, i;

  dest = fullscreen ? primary : image_area;
  dest->GetSize (dest, &x, &y);
  dest->SetFont (dest, small_font);
  SET_COLOR_LIGHT (dest);

  x /= 2;
  y = (y - num_help * 24) / 2;

  for (i = 0; i < num_help; i++, y+= 24)
    {
      SET_COLOR_LIGHT (dest);
      dest->DrawString (dest, help[i].key, -1, x - 10, y, DSTF_TOPRIGHT);
      SET_COLOR_DARK (dest);
      dest->DrawString (dest, help[i].str, -1, x + 10, y, DSTF_TOPLEFT);
    }
}
