//   Copyright (c)  2007-2013  John Abbott, Anna M. Bigatti

//   This file is part of the source of CoCoALib, the CoCoA Library.

//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.

//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.

//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.


#include "CoCoA/BigIntOps.H"
#include "CoCoA/BuildInfo.H"
#include "CoCoA/FractionField.H"
#include "CoCoA/FreeModule.H"
#include "CoCoA/GlobalManager.H"
#include "CoCoA/ModuleOrdering.H"
#include "CoCoA/RingDistrMPolyClean.H"
#include "CoCoA/RingQQ.H"
#include "CoCoA/SparsePolyOps-RingElem.H"
//#include "CoCoA/SparsePolyRing.H"
#include "CoCoA/degree.H"

#include <iostream>
using std::cout;
using std::cerr;
using std::endl;
#include <vector>
using std::vector;

namespace CoCoA
{

  //----------------------------------------------------------------------
  // Test for FreeModules over SparsePolyRing with many different orderings
  // functions:  LPosn(v),  LPP(v),  wdeg(v),  IsHomog(v)
  // environments: Q[x,y,z], WDegPosnOrd/OrdPosn, shifts
  //----------------------------------------------------------------------

  void trial(const FreeModule& F)
  {
    const vector<ModuleElem>& e = gens(F);
    const SparsePolyRing P = RingOf(F);
    const vector<RingElem>& x = indets(P);

    const ModuleElem u = power(x[1],5)*e[0] + x[0]*e[1];
    const ModuleElem v = x[2]*x[2]*e[1] + x[1]*x[2]*e[2];
    const RingElem a = 2 * one(RingOf(F));  // RingOf(F) is R

    cout << "---- F = " << F << " ----" << endl;
    cout << "-- " << ordering(F) << endl;
    //  cout << "-- " << ordering(PPM(P)) << endl;
    cout << "u[0] = " << u[0] << "  and  u[1] = " << u[1] << endl;
    cout << "u = " << u << endl;
    cout << "LPosn(u) = " << LPosn(u) << std::endl;
    cout << "LPP(u) = " << LPP(u) << endl;
    cout << "wdeg(u) = " << wdeg(u) << endl;
    if (GradingDim(P)==0)
      cout << "IsHomog(u) undefined because GradingDim is 0" << endl;
    else
      cout << "IsHomog(u) = " << IsHomog(u) << std::endl;
    cout << "v = " << v << endl;
    cout << "LPosn(v) = " << LPosn(v) << std::endl;

    cout << endl;
  }


  void program()
  {
    GlobalManager CoCoAFoundations;
    std::cout << std::boolalpha;

    ring QQ = RingQQ();
    SparsePolyRing PLex = NewPolyRing(QQ, symbols("x,y,z"), lex);
    SparsePolyRing PDegLex = NewPolyRing(QQ, symbols("x,y,z"), StdDegLex);
    long n = 4;
    std::vector<degree> sh(n, wdeg(one(PDegLex)));
    sh[1] = wdeg(power(indet(PDegLex,0),4));
    trial(NewFreeModule(PLex, n));
    trial(NewFreeModule(PDegLex, n));
    //  trial(NewFreeModule(PDegLex, n, PosnOrd));
    trial(NewFreeModule(PDegLex, n, OrdPosn));
    trial(NewFreeModule(PLex, n, WDegPosnOrd));
    trial(NewFreeModule(PDegLex, n, WDegPosnOrd));
    trial(NewFreeModule(PDegLex, sh, WDegPosnOrd));
  }

} // end of namespace CoCoA


// Use main() to handle any uncaught exceptions and warn the user about them.
int main()
{
  try
  {
    CoCoA::program();
    return 0;
  }
  catch (const CoCoA::ErrorInfo& err)
  {
    cerr << "***ERROR***  UNCAUGHT CoCoA error";
    ANNOUNCE(cerr, err);
  }
  catch (const std::exception& exc)
  {
    cerr << "***ERROR***  UNCAUGHT std::exception: " << exc.what() << endl;
  }
  catch(...)
  {
    cerr << "***ERROR***  UNCAUGHT UNKNOWN EXCEPTION" << endl;
  }
  CoCoA::BuildInfo::PrintAll(cerr);
  return 1;
}

//----------------------------------------------------------------------
// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/tests/test-FreeModule2.C,v 1.13 2018/09/28 15:54:05 abbott Exp $
// $Log: test-FreeModule2.C,v $
// Revision 1.13  2018/09/28 15:54:05  abbott
// Summary: Removed pseudo-ctors NewPolyRing which took just num of indets; now must specify their names
//
// Revision 1.12  2018/05/18 16:46:17  bigatti
// -- added include SparsePolyOps-RingElem.H
//
// Revision 1.11  2018/05/18 12:29:29  bigatti
// -- renamed IntOperations --> BigIntOps
//
// Revision 1.10  2016/09/16 16:36:37  abbott
// Summary: Changed TEST_ASSERT into CoCoA_ASSERT_ALWAYS; removed all include assert.H lines
//
// Revision 1.9  2014/10/29 10:59:12  bigatti
// -- improved comment
//
// Revision 1.8  2014/10/29 10:47:29  abbott
// Summary: moved code into namespace CoCoA; some minor tidying
// Author: JAA
//
// Revision 1.7  2014/07/30 14:28:52  abbott
// Summary: Changed BaseRing into RingOf
// Author: JAA
//
// Revision 1.6  2014/07/07 13:38:35  abbott
// Summary: Removed AsSparsePolyRing
// Author: JAA
//
// Revision 1.5  2013/08/02 16:05:50  bigatti
// -- LPos --> LPosn
//
// Revision 1.4  2013/06/03 09:12:24  bigatti
// renamed ModuleTermOrdering into ModuleOrdering
//
// Revision 1.3  2013/05/28 11:50:32  bigatti
// -- minor comments
//
// Revision 1.2  2013/05/28 06:58:47  bigatti
// -- improved test
//
// Revision 1.1  2013/05/27 16:52:35  bigatti
// -- first import
//
// Revision 1.6  2013/01/23 14:43:36  bigatti
// -- added "v==w"
//
// Revision 1.5  2012/02/08 17:52:17  bigatti
// -- changed: Z,Q -> ZZ,QQ
//
// Revision 1.4  2010/12/17 16:07:54  abbott
// Ensured that all i/o in examples is on standard C++ streams
// (rather than GlobalInput(), etc).
//
// Revision 1.3  2008/04/21 12:33:45  abbott
// Made a very minor change.
//
// Revision 1.2  2007/06/21 21:29:47  abbott
// Changed name of RingFloat into RingTwinFloat.
//
// Revision 1.1.1.1  2007/03/09 15:16:11  abbott
// Imported files
//
// Revision 1.6  2007/03/03 14:15:45  bigatti
// -- "foundations" renamed into "GlobalManager"
//
// Revision 1.5  2007/02/28 15:15:10  bigatti
// -- added sum and multiplication examples
//
// Revision 1.4  2007/02/26 17:40:34  bigatti
// -- getting ready for unique ring Z: using NewZmod(N), NewRingQ()
//
// Revision 1.3  2007/02/12 16:27:43  bigatti
// -- added strings ShortDescription and LongDescription for indexing
//
// Revision 1.2  2007/02/10 18:44:03  cocoa
// Added "const" twice to each test and example.
// Eliminated dependency on io.H in several files.
// Improved BuildInfo, and added an example about how to use it.
// Some other minor cleaning.
//
// Revision 1.1.1.1  2006/05/30 11:39:36  cocoa
// Imported files
//
// Revision 1.1  2005/12/16 17:53:01  cocoa
// --- first import
//
