//   Copyright (c)  2001-2017  John Abbott and Anna M. Bigatti
//   Author:  2005-2010  John Abbott

//   This file is part of the source of CoCoALib, the CoCoA Library.

//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.

//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.

//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.


// Implementation of classes PPMonoidOvImpl

#include "CoCoA/PPMonoidOv.H"

#include "CoCoA/BigInt.H"
#include "CoCoA/DivMask.H"
#include "CoCoA/MemPool.H"
#include "CoCoA/OrdvArith.H"
#include "CoCoA/PPMonoid.H"
#include "CoCoA/assert.H"
#include "CoCoA/convert.H"
#include "CoCoA/error.H"
#include "CoCoA/symbol.H"
#include "CoCoA/utils.H"
//#include "CoCoA/VectorOps.H" // for debugging only

#include <algorithm>
using std::min;
using std::max;
#include <iostream>
using std::ostream;
#include<limits>
using std::numeric_limits;
#include <memory>
using std::unique_ptr;
#include <vector>
using std::vector;


namespace CoCoA
{

  class PPMonoidOvImpl: public PPMonoidBase
  {
    typedef PPMonoidElemRawPtr RawPtr;           // just to save typing
    typedef PPMonoidElemConstRawPtr ConstRawPtr; // just to save typing
    typedef OrdvArith::OrdvElem OrdvElem;        // just to save typing

    static const unsigned long ourMaxExp;        // defined below, it is just numeric_limits<SmallExponent_t>::max()

  public:
    PPMonoidOvImpl(const std::vector<symbol>& IndetNames, const PPOrdering& ord);
    ~PPMonoidOvImpl();
  private: // disable copy ctor and assignment
    PPMonoidOvImpl(const PPMonoidOvImpl& copy);           // NEVER DEFINED -- copy ctor disabled
    PPMonoidOvImpl& operator=(const PPMonoidOvImpl& rhs); // NEVER DEFINED -- assignment disabled

  public:
    void contents() const; // FOR DEBUGGING ONLY

    const std::vector<PPMonoidElem>& myIndets() const;                  ///< std::vector whose n-th entry is n-th indet as PPMonoidElem

    // The functions below are operations on power products owned by PPMonoidOvImpl
    const PPMonoidElem& myOne() const;
    using PPMonoidBase::myNew;    // disable warnings of overloading
    PPMonoidElemRawPtr myNew() const;                                   ///< ctor from nothing
    PPMonoidElemRawPtr myNew(PPMonoidElemConstRawPtr rawpp) const;      ///< ctor by assuming ownership
    PPMonoidElemRawPtr myNew(const std::vector<long>& expv) const;      ///< ctor from exp vector
//NYI    PPMonoidElemRawPtr myNew(const std::vector<BigInt>& EXPV) const;///< ctor from exp vector
    void myDelete(RawPtr rawpp) const;                                  ///< dtor, frees pp
    void mySwap(RawPtr rawpp1, RawPtr rawpp2) const;                    ///< swap(pp1, pp2);
    void myAssignOne(RawPtr rawpp) const;                               ///< pp = 1
    void myAssign(RawPtr rawpp, ConstRawPtr rawpp1) const;              ///< p = pp1
    void myAssign(RawPtr rawpp, const std::vector<long>& expv) const;   ///< pp = expv (assign from exp vector)

    void myMul(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;   ///< pp = pp1*pp2
    using PPMonoidBase::myMulIndetPower;    // disable warnings of overloading
    void myMulIndetPower(RawPtr rawpp, long indet, long exp) const;           ///< pp *= indet^exp, assumes exp >= 0
    void myDiv(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;   ///< pp = pp1/pp2
    void myColon(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const; ///< pp = pp1/gcd(pp1,pp2)
    void myGcd(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;   ///< pp = gcd(pp1,pp2)
    void myLcm(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;   ///< pp = lcm(pp1,pp2)
    void myRadical(RawPtr rawpp, ConstRawPtr rawpp1) const;                  ///< pp = radical(pp1)
    void myPowerSmallExp(RawPtr rawpp, ConstRawPtr rawpp1, long exp) const;  ///< pp = pp1^exp, assumes exp >=  0
    void myPowerOverflowCheck(ConstRawPtr rawpp1, long exp) const;            ///< throw if pp1^exp would overflow, assumes exp >= 0

    bool myIsOne(ConstRawPtr rawpp) const;                                ///< is pp = 1?
    bool myIsIndet(long& index, ConstRawPtr rawpp) const;                 ///< true iff pp is an indet
    bool myIsCoprime(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;       ///< are pp1 & pp2 coprime?
    bool myIsEqual(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;         ///< is pp1 equal to pp2?
    bool myIsDivisible(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;     ///< does pp2 divide pp1?
    bool myIsSqFree(ConstRawPtr rawpp) const;                             ///< is pp equal to its radical?

    int myCmp(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;              ///< -1,0,1 as pp1 < = > pp2
    long myStdDeg(ConstRawPtr rawpp) const;                               ///< standard degree of pp
    void myWDeg(degree& d, ConstRawPtr rawpp) const;                      ///< d = grading(pp)
    int myCmpWDeg(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const;          ///< <0, =0, >0 as wdeg(pp1) < = > wdeg(pp2)
    int myCmpWDegPartial(ConstRawPtr rawpp1, ConstRawPtr rawpp2, long) const; ///< as myCmpWDeg wrt the first weights
    long myExponent(ConstRawPtr rawpp, long indet) const;                 ///< exponent of indet in pp
    void myBigExponent(BigInt& EXP, ConstRawPtr rawpp, long indet) const; ///< EXP = exponent of indet in pp
    void myExponents(std::vector<long>& expv, ConstRawPtr rawpp) const;   ///< expv[i] = exponent(pp,i)
    void myBigExponents(std::vector<BigInt>& v, ConstRawPtr rawpp) const; ///< get exponents, SHOULD BE myExponents ???
    void myIndetsIn(std::vector<bool>& v, ConstRawPtr rawpp) const;       ///< v[i] = true if i-th indet has exponent != 0
    void myOutputSelf(std::ostream& out) const;                           ///< out << PPM
    // INHERITED DEFINITION of virtual  void myOutput(std::ostream& out, ConstRawPtr rawpp) const;
    void myDebugPrint(std::ostream& out, ConstRawPtr rawpp) const;        ///< print pp in debugging format???


  private: // auxiliary functions
    OrdvElem* myOrdv(RawPtr) const;
    const OrdvElem* myOrdv(ConstRawPtr) const;

    void myComputeDivMask(DivMask& dm, const DivMaskRule& DivMaskImpl, ConstRawPtr rawpp) const; ///< used by PPWithMask
    void myComputeExpv(std::vector<long>& expv, RawPtr rawpp) const;
    bool myCheckExponents(const std::vector<long>& expv) const;
//???    void mySetExpv(RawPtr, const std::vector<long>& expv) const;

  private: // data members
    ///@name Class members
    //@{
    OrdvArith::reference myOrdvArith;  //??? should be const
    const long myEntrySize;
    mutable vector<long> myExpv1;  // buffer space
    mutable vector<long> myExpv2;  // buffer space
    mutable MemPool myMemMgr;     // IMPORTANT: this must come *before* myIndetVector and myOnePtr.
//???    std::vector<long> myDelta;
    vector<PPMonoidElem> myIndetVector; ///< the indets as PPMonoidElems
    unique_ptr<PPMonoidElem> myOnePtr;
    //@}
  };

  // static constant value
  const unsigned long PPMonoidOvImpl::ourMaxExp = numeric_limits<SmallExponent_t>::max();

  // File local inline functions

  inline PPMonoidOvImpl::OrdvElem* PPMonoidOvImpl::myOrdv(RawPtr rawpp) const
  {
    return static_cast<OrdvElem*>(rawpp.myRawPtr());

  }

  inline const PPMonoidOvImpl::OrdvElem* PPMonoidOvImpl::myOrdv(ConstRawPtr rawpp) const
  {
    return static_cast<const OrdvElem*>(rawpp.myRawPtr());
  }


  bool PPMonoidOvImpl::myCheckExponents(const std::vector<long>& expv) const
  {
    // Check len(expv) == myNumIndets.
    // Check exps are non-neg and not too big.
    if (len(expv) != myNumIndets) return false;
    for (long i=0; i < myNumIndets; ++i)
      if (expv[i] < 0 || static_cast<unsigned long>(expv[i]) > numeric_limits<SmallExponent_t>::max()) return false;
    return true;
  }


  //----   Constructors & destructor   ----//

  PPMonoidOvImpl::PPMonoidOvImpl(const std::vector<symbol>& IndetNames, const PPOrdering& ord):
      PPMonoidBase(ord, IndetNames),
      myOrdvArith(NewOrdvArith(ord)),
      myEntrySize(sizeof(OrdvElem)*OrdvWords(myOrdvArith)),
      myExpv1(myNumIndets),
      myExpv2(myNumIndets),
      myMemMgr(myEntrySize, "PPMonoidOvImpl.myMemMgr"),
      myIndetVector()
  {
    //    std::cout << "------PPMonoidOvImpl: called NewOrdvArith" << std::endl;
    myRefCountInc();  // this is needed for exception cleanliness, in case one of the lines below throws
    myOnePtr.reset(new PPMonoidElem(PPMonoid(this)));
    myIndetVector.reserve(myNumIndets);
    {
      // IMPORTANT: this block destroys pp *before* the call to myRefCountZero.
      PPMonoidElem pp(PPMonoid(this));
      vector<long> expv(myNumIndets);
      for (long i=0; i < myNumIndets; ++i)
      {
        expv[i] = 1;
        myAssign(raw(pp), expv);
        myIndetVector.push_back(pp);
        expv[i] = 0;
      }
    }
    myRefCountZero();
  }


  PPMonoidOvImpl::~PPMonoidOvImpl()
  {}


  // mySmartPtr is private
//   const OrdvArith::reference& OrdvA(const PPMonoid& PPM)
//   {
//     if (!IsPPMonoidOv(PPM))
//       CoCoA_ERROR("PPM must be PPMonoidOvImpl", OrdvA);
//     return dynamic_cast<const PPMonoidOvImpl*>(PPM.mySmartPtr()).myOrdvArith;
//   }


/////////////////////////////////////////////////////////////////////////////


  const std::vector<PPMonoidElem>& PPMonoidOvImpl::myIndets() const
  {
    return myIndetVector;
  }


  const PPMonoidElem& PPMonoidOvImpl::myOne() const
  {
    return *myOnePtr;
  }


  PPMonoidElemRawPtr PPMonoidOvImpl::myNew() const
  {
    PPMonoidElemRawPtr rawpp(myMemMgr.alloc());
    myAssignOne(rawpp); // cannot throw
    return rawpp;
  }

  PPMonoidElemRawPtr PPMonoidOvImpl::myNew(PPMonoidElemConstRawPtr rawcopypp) const
  {
    PPMonoidElemRawPtr rawpp(myMemMgr.alloc());
    myAssign(rawpp, rawcopypp); // cannot throw
    return rawpp;
  }


  PPMonoidElemRawPtr PPMonoidOvImpl::myNew(const std::vector<long>& expv) const
  {
    CoCoA_ASSERT(myCheckExponents(expv));
    PPMonoidElemRawPtr rawpp(myMemMgr.alloc());
    myAssign(rawpp, expv); // cannot throw
    return rawpp;
  }


  void PPMonoidOvImpl::myAssignOne(RawPtr rawpp) const
  {
    myOrdvArith->myAssignZero(myOrdv(rawpp));
  }


  void PPMonoidOvImpl::myAssign(RawPtr rawpp, ConstRawPtr rawpp1) const
  {
    if (rawpp == rawpp1) return;
    myOrdvArith->myAssign(myOrdv(rawpp), myOrdv(rawpp1));
  }

  void PPMonoidOvImpl::myAssign(RawPtr rawpp, const vector<long>& expv) const
  {
    CoCoA_ASSERT(myCheckExponents(expv));
    myOrdvArith->myAssignFromExpv(myOrdv(rawpp), expv);
  }


  void PPMonoidOvImpl::myDelete(RawPtr rawpp) const
  {
    myMemMgr.free(rawpp.myRawPtr());
  }


  void PPMonoidOvImpl::mySwap(RawPtr rawpp1, RawPtr rawpp2) const
  {
    if (rawpp1 == rawpp2) return;
    myOrdvArith->mySwap(myOrdv(rawpp1), myOrdv(rawpp2));
  }


  void PPMonoidOvImpl::myMul(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
    myOrdvArith->myMul(myOrdv(rawpp), myOrdv(rawpp1), myOrdv(rawpp2));
  }


  void PPMonoidOvImpl::myMulIndetPower(RawPtr rawpp, long indet, long exp) const  // assumes exp >= 0
  {
    CoCoA_ASSERT(exp >= 0);
    CoCoA_ASSERT(0 <= indet && indet < myNumIndets);
    CoCoA_ASSERT("Exponent Overflow" && ourMaxExp - myExponent(rawpp, indet) >= static_cast<unsigned long>(exp));
    myOrdvArith->myMulIndetPower(myOrdv(rawpp), indet, exp);
  }


  void PPMonoidOvImpl::myDiv(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
    myOrdvArith->myDiv(myOrdv(rawpp), myOrdv(rawpp1), myOrdv(rawpp2));
  }


  void PPMonoidOvImpl::myColon(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
    vector<long> myExpv2(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));
    myOrdvArith->myComputeExpv(myExpv2, myOrdv(rawpp2));

    for (long i = 0; i < myNumIndets; ++i)
    {
      if (myExpv1[i] > myExpv2[i])
        myExpv1[i] -= myExpv2[i];
      else
        myExpv1[i] = 0;
    }
    myOrdvArith->myAssignFromExpv(myOrdv(rawpp), myExpv1);
  }


  void PPMonoidOvImpl::myGcd(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
    vector<long> myExpv2(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));
    myOrdvArith->myComputeExpv(myExpv2, myOrdv(rawpp2));

    for (long i = 0; i < myNumIndets; ++i)
      myExpv1[i] = min(myExpv1[i], myExpv2[i]);

    myOrdvArith->myAssignFromExpv(myOrdv(rawpp), myExpv1);
  }


  void PPMonoidOvImpl::myLcm(RawPtr rawpp, ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
    vector<long> myExpv2(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));
    myOrdvArith->myComputeExpv(myExpv2, myOrdv(rawpp2));

    for (long i = 0; i < myNumIndets; ++i)
      myExpv1[i] = max(myExpv1[i], myExpv2[i]);

    myOrdvArith->myAssignFromExpv(myOrdv(rawpp), myExpv1);
  }


  void PPMonoidOvImpl::myRadical(RawPtr rawpp, ConstRawPtr rawpp1) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));

    for (long i = 0; i < myNumIndets; ++i)
      myExpv1[i] = (myExpv1[i] > 0);

    myOrdvArith->myAssignFromExpv(myOrdv(rawpp), myExpv1);
  }


  void PPMonoidOvImpl::myPowerSmallExp(RawPtr rawpp, ConstRawPtr rawpp1, long exp) const  // assumes exp >= 0
  {
    CoCoA_ASSERT(exp >= 0);
#ifdef CoCoA_DEBUG
    myPowerOverflowCheck(rawpp1, exp);
#endif
    myOrdvArith->myPower(myOrdv(rawpp), myOrdv(rawpp1), exp);
  }


  void PPMonoidOvImpl::myPowerOverflowCheck(ConstRawPtr rawpp, long exp) const
  {
    CoCoA_ASSERT(exp >= 0);
    myOrdvArith->myPowerOverflowCheck(myOrdv(rawpp), exp);
  }


  bool PPMonoidOvImpl::myIsOne(ConstRawPtr rawpp) const
  {
    return myOrdvArith->myIsZero(myOrdv(rawpp));
  }


  bool PPMonoidOvImpl::myIsIndet(long& index, ConstRawPtr rawpp) const
  {
    return myOrdvArith->myIsIndet(index, myOrdv(rawpp));
  }


  bool PPMonoidOvImpl::myIsCoprime(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
    vector<long> myExpv2(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));
    myOrdvArith->myComputeExpv(myExpv2, myOrdv(rawpp2));

    for (long i = 0; i < myNumIndets; ++i)
      if (myExpv1[i] != 0 && myExpv2[i] != 0) return false;

    return true;
  }


  bool PPMonoidOvImpl::myIsEqual(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
    return myOrdvArith->myCmp(myOrdv(rawpp1), myOrdv(rawpp2))==0;
  }


  bool PPMonoidOvImpl::myIsDivisible(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
    vector<long> myExpv2(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp1));
    myOrdvArith->myComputeExpv(myExpv2, myOrdv(rawpp2));

    for (long i = 0; i < myNumIndets; ++i)
      if (myExpv1[i] < myExpv2[i]) return false;

    return true;
  }


  bool PPMonoidOvImpl::myIsSqFree(ConstRawPtr rawpp) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp));
    for (long i = 0; i < myNumIndets; ++i)
      if (myExpv1[i] > 1) return false;
    return true;
  }


  int PPMonoidOvImpl::myCmp(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
    return myOrdvArith->myCmp(myOrdv(rawpp1), myOrdv(rawpp2));
  }


// // should potentially skip the first few packed ordv entries???
// int PPMonoidOvImpl::myHomogCmp(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
// {
//   return myOrdvArith->myCmp(myOrdv(rawpp1), myOrdv(rawpp2));
// }


  long PPMonoidOvImpl::myStdDeg(ConstRawPtr rawpp) const
  {
    return myOrdvArith->myStdDeg(myOrdv(rawpp));
  }


  void PPMonoidOvImpl::myWDeg(degree& d, ConstRawPtr rawpp) const
  {
    myOrdvArith->myWDeg(d, myOrdv(rawpp));
  }


  int PPMonoidOvImpl::myCmpWDeg(ConstRawPtr rawpp1, ConstRawPtr rawpp2) const
  {
    return myOrdvArith->myCmpWDeg(myOrdv(rawpp1), myOrdv(rawpp2));
  }


  int PPMonoidOvImpl::myCmpWDegPartial(ConstRawPtr rawpp1, ConstRawPtr rawpp2, long i) const
  {
    return myOrdvArith->myCmpWDegPartial(myOrdv(rawpp1), myOrdv(rawpp2), i);
  }


  long PPMonoidOvImpl::myExponent(ConstRawPtr rawpp, long indet) const
  {
    CoCoA_ASSERT(0 <= indet && indet < myNumIndets);
    return myOrdvArith->myExponent(myOrdv(rawpp), indet);
  }

  void PPMonoidOvImpl::myBigExponent(BigInt& EXP, ConstRawPtr rawpp, long indet) const
  {
    CoCoA_ASSERT(0 <= indet && indet < myNumIndets);
    EXP = myExponent(rawpp, indet);
  }


  void PPMonoidOvImpl::myExponents(std::vector<long>& expv, ConstRawPtr rawpp) const
  {
    CoCoA_ASSERT(len(expv) == myNumIndets);
    myOrdvArith->myComputeExpv(expv, myOrdv(rawpp));
  }


  void PPMonoidOvImpl::myBigExponents(std::vector<BigInt>& expv, ConstRawPtr rawpp) const
  {
    CoCoA_ASSERT(len(expv) == myNumIndets);
    std::vector<long> v(myNumIndets);
    myOrdvArith->myComputeExpv(v, myOrdv(rawpp));
    for (long i=0; i < myNumIndets; ++i)  expv[i] = v[i];
  }


  void PPMonoidOvImpl::myIndetsIn(std::vector<bool>& v, ConstRawPtr rawpp) const
  {
    CoCoA_ASSERT(len(v) == myNumIndets);
    // SLUG SLUG SLUG
    vector<long> expv(myNumIndets);
    myOrdvArith->myComputeExpv(expv, myOrdv(rawpp));
    for (int i=0; i < myNumIndets; ++i)
      if (expv[i] != 0) v[i] = true;
  }


  void PPMonoidOvImpl::myComputeDivMask(DivMask& dm, const DivMaskRule& DivMaskImpl, ConstRawPtr rawpp) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp));
    vector<SmallExponent_t> expv(myNumIndets);
    for (long i=0; i < myNumIndets; ++i)
      expv[i] = static_cast<SmallExponent_t>(myExpv1[i]); // no problem as exponent must be non-neg.
    DivMaskImpl->myAssignFromExpv(dm, &expv[0], myNumIndets);
  }


  void PPMonoidOvImpl::myOutputSelf(std::ostream& out) const
  {
    out << "PPMonoidOv(" << myNumIndets << ", " << myOrd << ")";
  }


  void PPMonoidOvImpl::myDebugPrint(std::ostream& out, ConstRawPtr rawpp) const
  {
#ifdef CoCoA_THREADSAFE_HACK
    vector<long> myExpv1(myNumIndets);
#endif
    myOrdvArith->myComputeExpv(myExpv1, myOrdv(rawpp));
    out << "DEBUG PP: myNumIndets=" << myNumIndets << ", exps=[";
    for (long i=0; i < myNumIndets; ++i)
      out << myExpv1[i] << " ";
    out << "]" << std::endl;
  }


  PPMonoid NewPPMonoidOv(const std::vector<symbol>& IndetNames, const PPOrdering& ord)
  {
    // Sanity check on the indet names given.
    const long nvars = NumIndets(ord);

    if (len(IndetNames) != nvars)
      CoCoA_ERROR(ERR::BadNumIndets, "NewPPMonoidOv(IndetNames,ord)");
    if (!AreDistinct(IndetNames))
      CoCoA_ERROR(ERR::BadIndetNames, "NewPPMonoidOv(IndetNames,ord)");
    if (!AreArityConsistent(IndetNames))
      CoCoA_ERROR(ERR::BadIndetNames, "NewPPMonoidOv(IndetNames,ord)");

    return PPMonoid(new PPMonoidOvImpl(IndetNames, ord));
  }

  PPMonoid NewPPMonoidOv(const std::vector<symbol>& IndetNames, const PPOrderingCtor& OrdCtor)
  {
    return NewPPMonoidOv(IndetNames, OrdCtor(len(IndetNames)));
  }


  bool IsPPMonoidOv(const PPMonoid& PPM)
  {
    return dynamic_cast<const PPMonoidOvImpl*>(PPM.operator->()) != nullptr;
  }

} // end of namespace CoCoA


// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/AlgebraicCore/PPMonoidOv.C,v 1.36 2019/10/15 11:54:08 abbott Exp $
// $Log: PPMonoidOv.C,v $
// Revision 1.36  2019/10/15 11:54:08  abbott
// Summary: Changed 0 into nullptr (where appropriate)
//
// Revision 1.35  2019/03/04 10:32:03  abbott
// Summary: Changed auto_ptr into unqiue_ptr
//
// Revision 1.34  2018/05/17 15:38:11  bigatti
// -- renamed VectorOperations --> VectorOps
//
// Revision 1.33  2018/04/20 16:15:31  bigatti
// -- minor
//
// Revision 1.32  2017/12/01 17:29:21  bigatti
// // -- updated Copyright line
// // -- removed doxygen initial comment
// // -- some commented out debugging info
//
// Revision 1.31  2017/04/18 15:52:27  abbott
// Summary: Corrected an ifdef directive
//
// Revision 1.30  2017/04/18 12:50:06  abbott
// Summary: Corrected ifdef use of CoCoA_THREADSAFE_HACK and CoCoA_DEBUG
//
// Revision 1.29  2016/11/03 12:25:25  abbott
// Summary: Changed IsRadical (for PPMonoidElem) into IsSqFree
//
// Revision 1.28  2015/12/01 13:11:01  abbott
// Summary: Changed mem fn PPOrderingCtor::myCtor into operator(); also for ModuleOrderingCtor; see issue 829
//
// Revision 1.27  2015/11/30 21:53:55  abbott
// Summary: Major update to matrices for orderings (not yet complete, some tests fail)
//
// Revision 1.26  2015/06/30 12:55:00  abbott
// Summary: Added new fn myIndetsIn
// Author: JAA
//
// Revision 1.25  2015/04/16 16:36:33  abbott
// Summary: Cleaned impls of myPowerOverflowCheck
// Author: JAA
//
// Revision 1.24  2015/04/13 14:42:08  abbott
// Summary: Added myPowerOverflowCheck (1st version)
// Author: JAA
//
// Revision 1.23  2014/07/03 15:36:35  abbott
// Summary: Cleaned up impl of PPMonoids: moved myIndetSymbols & myNumIndets to base class
// Author: JAA
//
// Revision 1.22  2014/05/14 15:57:15  bigatti
// -- added "using" for clang with superpedantic flag
//
// Revision 1.21  2014/01/30 17:28:42  abbott
// Summary: Added new fn IsPPMonoidOv
// Author: JAA
//
// Revision 1.20  2014/01/28 16:46:21  abbott
// Made code threadsafe; also some cleaning (buffer expv no longer needed).
//
// Revision 1.19  2012/01/26 16:50:55  bigatti
// -- changed back_inserter into insert
//
// Revision 1.18  2011/08/14 15:52:17  abbott
// Changed ZZ into BigInt (phase 1: just the library sources).
//
// Revision 1.17  2011/05/03 09:56:16  abbott
// Introduced static data member "ourMaxExp" to avoid compiler complaints
// (it does improve readability too).
//
// Revision 1.16  2011/03/22 22:46:01  abbott
// Corrected wrong static_cast in a CoCoA_ASSERT.
//
// Revision 1.15  2011/03/10 17:28:28  bigatti
// -- changed unsigned long into long in some CoCoA_ASSERT
// -- removed assert in myCmpWDegPartial (done in OrdvArith)
//
// Revision 1.14  2011/03/10 16:39:34  abbott
// Replaced (very many) size_t by long in function interfaces (for rings,
// PPMonoids and modules).  Also replaced most size_t inside fn defns.
//
// Revision 1.13  2010/11/30 11:18:11  bigatti
// -- renamed IndetName --> IndetSymbol
//
// Revision 1.12  2010/11/05 16:21:08  bigatti
// -- added ZZExponents
//
// Revision 1.11  2010/10/06 14:10:24  abbott
// Added increments to the ref count in ring and PPMonoid ctors to make
// them exception safe.
//
// Revision 1.10  2010/02/03 16:13:52  abbott
// Added new single word tags for specifying the ordering in PPMonoid
// pseudo-ctors.
//
// Revision 1.9  2010/02/02 16:44:31  abbott
// Added radical & IsRadical (via mem fns myRadical & myIsRadical)
// for PPMonoidElems.
//
// Revision 1.8  2009/09/22 14:01:33  bigatti
// -- added myCmpWDegPartial (ugly name, I know....)
// -- cleaned up and realigned code in PPMonoid*.C files
//
// Revision 1.7  2007/12/05 11:06:24  bigatti
// -- changed "size_t StdDeg/myStdDeg(f)" into "long"  (and related functions)
// -- changed "log/myLog(f, i)" into "MaxExponent/myMaxExponent(f, i)"
// -- fixed bug in "IsOne(ideal)" in SparsePolyRing.C
//
// Revision 1.6  2007/12/04 14:27:06  bigatti
// -- changed "log(pp, i)" into "exponent(pp, i)"
//
// Revision 1.5  2007/10/30 17:14:07  abbott
// Changed licence from GPL-2 only to GPL-3 or later.
// New version for such an important change.
//
// Revision 1.4  2007/05/31 14:54:31  bigatti
// -- now using AreDistinct and AreArityConsistent for sanity check on
//    indet names
//
// Revision 1.2  2007/05/03 10:35:23  abbott
// Added new PPMonoidEvZZ with (virtually) unlimited exponents.
// Modified test-PPMonoid1.C accordingly.
// Added warning in doc about silent/unchecked exponent overflow in other
// PPMonoids.
//
// Revision 1.1.1.1  2007/03/09 15:16:11  abbott
// Imported files
//
// Revision 1.11  2007/03/08 18:22:29  cocoa
// Just whitespace cleaning.
//
// Revision 1.10  2007/03/08 17:43:11  cocoa
// Swapped order of args to the NewPPMonoid pseudo ctors.
//
// Revision 1.9  2007/03/08 11:07:12  cocoa
// Made pseudo ctors for polynomial rings more uniform.  This allowed me to
// remove an include of CoCoA/symbol.H  from the RingDistrM*.H files, but then
// I had to put the include in several .C files.
//
// Revision 1.8  2006/12/06 17:35:58  cocoa
// -- style: RawPtr args are now called "raw.."
//
// Revision 1.7  2006/11/24 17:04:32  cocoa
// -- reorganized includes of header files
//
// Revision 1.6  2006/11/23 17:39:26  cocoa
// -- added #include
//
// Revision 1.5  2006/11/16 11:27:20  cocoa
// -- reinserted myRefCountZero(): sometimes really necessary, in general safe
//
// Revision 1.4  2006/11/14 17:29:20  cocoa
// -- commented out myRefCountZero() (not necessary???)
//
// Revision 1.3  2006/10/06 14:04:14  cocoa
// Corrected position of #ifndef in header files.
// Separated CoCoA_ASSERT into assert.H from config.H;
// many minor consequential changes (have to #include assert.H).
// A little tidying of #include directives (esp. in Max's code).
//
// Revision 1.2  2006/08/07 21:23:25  cocoa
// Removed almost all publicly visible references to SmallExponent_t;
// changed to long in all PPMonoid functions and SparsePolyRing functions.
// DivMask remains to sorted out.
//
// Revision 1.1.1.1  2006/05/30 11:39:37  cocoa
// Imported files
//
// Revision 1.6  2006/03/15 18:09:31  cocoa
// Changed names of member functions which print out their object
// into myOutputSelf -- hope this will appease the Intel C++ compiler.
//
// Revision 1.5  2006/03/14 17:21:18  cocoa
// Moved concrete PPMonoid impls entirely into their respective .C files.
// Now the corresponding .H files are very compact.
//
// Revision 1.4  2006/03/12 21:28:33  cocoa
// Major check in after many changes
//
// Revision 1.3  2006/02/20 22:41:20  cocoa
// All forms of the log function for power products now return SmallExponent_t
// (instead of int).  exponents now resizes the vector rather than requiring
// the user to pass in the correct size.
//
// Revision 1.2  2006/01/17 10:23:08  cocoa
// Updated DivMask; many consequential changes.
// A few other minor fixes.
//
// Revision 1.1.1.1  2005/10/17 10:46:54  cocoa
// Imported files
//
// Revision 1.7  2005/10/11 16:37:30  cocoa
// Added new small prime finite field class (see RingFpDouble).
//
// Cleaned makefiles and configuration script.
//
// Tidied PPMonoid code (to eliminate compiler warnings).
//
// Fixed bug in RingFloat::myIsInteger.
//
// Revision 1.6  2005/08/08 16:36:32  cocoa
// Just checking in before going on holiday.
// Don't really recall what changes have been made.
// Added IsIndet function for RingElem, PPMonoidElem,
// and a member function of OrdvArith.
// Improved the way failed assertions are handled.
//
// Revision 1.5  2005/07/19 15:30:20  cocoa
// A first attempt at iterators over sparse polynomials.
// Main additions are to SparsePolyRing, DistrMPoly*.
// Some consequential changes to PPMonoid*.
//
// Revision 1.4  2005/07/08 15:09:28  cocoa
// Added new symbol class (to represent names of indets).
// Integrated the new class into concrete polynomial rings
// and PPMonoid -- many consequential changes.
// Change ctors for the "inline" sparse poly rings: they no
// longer expect a PPMonoid, but build their own instead
// (has to be a PPMonoidOv).
//
// Revision 1.3  2005/07/01 16:08:15  cocoa
// Friday check-in.  Major change to structure under PolyRing:
// now SparsePolyRing and DUPolyRing are separated (in preparation
// for implementing iterators).
//
// A number of other relatively minor changes had to be chased through
// (e.g. IndetPower).
//
// Revision 1.2  2005/06/23 15:42:41  cocoa
// Fixed typo in GNU fdl -- all doc/*.txt files affected.
// Minor corrections to PPMonoid (discovered while writing doc).
//
// Revision 1.1  2005/06/22 14:47:56  cocoa
// PPMonoids and PPMonoidElems updated to mirror the structure
// used for rings and RingElems.  Many consequential changes.
//
