#!/bin/bash

SCRIPT_NAME=[[`basename "$0"`]]

# Auxiliary script for CoCoALib configuration process.
# Script expects the env variables CXX and CXXFLAGS to be set.

# Script to see whether the -std=c++14 compiler flag is needed/recognised.
# Exit with code 1 if we did not find a way to compile C++14 code.
# Exit with code 0 if we found a way to compile C++14 code; printed
# value is flag to give compiler to get C++14 compilation
# (printed value may be empty string or "-std=c++14")


if [ $# -ne 0 ]
then
  echo "ERROR: expected no args.   $SCRIPT_NAME"  > /dev/stderr
  exit 1
fi

# Check environment variable CXX
if [ -z "$CXX" ]
then
  echo "ERROR: environment variable CXX not set.   $SCRIPT_NAME"  > /dev/stderr
  exit 1
fi


# Create tmp directory, put test prog in it, compile and run.
umask 22
TODAY=`date "+%Y-%m-%d"`
TIME=`date "+%H:%M:%S"`
TMP_DIR=/tmp/CoCoALib-config-$USER-$TODAY/cxx03-$TIME-$$
/bin/rm -rf $TMP_DIR  &&  /bin/mkdir -p $TMP_DIR
if [ $? -ne 0 ]
then
  echo "ERROR: failed to create temporary directory \"$TMP_DIR\".   $SCRIPT_NAME"  > /dev/stderr
  exit 1
fi

cd $TMP_DIR

/bin/cat > test.C <<EOF
#include <iostream>
int main()
{
  int ReturnCode = 0; // will mean c++14 compliant
  std::cout << "C++ version: " << __cplusplus << std::endl;
#if __cplusplus < 201400L
  ReturnCode = 1;  // NOT C++14 compilant
#endif
  return ReturnCode;
}
EOF

# First try with no compiler flag...
"$CXX"  test.C  -o test  >> LogFile  2>& 1 
if [ $? -ne 0 ]
then
    echo "cxx14.sh: test compilation unexpectedly failed; is $CXX a c++ compiler?" > /dev/stderr
    exit 1
fi
./test  >> LogFile
if [ $? -eq 0 ]
then
   exit 0; # exit without printing (no flag needed for C++14)
fi

# Compilation without flag is not C++14 standard; try with -std=c++14

CXX14="-std=c++14"
"$CXX" $CXX14 test.C  >> LogFile  2>& 1 
if [ $? -ne 0 ]
then
    echo "cxx14.sh: test compilation with flag $CXX14 failed" > /dev/stderr
    exit 1  
fi

# Clean up TMP_DIR
cd
/bin/rm -rf "$TMP_DIR"
echo $CXX14
