#/bin/sh

# Script to choose the best ULong2Long defn (in ULong2Long.H).
# Expects env variables CXX and CXXFLAGS to be set.

SCRIPT_NAME=[[`basename "$0"`]]

# Try the three possible settings for the flag CoCoA_ULONG2LONG.
# Print out the first which works (and exit with code 0): prints 1, 2 or 3
# (this is the value to assign to preprocessor symbol CoCoA_ULONG2LONG).
# If none works, print an error message on stderr, and exit with non-zero code.
# Exit code 1 means some input was bad.
# Exit code 2 means no working defn of ULong2Long was found.
# Exit code 3 means a problem in compilation.

# taken from StackExchange 256434
is_absolute()
{
    case "$1" in
	///* | //) true;;
	//*) false;;
	/*) true;;
	*) false;;
    esac
}


if [ $# -ne 1 ]
then
  echo "ERROR: expected 1 arg (full path of CoCoALib include directory)   $SCRIPT_NAME"   > /dev/stderr
  exit 1
fi

# The following is a cryptic if...then block
is_absolute "$1" ||
(
  echo "ERROR: arg is not absolute \"$1\"   $SCRIPT_NAME"   > /dev/stderr
  exit 1
)

if [ \! -d "$1" ]
then
    echo "ERROR: arg is not a directory \"$1\"   $SCRIPT_NAME"  > /dev/stderr
    exit 1
fi
    
COCOA_INC_DIR=`dirname "$1"`
BASE=`basename "$1"`
if [ "$BASE" \!= "CoCoA" -o \! -f "$COCOA_INC_DIR/CoCoA/ULong2Long.H" ]
then
  echo "ERROR: cannot find CoCoA header file \"$COCOA_INC_DIR/CoCoA/ULong2Long.H\"   $SCRIPT_NAME"  > /dev/stderr
  exit 1
fi

if [ -z "$CXX" ]
then
  echo "ERROR: CXX environment variable is not defined   $SCRIPT_NAME"  > /dev/stderr
  exit 1
fi

# Create tmp directory, put test prog in it, compile and run.
umask 22
TODAY=`date "+%Y-%m-%d"`
TIME=`date "+%H:%M:%S"`
TMP_DIR=/tmp/CoCoALib-config-$USER-$TODAY/cpp-flags-ulong2long-$TIME-$$
/bin/rm -rf $TMP_DIR  &&  /bin/mkdir -p $TMP_DIR
if [ $? -ne 0 ]
then
  echo "ERROR: failed to create temporary directory \"$TMP_DIR\"   $SCRIPT_NAME"  >/dev/stderr
  exit 1
fi

cd $TMP_DIR
/bin/cat > CheckULong2Long.C  <<EOF
#include "CoCoA/ULong2Long.H"
using CoCoA::ULong2Long;

#include <limits>
using std::numeric_limits;

// Check whether ULong2Long works at four negative values:
//   MinLong, MinLong+1, (MinLong+1)/2, and -1
// Exits with 0 iff the fn works correctly at all four points.
// A non-zero exit code indicates which check failed, but this
// information is not currently used anywhere.

// This program is compiled and run by cpp-flags-ulong2long.sh
// trying the various possible values for CoCoA_ULONG2LONG.

int main()
{
  volatile unsigned long ul;
  long l = numeric_limits<long>::min();
  ul = l;
  if (ULong2Long(ul) != l) return 1;

  l = l+1;
  ul = l;
  if (ULong2Long(ul) != l) return 2;

  l = l/2;
  ul = l;
  if (ULong2Long(ul) != l) return 3;

  l = -1;
  ul = l;
  if (ULong2Long(ul) != l) return 4;

  return 0;
}
EOF


# Now try the various possible settings for CoCoA_ULONG2LONG
UL2L_DEFN=
for defn in 1 2 3
do
  CPPFLAG="-DCoCoA_ULONG2LONG=$defn"
  $CXX $CXXFLAGS $CPPFLAG -I"$COCOA_INC_DIR"  CheckULong2Long.C -o CheckULong2Long  > LogFile  2>&1
  if [ $? -ne 0 ]
  then
    echo "ERROR: Compilation of test program failed --> see LogFile   $SCRIPT_NAME"  > /dev/stderr
    exit 3  # do not clean TMP_DIR, for possible debugging
  fi
  ./CheckULong2Long  >> LogFile  2>&1
  if [ $? = 0 ]
  then
    UL2L_DEFN="$defn"
    break
  fi
done

if [ -z "$UL2L_DEFN" ]
then
  echo "ERROR: Failed to determine a working defn of ULong2Long   $SCRIPT_NAME"  > /dev/stderr
  exit 2 # do not clean TMP_DIR, for possible debugging
fi

# Clean up TMP_DIR
cd
/bin/rm -rf $TMP_DIR
echo $UL2L_DEFN
exit 0
