/*
 *  Copyright © 2004-2005 Jasper Huijsmans <jasper@xfce.org>
 *  Copyright (c) 2005 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published 
 *  by the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#include "xfmedia-handle.h"

#define DECOR_WIDTH          6
#define HANDLE_SIZE          8
#define DEFAULT_STYLE        XFMEDIA_HANDLE_STYLE_GRIP
#define DEFAULT_ORIENTATION  GTK_ORIENTATION_VERTICAL

/* should use glib-mkenums, but... lazy */
#define XFMEDIA_TYPE_HANDLE_STYLE   (xfmedia_handle_style_get_type())
static GType
xfmedia_handle_style_get_type()
{
    static GType handle_style_type = 0;
    
    if(!handle_style_type) {
        static const GEnumValue values[] = {
            { XFMEDIA_HANDLE_STYLE_DOT, "XFMEDIA_HANDLE_STYLE_DOT", "dot" },
            { XFMEDIA_HANDLE_STYLE_GRIP, "XFMEDIA_HANDLE_STYLE_GRIP", "grip" },
            { XFMEDIA_HANDLE_STYLE_BOX, "XFMEDIA_HANDLE_STYLE_BOX", "box" },
            { 0, NULL, NULL }
        };
        
        handle_style_type = g_enum_register_static("XfmediaHandleStyle",
                                                   values);
    }
    
    return handle_style_type;
}

static void xfmedia_handle_class_init     (XfmediaHandleClass *klass);
static void xfmedia_handle_init           (XfmediaHandle *handle);
static void xfmedia_handle_finalize       (GObject *object);
static void xfmedia_handle_set_property   (GObject *object,
                                           guint property_id,
                                           const GValue *value,
                                           GParamSpec *pspec);
static void xfmedia_handle_get_property   (GObject *object,
                                           guint property_id,
                                           GValue *value,
                                           GParamSpec *pspec);
static void xfmedia_handle_realize        (GtkWidget *widget);
static void xfmedia_handle_unrealize      (GtkWidget *widget);
static void xfmedia_handle_size_request   (GtkWidget *widget,
                                           GtkRequisition *requisition);
static void xfmedia_handle_size_allocate  (GtkWidget *widget,
                                           GtkAllocation *allocation);
static gboolean xfmedia_handle_expose     (GtkWidget *widget,
                                           GdkEventExpose *event);

static void xfmedia_handle_paint_dotted   (GtkWidget *widget,
                                           GdkRectangle *area,
                                           const gchar *detail,
                                           gint x,
                                           gint y,
                                           gint width,
                                           gint height);

static void xfmedia_handle_real_paint     (GtkWidget *widget,
                                           GdkRectangle *area,
                                           const gchar *detail,
                                           gint x,
                                           gint y,
                                           gint width,
                                           gint height);
static void xfmedia_handle_paint          (GtkWidget *widget,
                                           GdkRectangle *area);


enum
{
    PROP_0 = 0,
    PROP_STYLE,
    PROP_ORIENTATION,
    N_PROPS
};

struct _XfmediaHandlePrivate
{
    XfmediaHandleStyle style;
    GtkOrientation orientation;
    
    GdkBitmap *light_bmap;
    GdkBitmap *mid_bmap;
    GdkBitmap *dark_bmap;
};

static const gchar dark_bits[] = { 0x00, 0x0e, 0x02, 0x02, 0x00, 0x00, };
static const gchar light_bits[] = { 0x00, 0x00, 0x10, 0x10, 0x1c, 0x00, };
static const gchar mid_bits[] = { 0x00, 0x00, 0x0c, 0x0c, 0x00, 0x00, };


G_DEFINE_TYPE(XfmediaHandle, xfmedia_handle, GTK_TYPE_WIDGET);
              

static void
xfmedia_handle_class_init(XfmediaHandleClass *klass)
{
    GObjectClass *gobject_class = (GObjectClass *)klass;
    GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
    
    gobject_class->finalize = xfmedia_handle_finalize;
    gobject_class->set_property = xfmedia_handle_set_property;
    gobject_class->get_property = xfmedia_handle_get_property;
    
    widget_class->realize = xfmedia_handle_realize;
    widget_class->unrealize = xfmedia_handle_unrealize;
    widget_class->size_request = xfmedia_handle_size_request;
    widget_class->size_allocate = xfmedia_handle_size_allocate;
    widget_class->expose_event = xfmedia_handle_expose;
    
    g_object_class_install_property(gobject_class,
                                    PROP_STYLE,
                                    g_param_spec_enum("style",
                                                      _("Handle Style"),
                                                      _("This property defines the handle style of the XfmediaHandle widget."),
                                                      XFMEDIA_TYPE_HANDLE_STYLE,
                                                      DEFAULT_STYLE,
                                                      G_PARAM_READWRITE));
    
    g_object_class_install_property(gobject_class,
                                    PROP_ORIENTATION,
                                    g_param_spec_enum("orientation",
                                                      _("Orientation"),
                                                      _("This property defines whether this is a horizontal or vertical handle."),
                                                      GTK_TYPE_ORIENTATION,
                                                      DEFAULT_ORIENTATION,
                                                      G_PARAM_READWRITE));
}

static void
xfmedia_handle_init(XfmediaHandle *handle)
{
    handle->priv = g_new0(XfmediaHandlePrivate, 1);
    handle->priv->style = DEFAULT_STYLE;
    handle->priv->orientation = DEFAULT_ORIENTATION;
    
    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(handle), GTK_NO_WINDOW);
    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(handle), GTK_CAN_FOCUS);
}

static void
xfmedia_handle_finalize(GObject *object)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(object);
    
    if(G_LIKELY(handle->priv->light_bmap)) {
        g_object_unref(G_OBJECT(handle->priv->light_bmap));
        g_object_unref(G_OBJECT(handle->priv->mid_bmap));
        g_object_unref(G_OBJECT(handle->priv->dark_bmap));
    }
    
    g_free(handle->priv);
    handle->priv = NULL;
}

static void
xfmedia_handle_set_property(GObject *object,
                            guint property_id,
                            const GValue *value,
                            GParamSpec *pspec)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(object);
    
    switch(property_id) {
        case PROP_STYLE:
            xfmedia_handle_set_style(handle, g_value_get_enum(value));
            break;
        
        case PROP_ORIENTATION:
            xfmedia_handle_set_orientation(handle, g_value_get_enum(value));
            break;
        
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
            break;
    }
}

static void
xfmedia_handle_get_property(GObject *object,
                            guint property_id,
                            GValue *value,
                            GParamSpec *pspec)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(object);
    
    switch(property_id) {
        case PROP_STYLE:
            g_value_set_enum(value, xfmedia_handle_get_style(handle));
            break;
        
        case PROP_ORIENTATION:
            g_value_set_enum(value, xfmedia_handle_get_orientation(handle));
            break;
        
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
            break;
    }
}

static void
xfmedia_handle_realize(GtkWidget *widget)
{
    GdkWindowAttr wattr;
    gint wattr_mask;
    
    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

    wattr.window_type = GDK_WINDOW_CHILD;
    wattr.x = widget->allocation.x;
    wattr.y = widget->allocation.y;
    wattr.width = widget->allocation.width;
    wattr.height = widget->allocation.height;
    wattr.wclass = GDK_INPUT_OUTPUT;
    wattr.event_mask = gtk_widget_get_events(widget);
    wattr.event_mask |= GDK_BUTTON_PRESS_MASK
                        | GDK_BUTTON_RELEASE_MASK
                        | GDK_EXPOSURE_MASK;
    
    wattr_mask = GDK_WA_X | GDK_WA_Y;
    
    widget->window = gdk_window_new(gtk_widget_get_parent_window(widget),
                                    &wattr,
                                    wattr_mask);
    
    gdk_window_set_user_data(widget->window, widget);
                                           
    widget->style = gtk_style_attach(widget->style, widget->window);
}

static void
xfmedia_handle_unrealize(GtkWidget *widget)
{
    
    gtk_style_detach(widget->style);
    
    g_object_unref(G_OBJECT(widget->window));
    widget->window = NULL;
    
    GTK_WIDGET_UNSET_FLAGS(widget, GTK_REALIZED);
}

static void
xfmedia_handle_size_request(GtkWidget *widget,
                            GtkRequisition *requisition)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(widget);
    GtkWidget *parent = gtk_widget_get_parent(widget);
    gint xthick, ythick;
    
    xthick = widget->style->xthickness;
    ythick = widget->style->ythickness;
    
    switch(handle->priv->orientation) {
        case GTK_ORIENTATION_HORIZONTAL:
            requisition->width = parent->allocation.width + 2 * xthick;
            requisition->height = HANDLE_SIZE + 2 * ythick;
            break;
        
        case GTK_ORIENTATION_VERTICAL:
            requisition->width = HANDLE_SIZE + 2 * xthick;
            requisition->height = parent->allocation.height + 2 * ythick;
            break;
    }
}

static void
xfmedia_handle_size_allocate(GtkWidget *widget,
                             GtkAllocation *allocation)
{
    widget->allocation = *allocation;
    
    if(GTK_WIDGET_REALIZED(widget)) {
        gdk_window_move_resize(widget->window,
                               allocation->x,
                               allocation->y,
                               allocation->width,
                               allocation->height);
    }
}

static gboolean
xfmedia_handle_expose(GtkWidget *widget,
                      GdkEventExpose *event)
{
    
    if(GTK_WIDGET_DRAWABLE(widget))
        xfmedia_handle_paint(widget, &event->area);
    
    return TRUE;
}

static void
xfmedia_handle_paint_dotted(GtkWidget *widget,
                            GdkRectangle *area,
                            const gchar *detail,
                            gint x,
                            gint y,
                            gint width,
                            gint height)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(widget);
    gint rows, columns, w, h;
    
    if(G_UNLIKELY(!handle->priv->light_bmap)) {
        handle->priv->light_bmap = gdk_bitmap_create_from_data(widget->window,
                                                               light_bits,
                                                               DECOR_WIDTH,
                                                               DECOR_WIDTH);
        handle->priv->mid_bmap = gdk_bitmap_create_from_data(widget->window,
                                                             mid_bits,
                                                             DECOR_WIDTH,
                                                             DECOR_WIDTH);
        handle->priv->dark_bmap = gdk_bitmap_create_from_data(widget->window,
                                                              dark_bits,
                                                              DECOR_WIDTH,
                                                              DECOR_WIDTH);
    }
    
    gdk_gc_set_clip_rectangle(widget->style->light_gc[widget->state], area);
    gdk_gc_set_clip_rectangle(widget->style->dark_gc[widget->state], area);
    gdk_gc_set_clip_rectangle(widget->style->mid_gc[widget->state], area);
    
    if(GTK_ORIENTATION_HORIZONTAL == handle->priv->orientation) {
        rows = 1;
        columns = width / DECOR_WIDTH - 2;
    } else {
        rows = height / DECOR_WIDTH - 2;
        columns = 1;
    }
    
    w = columns * DECOR_WIDTH;
    h = rows * DECOR_WIDTH;
    x = x + (width - w) / 2;
    y = y + (height - h) / 2;
    
    gdk_gc_set_stipple(widget->style->light_gc[widget->state],
                       handle->priv->light_bmap);
    gdk_gc_set_stipple(widget->style->mid_gc[widget->state],
                       handle->priv->mid_bmap);
    gdk_gc_set_stipple(widget->style->dark_gc[widget->state],
                       handle->priv->dark_bmap);

    gdk_gc_set_fill(widget->style->light_gc[widget->state], GDK_STIPPLED);
    gdk_gc_set_fill(widget->style->mid_gc[widget->state], GDK_STIPPLED);
    gdk_gc_set_fill(widget->style->dark_gc[widget->state], GDK_STIPPLED);

    gdk_gc_set_ts_origin(widget->style->light_gc[widget->state], x, y);
    gdk_gc_set_ts_origin(widget->style->mid_gc[widget->state], x, y);
    gdk_gc_set_ts_origin(widget->style->dark_gc[widget->state], x, y);

    gdk_draw_rectangle(widget->window,
                       widget->style->light_gc[widget->state], TRUE, x,
                       y, w, h);
    gdk_draw_rectangle(widget->window,
                       widget->style->mid_gc[widget->state], TRUE, x, y,
                       w, h);
    gdk_draw_rectangle(widget->window,
                       widget->style->dark_gc[widget->state], TRUE, x, y,
                       w, h);

    gdk_gc_set_fill(widget->style->light_gc[widget->state], GDK_SOLID);
    gdk_gc_set_fill(widget->style->mid_gc[widget->state], GDK_SOLID);
    gdk_gc_set_fill(widget->style->dark_gc[widget->state], GDK_SOLID);

    gdk_gc_set_clip_rectangle(widget->style->light_gc[widget->state], NULL);
    gdk_gc_set_clip_rectangle(widget->style->dark_gc[widget->state], NULL);
    gdk_gc_set_clip_rectangle(widget->style->mid_gc[widget->state], NULL);
}

static void
xfmedia_handle_real_paint(GtkWidget *widget,
                          GdkRectangle *area,
                          const gchar *detail,
                          gint x,
                          gint y,
                          gint width,
                          gint height)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(widget);
    
    switch(handle->priv->style) {
        case XFMEDIA_HANDLE_STYLE_DOT:
            xfmedia_handle_paint_dotted(widget, area, detail,
                                        x, y, width, height);
            break;
        
        case XFMEDIA_HANDLE_STYLE_GRIP:
            gtk_paint_handle(widget->style,
                             widget->window,
                             GTK_WIDGET_STATE(widget),
                             GTK_SHADOW_OUT,
                             area, widget, "handlebox",
                             x, y, width, height,
                             handle->priv->orientation);
            break;
        
        default:
            gtk_paint_box(widget->style,
                          widget->window,
                          GTK_WIDGET_STATE(widget),
                          GTK_SHADOW_OUT, area, widget, detail, 
                          x, y, width, height);
    }
}

static void
xfmedia_handle_paint(GtkWidget *widget,
                     GdkRectangle *area)
{
    XfmediaHandle *handle = XFMEDIA_HANDLE(widget);
    gint x, y, w, h, xthick, ythick;
    gboolean horizontal = handle->priv->orientation == GTK_ORIENTATION_HORIZONTAL;
    
    xthick = widget->style->xthickness;
    ythick = widget->style->ythickness;
    
    if(horizontal) {
        w = widget->allocation.width - 2 * xthick;
        h = HANDLE_SIZE;
        
        x = widget->allocation.x + xthick;
        y = widget->allocation.y + ythick;
    } else {
        w = HANDLE_SIZE;
        h = widget->allocation.height - 2 * ythick;
        
        x = widget->allocation.x + xthick;
        y = widget->allocation.y + ythick;
    }
    
    /* background */
    gtk_paint_box(widget->style,
                  widget->window,
                  GTK_STATE_NORMAL,
                  GTK_SHADOW_NONE,
                  area,
                  widget,
                  "background",
                  area->x,
                  area->y,
                  area->width,
                  area->height);
    
    /* grip/handle */
    xfmedia_handle_real_paint(widget, area, "xfmedia-handle", x, y, w, h);
    
    /* border - should probably be using something else for this */
    gdk_draw_rectangle(GDK_DRAWABLE(widget->window),
                       widget->style->mid_gc[GTK_STATE_NORMAL],
                       FALSE,
                       0, 0,
                       widget->allocation.width - 1,
                       widget->allocation.height - 1);
}


/*
 * public api
 */

GtkWidget *
xfmedia_handle_new(XfmediaHandleStyle style,
                   GtkOrientation orientation)
{
    return g_object_new(XFMEDIA_TYPE_HANDLE,
                        "style", style,
                        "orientation", orientation,
                        NULL);
}

XfmediaHandleStyle
xfmedia_handle_get_style(XfmediaHandle *handle)
{
    g_return_val_if_fail(XFMEDIA_IS_HANDLE(handle), DEFAULT_STYLE);
    return handle->priv->style;
}

void
xfmedia_handle_set_style(XfmediaHandle *handle,
                         XfmediaHandleStyle style)
{
    g_return_if_fail(XFMEDIA_IS_HANDLE(handle)
                     && style >= XFMEDIA_HANDLE_STYLE_DOT
                     && style <= XFMEDIA_HANDLE_STYLE_BOX);
    
    handle->priv->style = style;
    
    g_object_notify(G_OBJECT(handle), "style");
}

GtkOrientation
xfmedia_handle_get_orientation(XfmediaHandle *handle)
{
    g_return_val_if_fail(XFMEDIA_IS_HANDLE(handle), DEFAULT_ORIENTATION);
    return handle->priv->orientation;
}

void
xfmedia_handle_set_orientation(XfmediaHandle *handle,
                               GtkOrientation orientation)
{
    g_return_if_fail(XFMEDIA_IS_HANDLE(handle)
                     && (GTK_ORIENTATION_HORIZONTAL == orientation
                         || GTK_ORIENTATION_VERTICAL == orientation));
    
    handle->priv->orientation = orientation;
    
    g_object_notify(G_OBJECT(handle), "orientation");
}
