/* 
 * Copyright © 2008-2013 Kristian Høgsberg
 * Copyright © 2013      Rafael Antognolli
 * Copyright © 2013      Jasper St. Pierre
 * Copyright © 2010-2013 Intel Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this
 * software and its documentation for any purpose is hereby granted
 * without fee, provided that the above copyright notice appear in
 * all copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of
 * the copyright holders not be used in advertising or publicity
 * pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#ifndef XDG_SHELL_SERVER_PROTOCOL_H
#define XDG_SHELL_SERVER_PROTOCOL_H

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include "wayland-util.h"

struct wl_client;
struct wl_resource;

struct xdg_shell;
struct xdg_surface;
struct xdg_popup;

extern const struct wl_interface xdg_shell_interface;
extern const struct wl_interface xdg_surface_interface;
extern const struct wl_interface xdg_popup_interface;

#ifndef XDG_SHELL_VERSION_ENUM
#define XDG_SHELL_VERSION_ENUM
/**
 * xdg_shell_version - latest protocol version
 * @XDG_SHELL_VERSION_CURRENT: Always the latest version
 *
 * Use this enum to check the protocol version, and it will be updated
 * automatically.
 */
enum xdg_shell_version {
	XDG_SHELL_VERSION_CURRENT = 1,
};
#endif /* XDG_SHELL_VERSION_ENUM */

/**
 * xdg_shell - create desktop-style surfaces
 * @use_unstable_version: enable use of this unstable version
 * @get_xdg_surface: create a shell surface from a surface
 * @get_xdg_popup: create a shell surface from a surface
 *
 * This interface is implemented by servers that provide desktop-style
 * user interfaces.
 *
 * It allows clients to associate a xdg_surface with a basic surface.
 */
struct xdg_shell_interface {
	/**
	 * use_unstable_version - enable use of this unstable version
	 * @version: (none)
	 *
	 * Use this request in order to enable use of this interface.
	 *
	 * Understand and agree that one is using an unstable interface,
	 * that will likely change in the future, breaking the API.
	 */
	void (*use_unstable_version)(struct wl_client *client,
				     struct wl_resource *resource,
				     int32_t version);
	/**
	 * get_xdg_surface - create a shell surface from a surface
	 * @id: (none)
	 * @surface: (none)
	 *
	 * Create a shell surface for an existing surface.
	 *
	 * Only one shell or popup surface can be associated with a given
	 * surface.
	 */
	void (*get_xdg_surface)(struct wl_client *client,
				struct wl_resource *resource,
				uint32_t id,
				struct wl_resource *surface);
	/**
	 * get_xdg_popup - create a shell surface from a surface
	 * @id: (none)
	 * @surface: (none)
	 * @parent: (none)
	 * @seat: the wl_seat whose pointer is used
	 * @serial: serial of the implicit grab on the pointer
	 * @x: (none)
	 * @y: (none)
	 * @flags: (none)
	 *
	 * Create a popup surface for an existing surface.
	 *
	 * Only one shell or popup surface can be associated with a given
	 * surface.
	 */
	void (*get_xdg_popup)(struct wl_client *client,
			      struct wl_resource *resource,
			      uint32_t id,
			      struct wl_resource *surface,
			      struct wl_resource *parent,
			      struct wl_resource *seat,
			      uint32_t serial,
			      int32_t x,
			      int32_t y,
			      uint32_t flags);
};

#ifndef XDG_SURFACE_RESIZE_EDGE_ENUM
#define XDG_SURFACE_RESIZE_EDGE_ENUM
/**
 * xdg_surface_resize_edge - edge values for resizing
 * @XDG_SURFACE_RESIZE_EDGE_NONE: (none)
 * @XDG_SURFACE_RESIZE_EDGE_TOP: (none)
 * @XDG_SURFACE_RESIZE_EDGE_BOTTOM: (none)
 * @XDG_SURFACE_RESIZE_EDGE_LEFT: (none)
 * @XDG_SURFACE_RESIZE_EDGE_TOP_LEFT: (none)
 * @XDG_SURFACE_RESIZE_EDGE_BOTTOM_LEFT: (none)
 * @XDG_SURFACE_RESIZE_EDGE_RIGHT: (none)
 * @XDG_SURFACE_RESIZE_EDGE_TOP_RIGHT: (none)
 * @XDG_SURFACE_RESIZE_EDGE_BOTTOM_RIGHT: (none)
 *
 * These values are used to indicate which edge of a surface is being
 * dragged in a resize operation. The server may use this information to
 * adapt its behavior, e.g. choose an appropriate cursor image.
 */
enum xdg_surface_resize_edge {
	XDG_SURFACE_RESIZE_EDGE_NONE = 0,
	XDG_SURFACE_RESIZE_EDGE_TOP = 1,
	XDG_SURFACE_RESIZE_EDGE_BOTTOM = 2,
	XDG_SURFACE_RESIZE_EDGE_LEFT = 4,
	XDG_SURFACE_RESIZE_EDGE_TOP_LEFT = 5,
	XDG_SURFACE_RESIZE_EDGE_BOTTOM_LEFT = 6,
	XDG_SURFACE_RESIZE_EDGE_RIGHT = 8,
	XDG_SURFACE_RESIZE_EDGE_TOP_RIGHT = 9,
	XDG_SURFACE_RESIZE_EDGE_BOTTOM_RIGHT = 10,
};
#endif /* XDG_SURFACE_RESIZE_EDGE_ENUM */

/**
 * xdg_surface - desktop-style metadata interface
 * @destroy: remove xdg_surface interface
 * @set_transient_for: surface is a child of another surface
 * @set_title: set surface title
 * @set_app_id: set surface class
 * @pong: respond to a ping event
 * @move: start an interactive move
 * @resize: start an interactive resize
 * @set_output: set the default output used by this surface
 * @set_fullscreen: set the surface state as fullscreen
 * @unset_fullscreen: unset the surface state as fullscreen
 * @set_maximized: set the surface state as maximized
 * @unset_maximized: unset the surface state as maximized
 * @set_minimized: set the surface state as minimized
 *
 * An interface that may be implemented by a wl_surface, for
 * implementations that provide a desktop-style user interface.
 *
 * It provides requests to treat surfaces like windows, allowing to set
 * properties like maximized, fullscreen, minimized, and to move and resize
 * them, and associate metadata like title and app id.
 *
 * On the server side the object is automatically destroyed when the
 * related wl_surface is destroyed. On client side, xdg_surface.destroy()
 * must be called before destroying the wl_surface object.
 */
struct xdg_surface_interface {
	/**
	 * destroy - remove xdg_surface interface
	 *
	 * The xdg_surface interface is removed from the wl_surface
	 * object that was turned into a xdg_surface with
	 * xdg_shell.get_xdg_surface request. The xdg_surface properties,
	 * like maximized and fullscreen, are lost. The wl_surface loses
	 * its role as a xdg_surface. The wl_surface is unmapped.
	 */
	void (*destroy)(struct wl_client *client,
			struct wl_resource *resource);
	/**
	 * set_transient_for - surface is a child of another surface
	 * @parent: (none)
	 *
	 * Setting a surface as transient of another means that it is
	 * child of another surface.
	 *
	 * Child surfaces are stacked above their parents, and will be
	 * unmapped if the parent is unmapped too. They should not appear
	 * on task bars and alt+tab.
	 */
	void (*set_transient_for)(struct wl_client *client,
				  struct wl_resource *resource,
				  struct wl_resource *parent);
	/**
	 * set_title - set surface title
	 * @title: (none)
	 *
	 * Set a short title for the surface.
	 *
	 * This string may be used to identify the surface in a task bar,
	 * window list, or other user interface elements provided by the
	 * compositor.
	 *
	 * The string must be encoded in UTF-8.
	 */
	void (*set_title)(struct wl_client *client,
			  struct wl_resource *resource,
			  const char *title);
	/**
	 * set_app_id - set surface class
	 * @app_id: (none)
	 *
	 * Set an id for the surface.
	 *
	 * The app id identifies the general class of applications to which
	 * the surface belongs.
	 *
	 * It should be the ID that appears in the new desktop entry
	 * specification, the interface name.
	 */
	void (*set_app_id)(struct wl_client *client,
			   struct wl_resource *resource,
			   const char *app_id);
	/**
	 * pong - respond to a ping event
	 * @serial: serial of the ping event
	 *
	 * A client must respond to a ping event with a pong request or
	 * the client may be deemed unresponsive.
	 */
	void (*pong)(struct wl_client *client,
		     struct wl_resource *resource,
		     uint32_t serial);
	/**
	 * move - start an interactive move
	 * @seat: the wl_seat whose pointer is used
	 * @serial: serial of the implicit grab on the pointer
	 *
	 * Start a pointer-driven move of the surface.
	 *
	 * This request must be used in response to a button press event.
	 * The server may ignore move requests depending on the state of
	 * the surface (e.g. fullscreen or maximized).
	 */
	void (*move)(struct wl_client *client,
		     struct wl_resource *resource,
		     struct wl_resource *seat,
		     uint32_t serial);
	/**
	 * resize - start an interactive resize
	 * @seat: the wl_seat whose pointer is used
	 * @serial: serial of the implicit grab on the pointer
	 * @edges: which edge or corner is being dragged
	 *
	 * Start a pointer-driven resizing of the surface.
	 *
	 * This request must be used in response to a button press event.
	 * The server may ignore resize requests depending on the state of
	 * the surface (e.g. fullscreen or maximized).
	 */
	void (*resize)(struct wl_client *client,
		       struct wl_resource *resource,
		       struct wl_resource *seat,
		       uint32_t serial,
		       uint32_t edges);
	/**
	 * set_output - set the default output used by this surface
	 * @output: (none)
	 *
	 * Set the default output used by this surface when it is first
	 * mapped.
	 *
	 * If this value is NULL (default), it's up to the compositor to
	 * choose which display will be used to map this surface.
	 *
	 * When fullscreen or maximized state are set on this surface, and
	 * it wasn't mapped yet, the output set with this method will be
	 * used. Otherwise, the output where the surface is currently
	 * mapped will be used.
	 */
	void (*set_output)(struct wl_client *client,
			   struct wl_resource *resource,
			   struct wl_resource *output);
	/**
	 * set_fullscreen - set the surface state as fullscreen
	 *
	 * Set the surface as fullscreen.
	 *
	 * After this request, the compositor should send a configure event
	 * informing the output size.
	 *
	 * This request informs the compositor that the next attached
	 * buffer committed will be in a fullscreen state. The buffer size
	 * should be the same size as the size informed in the configure
	 * event, if the client doesn't want to leave any empty area.
	 *
	 * In other words: the next attached buffer after set_maximized is
	 * the new maximized buffer. And the surface will be positioned at
	 * the maximized position on commit.
	 *
	 * A simple way to synchronize and wait for the correct configure
	 * event is to use a wl_display.sync request right after the
	 * set_fullscreen request. When the sync callback returns, the last
	 * configure event received just before it will be the correct one,
	 * and should contain the right size for the surface to maximize.
	 *
	 * Setting one state won't unset another state. Use
	 * xdg_surface.unset_fullscreen for unsetting it.
	 */
	void (*set_fullscreen)(struct wl_client *client,
			       struct wl_resource *resource);
	/**
	 * unset_fullscreen - unset the surface state as fullscreen
	 *
	 * Unset the surface fullscreen state.
	 *
	 * Same negotiation as set_fullscreen must be used.
	 */
	void (*unset_fullscreen)(struct wl_client *client,
				 struct wl_resource *resource);
	/**
	 * set_maximized - set the surface state as maximized
	 *
	 * Set the surface as maximized.
	 *
	 * After this request, the compositor will send a configure event
	 * informing the output size minus panel and other MW decorations.
	 *
	 * This request informs the compositor that the next attached
	 * buffer committed will be in a maximized state. The buffer size
	 * should be the same size as the size informed in the configure
	 * event, if the client doesn't want to leave any empty area.
	 *
	 * In other words: the next attached buffer after set_maximized is
	 * the new maximized buffer. And the surface will be positioned at
	 * the maximized position on commit.
	 *
	 * A simple way to synchronize and wait for the correct configure
	 * event is to use a wl_display.sync request right after the
	 * set_maximized request. When the sync callback returns, the last
	 * configure event received just before it will be the correct one,
	 * and should contain the right size for the surface to maximize.
	 *
	 * Setting one state won't unset another state. Use
	 * xdg_surface.unset_maximized for unsetting it.
	 */
	void (*set_maximized)(struct wl_client *client,
			      struct wl_resource *resource);
	/**
	 * unset_maximized - unset the surface state as maximized
	 *
	 * Unset the surface maximized state.
	 *
	 * Same negotiation as set_maximized must be used.
	 */
	void (*unset_maximized)(struct wl_client *client,
				struct wl_resource *resource);
	/**
	 * set_minimized - set the surface state as minimized
	 *
	 * Set the surface minimized state.
	 *
	 * Setting one state won't unset another state.
	 */
	void (*set_minimized)(struct wl_client *client,
			      struct wl_resource *resource);
};

#define XDG_SURFACE_PING	0
#define XDG_SURFACE_CONFIGURE	1
#define XDG_SURFACE_REQUEST_SET_FULLSCREEN	2
#define XDG_SURFACE_REQUEST_UNSET_FULLSCREEN	3
#define XDG_SURFACE_REQUEST_SET_MAXIMIZED	4
#define XDG_SURFACE_REQUEST_UNSET_MAXIMIZED	5
#define XDG_SURFACE_FOCUSED_SET	6
#define XDG_SURFACE_FOCUSED_UNSET	7

static inline void
xdg_surface_send_ping(struct wl_resource *resource_, uint32_t serial)
{
	wl_resource_post_event(resource_, XDG_SURFACE_PING, serial);
}

static inline void
xdg_surface_send_configure(struct wl_resource *resource_, uint32_t edges, int32_t width, int32_t height)
{
	wl_resource_post_event(resource_, XDG_SURFACE_CONFIGURE, edges, width, height);
}

static inline void
xdg_surface_send_request_set_fullscreen(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_REQUEST_SET_FULLSCREEN);
}

static inline void
xdg_surface_send_request_unset_fullscreen(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_REQUEST_UNSET_FULLSCREEN);
}

static inline void
xdg_surface_send_request_set_maximized(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_REQUEST_SET_MAXIMIZED);
}

static inline void
xdg_surface_send_request_unset_maximized(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_REQUEST_UNSET_MAXIMIZED);
}

static inline void
xdg_surface_send_focused_set(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_FOCUSED_SET);
}

static inline void
xdg_surface_send_focused_unset(struct wl_resource *resource_)
{
	wl_resource_post_event(resource_, XDG_SURFACE_FOCUSED_UNSET);
}

/**
 * xdg_popup - desktop-style metadata interface
 * @destroy: remove xdg_surface interface
 * @pong: respond to a ping event
 *
 * An interface that may be implemented by a wl_surface, for
 * implementations that provide a desktop-style popups/menus. A popup
 * surface is a transient surface with an added pointer grab.
 *
 * An existing implicit grab will be changed to owner-events mode, and the
 * popup grab will continue after the implicit grab ends (i.e. releasing
 * the mouse button does not cause the popup to be unmapped).
 *
 * The popup grab continues until the window is destroyed or a mouse button
 * is pressed in any other clients window. A click in any of the clients
 * surfaces is reported as normal, however, clicks in other clients
 * surfaces will be discarded and trigger the callback.
 *
 * The x and y arguments specify the locations of the upper left corner of
 * the surface relative to the upper left corner of the parent surface, in
 * surface local coordinates.
 *
 * xdg_popup surfaces are always transient for another surface.
 */
struct xdg_popup_interface {
	/**
	 * destroy - remove xdg_surface interface
	 *
	 * The xdg_surface interface is removed from the wl_surface
	 * object that was turned into a xdg_surface with
	 * xdg_shell.get_xdg_surface request. The xdg_surface properties,
	 * like maximized and fullscreen, are lost. The wl_surface loses
	 * its role as a xdg_surface. The wl_surface is unmapped.
	 */
	void (*destroy)(struct wl_client *client,
			struct wl_resource *resource);
	/**
	 * pong - respond to a ping event
	 * @serial: serial of the ping event
	 *
	 * A client must respond to a ping event with a pong request or
	 * the client may be deemed unresponsive.
	 */
	void (*pong)(struct wl_client *client,
		     struct wl_resource *resource,
		     uint32_t serial);
};

#define XDG_POPUP_PING	0
#define XDG_POPUP_POPUP_DONE	1

static inline void
xdg_popup_send_ping(struct wl_resource *resource_, uint32_t serial)
{
	wl_resource_post_event(resource_, XDG_POPUP_PING, serial);
}

static inline void
xdg_popup_send_popup_done(struct wl_resource *resource_, uint32_t serial)
{
	wl_resource_post_event(resource_, XDG_POPUP_POPUP_DONE, serial);
}

#ifdef  __cplusplus
}
#endif

#endif
