# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# See the COPYING file for license information.
#
# Copyright (c) 2006, 2007 Guillaume Chazarain <guichaz@yahoo.fr>

import os
import stat

from pysize.core import chdir_browsing
from pysize.core.observable import observable
from pysize.core.pysize_global_fs_cache import get_dev_ino, cache_add_dir
from pysize.core.pysize_global_fs_cache import cache_add_hardlink
from pysize.core.pysize_global_fs_cache import cache_get_dir_size
from pysize.core.deletion import get_subtracted

size_observable = observable()

def log_error(text, e):
    print text, e

def _fast(path, dir_ino, cross_device, error_cb):
    """Return the size of the file or directory at path, using or updating the
    cache. dir_ino is the inode of the directory containing the path, it is
    used only with hardlinks.
    """
    try:
        size_observable.fire_observers()
        st = os.lstat(path)
        if stat.S_ISDIR(st.st_mode): # Directory
            dev_ino = st.st_dev, st.st_ino
            cached_size = cache_get_dir_size(dev_ino)
            if cached_size is None:
                size = st.st_blocks * 512
                dir_ino = st.st_ino
                cookie = chdir_browsing.init(path)
                try:
                    for child in chdir_browsing.browsedir(cookie, cross_device,
                                                        account_deletion=False):
                        size += _fast(child, dir_ino, cross_device, error_cb)
                finally:
                    chdir_browsing.finalize(cookie)
                cache_add_dir(dev_ino, size)
            else:
                size = cached_size
        elif st.st_nlink > 1: # Hardlink
            if cache_add_hardlink(st.st_dev, st.st_ino, dir_ino, path):
                size = st.st_blocks * 512
            else:
                size = 0
        else: # File
            size = st.st_blocks * 512
        return size
    except OSError, e:
        error_cb('(%s) size.fast(%s)' % (os.getcwd(), path), e)
        return 0

def slow(path, cross_device=True, error_cb=log_error, account_deletion=True):
    """Same as _size_fast(path, dir_ino), except that dir_ino is computed."""
    def parent_inode():
        """Return the inode of the parent directory"""
        path_st = os.lstat(path)
        if stat.S_ISDIR(path_st.st_mode):
            parent_path = path + '/..'
        else:
            parent_path = os.path.dirname(path) or '.'
        return get_dev_ino(parent_path)[1]
    try:
        parent_ino = parent_inode()
    except OSError, e:
        error_cb('size.slow(%s)' % (path), e)
        return 0
    size = _fast(path, parent_ino, cross_device, error_cb)
    if account_deletion:
        size -= get_subtracted(path)
    return size

def slow_sum(paths, cross_device=True, error_cb=log_error):
    res = 0
    for p in paths:
        res += slow(p, cross_device, error_cb)
    return res
