# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# See the COPYING file for license information.
#
# Copyright (c) 2006, 2007 Guillaume Chazarain <guichaz@yahoo.fr>

import os

from pysize.core.pysize_global_fs_cache import get_dev_ino

# ['/path/to/file1', '/path/to/file2']
TO_DELETE = []

# {device: {inode: (size_to_subtract|DELETED)}}
DEV_INO_TO_SUBTRACT = {}

DELETED = 'deleted'

def _get_subtracted(path):
    dev, ino = get_dev_ino(path)
    return DEV_INO_TO_SUBTRACT.setdefault(dev, {}).get(ino, None)

def _iter_to_the_root(path):
    while path not in ('/', ''):
        yield get_dev_ino(path), path
        path = os.path.dirname(path)

def get_subtracted(path):
    if DEV_INO_TO_SUBTRACT:
        for dev_ino, tmp_path in _iter_to_the_root(path):
            if _get_subtracted(tmp_path) is DELETED:
                return DELETED
        subtracted = _get_subtracted(path)
        if subtracted is not None:
            return subtracted
    return 0

def _add_hash(path):
    from pysize.core import compute_size
    size = compute_size.slow(path)
    dev, ino = get_dev_ino(path)
    DEV_INO_TO_SUBTRACT.setdefault(dev, {})[ino] = DELETED
    for (dev, ino), path in _iter_to_the_root(os.path.dirname(path)):
        inodes = DEV_INO_TO_SUBTRACT.setdefault(dev, {})
        subtract = inodes.get(ino, 0)
        subtract += size
        inodes[ino] = subtract

# TODO: Deletion of hard links
def schedule(nodes):
    for node in nodes:
        for path in node.get_fullpaths():
            subtract = get_subtracted(path)
            if subtract is not DELETED:
                TO_DELETE.append(path)
                _add_hash(path)
    _prune_list()
    _rebuild_hash()

def _prune_list():
    """Remove children if we have the parent"""
    while True:
        for i in xrange(len(TO_DELETE)):
            parent = os.path.dirname(TO_DELETE[i])
            if get_subtracted(parent) is DELETED:
                del TO_DELETE[i]
                break
        else:
            break

def _rebuild_hash():
    DEV_INO_TO_SUBTRACT.clear()
    for path in TO_DELETE:
        _add_hash(path)

def filter_deleted(paths):
    if DEV_INO_TO_SUBTRACT:
        def is_not_deleted(path):
            return get_subtracted(path) is not DELETED
        return filter(is_not_deleted, paths)
    return paths

def get_deleted():
    return TO_DELETE[:]

def restore(path):
    from pysize.core import compute_size
    TO_DELETE.remove(path)
    dev, ino = get_dev_ino(path)
    del DEV_INO_TO_SUBTRACT[dev][ino]
    size = compute_size.slow(path, account_deletion=False)
    for (dev, ino), path in _iter_to_the_root(os.path.dirname(path)):
        inodes = DEV_INO_TO_SUBTRACT[dev]
        if inodes[ino] == size:
            del inodes[ino]
        else:
            inodes[ino] -= size
