# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# See the COPYING file for license information.
#
# Copyright (c) 2006, 2007 Guillaume Chazarain <guichaz@yahoo.fr>

import os
import stat

from pysize.core.deletion import filter_deleted

# We cannot use a generator because we rely on the finally block to
# be sure finalize() will be called. Python 2.5 allows finally block
# in generator, but it relies on the GC so it's not sure it will be
# called at the right time.
#
# The expected use of these browsing functions is:
#
# cookie = chdir_browsing.init(path)
# try:
#     for child in chdir_browsing.browsedir(cookie, (True|False)):
#         .. Do something with child ..
# finally:
#     chdir_browsing.finalize(cookie)

def init(path):
    previous_cwd = os.open('.', os.O_RDONLY)
    try:
        os.chdir(path)
        return previous_cwd
    except OSError, e:
        os.close(previous_cwd)
        raise e

def browsedir(previous_cwd, cross_device, account_deletion=True):
    res = os.listdir('.')
    if not cross_device:
        def same_dev(dev, path):
            try:
                return os.lstat(path)[stat.ST_DEV] == dev
            except OSError:
                return True
        dev = os.lstat('.')[stat.ST_DEV]
        res = [p for p in res if same_dev(dev, p)]
    if account_deletion:
        res = filter_deleted(res)
    # We want the order to be always so same so that hard links
    # detection will be consistent across filesystems
    res.sort()
    return res

def finalize(previous_cwd):
    if previous_cwd:
        try:
            os.fchdir(previous_cwd)
        finally:
            os.close(previous_cwd)

def listdir(path, cross_device):
    cookie = init(path)
    try:
        return browsedir(cookie, cross_device)
    finally:
        finalize(cookie)
