/*

    Copyright (C) 2013  Hong Jen Yee (PCMan) <pcman.tw@gmail.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


#include "application.h"
#include "mainwindow.h"
#include "desktopwindow.h"
#include <QDBusConnection>
#include <QDBusInterface>
#include <QDir>
#include <QDesktopWidget>
#include <QVector>
#include <QLocale>
#include <QLibraryInfo>
#include <QPixmapCache>
#include <QFile>
#include <QMessageBox>
#include <gio/gio.h>

#include "applicationadaptor.h"
#include "preferencesdialog.h"
#include "desktoppreferencesdialog.h"
#include "mountoperation.h"
#include "autorundialog.h"
#include "launcher.h"

#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
#include <QScreen>
#include <QWindow>
#endif

#include <X11/Xlib.h>

using namespace PCManFM;
static const char* serviceName = "org.pcmanfm.PCManFM";
static const char* ifaceName = "org.pcmanfm.Application";

int ProxyStyle::styleHint(StyleHint hint, const QStyleOption* option, const QWidget* widget, QStyleHintReturn* returnData) const {
  Application* app = static_cast<Application*>(qApp);
  if(hint == QStyle::SH_ItemView_ActivateItemOnSingleClick)
    return app->settings().singleClick();
  return QProxyStyle::styleHint(hint, option, widget, returnData);
}

Application::Application(int& argc, char** argv):
  QApplication(argc, argv),
  libFm_(),
  settings_(),
  profileName_("default"),
  daemonMode_(false),
  desktopWindows_(),
  enableDesktopManager_(false),
  preferencesDialog_(),
  volumeMonitor_(NULL),
  editBookmarksialog_() {

  argc_ = argc;
  argv_ = argv;

  // QDBusConnection::sessionBus().registerObject("/org/pcmanfm/Application", this);
  QDBusConnection dbus = QDBusConnection::sessionBus();
  if(dbus.registerService(serviceName)) {
    // we successfully registered the service
    isPrimaryInstance = true;
    setStyle(new ProxyStyle());
    desktop()->installEventFilter(this);

    new ApplicationAdaptor(this);
    dbus.registerObject("/Application", this);

    connect(this, SIGNAL(aboutToQuit()), SLOT(onAboutToQuit()));
    settings_.load(profileName_);

    // decrease the cache size to reduce memory usage
    QPixmapCache::setCacheLimit(2048);

    if(settings_.useFallbackIconTheme()) {
      QIcon::setThemeName(settings_.fallbackIconThemeName());
      Fm::IconTheme::checkChanged();
    }
  }
  else {
    // an service of the same name is already registered.
    // we're not the first instance
    isPrimaryInstance = false;
  }
}

Application::~Application() {
  desktop()->removeEventFilter(this);

  if(volumeMonitor_) {
    g_signal_handlers_disconnect_by_func(volumeMonitor_, gpointer(onVolumeAdded), this);
    g_object_unref(volumeMonitor_);
  }

#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
  if(enableDesktopManager_)
    removeNativeEventFilter(this);
#endif
}

struct FakeTr {
  FakeTr(int reserved = 20) {
    strings.reserve(reserved);
  }

  const char* operator()(const char* str) {
    QString translated = QApplication::translate(NULL, str);
    strings.push_back(translated.toUtf8());
    return strings.back().constData();
  }
  QVector<QByteArray> strings;
};

bool Application::parseCommandLineArgs() {
  bool keepRunning = false;

  // It's really a shame that the great Qt library does not come
  // with any command line parser.
  // After trying some Qt ways, I finally realized that glib is the best.
  // Simple, efficient, effective, and does not use signal/slot!
  // The only drawback is the translated string returned by tr() is
  // a temporary one. We need to store them in a list to keep them alive. :-(

  char* profile = NULL;
  gboolean daemon_mode = FALSE;
  gboolean ask_quit = FALSE;
  gboolean desktop = FALSE;
  gboolean desktop_off = FALSE;
  char* desktop_pref = NULL;
  char* wallpaper = NULL;
  char* wallpaper_mode = NULL;
  char* show_pref = NULL;
  gboolean new_window = FALSE;
  gboolean find_files = FALSE;
  char** file_names = NULL;
  {
    FakeTr tr; // a functor used to override QObject::tr().
    // it convert the translated strings to UTF8 and add them to a list to
    // keep them alive during the option parsing process.
    GOptionEntry option_entries[] = {
      /* options only acceptable by first pcmanfm instance. These options are not passed through IPC */
      {"profile", 'p', 0, G_OPTION_ARG_STRING, &profile, tr("Name of configuration profile"), tr("PROFILE") },
      {"daemon-mode", 'd', 0, G_OPTION_ARG_NONE, &daemon_mode, tr("Run PCManFM as a daemon"), NULL },
      // options that are acceptable for every instance of pcmanfm and will be passed through IPC.
      {"quit", 'p', 0, G_OPTION_ARG_NONE, &ask_quit, tr("Quit PCManFM"), NULL},
      {"desktop", '\0', 0, G_OPTION_ARG_NONE, &desktop, tr("Launch desktop manager"), NULL },
      {"desktop-off", '\0', 0, G_OPTION_ARG_NONE, &desktop_off, tr("Turn off desktop manager if it's running"), NULL },
      {"desktop-pref", '\0', 0, G_OPTION_ARG_STRING, &desktop_pref, tr("Open desktop preference dialog on the page with the specified name"), tr("NAME") },
      {"set-wallpaper", 'w', 0, G_OPTION_ARG_FILENAME, &wallpaper, tr("Set desktop wallpaper from image FILE"), tr("FILE") },
      // don't translate list of modes in description, please
      {"wallpaper-mode", '\0', 0, G_OPTION_ARG_STRING, &wallpaper_mode, tr("Set mode of desktop wallpaper. MODE=(color|stretch|fit|center|tile)"), tr("MODE") },
      {"show-pref", '\0', 0, G_OPTION_ARG_STRING, &show_pref, tr("Open Preferences dialog on the page with the specified name"), tr("NAME") },
      {"new-window", 'n', 0, G_OPTION_ARG_NONE, &new_window, tr("Open new window"), NULL },
      {"find-files", 'f', 0, G_OPTION_ARG_NONE, &find_files, tr("Open Find Files utility"), NULL },
      {G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &file_names, NULL, tr("[FILE1, FILE2,...]")},
      { NULL }
    };

    GOptionContext* context = g_option_context_new("");
    g_option_context_add_main_entries(context, option_entries, NULL);
    GError* error = NULL;
    if(!g_option_context_parse(context, &argc_, &argv_, &error)) {
      // show error and exit
      g_fprintf(stderr, "%s\n\n", error->message);
      g_error_free(error);
      g_option_context_free(context);
      return false;
    }
    g_option_context_free(context);
  }

  if(isPrimaryInstance) {
    qDebug("isPrimaryInstance");

    if(daemon_mode)
      daemonMode_ = true;
    if(profile)
      profileName_ = profile;

    // load settings
    settings_.load(profileName_);

    // desktop icon management
    if(desktop) {
      desktopManager(true);
      keepRunning = true;
    }
    else if(desktop_off)
      desktopManager(false);

    if(desktop_pref) { // desktop preference dialog
      desktopPrefrences(desktop_pref);
      keepRunning = true;
    }
    else if(find_files) { // file searching utility
      QStringList paths;
      if(file_names) {
        for(char** filename = file_names; *filename; ++filename) {
          QString path(*filename);
          paths.push_back(path);
        }
      }
      findFiles(paths);
      keepRunning = true;
    }
    else if(show_pref) { // preferences dialog
      preferences(show_pref);
      keepRunning = true;
    }
    else if(wallpaper || wallpaper_mode) // set wall paper
      setWallpaper(wallpaper, wallpaper_mode);
    else {
      if(!desktop && !desktop_off) {
        QStringList paths;
        if(file_names) {
          for(char** filename = file_names; *filename; ++filename) {
            QString path = QString::fromLocal8Bit(*filename);
            paths.push_back(path);
          }
        }
        else {
          // if no path is specified and we're using daemon mode,
          // don't open current working directory
          if(!daemonMode_)
            paths.push_back(QDir::currentPath());
        }
        if(!paths.isEmpty())
          launchFiles(paths, (bool)new_window);
        keepRunning = true;
      }
    }
  }
  else {
    QDBusConnection dbus = QDBusConnection::sessionBus();
    QDBusInterface iface(serviceName, "/Application", ifaceName, dbus, this);
    if(ask_quit) {
      iface.call("quit");
      return false;
    }

    if(desktop)
      iface.call("desktopManager", true);
    else if(desktop_off)
      iface.call("desktopManager", false);

    if(desktop_pref) { // desktop preference dialog
      iface.call("desktopPrefrences", QString(desktop_pref));
    }
    else if(find_files) { // file searching utility
      QStringList paths;
      if(file_names) {
        for(char** filename = file_names; *filename; ++filename) {
          QString path(*filename);
          paths.push_back(path);
        }
      }
      iface.call("findFiles", paths);
    }
    else if(show_pref) { // preferences dialog
      iface.call("preferences", QString(show_pref));
    }
    else if(wallpaper || wallpaper_mode) { // set wall paper
      iface.call("setWallpaper", QString(wallpaper), QString(wallpaper_mode));
    }
    else {
      if(!desktop && !desktop_off) {
        QStringList paths;
        if(file_names) {
          for(char** filename = file_names; *filename; ++filename) {
            QString path = QString::fromLocal8Bit(*filename);
            paths.push_back(path);
          }
        }
        else
          paths.push_back(QDir::currentPath());
        // the function requires bool, but new_window is gboolean, the casting is needed.
        iface.call("launchFiles", paths, (bool)new_window);
      }
    }
  }

  // cleanup
  g_free(desktop_pref);
  g_free(show_pref);
  g_free(wallpaper);
  g_free(wallpaper_mode);
  g_free(profile);
  g_strfreev(file_names);

  return keepRunning;
}

void Application::init() {

  // install the translations built-into Qt itself
  qtTranslator.load("qt_" + QLocale::system().name(), QLibraryInfo::location(QLibraryInfo::TranslationsPath));
  installTranslator(&qtTranslator);

  // install libfm-qt translator
  installTranslator(libFm_.translator());

  // install our own tranlations
  translator.load("pcmanfm-qt_" + QLocale::system().name(), PCMANFM_DATA_DIR "/translations");
  installTranslator(&translator);
}

int Application::exec() {

  if(!parseCommandLineArgs())
    return 0;

  if(daemonMode_) // keep running even when there is no window opened.
    setQuitOnLastWindowClosed(false);

  volumeMonitor_ = g_volume_monitor_get();
  // delay the volume manager a little because in newer versions of glib/gio there's a problem.
  // when the first volume monitor object is created, it discovers volumes asynchonously.
  // g_volume_monitor_get() immediately returns while the monitor is still discovering devices.
  // So initially g_volume_monitor_get_volumes() returns nothing, but shortly after that
  // we get volume-added signals for all of the volumes. This is not what we want.
  // So, we wait for 3 seconds here to let it finish device discovery.
  QTimer::singleShot(3000, this, SLOT(initVolumeManager()));

  return QCoreApplication::exec();
}

void Application::onAboutToQuit() {
  qDebug("aboutToQuit");
  settings_.save();
}

bool Application::eventFilter(QObject* watched, QEvent* event) {
  if(watched == desktop()) {
    // qDebug() << "filter" << event;
    switch(event->type()) {
      case QEvent::StyleChange:
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
      case QEvent::ThemeChange:
#endif
        setStyle(new ProxyStyle());
    };
  }
  return QObject::eventFilter(watched, event);
}

void Application::commitData(QSessionManager& manager) {
  qDebug("commitData");
  // FIXME: where should we write the config file?
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
#else
  QApplication::commitData(manager);
#endif
}

void Application::onLastWindowClosed() {

}

void Application::onSaveStateRequest(QSessionManager& manager) {

}

void Application::desktopManager(bool enabled) {
  // TODO: turn on or turn off desktpo management (desktop icons & wallpaper)
  qDebug("desktopManager: %d", enabled);
  QDesktopWidget* desktopWidget = desktop();
  if(enabled) {
    if(!enableDesktopManager_) {
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
      installNativeEventFilter(this);
      Q_FOREACH(QScreen* screen, screens()) {
        connect(screen, &QScreen::virtualGeometryChanged, this, &Application::onVirtualGeometryChanged);
        connect(screen, &QObject::destroyed, this, &Application::onScreenDestroyed);
      }
      connect(this, &QApplication::screenAdded, this, &Application::onScreenAdded);
#else
      connect(desktopWidget, SIGNAL(workAreaResized(int)), SLOT(onWorkAreaResized(int)));
#endif
      connect(desktopWidget, SIGNAL(resized(int)), SLOT(onScreenResized(int)));
      connect(desktopWidget, SIGNAL(screenCountChanged(int)), SLOT(onScreenCountChanged(int)));

      // NOTE: there are two modes
      // When virtual desktop is used (all screens are combined to form a large virtual desktop),
      // we only create one DesktopWindow. Otherwise, we create one for each screen.
      if(desktopWidget->isVirtualDesktop()) {
        DesktopWindow* window = createDesktopWindow(-1);
        desktopWindows_.push_back(window);
      }
      else {
        int n = desktopWidget->numScreens();
        desktopWindows_.reserve(n);
        for(int i = 0; i < n; ++i) {
          DesktopWindow* window = createDesktopWindow(i);
          desktopWindows_.push_back(window);
        }
      }
    }
  }
  else {
    if(enableDesktopManager_) {
      disconnect(desktopWidget, SIGNAL(resized(int)), this, SLOT(onScreenResized(int)));
      disconnect(desktopWidget, SIGNAL(screenCountChanged(int)), this, SLOT(onScreenCountChanged(int)));
      int n = desktopWindows_.size();
      for(int i = 0; i < n; ++i) {
        DesktopWindow* window = desktopWindows_.at(i);
        delete window;
      }
      desktopWindows_.clear();
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
      Q_FOREACH(QScreen* screen, screens()) {
        disconnect(screen, &QScreen::virtualGeometryChanged, this, &Application::onVirtualGeometryChanged);
        disconnect(screen, &QObject::destroyed, this, &Application::onScreenDestroyed);
      }
      disconnect(this, &QApplication::screenAdded, this, &Application::onScreenAdded);
      removeNativeEventFilter(this);
#else
      disconnect(desktopWidget, SIGNAL(workAreaResized(int)), this, SLOT(onWorkAreaResized(int)));
#endif
    }
  }
  enableDesktopManager_ = enabled;
}

void Application::desktopPrefrences(QString page) {
  // show desktop preference window
  if(!desktopPreferencesDialog_) {
    desktopPreferencesDialog_ = new DesktopPreferencesDialog();
  }
  desktopPreferencesDialog_.data()->selectPage(page);
  desktopPreferencesDialog_.data()->show();
  desktopPreferencesDialog_.data()->raise();
  desktopPreferencesDialog_.data()->activateWindow();
}

void Application::findFiles(QStringList paths) {
  // TODO: add a file searching utility here.
  qDebug("findFiles");
}

void Application::launchFiles(QStringList paths, bool inNewWindow) {
  FmPathList* pathList = fm_path_list_new();
  QStringList::iterator it;
  for(it = paths.begin(); it != paths.end(); ++it) {
    QString& pathName = *it;
    FmPath* path = fm_path_new_for_path(pathName.toLocal8Bit().constData());
    fm_path_list_push_tail(pathList, path);
    fm_path_unref(path);
  }
  Launcher(NULL).launchPaths(NULL, pathList);
  fm_path_list_unref(pathList);
}

void Application::openFolders(FmFileInfoList* files) {
  Launcher(NULL).launchFiles(NULL, files);
}

void Application::openFolderInTerminal(FmPath* path) {
  if(!settings_.terminal().isEmpty()) {
    char* cwd_str;
    if(fm_path_is_native(path))
      cwd_str = fm_path_to_str(path);
    else { // gio will map remote filesystems to local FUSE-mounted paths here.
      GFile* gf = fm_path_to_gfile(path);
      cwd_str = g_file_get_path(gf);
      g_object_unref(gf);
    }
    GError* err = NULL;
    if(!fm_terminal_launch(cwd_str, &err)) {
      QMessageBox::critical(NULL, tr("Error"), QString::fromUtf8(err->message));
      g_error_free(err);
    }
    g_free(cwd_str);
  }
  else {
    // show an error message and ask the user to set the command
    QMessageBox::critical(NULL, tr("Error"), tr("Terminal emulator is not set."));
    preferences("advanced");
  }
}

void Application::preferences(QString page) {
  // open preference dialog
  if(!preferencesDialog_) {
    preferencesDialog_ = new PreferencesDialog(page);
  }
  else {
    // TODO: set page
  }
  preferencesDialog_.data()->show();
  preferencesDialog_.data()->raise();
  preferencesDialog_.data()->activateWindow();
}

void Application::setWallpaper(QString path, QString modeString) {
  static const char* valid_wallpaper_modes[] = {"color", "stretch", "fit", "center", "tile"};
  DesktopWindow::WallpaperMode mode = settings_.wallpaperMode();
  bool changed = false;

  if(!path.isEmpty() && path != settings_.wallpaper()) {
    if(QFile(path).exists()) {
      settings_.setWallpaper(path);
      changed = true;
    }
  }
  // convert mode string to value
  for(int i = 0; i < G_N_ELEMENTS(valid_wallpaper_modes); ++i) {
    if(modeString == valid_wallpaper_modes[i]) {
      mode = (DesktopWindow::WallpaperMode)i;
      if(mode != settings_.wallpaperMode())
        changed = true;
      break;
    }
  }
  // FIXME: support different wallpapers on different screen.
  // update wallpaper
  if(changed) {
    if(enableDesktopManager_) {
      Q_FOREACH(DesktopWindow * desktopWindow, desktopWindows_) {
        if(!path.isEmpty())
          desktopWindow->setWallpaperFile(path);
        if(mode != settings_.wallpaperMode())
          desktopWindow->setWallpaperMode(mode);
        desktopWindow->updateWallpaper();
      }
      settings_.save(); // save the settings to the config file
    }
  }
}

void Application::onScreenResized(int num) {
  if(desktop()->isVirtualDesktop()) {
    // in virtual desktop mode, we only have one desktop window. that is the first one.
    DesktopWindow* window = desktopWindows_.at(0);
    window->setGeometry(desktop()->geometry());
  }
  else {
    DesktopWindow* window = desktopWindows_.at(num);
    QRect rect = desktop()->screenGeometry(num);
    window->setGeometry(rect);
  }
}

// This slot is for Qt4 only
void Application::onWorkAreaResized(int num) {
#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
  if(desktop()->isVirtualDesktop())
    num = 0; // in virtual desktop mode, we only have one desktop window. that is the first one.
  DesktopWindow* window = desktopWindows_.at(num);
  QRect rect = desktop()->availableGeometry(num);
  qDebug() << "workAreaResized" << num << rect;
  window->queueRelayout();
#endif
}

DesktopWindow* Application::createDesktopWindow(int screenNum) {
  DesktopWindow* window = new DesktopWindow(screenNum);
  if(screenNum == -1) { // one large virtual desktop only
    QRect rect = desktop()->geometry();
    window->setGeometry(rect);
  }
  else {
    QRect rect = desktop()->screenGeometry(screenNum);
    window->setGeometry(rect);
  }
  window->updateFromSettings(settings_);
  window->show();
  return window;
}

void Application::onScreenCountChanged(int newCount) {
  QDesktopWidget* desktopWidget = desktop();
  bool oldVirtual = (desktopWindows_.size() == 1 && desktopWindows_.at(0)->screenNum() == -1);
  bool isVirtual = desktopWidget->isVirtualDesktop();

  if(oldVirtual && isVirtual) {
    // if we are using virtual desktop mode previously, and the new mode is sitll virtual
    // no further change is needed, only do relayout.
    desktopWindows_.at(0)->queueRelayout();
    return;
  }

  // we used non-virtual mode originally, but now we're switched to virtual mode
  if(isVirtual)
      newCount = 1; // we only want one desktop window for all screens in virtual mode

  if(newCount > desktopWindows_.size()) {
    // add more desktop windows
    for(int i = desktopWindows_.size(); i < newCount; ++i) {
      DesktopWindow* desktop = createDesktopWindow(i);
      desktopWindows_.push_back(desktop);
    }
  }
  else if(newCount < desktopWindows_.size()) {
    // delete excessive desktop windows
    for(int i = newCount; i < desktopWindows_.size(); ++i) {
      DesktopWindow* desktop = desktopWindows_.at(i);
      delete desktop;
    }
    desktopWindows_.resize(newCount);
  }

  if(newCount == 1) { // now only 1 screen is in use
    DesktopWindow* desktop = desktopWindows_.at(0);
    if(isVirtual)
      desktop->setScreenNum(-1);
    else // non-virtual mode, and we only have 1 screen
      desktop->setScreenNum(0);
    desktop->updateWallpaper();
  }
}

// called when Settings is changed to update UI
void Application::updateFromSettings() {
  // if(iconTheme.isEmpty())
  //  Fm::IconTheme::setThemeName(settings_.fallbackIconThemeName());

  // update main windows and desktop windows
  QWidgetList windows = this->topLevelWidgets();
  QWidgetList::iterator it;
  for(it = windows.begin(); it != windows.end(); ++it) {
    QWidget* window = *it;
    if(window->inherits("PCManFM::MainWindow")) {
      MainWindow* mainWindow = static_cast<MainWindow*>(window);
      mainWindow->updateFromSettings(settings_);
    }
  }
  if(desktopManagerEnabled())
    updateDesktopsFromSettings();
}

void Application::updateDesktopsFromSettings() {
  QVector<DesktopWindow*>::iterator it;
  for(it = desktopWindows_.begin(); it != desktopWindows_.end(); ++it) {
    DesktopWindow* desktopWindow = static_cast<DesktopWindow*>(*it);
    desktopWindow->updateFromSettings(settings_);
  }
}

void Application::editBookmarks() {
  if(!editBookmarksialog_) {
    FmBookmarks* bookmarks = fm_bookmarks_dup();
    editBookmarksialog_ = new Fm::EditBookmarksDialog(bookmarks);
    g_object_unref(bookmarks);
  }
  editBookmarksialog_.data()->show();
}

void Application::initVolumeManager() {

  g_signal_connect(volumeMonitor_, "volume-added", G_CALLBACK(onVolumeAdded), this);

  if(settings_.mountOnStartup()) {
    /* try to automount all volumes */
    GList* vols = g_volume_monitor_get_volumes(volumeMonitor_);
    for(GList* l = vols; l; l = l->next) {
      GVolume* volume = G_VOLUME(l->data);
      if(g_volume_should_automount(volume))
        autoMountVolume(volume, false);
      g_object_unref(volume);
    }
    g_list_free(vols);
  }
}

bool Application::autoMountVolume(GVolume* volume, bool interactive) {
  if(!g_volume_should_automount(volume) || !g_volume_can_mount(volume))
    return FALSE;

  GMount* mount = g_volume_get_mount(volume);
  if(!mount) { // not mounted, automount is needed
    // try automount
    Fm::MountOperation* op = new Fm::MountOperation(interactive);
    op->mount(volume);
    if(!op->wait())
      return false;
    if(!interactive)
      return true;
    mount = g_volume_get_mount(volume);
  }

  if(mount) {
    if(interactive && settings_.autoRun()) { // show autorun dialog
      AutoRunDialog* dlg = new AutoRunDialog(volume, mount);
      dlg->show();
    }
    g_object_unref(mount);
  }
  return true;
}

// static
void Application::onVolumeAdded(GVolumeMonitor* monitor, GVolume* volume, Application* pThis) {
  if(pThis->settings_.mountRemovable())
    pThis->autoMountVolume(volume, true);
}

#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)

bool Application::nativeEventFilter(const QByteArray & eventType, void * message, long * result) {
  if(eventType == "xcb_generic_event_t") { // XCB event
    // filter all native X11 events (xcb)
    xcb_generic_event_t* generic_event = reinterpret_cast<xcb_generic_event_t*>(message);
    // qDebug("XCB event: %d", generic_event->response_type & ~0x80);
    Q_FOREACH(DesktopWindow * window, desktopWindows_) {
      window->xcbEvent(generic_event);
    }
  }
  return false;
}

#endif

// This slot is for Qt 5 onlt, but the stupid Qt moc cannot do conditional compilation
// so we have to define it for Qt 4 as well.
void Application::onScreenAdded(QScreen* newScreen) {
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
  if(enableDesktopManager_) {
    connect(newScreen, &QScreen::virtualGeometryChanged, this, &Application::onVirtualGeometryChanged);
    connect(newScreen, &QObject::destroyed, this, &Application::onScreenDestroyed);
  }
#endif
}

// This slot is for Qt 5 onlt, but the stupid Qt moc cannot do conditional compilation
// so we have to define it for Qt 4 as well.
void Application::onScreenDestroyed(QObject* screenObj) {
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
  // NOTE by PCMan: This is a workaround for Qt 5 bug #40681.
  // With this very dirty workaround, we can fix lxde/lxde-qt bug #204, #205, and #206.
  // Qt 5 has two new regression bugs which breaks lxqt-panel in a multihead environment.
  // #40681: Regression bug: QWidget::winId() returns old value and QEvent::WinIdChange event is not emitted sometimes. (multihead setup) 
  // #40791: Regression: QPlatformWindow, QWindow, and QWidget::winId() are out of sync.
  // Explanations for the workaround:
  // Internally, Qt mantains a list of QScreens and update it when XRandR configuration changes.
  // When the user turn off an monitor with xrandr --output <xxx> --off, this will destroy the QScreen
  // object which represent the output. If the QScreen being destroyed contains our panel widget,
  // Qt will call QWindow::setScreen(0) on the internal windowHandle() of our panel widget to move it
  // to the primary screen. However, moving a window to a different screen is more than just changing
  // its position. With XRandR, all screens are actually part of the same virtual desktop. However,
  // this is not the case in other setups, such as Xinerama and moving a window to another screen is 
  // not possible unless you destroy the widget and create it again for a new screen.
  // Therefore, Qt destroy the widget and re-create it when moving our panel to a new screen.
  // Unfortunately, destroying the window also destroy the child windows embedded into it,
  // using XEMBED such as the tray icons. (#206)
  // Second, when the window is re-created, the winId of the QWidget is changed, but Qt failed to
  // generate QEvent::WinIdChange event so we have no way to know that. We have to set
  // some X11 window properties using the native winId() to make it a dock, but this stop working
  // because we cannot get the correct winId(), so this causes #204 and #205.
  //
  // The workaround is very simple. Just completely destroy the window before Qt has a chance to do
  // QWindow::setScreen() for it. Later, we recreate the window ourselves. So this can bypassing the Qt bugs.
  QScreen* screen = static_cast<QScreen*>(screenObj);
  if(enableDesktopManager_) {
    bool reloadNeeded = false;
    // FIXME: add workarounds for Qt5 bug #40681 and #40791 here.
    Q_FOREACH(DesktopWindow* desktop, desktopWindows_) {
      if(desktop->windowHandle()->screen() == screenObj) {
        desktop->destroy(); // destroy the underlying native window
        reloadNeeded = true;
      }
    }
    if(reloadNeeded)
        QTimer::singleShot(0, this, SLOT(reloadDesktopsAsNeeded()));
  }
#endif
}

// This slot is for Qt 5 onlt, but the stupid Qt moc cannot do conditional compilation
// so we have to define it for Qt 4 as well.
void Application::reloadDesktopsAsNeeded() {
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
  if(enableDesktopManager_) {
    // workarounds for Qt5 bug #40681 and #40791 here.
    Q_FOREACH(DesktopWindow* desktop, desktopWindows_) {
      if(!desktop->windowHandle()) {
        desktop->create(); // re-create the underlying native window
        desktop->queueRelayout();
        desktop->show();
      }
    }
  }
#endif
}

// This slot is for Qt 5 onlt, but the stupid Qt moc cannot do conditional compilation
// so we have to define it for Qt 4 as well.
void Application::onVirtualGeometryChanged(const QRect& rect) {
  // NOTE: the following is a workaround for Qt bug 32567.
  // https://bugreports.qt-project.org/browse/QTBUG-32567
  // Though the status of the bug report is closed, it's not yet fixed for X11.
  // In theory, QDesktopWidget should emit "workAreaResized()" signal when the work area
  // of any screen is changed, but in fact it does not do it.
  // However, QScreen provided since Qt5 does not have the bug and
  // virtualGeometryChanged() is emitted correctly when the workAreas changed.
  // So we use it in Qt5.
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
  if(enableDesktopManager_) {
    QScreen* screeb = static_cast<QScreen*>(sender());
    // qDebug() << "onVirtualGeometryChanged";
    Q_FOREACH(DesktopWindow* desktop, desktopWindows_) {
      desktop->queueRelayout();
    }
  }
#endif
}
