/*  
 * Copyright (C) 2011 ammonkey <am.monkeyd@gmail.com>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 */ 

using GLib;

namespace ExtendedActions
{
    [DBus (name = "org.magma.ExtendedActions")]
    public class ExtendedActions : Object {
        construct {
            /*application_id = "org.magma.extended-actions";
              flags = GLib.ApplicationFlags.IS_SERVICE;*/
            startup ();
        }

        private EActionFileService cfs;

        //protected override void startup () 
        private void startup () 
        {
            cfs = new EActionFileService ();
        }
        

        private bool all_native;
        private bool is_native;
        private string common_parent;
        private string cmd_uris;
        private Gee.Set<string>? mimes;

        GLib.HashTable<string,string>[] filtered;

        /* generate possible eactions for list of arguments and filter them by 
           common parent mimetype and all.
           We don't want to waste time and ressources determining the common eactions 
           for each files one by one */
        public GLib.HashTable<string,string>[]? GetServicesByLocationsList (GLib.HashTable<string,string>[] locations)
        {
            filtered = null;
            if (!cfs.initialized)
                return null;

            var nbargs = locations.length;
            //message ("locationslist %d", nbargs);

            if (nbargs == 1) {
                GLib.HashTable<string,string> location = locations[0];
                return GetServicesByLocation (location.lookup ("uri"),
                                              location.lookup ("mimetype"));
            }

            analyse_similarities (locations);

            //message ("common parent %s", common_parent);
            var list_for_all = cfs.get_eaction_files_for_type ("all");
            if (list_for_all.size > 0)
            {
                foreach (var entry in list_for_all) {
                    multi_args_add_eaction_to_filtered_table (entry);
                }
            }

            if (common_parent != null) 
            {
                var list_for_parent = cfs.get_eaction_files_for_type (common_parent);
                if (list_for_parent.size > 0)
                {
                    foreach (var entry in list_for_parent) {
                        multi_args_add_eaction_to_filtered_table (entry);
                    }
                }
            }

            /* get all common eaction_files mime by mime and intersect them */
            Gee.List<EActionFileInfo>? list_mime0 = null;
            Gee.List<EActionFileInfo>? list_intersect = null;
            foreach (var mime in mimes) {
                //message ("mime %s", mime);
                if (list_mime0 == null) {
                    list_mime0 = cfs.get_eaction_files_for_type (mime);
                    if (list_mime0.size == 0)
                        break;
                    continue;
                }
                var list_mime1 = cfs.get_eaction_files_for_type (mime);
                if (list_mime1.size > 0) {
                    list_intersect = new Gee.ArrayList<EActionFileInfo> ();
                    foreach (var entry0 in list_mime0) {
                        foreach (var entry1 in list_mime1)
                            if (entry0.name == entry1.name) { 
                                //message ("entry match %s", entry0.name);
                                list_intersect.add (entry0);
                            }
                    }
                    if (list_intersect.size == 0) {
                        //message ("xxxxxxxxxxxx");
                        break;
                    }
                    list_mime0 = list_intersect;

                } else {
                    //message ("xxxxxxxxxxxx");
                    break;
                }
            }

            if (list_intersect != null)
            foreach (var entry in list_intersect)
                multi_args_add_eaction_to_filtered_table (entry);


            /* conditional eactions are eactions which own mime_type entries containing special conditional character(s) like ! for negation. Thoses conditionnals characters can apply to another eaction mime group, a parent_mime or a mime type. At the moment it's relatively simple but maybe later we can extend conditionals to characters like & | () */
            foreach (var cc in cfs.conditional_eactions) {
                debug ("CC %s %s", cc.name, cc.conditional_mime);
                var len = cc.conditional_mime.length;
                if (len >= 2) {
                    string str = cc.conditional_mime.slice (1, len);
                    /* check if the conditional target another eaction name, check if the first letter is a maj */
                    if (str.get_char (0) >= 'A' && str.get_char (0) <= 'Z') {
                        /* check if the eaction exist */ 
                        var eaction_target = cfs.name_id_map[str];
                        if (eaction_target != null && !is_eaction_in_filtered (str)) 
                        {
                            /* check if the eaction isn't filtered 
                               and the matches on corresponding mimetype */
                            if (!is_eaction_in_filtered (str) 
                                && eaction_target.mime_types != null) 
                            {
                                bool ret;
                                if (cc.strict_condition)
                                    ret = are_locations_mimes_match_strict_conditional_eaction_mimes (eaction_target.mime_types, locations); 
                                else
                                    ret = are_locations_mimes_match_conditional_eaction_mimes (eaction_target.mime_types, locations);
                                if (ret) 
                                    multi_args_add_eaction_to_filtered_table (cc);
                            }
                        } else {
                            warning ("%s, Conditional MimeType %s doesn't match anything", cc.name, cc.conditional_mime);
                        }
                    } else if (str.contains("/")) {
                        /* could be a mimetype */
                        if (are_locations_match_conditional_mime (locations, str)) {
                            multi_args_add_eaction_to_filtered_table (cc);
                        }
                    } else {
                        /* could be a parent */
                        if (common_parent != null && common_parent != str) {
                            multi_args_add_eaction_to_filtered_table (cc);
                        }
                    }
                }
            }

            return filtered;
        }
        
        public GLib.HashTable<string,string>[]? GetServicesByLocation (string strlocation, string? file_mime = null) 
        {
            File file = File.new_for_commandline_arg (strlocation);
            
            filtered = null;
            if (!cfs.initialized)
                return null;
            
            is_native = file.is_native ();

            /*if (file.query_exists ()) {
              message ("file exist");
              }*/
            string mimetype;
            string parent_mime = null;

            if (file_mime == null || file_mime.length <= 0)
                mimetype = query_content_type (file);
            else
                mimetype = file_mime;

            //message ("test path %s %s %s", file.get_path (), file.get_uri (), mimetype);
            if (mimetype != null) 
            {
                var list_for_all = cfs.get_eaction_files_for_type ("all");
                if (list_for_all.size > 0)
                {
                    foreach (var entry in list_for_all)
                    {
                        single_arg_add_eaction_to_filtered_table (entry, file);
                    }
                }
                parent_mime = get_parent_mime (mimetype);
                if (parent_mime != null) 
                {
                    var list_for_parent = cfs.get_eaction_files_for_type (parent_mime);
                    if (list_for_parent.size > 0)
                    {
                        foreach (var entry in list_for_parent) {
                            single_arg_add_eaction_to_filtered_table (entry, file);
                        }
                    }
                }
                var list_for_mimetype = cfs.get_eaction_files_for_type (mimetype);
                if (list_for_mimetype.size > 0)
                {
                    foreach (var entry in list_for_mimetype) {
                        single_arg_add_eaction_to_filtered_table (entry, file);
                    }
                }
            }

            /* conditional eactions are eactions which own mime_type entries containing special conditional character(s) like ! for negation. Thoses conditionnals characters can apply to another eaction mime group, a parent_mime or a mime type. At the moment it's relatively simple but maybe later we can extend conditionals to characters like & | () */
            foreach (var cc in cfs.conditional_eactions) {
                debug ("CC %s %s", cc.name, cc.conditional_mime);
                var len = cc.conditional_mime.length;
                if (len >= 2) {
                    string str = cc.conditional_mime.slice (1, len);
                    /* check if the conditional target another eaction name, check if the first letter is a maj */
                    if (str.get_char (0) >= 'A' && str.get_char (0) <= 'Z') {
                        /* check if the eaction exist */
                        var eaction_target = cfs.name_id_map[str];
                        if (eaction_target != null) {
                            /* check is the eaction isn't filtered and the matches on corresponding mimetype */
                            if (!is_eaction_in_filtered (str) 
                                && eaction_target.mime_types != null 
                                && is_mime_match_conditional_eaction_mimes (eaction_target.mime_types, mimetype))
                            {
                                single_arg_add_eaction_to_filtered_table (cc, file);
                            }
                        } else {
                            warning ("%s, Conditional MimeType %s doesn't match anything", cc.name, cc.conditional_mime);
                        }
                    } else if (str.contains("/")) {
                        /* could be a mimetype */
                        if (mimetype != str) {
                            single_arg_add_eaction_to_filtered_table (cc, file);
                        }
                    } else {
                        /* could be a parent */
                        if (parent_mime != null && parent_mime != str) {
                            single_arg_add_eaction_to_filtered_table (cc, file);
                        }
                    }
                }
            }

            return filtered;
        }

        public GLib.HashTable<string,string>[] GetServicesForString () 
        {
            GLib.HashTable<string,string>[] filtered = null;
            
            //message ("GetServicesForString");
            foreach (var cfi in cfs.exec_string_map.values) {
                filtered += get_filtered_entry_for_string (cfi);
                //message ("exec_string %s", cfi.name);
            }

            return filtered;
        }

        private void multi_args_add_eaction_to_filtered_table (EActionFileInfo entry)
        {
            if (!(!all_native && !entry.take_uri_args)
                && !(!entry.take_multi_args)
                && entry.filter_default_app ())
            {
                var filtered_entry = get_filtered_entry_for_list (entry);
                //debug ("desc: %s exec: %s", filtered_entry.lookup ("Description"), filtered_entry.lookup ("Exec"));
                filtered += filtered_entry;
            }
        }

        private void single_arg_add_eaction_to_filtered_table (EActionFileInfo entry, File file)
        {
            if (!(!is_native && !entry.take_uri_args)
                && entry.filter_default_app ())
            {
                var filtered_entry = get_filtered_entry (entry, file);
                //debug ("desc: %s exec: %s", filtered_entry.lookup ("Description"), filtered_entry.lookup ("Exec"));
                filtered += filtered_entry;
            }
        }

        private void analyse_similarities (GLib.HashTable<string,string>[] locations)
        {
            string[] pmimes = null;
            bool[] natives = null;

            all_native = true;
            common_parent = null;
            cmd_uris = "";
            mimes = new Gee.HashSet<string> ();

            foreach (var location in locations) {       
                var uri = location.lookup ("uri");
                cmd_uris += uri + " ";
                File file = File.new_for_commandline_arg (uri);
                string mimetype = location.lookup ("mimetype");
                if (mimetype == null || mimetype.length <= 0)
                    mimetype = query_content_type (file);
                if (mimetype != null)
                    mimes.add (mimetype);
                pmimes += get_parent_mime (mimetype);
                natives += file.is_native ();
            }

            foreach (var pmime in pmimes) {
                if (pmime != null) {
                    if (common_parent == null)
                        common_parent = pmime;
                    else {
                        if (pmime != common_parent) {
                            common_parent = null;
                            break;
                        }
                    }
                }else {
                    common_parent = null;
                    break;
                }
            }
            
            foreach (var native in natives) {
                if (!native) {
                    all_native = false;
                    break;
                }
            }
        }
        
        private bool are_locations_match_conditional_mime (GLib.HashTable<string,string>[] locations, string mime) 
        {
            foreach (var location in locations) {
                if (location.lookup ("mimetype") == mime)
                    return false;
            }

            return true;
        }
        
        private bool is_mime_match_conditional_eaction_mimes (string[] mime_types, string mime) 
        {
            foreach (var umime in mime_types) {
                if (umime == mime)
                    return false;
            }

            return true;
        }
        
        private bool are_locations_mimes_match_strict_conditional_eaction_mimes (string[] mime_types, GLib.HashTable<string,string>[] locations) 
        {
            foreach (var location in locations) {
                var mime = location.lookup ("mimetype");
                foreach (var umime in mime_types) 
                {
                    if (umime == mime)
                        return false;
                }
            }

            return true; 
        }
        
        /* at least one arg should respect the condition */
        private bool are_locations_mimes_match_conditional_eaction_mimes (string[] mime_types, GLib.HashTable<string,string>[] locations) 
        {
            uint mlength = mime_types.length;
            foreach (var location in locations) {
                var mime = location.lookup ("mimetype");
                var count = 0;
                foreach (var umime in mime_types) 
                {
                    if (umime != mime)
                        count++;
                }
                if (count == mlength)
                    return true;
            }

            return false;
        }
        
        private bool is_eaction_in_filtered (string eaction_name) {
            foreach (var entry in filtered) {
                if (entry.lookup ("Name") == eaction_name)
                    return true;
            }

            return false;
        }

        private GLib.HashTable<string,string> get_filtered_entry_for_list (EActionFileInfo entry)
        {
            GLib.HashTable<string,string> filtered_entry;
            
            filtered_entry = new GLib.HashTable<string,string> (str_hash, str_equal);
            filtered_entry.insert ("Name", entry.name);
            filtered_entry.insert ("Description", entry.description);
            filtered_entry.insert ("Exec", entry.exec.printf (cmd_uris));
            filtered_entry.insert ("IconName", entry.icon_name);

            return filtered_entry;
        }

        private GLib.HashTable<string,string> get_filtered_entry (EActionFileInfo entry, File file)
        {
            GLib.HashTable<string,string> filtered_entry;
            
            filtered_entry = new GLib.HashTable<string,string> (str_hash, str_equal);
            filtered_entry.insert ("Name", entry.name);
            filtered_entry.insert ("Description", entry.description);
            filtered_entry.insert ("Exec", get_exec_from_entry (entry, file));
            filtered_entry.insert ("IconName", entry.icon_name);

            return filtered_entry;
        }

        private GLib.HashTable<string,string> get_filtered_entry_for_string (EActionFileInfo entry)
        {
            GLib.HashTable<string,string> filtered_entry;
            
            filtered_entry = new GLib.HashTable<string,string> (str_hash, str_equal);
            filtered_entry.insert ("Name", entry.name);
            filtered_entry.insert ("Description", entry.description);
            filtered_entry.insert ("Exec", entry.exec_string);
            filtered_entry.insert ("IconName", entry.icon_name);

            return filtered_entry;
        }

        private string get_exec_from_entry (EActionFileInfo cfi, File file)
        {
            if (cfi.take_uri_args)
                return (cfi.exec.printf (file.get_uri ()));
            else
                return (cfi.exec.printf (file.get_path ()));
        }

        private string get_parent_mime (string mimetype)
        {
            string parentmime = null;
            var arr = mimetype.split ("/", 2);
            if (arr.length == 2)
                parentmime = arr[0];

            return parentmime;
        }

        private string query_content_type (File file)
        {
            string mimetype = null;

            try {
                var file_info = file.query_info ("standard::content-type", FileQueryInfoFlags.NONE);
                mimetype = file_info.get_content_type ();
            } catch (Error e) {
                warning ("file query_info error %s: %s\n", file.get_uri (), e.message);
            }

            return mimetype;
        }
    }

    public class EActionFileInfo: Object
    {
        public string name { get; construct set; }
        public string desktop_id { get; set; }
        public string exec { get; set; }
        public string exec_string { get; set; }
        public string description { get; set; }
        public string[] mime_types = null;
        public string conditional_mime;
        public string icon_name { get; construct set; default = ""; }
        public bool take_multi_args { get; set; }
        public bool take_uri_args { get; set; }

        public string filename { get; construct set; }
        public bool is_valid { get; private set; default = true; }
        public bool is_conditional { get; private set; default = false; }
        /* used in the context of multiples arguments. If true, all arguments should respect the condition. If false, at least one argument should respect it. Default true */
        public bool strict_condition { get; private set; default = true; }

        private static const string GROUP = "Extended Action Entry";

        public EActionFileInfo.for_keyfile (string path, KeyFile keyfile)
        {
            Object (filename: path);

            init_from_keyfile (keyfile);
        }

        private void init_from_keyfile (KeyFile keyfile)
        {
            try
            {
                name = keyfile.get_locale_string (GROUP, "Name");
                exec = keyfile.get_string (GROUP, "Exec");
                description = keyfile.get_locale_string (GROUP, "Description");
                conditional_mime = keyfile.get_string (GROUP, "MimeType");
                if (conditional_mime.contains ("!")) {
                    is_conditional = true;
                    strict_condition = keyfile.get_boolean (GROUP, "StrictCondition");
                    if (conditional_mime.contains (";"))
                        warning ("%s: multi arguments in conditional mimetype are not currently supported: %s", name, conditional_mime);
                } else {
                    mime_types = keyfile.get_string_list (GROUP, "MimeType");
                }

                if (keyfile.has_key (GROUP, "Icon"))
                {
                    icon_name = keyfile.get_locale_string (GROUP, "Icon");
                    if (!Path.is_absolute (icon_name) &&
                        (icon_name.has_suffix (".png") ||
                         icon_name.has_suffix (".svg") ||
                         icon_name.has_suffix (".xpm")))
                    {
                        icon_name = icon_name.substring (0, icon_name.length - 4);
                    }
                }
                
                if (keyfile.has_key (GROUP, "ExecString"))
                    exec_string = keyfile.get_string (GROUP, "ExecString");
                
                if (keyfile.has_key (GROUP, "DesktopID")) {
                    desktop_id = keyfile.get_string (GROUP, "DesktopID");

                    /* check if the DesktopID is valid */
                    AppInfo app_mail = (AppInfo) new DesktopAppInfo (desktop_id + ".desktop");
                    if (app_mail == null) {
                        warning ("no desktop file installed for : %s", desktop_id);
                        is_valid = false;
                    }
                }
            }
            catch (Error err)
            {
                warning ("cannot init keyfile: %s", err.message);
                is_valid = false;
            }
        }
       
        /* Filter some default app to avoid having multiple Send by Email for example 
        The default apps are defined via gnome-control-center with some specific type */
        public bool filter_default_app () {
            if (name == "Mail" && desktop_id != null) {
                AppInfo app_mail_default = AppInfo.get_default_for_type("x-scheme-handler/mailto", false);
                AppInfo app_mail = (AppInfo) new DesktopAppInfo (desktop_id + ".desktop");
                //message (">>mail %s %s", app_mail_default.get_name (), app_mail.get_name ());
                if (app_mail_default.get_name () != app_mail.get_name ())
                    return false;
            }

            return true;
        }

    }

/* 
 * EActionFileService heavily inspired from Synapse DesktopFileService.
 * Kudos to the Synapse's developpers ! 
 */

    public class EActionFileService : Object
    {
        //private static unowned EActionFileService? cfservice;
        private File directory;
        private FileMonitor monitor = null;
        private File local_dir;
        private FileMonitor local_monitor = null;

        private Gee.Set<string> filenames_cache;

        private Gee.List<EActionFileInfo> all_eaction_files;
        public Gee.List<EActionFileInfo> conditional_eactions;

        private Gee.Map<unowned string, Gee.List<EActionFileInfo> > mimetype_map;
        private Gee.Map<string, Gee.List<EActionFileInfo> > exec_map;
        private Gee.Map<string, EActionFileInfo> eaction_id_map;

        public Gee.Map<string, EActionFileInfo> name_id_map;
        public Gee.Map<string, EActionFileInfo> exec_string_map;
        
        public bool initialized { get; private set; default = false; }

        public signal void initialization_done ();

        public EActionFileService ()
        {
            all_eaction_files = new Gee.ArrayList<EActionFileInfo> ();
            conditional_eactions = new Gee.ArrayList<EActionFileInfo> ();
            initialize ();
        }

        private async void initialize ()
        {
            yield load_all_eaction_files ();
            initialized = true;
            initialization_done ();
        }

        private async void load_all_eaction_files (bool should_monitor=true)
        {
            filenames_cache = new Gee.HashSet<string> ();

            local_dir = File.new_for_path (Environment.get_home_dir () + "/.local/share/extended-actions/");
            yield process_directory (local_dir);

            directory = File.new_for_path (Build.PREFIX + "/share/extended-actions/");
            yield process_directory (directory);

            if (should_monitor) {
                try {
                    monitor = directory.monitor_directory (0);
                    local_monitor = local_dir.monitor_directory (0);
                } catch (IOError e) {
                    error ("directory monitor failed: %s", e.message);
                }
                monitor.changed.connect (eaction_file_directory_changed);
                if (local_monitor != null)
                    local_monitor.changed.connect (eaction_file_directory_changed);
            }
            
            create_maps ();

        }

        private async void process_directory (File directory)
        {
            try {
                /*bool exists = yield Utils.query_exists_async (directory);
                  if (!exists) return;*/
                var enumerator = yield directory.enumerate_children_async (FILE_ATTRIBUTE_STANDARD_NAME + "," + FILE_ATTRIBUTE_STANDARD_TYPE, 0, 0);
                var files = yield enumerator.next_files_async (1024, 0);
                foreach (var f in files)
                {
                    unowned string name = f.get_name ();
                    if (f.get_file_type () == FileType.REGULAR && name.has_suffix (".desktop")
                        && !filenames_cache.contains (name))
                    {
                        yield load_eaction_file (directory.get_child (name));
                        filenames_cache.add (name);
                        message ("found: %s", name);
                    }
                }
            } catch (Error err) {
                warning ("%s", err.message);
            }
        }

        private async void load_eaction_file (File file)
        {
            try {
                uint8[] contents;
                bool success = yield file.load_contents_async (null, out contents, null);
                var contents_str = (string) contents;
                size_t len = contents_str.length;
                
                if (success && len>0)
                {
                    var keyfile = new KeyFile ();
                    keyfile.load_from_data (contents_str, len, 0);
                    var cfi = new EActionFileInfo.for_keyfile (file.get_path (), keyfile);
                    if (cfi.is_valid) {
                        all_eaction_files.add (cfi);
                        if (cfi.is_conditional)
                            conditional_eactions.add (cfi);
                    }
                }
            } catch (Error err) {
                warning ("%s", err.message);
            }
        }

        private void create_maps ()
        {
            // create mimetype maps
            mimetype_map =
                new Gee.HashMap<unowned string, Gee.List<EActionFileInfo> > ();
            // and exec map
            exec_map =
                new Gee.HashMap<string, Gee.List<EActionFileInfo> > ();
            // and exec string map
            exec_string_map =
                new Gee.HashMap<string, EActionFileInfo> ();
            // and desktop id map
            eaction_id_map =
                new Gee.HashMap<string, EActionFileInfo> ();
            // and name id map
            name_id_map =
                new Gee.HashMap<string, EActionFileInfo> ();

            Regex exec_re;
            try {
                exec_re = new Regex ("%[fFuU]");
            } catch (Error err) {
                critical ("%s", err.message);
                return;
            }

            foreach (var cfi in all_eaction_files)
            {
                //message ("create_map %s", cfi.name);
                string exec = "";

                string[] parameter = null;
                MatchInfo info = null;

                try {
                    if (exec_re.match (cfi.exec, 0, out info)) {
                        parameter = info.fetch_all();
                        if (parameter.length != 1) {
                            warning ("argument definition eroned in %s", cfi.name);
                        } else {
                            var argt = parameter[0];
                            if (argt == "%u" || argt == "%f")
                                cfi.take_multi_args = false;
                            else
                                cfi.take_multi_args = true;
                            if (argt == "%u" || argt == "%U")
                                cfi.take_uri_args = true;
                            else
                                cfi.take_uri_args = false;
                            //cfi.args = parameter[0];
                        }
                    }
                    exec = exec_re.replace_literal (cfi.exec, -1, 0, "%s");
                    //message ("exec: %s", exec);
                } catch (RegexError err) {
                    error ("%s", err.message);
                }
                exec = exec.strip ();
                cfi.exec = exec;
                // update exec map
                Gee.List<EActionFileInfo>? exec_list = exec_map[exec];
                if (exec_list == null)
                {
                    exec_list = new Gee.ArrayList<EActionFileInfo> ();
                    exec_map[exec] = exec_list;
                }
                exec_list.add (cfi);

                // update exec sting map
                if (cfi.exec_string != null)
                    exec_string_map [Path.get_basename (cfi.filename)] = cfi;

                // update eaction id map
                eaction_id_map[Path.get_basename (cfi.filename)] = cfi;

                // update name id map
                name_id_map[cfi.name] = cfi;

                // update mimetype map
                if (cfi.mime_types == null) continue;

                foreach (unowned string mime_type in cfi.mime_types)
                {
                    Gee.List<EActionFileInfo>? list = mimetype_map[mime_type];
                    if (list == null)
                    {
                        list = new Gee.ArrayList<EActionFileInfo> ();
                        mimetype_map[mime_type] = list;
                    }
                    list.add (cfi);
                }
            }
        }

        private uint timer_id = 0;

        private void eaction_file_directory_changed (File file, File? other_file, FileMonitorEvent event)
        {
            //message ("file_directory_changed");
            if (timer_id != 0)
            {
                Source.remove (timer_id);
            }

            timer_id = Timeout.add (1000, () =>
            {
                timer_id = 0;
                reload_eaction_files ();
                return false;
            });
        }

        private async void reload_eaction_files ()
        {
            debug ("Reloading eaction files...");
            all_eaction_files.clear ();
            conditional_eactions.clear ();
            filenames_cache.clear ();
            yield load_all_eaction_files (false);
        }

        private void add_cfi_for_mime (string mime, Gee.Set<EActionFileInfo> ret)
        {
            var cfis = mimetype_map[mime];
            if (cfis != null) ret.add_all (cfis);
        }

        public Gee.List<EActionFileInfo> get_eaction_files_for_type (string mime_type)
        {
            var cfi_set = new Gee.HashSet<EActionFileInfo> ();
            add_cfi_for_mime (mime_type, cfi_set);
            var ret = new Gee.ArrayList<EActionFileInfo> ();
            ret.add_all (cfi_set);
            return ret;
        }

        /*public void get_eactions_for_string ()
        {
            GLib.HashTable<string,string>[] filtered = null;
            //GLib.HashTable<string,string>[] filtered = null;
            //foreach (EActionFileInfo cfi in exec_string_map)
            foreach (var cfi in exec_string_map.values) {
                filtered += get_filtered_entry_for_string (cfi);
                message ("exec_string %s", cfi.name);
            }
        }*/
    }


    void on_bus_aquired (DBusConnection conn) {
        try {
            conn.register_object ("/org/magma/ExtendedActions", new ExtendedActions ());
        } catch (IOError e) {
            error ("Could not register service\n");
        }
    }

    public static int main (string[] args) {
        //var app = new ExtendedActions ();
        //app.run (args);
        Bus.own_name (BusType.SESSION, "org.magma.ExtendedActions", BusNameOwnerFlags.NONE,
                      on_bus_aquired,
                      () => {},
                      () => error ("Could not aquire name\n"));

        new MainLoop ().run ();

        return 0;
    }

}
