#ifndef TABLEMODIFIER_H
#define TABLEMODIFIER_H

#include "db/db.h"
#include "parser/ast/sqlitecreatetable.h"
#include "parser/ast/sqliteupdate.h"
#include "parser/ast/sqliteinsert.h"
#include "parser/ast/sqlitedelete.h"
#include "parser/ast/sqlitecreateindex.h"
#include "parser/ast/sqlitecreatetrigger.h"
#include "parser/ast/sqlitecreateview.h"

class API_EXPORT TableModifier
{
    public:
        TableModifier(Db* db, const QString& table);
        TableModifier(Db* db, const QString& database, const QString& table);

        void alterTable(SqliteCreateTablePtr newCreateTable);

        QStringList generateSqls() const;
        bool isValid() const;
        QStringList getErrors() const;
        QStringList getWarnings() const;
        QStringList getModifiedTables() const;
        QStringList getModifiedIndexes() const;
        QStringList getModifiedTriggers() const;
        QStringList getModifiedViews() const;
        bool hasMessages() const;

    private:
        void init();
        void parseDdl();
        QString getTempTableName();
        void copyDataTo(const QString& targetTable, const QStringList& srcCols, const QStringList& dstCols);
        void renameTo(const QString& newName);
        QString renameToTemp();
        void copyDataTo(const QString& table);
        void copyDataTo(SqliteCreateTablePtr newCreateTable);

        void handleIndexes();
        void handleIndex(SqliteCreateIndexPtr index);
        void handleTriggers();
        void handleTrigger(SqliteCreateTriggerPtr trigger);
        void handleTriggerQueries(SqliteCreateTriggerPtr trigger);
        void handleViews();
        void handleView(SqliteCreateViewPtr view);
        SqliteQuery* handleTriggerQuery(SqliteQuery* query, const QString& trigName, const QString& trigTable);
        SqliteSelect* handleSelect(SqliteSelect* select, const QString& trigTable = QString());
        SqliteUpdate* handleTriggerUpdate(SqliteUpdate* update, const QString& trigName, const QString& trigTable);
        SqliteInsert* handleTriggerInsert(SqliteInsert* insert, const QString& trigName, const QString& trigTable);
        SqliteDelete* handleTriggerDelete(SqliteDelete* del, const QString& trigName, const QString& trigTable);
        bool handleSubSelects(SqliteStatement* stmt, const QString& trigTable);
        bool handleExprWithSelect(SqliteExpr* expr, const QString& trigTable);
        bool handleAllExprWithTrigTable(SqliteStatement* stmt, const QString& contextTable);
        bool handleExprListWithTrigTable(const QList<SqliteExpr*>& exprList);
        bool handleExprWithTrigTable(SqliteExpr* expr);
        void simpleHandleIndexes();
        void simpleHandleTriggers(const QString& view = QString::null);
        SqliteQueryPtr parseQuery(const QString& ddl);

        /**
         * @brief alterTableHandleFks
         * @param newCreateTable
         * Finds all tables referencing currently modified table and updates their referenced table name and columns.
         */
        void handleFks(const QString& tempTableName);
        void subHandleFks(const QString& oldName, const QString& oldTempName);
        bool subHandleFks(SqliteForeignKey* fk, const QString& oldName, const QString& oldTempName);

        bool handleName(const QString& oldName, QString& valueToUpdate);
        bool handleIndexedColumns(QList<SqliteIndexedColumn*>& columnsToUpdate);
        bool handleColumnNames(QStringList& columnsToUpdate);
        bool handleColumnTokens(TokenList& columnsToUpdate);
        bool handleUpdateColumns(SqliteUpdate* update);

        Db* db = nullptr;
        Dialect dialect;

        /**
         * @brief Database name. The "main" is default.
         * Other databases (temp, attached...) are not supported at the moment.
         */
        QString database;

        /**
         * @brief Current table name (after renaming)
         */
        QString table;

        /**
         * @brief Initial table name, before any renaming.
         */
        QString originalTable;

        /**
         * @brief createTable Original DDL.
         */
        SqliteCreateTablePtr createTable;

        /**
         * @brief Statements to be executed to make changes real.
         */
        QStringList sqls;

        QStringList warnings;
        QStringList errors;

        QString newName;
        QStringList existingColumns;
        QHash<QString, QString> tableColMap;
        QHash<QString, QString> triggerNameToDdlMap;
        QStringList modifiedTables;
        QStringList modifiedIndexes;
        QStringList modifiedTriggers;
        QStringList modifiedViews;
        QStringList usedTempTableNames;
};

#endif // TABLEMODIFIER_H
