/*
 * Copyright (C) 2011 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#include "scheduled_recording_manager.h"
#include "../common/exception.h"
#include "../common/i18n.h"
#include "me-tv-server.h"

void ScheduledRecordingManager::initialise()
{
	g_static_rec_mutex_init(mutex.gobj());
}

void ScheduledRecordingManager::load(Data::Schema& schema, Data::Connection& connection)
{
	Glib::RecMutex::Lock lock(mutex);
	
	g_debug("Loading scheduled recordings");
	
	Data::Table table = schema.tables["scheduled_recording"];
	Data::TableAdapter adapter(connection, table);
	
	Glib::ustring where = Glib::ustring::compose(
		"((start_time + duration) > %1 OR recurring_type != 0)", time(NULL));
	Data::DataTable data_table = adapter.select_rows(where, "start_time");

	dirty = false;
	
	scheduled_recordings.clear();
	for (Data::Rows::iterator i = data_table.rows.begin(); i != data_table.rows.end(); i++)
	{
		ScheduledRecording scheduled_recording;
		Data::Row& row = *i;
		
		scheduled_recording.id					= row["scheduled_recording_id"].int_value;
		scheduled_recording.channel_id				= row["channel_id"].int_value;
		scheduled_recording.description				= row["description"].string_value;
		scheduled_recording.recurring_type			= row["recurring_type"].int_value;
		scheduled_recording.action_after			= row["action_after"].int_value;
		scheduled_recording.start_time				= row["start_time"].int_value;
		scheduled_recording.duration				= row["duration"].int_value;
		scheduled_recording.device					= row["device"].string_value;

		scheduled_recordings.push_back(scheduled_recording);

		guint now = time(NULL);
		if (scheduled_recording.start_time + scheduled_recording.duration < now)
		{
			dirty = true;
		}
	}

	g_debug("Scheduled recordings loaded");
}

void ScheduledRecordingManager::save(Data::Schema& schema, Data::Connection& connection)
{
	if (!dirty)
	{
		g_debug("Scheduled recordings are not dirty, not saving");
		return;
	}
	g_debug("Scheduled recordings are dirty, saving");
	
	Glib::RecMutex::Lock lock(mutex);

	g_debug("Saving %d scheduled recordings", (int)scheduled_recordings.size());
	
	Data::Table table = schema.tables["scheduled_recording"];	
	Data::DataTable data_table(table);
	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
	{
		ScheduledRecording& scheduled_recording = *i;
		time_t now = time(NULL);
		if (scheduled_recording.get_end_time() > now || scheduled_recording.recurring_type != 0)
		{
			Data::Row row;
			row.auto_increment 						= &(scheduled_recording.id);
			row["scheduled_recording_id"].int_value	= scheduled_recording.id;
			row["channel_id"].int_value				= scheduled_recording.channel_id;
			row["description"].string_value			= scheduled_recording.description;
			row["recurring_type"].int_value			= scheduled_recording.recurring_type;
			row["action_after"].int_value			= scheduled_recording.action_after;
			row["start_time"].int_value				= scheduled_recording.start_time;
			row["duration"].int_value				= scheduled_recording.duration;
			row["device"].string_value				= scheduled_recording.device;

			data_table.rows.add(row);
		}

		g_debug("Scheduled recording '%s' (%d) saved", scheduled_recording.description.c_str(), scheduled_recording.id);
	}
	
	Data::TableAdapter adapter(connection, table);
	adapter.replace_rows(data_table);

	guint now = time(NULL);
	g_debug("Deleting old scheduled recordings ending before %d", now);
	Glib::ustring where = Glib::ustring::compose("recurring_type != %1 AND (start_time + duration) < %2", SCHEDULED_RECORDING_RECURRING_TYPE_ONCE, now);
	data_table = adapter.select_rows(where, "start_time");

	gboolean updated = false;
	for (Data::Rows::iterator i = data_table.rows.begin(); i != data_table.rows.end(); i++)
	{
		Data::Row& row = *i;
		g_debug("ScheduledRecordingManager::save/clear ID: %d", row["scheduled_recording_id"].int_value); 

		if(row["recurring_type"].int_value == SCHEDULED_RECORDING_RECURRING_TYPE_EVERYDAY)
		{
			row["start_time"].int_value += 86400;
		}
		else if(row["recurring_type"].int_value == SCHEDULED_RECORDING_RECURRING_TYPE_EVERYWEEK)
		{
			row["start_time"].int_value += 604800;
		}
		else if(row["recurring_type"].int_value == SCHEDULED_RECORDING_RECURRING_TYPE_EVERYWEEKDAY)
		{
			time_t tim = row["start_time"].int_value;
			struct tm *ts;
			char buf[80];
			ts = localtime(&tim);
			strftime(buf, sizeof(buf), "%w", ts);
			switch(atoi(buf))
			{
				case 5 : row["start_time"].int_value += 259200; break;
				case 6 : row["start_time"].int_value += 172800; break;
				default: row["start_time"].int_value +=  86400; break;
			}
		}  
		updated = true;
	}
	if(updated)
	{
		adapter.replace_rows(data_table);
		load(schema, connection);
	}
	Glib::ustring clause = Glib::ustring::compose("(start_time + duration) < %1 AND recurring_type = %2", now, SCHEDULED_RECORDING_RECURRING_TYPE_ONCE);
	adapter.delete_rows(clause);

	dirty = false;

	g_debug("Scheduled recordings saved");
}

gboolean ScheduledRecordingManager::is_device_available(const Glib::ustring& device, const ScheduledRecording& scheduled_recording)
{
	Channel& channel = channel_manager.get_channel_by_id(scheduled_recording.channel_id);

	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
	{
		ScheduledRecording& current = *i;

		Channel& current_channel = channel_manager.get_channel_by_id(current.channel_id);

		if (
			current.id != scheduled_recording.id &&
			channel.transponder != current_channel.transponder &&
			scheduled_recording.overlaps(current) &&
			device == current.device
		    )
		{
			g_debug("Frontend '%s' is busy recording '%s'", device.c_str(), scheduled_recording.description.c_str());
			return false;
		}
	}

	g_debug("Found available frontend '%s'", device.c_str());

	return true;
}

void ScheduledRecordingManager::select_device(ScheduledRecording& scheduled_recording)
{
	g_debug("Looking for an available device for scheduled recording");
	
	Channel& channel = channel_manager.get_channel_by_id(scheduled_recording.channel_id);

	FrontendList& frontends = device_manager.get_frontends();
	for (FrontendList::iterator j = frontends.begin(); j != frontends.end(); j++)
	{
		Dvb::Frontend* device = (*j);
		const Glib::ustring& device_path = device->get_path();

		if (device->get_frontend_type() != channel.transponder.frontend_type)
		{
			g_debug("Device %s is the wrong type", device_path.c_str());
		}
		else
		{
			if (is_device_available(device_path, scheduled_recording))
			{
				scheduled_recording.device = device_path;
				if (!stream_manager.is_broadcasting(device_path))
				{
					return;
				}
				g_debug("Device '%s' is currently broadcasting, looking for something better", device_path.c_str());
			}
		}
	}
}

void ScheduledRecordingManager::add_scheduled_recording(Data::Schema& schema, Data::Connection& connection, EpgEvent& epg_event)
{
	ScheduledRecording scheduled_recording;
	Channel& channel = channel_manager.get_channel_by_id(epg_event.channel_id);
	
	int before = channel.record_extra_before;
	int after = channel.record_extra_after;
	
	scheduled_recording.channel_id		= epg_event.channel_id;
	scheduled_recording.description		= epg_event.get_title();
	scheduled_recording.recurring_type	= SCHEDULED_RECORDING_RECURRING_TYPE_ONCE;
	scheduled_recording.action_after	= SCHEDULED_RECORDING_ACTION_AFTER_NONE;
	scheduled_recording.start_time		= epg_event.start_time - (before * 60);
 	scheduled_recording.duration		= epg_event.duration + ((before + after) * 60);	
	scheduled_recording.device			= "";

	add_scheduled_recording(schema, connection, scheduled_recording);
}

void ScheduledRecordingManager::add_scheduled_recording(Data::Schema& schema, Data::Connection& connection, ScheduledRecording& scheduled_recording)
{
	Glib::RecMutex::Lock lock(mutex);
	ScheduledRecordingList::iterator iupdated;
	gboolean updated  = false;
	gboolean is_same  = false;
	gboolean conflict = false;

	g_debug("Setting scheduled recording");
	
	Channel& channel = channel_manager.get_channel_by_id(scheduled_recording.channel_id);

	if (scheduled_recording.device.empty())
	{
		select_device(scheduled_recording);

		if (scheduled_recording.device.empty())
		{
			Glib::ustring message = Glib::ustring::compose(_(
				"Failed to set scheduled recording for '%1' at %2: There are no devices available at that time"),
				scheduled_recording.description, scheduled_recording.get_start_time_text());
			throw Exception(message);
		}

		g_debug("Device selected: '%s'", scheduled_recording.device.c_str());
	}

	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
	{
		ScheduledRecording& current = *i;

		Channel& current_channel = channel_manager.get_channel_by_id(current.channel_id);

		// Check for conflict
		if (current.id != scheduled_recording.id &&
		    current_channel.transponder != channel.transponder &&
		    scheduled_recording.device == current.device &&
		    scheduled_recording.overlaps(current))
		{
			conflict = true;
			Glib::ustring message =  Glib::ustring::compose(
				_("Failed to save scheduled recording because it conflicts with another scheduled recording called '%1'."),
				current.description);
			throw Exception(message);
		}

		// Check if it's an existing scheduled recording
		if (scheduled_recording.id != 0 &&
		    scheduled_recording.id == current.id)
		{
			g_debug("Updating scheduled recording");
			updated = true;
			iupdated = i;
		}

		// Check if we are scheduling the same program
		if (scheduled_recording.id == 0 &&
		    current.channel_id 		== scheduled_recording.channel_id &&
		    current.start_time 		== scheduled_recording.start_time &&
		    current.duration 		== scheduled_recording.duration)
		{
			Glib::ustring message =  Glib::ustring::compose(
				_("Failed to save scheduled recording because you have already have a scheduled recording called '%1' which is scheduled for the same time on the same channel."),
				current.description);
			throw Exception(message);
		}

		if (current.id == scheduled_recording.id &&
			current.recurring_type	== scheduled_recording.recurring_type &&
			current.action_after	== scheduled_recording.action_after &&
			current.channel_id 		== scheduled_recording.channel_id &&
			current.start_time 		== scheduled_recording.start_time &&
			current.duration 		== scheduled_recording.duration)
		{
			is_same = true;
		}
	}
	
	// If there is an update an not conflict on scheduled recording, update it.
	if (updated && !conflict && !is_same)
	{
		ScheduledRecording& current = *iupdated;

		current.device = scheduled_recording.device;
		current.recurring_type = scheduled_recording.recurring_type;
		current.action_after = scheduled_recording.action_after;
		current.description = scheduled_recording.description;
		current.channel_id = scheduled_recording.channel_id;
		current.start_time = scheduled_recording.start_time;
		current.duration = scheduled_recording.duration;
		dirty = true;
	}
	
	// If the scheduled recording is new then add it
	if (scheduled_recording.id == 0)
	{
		g_debug("Adding scheduled recording");
		scheduled_recordings.push_back(scheduled_recording);
		dirty = true;
	}

	// Have to save to update the scheduled recording ID
	save(schema, connection);
}

void ScheduledRecordingManager::remove_scheduled_recording(Data::Schema& schema, Data::Connection& connection, guint scheduled_recording_id)
{
	Glib::RecMutex::Lock lock(mutex);

	g_debug("Deleting scheduled recording %d", scheduled_recording_id);
	
	gboolean found = false;
	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end() && !found; i++)
	{
		ScheduledRecording& scheduled_recording = *i;
		if (scheduled_recording_id == scheduled_recording.id)
		{
			g_debug("Deleting scheduled recording '%s' (%d)",
				scheduled_recording.description.c_str(),
				scheduled_recording.id);
			scheduled_recordings.erase(i);

			Data::Table table = schema.tables["scheduled_recording"];
			Data::TableAdapter adapter(connection, table);
			adapter.delete_row(scheduled_recording_id);
			
			found = true;
			dirty = true;
		}
	}	
	g_debug("Scheduled recording deleted");
}

void ScheduledRecordingManager::remove_scheduled_recording(Data::Schema& schema, Data::Connection& connection, Channel& channel)
{
	Glib::RecMutex::Lock lock(mutex);

	ScheduledRecordingList::iterator i = scheduled_recordings.begin();
	while (i != scheduled_recordings.end())
	{
		ScheduledRecording& scheduled_recording = *i;
		if (scheduled_recording.channel_id == channel.id)
		{
			remove_scheduled_recording(schema, connection, scheduled_recording.id);
			i = scheduled_recordings.begin();
		}
		else
		{
			i++;
		}
	}
}

void ScheduledRecordingManager::check_scheduled_recordings(Data::Schema& schema, Data::Connection& connection)
{
	ScheduledRecordingList active_scheduled_recordings;
	
	time_t now = time(NULL);

	g_debug("Checking scheduled recordings");
	Glib::RecMutex::Lock lock(mutex);

	g_debug("Now: %u", (guint)now);
	g_debug("Removing scheduled recordings older than %u", (guint)now);
	
	ScheduledRecordingList::iterator i = scheduled_recordings.begin();
	while (i != scheduled_recordings.end())
	{
		if ((*i).is_old(now))
		{
			remove_scheduled_recording(schema, connection, (*i).id);
			i = scheduled_recordings.begin();
		}
		else
		{
			i++;
		}
	}
	
	if (!scheduled_recordings.empty())
	{
		g_debug("=============================================================================================");
		g_debug("#ID | Start Time | Duration | Record | Channel    | Device                      | Description");
		g_debug("=============================================================================================");
		for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
		{			
			ScheduledRecording& scheduled_recording = *i;
				
			gboolean record = scheduled_recording.contains(now);
			g_debug("%3d | %u | %8d | %s | %10s | %27s | %s",
				scheduled_recording.id,
				(guint)scheduled_recording.start_time,
				scheduled_recording.duration,
				record ? "true  " : "false ",
				channel_manager.get_channel_by_id(scheduled_recording.channel_id).name.c_str(),
				scheduled_recording.device.c_str(),
				scheduled_recording.description.c_str());
			
			if (record)
			{
				active_scheduled_recordings.push_back(scheduled_recording);
			}
			
			if (scheduled_recording.get_end_time() < now)
			{
				dirty = true;
			}
		}
	}
		
	for (ScheduledRecordingList::iterator i = active_scheduled_recordings.begin(); i != active_scheduled_recordings.end(); i++)
	{
		const ScheduledRecording& scheduled_recording = *i;
		stream_manager.start_recording(scheduled_recording, remove_colon);
	}

	gboolean check = true;
	while (check)
	{
		check = false;

		FrontendThreadList& frontend_threads = stream_manager.get_frontend_threads();
		for (FrontendThreadList::iterator i = frontend_threads.begin(); i != frontend_threads.end(); i++)
		{
			FrontendThread& frontend_thread = **i;
			ChannelStreamList& streams = frontend_thread.get_streams();
			for (ChannelStreamList::iterator j = streams.begin(); j != streams.end(); j++)
			{
				ChannelStream& channel_stream = **j;
				guint scheduled_recording_id = is_recording(channel_stream.channel);
				if (channel_stream.type == CHANNEL_STREAM_TYPE_SCHEDULED_RECORDING && scheduled_recording_id == -1)
				{
					stream_manager.stop_recording(channel_stream.channel);
					check = true;
					break;
				}
			}
		}
	}
}

ScheduledRecording ScheduledRecordingManager::get_scheduled_recording(guint scheduled_recording_id)
{
	Glib::RecMutex::Lock lock(mutex);

	ScheduledRecording* result = NULL;
	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end() && result == NULL; i++)
	{
		ScheduledRecording& scheduled_recording = *i;
		if (scheduled_recording.id == scheduled_recording_id)
		{
			result = &scheduled_recording;			
		}
	}
	
	if (result == NULL)
	{
		Glib::ustring message = Glib::ustring::compose(
			_("Scheduled recording '%1' not found"), scheduled_recording_id);
		throw Exception(message);
	}
	
	return *result;
}

gint ScheduledRecordingManager::is_recording(const Channel& channel)
{
	Glib::RecMutex::Lock lock(mutex);

	guint now = time(NULL);
	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
	{			
		ScheduledRecording& scheduled_recording = *i;
		if (scheduled_recording.contains(now) && scheduled_recording.channel_id == channel.id)
		{
			return scheduled_recording.id;
		}
	}

	return -1;
}

gint ScheduledRecordingManager::is_recording(const EpgEvent& epg_event)
{
	Glib::RecMutex::Lock lock(mutex);

	for (ScheduledRecordingList::iterator i = scheduled_recordings.begin(); i != scheduled_recordings.end(); i++)
	{
		ScheduledRecording& scheduled_recording = *i;
		if (scheduled_recording.channel_id == epg_event.channel_id &&
			scheduled_recording.contains(epg_event.start_time, epg_event.get_end_time()))
		{
			return scheduled_recording.id;
		}
	}
	
	return -1;
}

void ScheduledRecordingManager::check_auto_recordings(Data::Schema& schema, Data::Connection& connection)
{
	g_debug("Checking auto recordings");

	ChannelList& channels = channel_manager.get_channels();
	StringList auto_record_list;

	Data::Table table = schema.tables["auto_record"];
	Data::TableAdapter adapter(connection, table);
	Data::DataTable data_table = adapter.select_rows("", "priority");

	for (Data::Rows::iterator i = data_table.rows.begin(); i != data_table.rows.end(); i++)
	{
		Data::Row row = *i;
		auto_record_list.push_back(row["title"].string_value);
	}

	g_debug("Searching for auto record EPG events");
	for (StringList::iterator iterator = auto_record_list.begin(); iterator != auto_record_list.end(); iterator++)
	{
		Glib::ustring title = (*iterator).uppercase();
		g_debug("Searching for '%s'", title.c_str());

		for (ChannelList::iterator i = channels.begin(); i != channels.end(); i++)
		{
			Channel& channel = *i;

			EpgEventList list = channel.epg_events.search(title, false);
			for (EpgEventList::iterator j = list.begin(); j != list.end(); j++)
			{
				EpgEvent& epg_event = *j;

				g_debug("Checking candidate: %s at %d", epg_event.get_title().c_str(), (int)epg_event.start_time);
				gint scheduled_recording_id = is_recording(epg_event);
				if (scheduled_recording_id == -1)
				{
					try
					{
						g_debug("Trying to auto record '%s' (%d)", epg_event.get_title().c_str(), epg_event.event_id);
						add_scheduled_recording(schema, connection, epg_event);
					}
					catch(...)
					{
						g_debug("If there's an error here then we need to know!");
						handle_error();
					}
				}
				else
				{
					g_debug("EPG event '%s' at %d is already being recorded", epg_event.get_title().c_str(), (int)epg_event.start_time);
				}
			}
		}
	}
}

