/*
 * Copyright (C) 2011 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif /* HAVE_CONFIG_H */

#include "me-tv-server.h"
#include "data.h"
#include "crc32.h"
#include "../common/i18n.h"
#include "../common/exception.h"
#include "../common/common.h"
#include <giomm/socketservice.h>
#include "request_handler.h"
#include <glib/gprintf.h>
#include <sys/types.h> 
#include <sys/socket.h>
#include <netinet/in.h>

#define CURRENT_DATABASE_VERSION	409

#define ME_TV_SUMMARY _("Me TV is a digital television viewer for GTK")
#define ME_TV_DESCRIPTION _("Me TV was developed for the modern digital lounge room with a PC for a media centre that is capable "\
	"of normal PC tasks (web surfing, word processing and watching TV).\n")

ChannelManager				channel_manager;
ScheduledRecordingManager	scheduled_recording_manager;
DeviceManager				device_manager;
StreamManager				stream_manager;
Data::Schema				schema;
Data::Connection			connection;
RequestHandler				request_handler;
ConfigurationManager			configuration_manager;

Glib::ustring					preferred_language;
Glib::StaticRecMutex			mutex;
bool			verbose_logging			= false;
bool			disable_epg_thread		= false;
Glib::ustring	devices;
Glib::ustring	text_encoding;
gboolean		ignore_teletext = true;
Glib::ustring	recording_directory;
gboolean		remove_colon = false;
int				read_timeout = 5000;
int				server_port = 1999;
Glib::RefPtr<Glib::MainLoop>		main_loop;

sigc::signal<void>					signal_update;
sigc::signal<void, Glib::ustring>	signal_error;

void log_handler(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data)
{
	if (log_level != G_LOG_LEVEL_DEBUG || verbose_logging)
	{
		Glib::ustring time_text = get_local_time_text("%F %T");
		g_printf("%s: %s\n", time_text.c_str(), message);
	}
}

guint convert_string_to_value(const StringTable* table, const Glib::ustring& text)
{
	gboolean found = false;
	const StringTable*	current = table;

	while (current->text != NULL && !found)
	{
		if (text == current->text)
		{
			found = true;
		}
		else
		{
			current++;
		}
	}
	
	if (!found)
	{
		throw Exception(Glib::ustring::compose(_("Failed to find a value for '%1'"), text));
	}
	
	return (guint)current->value;
}

Glib::ustring convert_value_to_string(const StringTable* table, guint value)
{
	gboolean found = false;
	const StringTable*	current = table;

	while (current->text != NULL && !found)
	{
		if (value == current->value)
		{
			found = true;
		}
		else
		{
			current++;
		}
	}
	
	if (!found)
	{
		throw Exception(Glib::ustring::compose(_("Failed to find a text value for '%1'"), value));
	}
	
	return current->text;
}

void split_string(StringArray& parts, const Glib::ustring& text, const char* delimiter, gboolean remove_empty, gsize max_length)
{
	gchar** temp_parts = g_strsplit_set(text.c_str(), delimiter, max_length);	
	gchar** iterator = temp_parts;
	guint count = 0;
	while (*iterator != NULL)
	{
		gchar* part = *iterator++;
		if (part[0] != 0)
		{
			count++;
			parts.push_back(part);
		}
	}
	g_strfreev(temp_parts);
}

gboolean initialise_database()
{
	gboolean result = false;
	
	Data::Table table_channel;
	table_channel.name = "channel";
	table_channel.columns.add("channel_id",				Data::DATA_TYPE_INTEGER, 0, false);
	table_channel.columns.add("name",					Data::DATA_TYPE_STRING, 50, false);
	table_channel.columns.add("type",					Data::DATA_TYPE_INTEGER, 0, false);
	table_channel.columns.add("sort_order",				Data::DATA_TYPE_INTEGER, 0, false);
	table_channel.columns.add("mrl",					Data::DATA_TYPE_STRING, 1024, true);
	table_channel.columns.add("service_id",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("frequency",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("inversion",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("bandwidth",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("code_rate_hp",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("code_rate_lp",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("constellation",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("transmission_mode",		Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("guard_interval",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("hierarchy_information",	Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("symbol_rate",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("fec_inner",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("modulation",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("polarisation",			Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("record_extra_before",	Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("record_extra_after",		Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.columns.add("sat_number",				Data::DATA_TYPE_INTEGER, 0, true);
	table_channel.primary_key = "channel_id";
	StringList table_channel_unique_columns;
	table_channel_unique_columns.push_back("name");
	table_channel.constraints.add_unique(table_channel_unique_columns);
	schema.tables.add(table_channel);
	
	Data::Table table_epg_event;
	table_epg_event.name = "epg_event";
	table_epg_event.columns.add("epg_event_id",		Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.columns.add("channel_id",		Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.columns.add("version_number",	Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.columns.add("event_id",			Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.columns.add("start_time",		Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.columns.add("duration",			Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event.primary_key = "epg_event_id";
	StringList table_epg_event_unique_columns;
	table_epg_event_unique_columns.push_back("channel_id");
	table_epg_event_unique_columns.push_back("event_id");
	table_epg_event.constraints.add_unique(table_epg_event_unique_columns);
	schema.tables.add(table_epg_event);
			
	Data::Table table_epg_event_text;
	table_epg_event_text.name = "epg_event_text";
	table_epg_event_text.columns.add("epg_event_text_id",	Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event_text.columns.add("epg_event_id",		Data::DATA_TYPE_INTEGER, 0, false);
	table_epg_event_text.columns.add("language",			Data::DATA_TYPE_STRING, 3, false);
	table_epg_event_text.columns.add("title",				Data::DATA_TYPE_STRING, 200, false);
	table_epg_event_text.columns.add("subtitle",			Data::DATA_TYPE_STRING, 200, false);
	table_epg_event_text.columns.add("description",			Data::DATA_TYPE_STRING, 1000, false);
	table_epg_event_text.primary_key = "epg_event_text_id";
	StringList table_epg_event_text_unique_columns;
	table_epg_event_text_unique_columns.push_back("epg_event_id");
	table_epg_event_text_unique_columns.push_back("language");
	table_epg_event_text.constraints.add_unique(table_epg_event_text_unique_columns);
	schema.tables.add(table_epg_event_text);

	Data::Table table_scheduled_recording;
	table_scheduled_recording.name = "scheduled_recording";
	table_scheduled_recording.columns.add("scheduled_recording_id",	Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("description",			Data::DATA_TYPE_STRING, 200, false);
	table_scheduled_recording.columns.add("recurring_type",			Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("action_after",			Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("channel_id",				Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("start_time",				Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("duration",				Data::DATA_TYPE_INTEGER, 0, false);
	table_scheduled_recording.columns.add("device",					Data::DATA_TYPE_STRING, 200, false);
	table_scheduled_recording.primary_key = "scheduled_recording_id";
	schema.tables.add(table_scheduled_recording);

	Data::Table table_auto_record;
	table_auto_record.name = "auto_record";
	table_auto_record.columns.add("auto_record_id",	Data::DATA_TYPE_INTEGER, 0, false);
	table_auto_record.columns.add("title",			Data::DATA_TYPE_STRING, 200, false);
	table_auto_record.columns.add("priority",		Data::DATA_TYPE_INTEGER, 0, false);
	table_auto_record.primary_key = "auto_record_id";
	StringList table_auto_record_unique_columns;
	table_auto_record_unique_columns.push_back("title");
	table_auto_record_unique_columns.push_back("priority");
	table_auto_record.constraints.add_unique(table_auto_record_unique_columns);
	schema.tables.add(table_auto_record);

	Data::Table table_configuration;
	table_configuration.name = "configuration";
	table_configuration.columns.add("configuration_id",	Data::DATA_TYPE_INTEGER, 0, false);
	table_configuration.columns.add("name",				Data::DATA_TYPE_STRING, 200, false);
	table_configuration.columns.add("value",			Data::DATA_TYPE_STRING, 1024, false);
	table_configuration.primary_key = "configuration_id";
	StringList table_configuration_unique_columns;
	table_configuration_unique_columns.push_back("name");
	table_configuration.constraints.add_unique(table_configuration_unique_columns);
	schema.tables.add(table_configuration);

	Data::Table table_version;
	table_version.name = "version";
	table_version.columns.add("value",	Data::DATA_TYPE_INTEGER, 0, false);
	schema.tables.add(table_version);
	
	Data::SchemaAdapter adapter(connection, schema);
	adapter.initialise_table(table_version);
	
	Data::TableAdapter adapter_version(connection, table_version);
	
	if (connection.get_database_created())
	{
		adapter.initialise_schema();
		
		Data::DataTable data_table_version(table_version);
		Data::Row row;
		row["value"].int_value = CURRENT_DATABASE_VERSION;
		data_table_version.rows.add(row);
		adapter_version.replace_rows(data_table_version);
		
		result = true;
	}
	else
	{
		guint database_version = 0;

		Data::DataTable data_table = adapter_version.select_rows();
		if (!data_table.rows.empty())
		{
			database_version = data_table.rows[0]["value"].int_value;
		}
		
		g_debug("Required Database version: %d", CURRENT_DATABASE_VERSION);
		g_debug("Actual Database version: %d", database_version);

		if (database_version != CURRENT_DATABASE_VERSION)
		{
			throw Exception(_("The Me TV database version does not match the Me TV server version.")); 
		}

		result = true;
	}
		
	return result;
}

gboolean on_timeout()
{
	static guint last_seconds = 60;

	try
	{
		guint now = time(NULL);
		guint seconds = now % 60;
		bool update = last_seconds > seconds;
		last_seconds = seconds;

		if (update)
		{
			if (channel_manager.is_dirty())
			{
				channel_manager.save(schema, connection);
			}

			request_handler.clients.check();
			scheduled_recording_manager.save(schema, connection);
			channel_manager.prune_epg();

			scheduled_recording_manager.check_auto_recordings(schema, connection);
			scheduled_recording_manager.check_scheduled_recordings(schema, connection);

			signal_update();
		}

	}
	catch(...)
	{
		handle_error();
	}
	
	return true;
}

void handle_error()
{
	try
	{
		throw;
	}
	catch (const Exception& exception)
	{
		g_debug("Signal error '%s'", exception.what().c_str());
		signal_error(exception.what());
	}
	catch (const Glib::Error& exception)
	{
		g_debug("Signal error '%s'", exception.what().c_str());
		signal_error(exception.what());
	}
	catch (...)
	{
		g_message("Unhandled exception");
		signal_error("Unhandled exception");
	}
}

void on_error(const Glib::ustring& message)
{
	g_message("Error: %s", message.c_str());
}

void make_directory_with_parents(const Glib::ustring& path)
{
        Glib::RefPtr<Gio::File> file = Gio::File::create_for_path(path);
        if (!file->query_exists())
        {
                Glib::RefPtr<Gio::File> parent = file->get_parent();
                if (!parent->query_exists())
                {
                        make_directory_with_parents(parent->get_path());
                }

                g_debug("Creating directory '%s'", path.c_str());
                file->make_directory();
        }
}

class NetworkServerThread : public Thread
{
private:
	int sockfd;

protected:
	void run()
	{
		while (!is_terminated())
		{
			struct sockaddr_in client_addr;
			socklen_t client_length = sizeof(client_addr);
			int newsockfd = accept(sockfd, (struct sockaddr *)&client_addr, &client_length);
			gboolean do_terminate = request_handler.handle_request(schema, connection, newsockfd);
			::close(newsockfd);

			if (do_terminate)
			{
				main_loop->quit();
				terminate();
			}
		}
	}

public:
	NetworkServerThread() : Thread("Network Server")
	{
		g_message("Starting socket service on port %d", server_port);

		sockfd = socket(AF_INET, SOCK_STREAM, 0);
		if (sockfd < 0)
		{
			throw SystemException("Failed to open socket");
		}

		struct sockaddr_in serv_addr;
		bzero((char *) &serv_addr, sizeof(serv_addr));
		serv_addr.sin_family = AF_INET;
		serv_addr.sin_addr.s_addr = INADDR_ANY;
		serv_addr.sin_port = htons(server_port);
		if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) 
		{
			throw SystemException("Failed to on bind");
		}

		listen(sockfd, 5);
	}
};

int main(int argc, char** argv)
{
	#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
	#endif
		
	gboolean database_initialised = false;

	try
	{
		Glib::init();

		signal_error.connect(sigc::ptr_fun(&on_error));

		g_log_set_handler(G_LOG_DOMAIN,
			(GLogLevelFlags)(G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION),
			log_handler, NULL);

		g_message("Me TV Server %s started", VERSION);

		if (!Glib::thread_supported())
		{
			Glib::thread_init();
		}

		Crc32::init();
		Gio::init();

		g_debug("sqlite3_threadsafe() = %d", sqlite3_threadsafe());
		if (sqlite3_threadsafe() == 0)
		{
			throw Exception(_("The SQLite version is not thread-safe"));
		}

		Glib::ustring broadcast_address = "127.0.0.1";

		Glib::OptionEntry verbose_option_entry;
		verbose_option_entry.set_long_name("verbose");
		verbose_option_entry.set_short_name('v');
		verbose_option_entry.set_description(_("Enable verbose messages"));

		Glib::OptionEntry disable_epg_thread_option_entry;
		disable_epg_thread_option_entry.set_long_name("disable-epg-thread");
		disable_epg_thread_option_entry.set_description(_("Disable the EPG thread.  Me TV will stop collecting EPG events."));

		Glib::OptionEntry devices_option_entry;
		devices_option_entry.set_long_name("devices");
		devices_option_entry.set_description(_("Only use the specified frontend devices (e.g. --devices=/dev/dvb/adapter0/frontend0,/dev/dvb/adapter0/frontend1)"));

		Glib::OptionEntry read_timeout_option_entry;
		read_timeout_option_entry.set_long_name("read-timeout");
		read_timeout_option_entry.set_description(_("How long to wait (in seconds) before timing out while waiting for data from demuxer (default 5)."));

		Glib::OptionEntry broadcast_address_option_entry;
		broadcast_address_option_entry.set_long_name("broadcast-address");
		broadcast_address_option_entry.set_description(_("The network broadcast address to send video streams for clients to display (default 127.0.0.1)."));

		Glib::OptionEntry server_port_option_entry;
		server_port_option_entry.set_long_name("server-port");
		server_port_option_entry.set_description(_("The network port for clients to connect to (default 1999)."));

		Glib::OptionGroup option_group(PACKAGE_NAME, "", _("Show Me TV Client help options"));
		option_group.add_entry(verbose_option_entry, verbose_logging);
		option_group.add_entry(disable_epg_thread_option_entry, disable_epg_thread);
		option_group.add_entry(devices_option_entry, devices);
		option_group.add_entry(read_timeout_option_entry, read_timeout);
		option_group.add_entry(broadcast_address_option_entry, broadcast_address);
		option_group.add_entry(server_port_option_entry, server_port);

		Glib::OptionContext option_context;
		option_context.set_summary(ME_TV_SUMMARY);
		option_context.set_description(ME_TV_DESCRIPTION);
		option_context.set_main_group(option_group);

		option_context.parse(argc, argv);

		g_static_rec_mutex_init(mutex.gobj());
			
		Glib::ustring data_directory = Glib::get_home_dir() + "/.local/share/me-tv";
		make_directory_with_parents (data_directory);
	
		Glib::ustring database_filename = Glib::build_filename(data_directory, "me-tv.db");
		connection.open(database_filename);

		if (!initialise_database())
		{
			throw Exception(_("Failed to initialise database"));
		}
		database_initialised = true;

		g_debug("Me TV database initialised");

		configuration_manager.load(schema, connection);
		configuration_manager.set_default_value("recording_directory", Glib::get_home_dir());
		configuration_manager.set_default_value("preferred_language", "");
		configuration_manager.set_default_value("remove_colon", "false");
		configuration_manager.save(schema, connection);

		recording_directory = configuration_manager.get_value("recording_directory");
		preferred_language = configuration_manager.get_value("preferred_language");
		remove_colon = configuration_manager.get_value("remove_colon") == "true";

		device_manager.initialise(devices);
		channel_manager.initialise();
		channel_manager.load(schema, connection);
		scheduled_recording_manager.initialise();
		stream_manager.initialise(text_encoding, read_timeout, ignore_teletext);
		stream_manager.start();

		ChannelList& channels = channel_manager.get_channels();	

		const FrontendList& frontends = device_manager.get_frontends();	
		if (!frontends.empty())
		{
			scheduled_recording_manager.load(schema, connection);
		}

		Glib::signal_timeout().connect_seconds(sigc::ptr_fun(&on_timeout), 1);

		request_handler.set_broadcast_address(broadcast_address);

		NetworkServerThread server_thread;
		server_thread.start();

		g_message("Me TV Server entering main loop");
		main_loop = Glib::MainLoop::create();
		main_loop->run();

		server_thread.terminate();

		if (database_initialised)
		{
			scheduled_recording_manager.save(schema, connection);
			channel_manager.save(schema, connection);
			configuration_manager.save(schema, connection);
		}

		g_message("Me TV Server exiting");
	}
	catch(...)
	{
		g_message("An unhandled exception was generated");
		handle_error();
	}
	
	return 0;
}

