/*
 * Copyright (C) 2011 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "configuration_manager.h"
#include "../common/exception.h"

void ConfigurationManager::load(Data::Schema& schema, Data::Connection& connection)
{
	Data::Table table = schema.tables["configuration"];
	Data::TableAdapter adapter(connection, table);
	Data::DataTable data_table = adapter.select_rows();

	entries.clear();
	for (Data::Rows::iterator i = data_table.rows.begin(); i != data_table.rows.end(); i++)	
	{
		Data::Row row = *i;
		entries.add(row["id"].int_value, row["name"].string_value, row["value"].string_value);
	}
}

void ConfigurationManager::save(Data::Schema& schema, Data::Connection& connection)
{
	Data::Table table = schema.tables["configuration"];
	Data::TableAdapter adapter(connection, table);
	Data::DataTable data_table = adapter.select_rows();

	for (EntryList::iterator i = entries.begin(); i != entries.end(); i++)
	{
		Entry entry = *i;
		Data::Row row;
		row.auto_increment = &(entry.id);
		row["id"].int_value = entry.id;
		row["name"].string_value = entry.name;
		row["value"].string_value = entry.value;
		data_table.rows.push_back(row);
	}

	adapter.replace_rows(data_table);
}

gboolean ConfigurationManager::EntryList::contains(const Glib::ustring& name)
{
	for (iterator i = begin(); i != end(); i++)
	{
		if (name == (*i).name)
		{
			return true;
		}
	}

	return false;
}

void ConfigurationManager::EntryList::add(const Glib::ustring& name, const Glib::ustring& value)
{
	add(0, name, value);
}

void ConfigurationManager::EntryList::add(int id, const Glib::ustring& name, const Glib::ustring& value)
{
	Entry entry;
	entry.id = id;
	entry.name = name;
	entry.value = value;
	push_back(entry);
}

Glib::ustring ConfigurationManager::EntryList::get(const Glib::ustring& name)
{
	for (iterator i = begin(); i != end(); i++)
	{
		if (name == (*i).name)
		{
			return (*i).value;
		}
	}

	throw Exception(Glib::ustring::format("Configuration %1 value not found", name));
}

void ConfigurationManager::set_value(const Glib::ustring& name, const Glib::ustring& value)
{
	entries.add(name, value);
}

Glib::ustring ConfigurationManager::get_value(const Glib::ustring& name)
{
	return entries.get(name);
}

void ConfigurationManager::set_value(const Glib::ustring& name, int value)
{
	entries.add(name, Glib::ustring::compose("%1", value));
}

int ConfigurationManager::get_int_value(const Glib::ustring& name)
{
	return ::atoi(entries.get(name).c_str());
}

void ConfigurationManager::set_default_value(const Glib::ustring& name, const Glib::ustring& value)
{
	if (!entries.contains(name))
	{
		entries.add(name, value);
	}
}

void ConfigurationManager::set_default_value(const Glib::ustring& name, int value)
{
	if (!entries.contains(name))
	{
		entries.add(name, Glib::ustring::compose("%1", value));
	}
}

