\documentclass{article}
\usepackage{axiom}
\begin{document}
\title{\$SPAD/src/algebra nsfip.as}
\author{Michael Richardson}
\maketitle
\begin{abstract}
\end{abstract}
\eject
\tableofcontents
\eject
\section{NagSpecialFunctionsInterfacePackage}
<<NagSpecialFunctionsInterfacePackage>>=
+++ Author: M.G. Richardson
+++ Date Created: 1995 Nov. 27
+++ Date Last Updated:
+++ Basic Functions:
+++ Related Constructors:
+++ Also See:
+++ AMS Classifications:
+++ Keywords:
+++ References:
+++ Description:
+++ This package provides Axiom-like interfaces to those of the NAG
+++ special functions in the NAGlink for which no equivalent
+++ functionality is transparently present in Axiom.

NagSpecialFunctionsInterfacePackage: with {

  nagExpInt : DF -> DF ;

  ++ nagExpInt calculates an approximation to the exponential integral,
  ++ \spad{E1}, defined by
#if saturn
  ++ \[E_{1}(x) = \int_{x}^{\infty }\frac{e^{-u}}{u}\,du\]
#else
  ++ \spad{E1(x) = integrate(1/u*%e^u, u=x..%infinity)}
#endif
  ++ using the NAG routine S13AAF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s13aaf.

  nagSinInt : DF -> DF ;

  ++ nagSinInt calculates an approximation to the sine integral,
  ++ \spad{Si}, defined by
#if saturn
  ++ \[{\rm Si} (x) = \int_{0}^{x}\frac{\sin u}{u}\,du\]
#else
  ++ \spad{Si(x) = integrate(1/u*sin(u), u=0..x)}
#endif
  ++ using the NAG routine S13ADF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s13adf.

  nagCosInt : DF -> DF ;

  ++ nagCosInt calculates an approximation to the cosine integral,
  ++ \spad{Ci}, defined by
#if saturn
  ++ \[{\rm Ci} (x) =
  ++ \gamma + \ln x+ \int_{0}^{x}\frac{\cos u- 1}{u}\,du\]
#else
  ++ \spad{Ci(x) = gamma + log x + integrate(1/u*cos(u), u=0..x)}
  ++ where \spad{gamma} is Euler's constant,
#endif
  ++ using the NAG routine S13ACF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s13acf.

  nagIncompleteGammaP : (DF, DF) -> DF ; -- to machine precision

  ++ nagIncompleteGammaP evaluates the incomplete gamma function
  ++ \spad{P}, defined by
#if saturn
  ++ \[P(a,x) & = & \frac{1}{\Gamma(a)}\int_{0}^{x}t^{a-1}e^{-t}\,dt\]
#else
  ++ \spad{P(a,x) = 1/Gamma(a)*integrate(t^(a-1)%e^(-t),t=0..x)}
#endif
  ++ to machine precision, using the NAG routine S14BAF.

  nagIncompleteGammaP : (DF, DF, DF) -> DF ;

  ++ nagIncompleteGammaP(a,x,tol) evaluates the incomplete gamma
  ++ function \spad{P}, defined by
#if saturn
  ++ \[P(a,x) & = & \frac{1}{\Gamma(a)}\int_{0}^{x}t^{a-1}e^{-t}\,dt\]
#else
  ++ \spad{P(a,x) = 1/Gamma(a)*integrate(t^(a-1)%e^(-t),t=0..x)}
#endif
  ++ to a relative accuracy \spad{tol}, using the NAG routine S14BAF.

  nagIncompleteGammaQ : (DF, DF) -> DF ;

  ++ nagIncompleteGammaQ evaluates the incomplete gamma function
  ++ \spad{Q}, defined by
#if saturn
  ++ \[Q(a,x)&=&\frac{1}{\Gamma(a)}\int_{x}^{\infty}t^{a-1}e^{-t}\,dt\]
#else
  ++ \spad{Q(a,x) = 1/Gamma(a)*integrate(t^(a-1)%e^(-t),t=x..%infinity)}
#endif
  ++ to machine precision, using the NAG routine S14BAF.

  nagIncompleteGammaQ : (DF, DF, DF) -> DF ;

  ++ nagIncompleteGammaQ(a,x,tol) evaluates the incomplete gamma
  ++ function \spad{Q}, defined by
#if saturn
  ++ \[Q(a,x)&=&\frac{1}{\Gamma(a)}\int_{x}^{\infty}t^{a-1}e^{-t}\,dt\]
#else
  ++ \spad{Q(a,x) = 1/Gamma(a)*integrate(t^(a-1)%e^(-t),t=x..%infinity)}
#endif
  ++ to a relative accuracy \spad{tol}, using the NAG routine S14BAF.

  nagErf : DF -> DF ;

  ++ nagErf calculates an approximation to the error function,
  ++ \spad{erf}, defined by
#if saturn
  ++ \[{\rm erf}\, x = \frac{2}{\sqrt{\pi}}\int_{0}^{x}e^{-t^{2}}\,dt\]
#else
  ++ \spad{erf(x) = 2/sqrt(\%pi)*integrate(\%e^(-t^2),t=0..x)}
#endif
  ++ using the NAG routine S15AEF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s15aef.

  nagErfC : DF -> DF ;

  ++ nagErfC calculates an approximation to the complementary error
  ++ function \spad{erfc}, defined by
#if saturn
  ++ \[{\rm erfc}\,x =
  ++       \frac{2} {\sqrt{\pi}}\int_{x}^{\infty}e^{-t^{2}}\,dt\]
#else
  ++ \spad{erfc(x) = 2/sqrt(%pi)*integrate(%e^(-t^2),t=x..%infinity)}
#endif
  ++ using the NAG routine S15ADF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s15adf.

  nagDAiryAi : DF -> DF ;

  ++ nagDAiryAi calculates an approximation to \spad{Ai'}, the
  ++ derivative of the Airy function \spad{Ai}, using the NAG routine
  ++ S17AJF.

  nagDAiryAi : CDF -> CDF ;

  ++ nagDAiryAi calculates an approximation to \spad{Ai'}, the
  ++ derivative of the Airy function \spad{Ai}, using the NAG routine
  ++ S17DGF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dgf.

  nagDAiryBi : DF -> DF ;

  ++ nagDAiryBi calculates an approximation to \spad{Bi'}, the
  ++ derivative of the Airy function \spad{Bi}, using the NAG routine
  ++ S17AKF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17akf.

  nagDAiryBi : CDF -> CDF ;

  ++ nagDAiryBi calculates an approximation to \spad{Bi'}, the
  ++ derivative of the Airy function \spad{Bi}, using the NAG routine
  ++ S17DHF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dhf.

  nagScaledDAiryAi : CDF -> CDF ;

  ++ nagDAiryAi(z) calculates an approximation to \spad{Ai'(z)}, the
  ++ derivative of the Airy function \spad{Ai(z)}, with the result
  ++ scaled by a factor
#if saturn
  ++ $e^{2z\sqrt{z}/ 3}$
#else
  ++ \spad{%e^(2*z*sqrt(z)/3)}
#endif
  ++ using the NAG routine S17DGF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dgf.

  nagScaledDAiryBi : CDF -> CDF ;

  ++ nagDAiryBi(z) calculates an approximation to \spad{Bi'(z)}, the
  ++ derivative of the Airy function \spad{Bi(z)}, with the result
  ++ scaled by a factor
#if saturn
  ++ $e^{2z\sqrt{z}/ 3}$
#else
  ++ \spad{%e^(2*z*sqrt(z)/3)}
#endif
  ++ using the NAG routine S17DHF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dhf.

  nagHankelH1 : (DF, CDF, INT) -> MCDF ;

  ++ nagHankelH1(nu,z,n) calculates an approximation to a sequence of n
  ++ values of the Hankel function
#if saturn
  ++ $H_{\nu + k}^{(1)}(z)$
#else
  ++ \spad{H1(nu + k, z)}
#endif
  ++ for non-negative nu and \spad{k = 0,1 ... n-1}, using the NAG
  ++ routine S17DLF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dlf.

  nagHankelH2 : (DF, CDF, INT) -> MCDF ;

  ++ nagHankelH2(nu,z,n) calculates an approximation to a sequence of n
  ++ values of the Hankel function
#if saturn
  ++ $H_{\nu + k}^{(2)}(z)$
#else
  ++ \spad{H2(nu + k, z)}
#endif
  ++ for non-negative nu and \spad{k = 0,1 ... n-1}, using the NAG
  ++ routine S17DLF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dlf.

  nagScaledHankelH1 : (DF, CDF, INT) -> MCDF ;

  ++ nagHankelH1(nu,z,n) calculates an approximation to a sequence of n
  ++ values of the Hankel function
#if saturn
  ++ $H_{\nu + k}^{(1)}(z)$
#else
  ++ \spad{H1(nu + k, z)}
#endif
  ++ for non-negative nu and \spad{k = 0,1 ... n-1}, with the result
  ++ scaled by a factor
#if saturn
  ++ $e^{-iz}
#else
  ++ \spad{%e^(-%i*z)}
#endif
  ++ using the NAG routine S17DLF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dlf.

  nagScaledHankelH2 : (DF, CDF, INT) -> MCDF ;

  ++ nagHankelH2(nu,z,n) calculates an approximation to a sequence of n
  ++ values of the Hankel function
#if saturn
  ++ $H_{\nu + k}^{(2)}(z)$
#else
  ++ \spad{H2(nu + k, z)}
#endif
  ++ for non-negative nu and \spad{k = 0,1 ... n-1}, with the result
  ++ scaled by a factor
#if saturn
  ++ $e^{iz}
#else
  ++ \spad{%e^(%i*z)}
#endif
  ++ using the NAG routine S17DLF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s17dlf.

  nagKelvinBer : DF -> DF ;

  ++ nagKelvinBer calculates an approximation to the Kelvin function
  ++ \spad{ber}, using the NAG routine S19AAF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s19aaf.

  nagKelvinBei : DF -> DF ;

  ++ nagKelvinBei calculates an approximation to the Kelvin function
  ++ \spad{bei}, using the NAG routine S19ABF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s19abf.

  nagKelvinKer : DF -> DF ;

  ++ nagKelvinKer calculates an approximation to the Kelvin function
  ++ \spad{ker}, using the NAG routine S19ACF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s19acf.

  nagKelvinKei : DF -> DF ;

  ++ nagKelvinKei calculates an approximation to the Kelvin function
  ++ \spad{kei}, using the NAG routine S19ADF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s19adf.

  nagFresnelS : DF -> DF ;

  ++ nagFresnelS calculates an approximation to the Fresnel integral
  ++ \spad{S}, defined by
#if saturn
  ++ \[S(x) = \int_{0}^{x}\sin\left(\frac{\pi}{2}t^{2}\right)\,dt\]
#else
  ++ \spad{S(x) = integrate(sin(%pi/2*t^2),t=0..x)}
#endif
  ++ using the NAG routine S20ACF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s20acf.

  nagFresnelC : DF -> DF ;

  ++ nagFresnelC calculates an approximation to the Fresnel integral
  ++ \spad{C}, defined by
#if saturn
  ++ \[C(x) = \int_{0}^{x}\cos\left(\frac{\pi}{2}t^{2}\right)\,dt\]
#else
  ++ \spad{C(x) = integrate(cos(%pi/2*t^2),t=0..x)}
#endif
  ++ using the NAG routine S20ADF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s20adf.

  nagEllipticIntegralRC : (DF, DF) -> DF ;

  ++ nagEllipticIntegralRC(x,y) calculates an approximation to the
  ++ elementary (degenerate symmetrised elliptic) integral
#if saturn
  ++ \[R_{C}(x,y) =
  ++       \frac{1}{2}\int_{0}^{\infty}\frac{dt}{\sqrt{t+x}(t+y)}\]
#else
  ++ \spad{RC(x,y) = 1/2*integrate(1/(sqrt(t+x)*(t+y)),t=0..\infinity)}
#endif
  ++ using the NAG routine S21BAF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s21baf.

  nagEllipticIntegralRF : (DF, DF, DF) -> DF ;

  ++ nagEllipticIntegralRF(x,y,z) calculates an approximation to the
  ++ symmetrised elliptic integral of the first kind,
#if saturn
  ++ \[R_{F}(x, y, z) =
  ++     \frac{1}{2}\int_{0}^{\infty}\frac{dt}{\sqrt{(t+x)(t+y)(t+z)}}\]
#else
  ++ \spad{RF(x,y,z) =
  ++       1/2*integrate(1/sqrt((t+x)*(t+y)*(t+z)),t=0..\infinity)}
#endif
  ++ using the NAG routine S21BBF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s21bbf.

  nagEllipticIntegralRD : (DF, DF, DF) -> DF ;

  ++ nagEllipticIntegralRD(x,y,z) calculates an approximation to the
  ++ symmetrised elliptic integral of the second kind,
#if saturn
  ++ \[R_{D}(x, y, z) =
  ++ \frac{3}{2}\int_{0}^{\infty}\frac{dt}{\sqrt{(t+x)(t+y)(t+z)^{3}}}\]
#else
  ++ \spad{RD(x,y,z) =
  ++       1/2*integrate(1/sqrt((t+x)*(t+y)*(t+z)^3),t=0..\infinity)}
#endif
  ++ using the NAG routine S21BCF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s21bcf.

  nagEllipticIntegralRJ : (DF, DF, DF, DF) -> DF ;

  ++ nagEllipticIntegralRJ(x,y,z,rho) calculates an approximation to
  ++ the symmetrised elliptic integral of the third kind,
#if saturn
  ++ \[R_{J}(x, y, z, \rho ) =
  ++       \frac{3}{2}\int_{0}^{\infty}
  ++             \frac{dt}{(t+\rho)\sqrt{(t+x)(t+y)(t+z)}}\]
#else
  ++ \spad{RJ(x,y,z,rho) =
  ++ 3/2*integrate(1/((t+rho)*sqrt((t+x)*(t+y)*(t+z))),t=0..\infinity))u}
#endif
  ++ using the NAG routine S21BDF.
  ++ For detailed information on the accuracy, please consult the NAG
  ++ manual via the Browser page for the operation s21bdf.

} == add {

  import from NagSpecialFunctionsPackage ;
  import from NagResultChecks ;

  local ipIfail : Integer := -1 ;

  nagExpInt(x : DF) : DF ==
    checkResult(s13aaf(x,ipIfail), "s13aafResult", "S13AAF") ;

  nagCosInt(x : DF) : DF ==
    checkResult(s13acf(x,ipIfail), "s13acfResult", "S13ACF") ;

  nagSinInt(x : DF) : DF ==
    checkResult(s13adf(x,ipIfail), "s13adfResult", "S13ADF") ;

  nagIncompleteGammaP(a : DF, x : DF) : DF ==
    checkResult(s14baf(a,x,0.0,ipIfail), "p", "S14BAF") ;

  nagIncompleteGammaP(a : DF, x : DF, tol : DF) : DF ==
    checkResult(s14baf(a,x,tol,ipIfail), "p", "S14BAF") ;

  nagIncompleteGammaQ(a : DF, x : DF) : DF ==
    checkResult(s14baf(a,x,0.0,ipIfail), "q", "S14BAF") ;

  nagIncompleteGammaQ(a : DF, x : DF, tol : DF) : DF ==
    checkResult(s14baf(a,x,tol,ipIfail), "q", "S14BAF") ;

  nagErfC(x : DF) : DF ==
    checkResult(s15adf(x,ipIfail), "s15adfResult", "S15ADF") ;

  nagErf(x : DF) : DF ==
    checkResult(s15aef(x,ipIfail), "s15aefResult", "S15AEF") ;

  nagDAiryAi(x : DF) : DF ==
    checkResult(s17ajf(x,ipIfail), "s17ajfResult", "S17AJF") ;

  nagDAiryAi(z : CDF) : CDF ==
    checkCxResult(s17dgf("d",z,"u",ipIfail), "ai", "S17DGF") ;

  nagDAiryBi(x : DF) : DF ==
    checkResult(s17akf(x,ipIfail), "s17akfResult", "S17AKF") ;

  nagDAiryBi(z : CDF) : CDF ==
    checkCxResult(s17dhf("d",z,"u",ipIfail), "bi", "S17DHF") ;

  nagScaledDAiryAi(z : CDF) : CDF ==
    checkCxResult(s17dgf("d",z,"s",ipIfail), "ai", "S17DGF") ;

  nagScaledDAiryBi(z : CDF) : CDF ==
    checkCxResult(s17dhf("d",z,"s",ipIfail), "bi", "S17DHF") ;

  nagHankelH1(order : DF, z : CDF, n : INT) : Matrix CDF ==
    checkMxCDF(s17dlf(1,order,z,n,"u",ipIfail), "cy", "S17DLF") ;

  nagHankelH2(order : DF, z : CDF, n : INT) : Matrix CDF ==
    checkMxCDF(s17dlf(2,order,z,n,"u",ipIfail), "cy", "S17DLF") ;

  nagScaledHankelH1(order : DF, z : CDF, n : INT) : Matrix CDF ==
    checkMxCDF(s17dlf(1,order,z,n,"s",ipIfail), "cy", "S17DLF") ;

  nagScaledHankelH2(order : DF, z : CDF, n : INT) : Matrix CDF ==
    checkMxCDF(s17dlf(2,order,z,n,"s",ipIfail), "cy", "S17DLF") ;

  nagKelvinBer(x : DF) : DF ==
    checkResult(s19aaf(x,ipIfail), "s19aafResult", "S19AAF") ;

  nagKelvinBei(x : DF) : DF ==
    checkResult(s19abf(x,ipIfail), "s19abfResult", "S19ABF") ;

  nagKelvinKer(x : DF) : DF ==
    checkResult(s19acf(x,ipIfail), "s19acfResult", "S19ACF") ;

  nagKelvinKei(x : DF) : DF ==
    checkResult(s19adf(x,ipIfail), "s19adfResult", "S19ADF") ;

  nagFresnelS(x : DF) : DF ==
    checkResult(s20acf(x,ipIfail), "s20acfResult", "S20ACF") ;

  nagFresnelC(x : DF) : DF ==
    checkResult(s20adf(x,ipIfail), "s20adfResult", "S20ADF") ;

  nagEllipticIntegralRC(x : DF, y : DF) : DF ==
    checkResult(s21baf(x,y,ipIfail), "s21bafResult", "S21BAF") ;

  nagEllipticIntegralRF(x : DF, y : DF, z : DF) : DF ==
    checkResult(s21bbf(x,y,z,ipIfail), "s21bbfResult", "S21BBF") ;

  nagEllipticIntegralRD(x : DF, y : DF, z : DF) : DF ==
    checkResult(s21bcf(x,y,z,ipIfail), "s21bcfResult", "S21BCF") ;

  nagEllipticIntegralRJ(x : DF, y : DF, z : DF, rho : DF) : DF ==
    checkResult(s21bdf(x,y,z,rho,ipIfail), "s21bdfResult", "S21BDF") ;
}

#if NeverAssertThis

-- Note that the conversions of Results from DoubleFloat to Float
-- will become unnecessary if outputGeneral is extended to apply to
-- DoubleFloat quantities.

)lib nrc
)lib nsfip

outputGeneral 4

-- DF here means DoubleFloat.
-- Results converted to Float as outputGeneral not working on DF.

-- nagExpInt : DF -> DF ;

nagExpInt(2) :: Float

--       0.0489

nagExpInt(-1) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S13AAF: IFAIL =     1
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S13AAF. The error number (IFAIL value) is 1, please consult the
--      NAG manual via the Browser for diagnostic information.

-- nagSinInt : DF -> DF ;

nagSinInt(0) :: Float

--       0.0

nagSinInt(0.2) :: Float

--       0.1996

nagSinInt(0.4) :: Float

--       0.3965

nagSinInt(0.6) :: Float

--       0.5881

nagSinInt(0.8) :: Float

--       0.7721

nagSinInt(1) :: Float

--       0.9461

-- nagCosInt : DF -> DF ;

nagCosInt(0.2) :: Float

--       - 1.042

nagCosInt(0.4) :: Float

--       - 0.3788

nagCosInt(0.6) :: Float

--       - 0.02227

nagCosInt(0.8) :: Float

--       0.1983

nagCosInt(1) :: Float

--       0.3374

-- nagIncompleteGammaP : (DF, DF) -> DF ; (to machine precision)

nagIncompleteGammaP(2,3) :: Float

--       0.8009

nagIncompleteGammaP(7,1) :: Float

--       0.00008324

nagIncompleteGammaP(0.5,99) :: Float

--       1.0

nagIncompleteGammaP(20,21) :: Float

--       0.6157

nagIncompleteGammaP(21,20) :: Float

--       0.4409

-- nagIncompleteGammaP : (DF, DF, DF) -> DF ; (to specified precision)

nagIncompleteGammaP(7,1,0.1) :: Float

--       0.00008313

-- nagIncompleteGammaQ : (DF, DF) -> DF ; (to machine precision)

nagIncompleteGammaQ(2,3) :: Float

--       0.1991

nagIncompleteGammaQ(7,1) :: Float

--       0.9999

nagIncompleteGammaQ(0.5,99) :: Float

--       0.5705 E -44

nagIncompleteGammaQ(20,21) :: Float

--       0.3843

nagIncompleteGammaQ(21,20) :: Float

--       0.5591

nagIncompleteGammaQ(25,14) :: Float

--       0.995

-- nagIncompleteGammaQ : (DF, DF, DF) -> DF ; (to specified precision)

nagIncompleteGammaQ(25,14,0.1) :: Float

--       0.9953

-- nagErf : DF -> DF ;

nagErf(-6) :: Float

--       - 1.0

nagErf(-4.5) :: Float

--       - 1.0

nagErf(-1) :: Float

--       - 0.8427

nagErf(1) :: Float

--       0.8427

nagErf(4.5) :: Float

--       1.0

nagErf(6) :: Float

--       1.0

-- nagErfC : DF -> DF ;

nagErfC(-10) :: Float

--       2.0

nagErfC(-1) :: Float

--       1.843

nagErfC(0) :: Float

--       1.0

nagErfC(1) :: Float

--       0.1573

nagErfC(15) :: Float

--       0.7213 E -99

-- nagDAiryAi : DF -> DF ;

nagDAiryAi(-10) :: Float

--       0.9963

nagDAiryAi(-1) :: Float

--       - 0.01016

nagDAiryAi(0) :: Float

--       - 0.2588

nagDAiryAi(1) :: Float

--       - 0.1591

nagDAiryAi(5) :: Float

--       - 0.0002474

nagDAiryAi(10) :: Float

--       - 0.3521 E -9

nagDAiryAi(20) :: Float

--       - 0.7586 E -26

-- nagDAiryAi : CDF -> CDF ;

nagDAiryAi(0.3+0.4*%i) :: Complex Float

--       - 0.2612 + 0.03848 %i

-- nagDAiryBi : DF -> DF ;

nagDAiryBi(-10) :: Float

--       0.1194

nagDAiryBi(-1) :: Float

--       0.5924

nagDAiryBi(0) :: Float

--       0.4483

nagDAiryBi(1) :: Float

--       0.9324

nagDAiryBi(5) :: Float

--       1436.0

nagDAiryBi(10) :: Float

--       0.1429 E 10

nagDAiryBi(20) :: Float

--       0.9382 E 26

-- nagDAiryBi : CDF -> CDF ;

nagDAiryBi(0.3+0.4*%i) :: Complex Float

--       0.4093 + 0.07966 %i

-- nagScaledDAiryAi : CDF -> CDF ;

nagScaledDAiryAi(0.3+0.4*%i) :: Complex Float

--       - 0.2744 - 0.02356 %i

-- nagScaledDAiryBi : CDF -> CDF ;

nagScaledDAiryBi(0.3+0.4*%i) :: Complex Float

--       0.3924 + 0.07638 %i

-- nagHankelH1 : (DF, CDF, Int) -> List CDF ;

nagHankelH1(0,0.3+0.4*%i,2) :: Matrix Complex Float

--       [0.3466 - 0.5588 %i  - 0.7912 - 0.8178 %i]

nagHankelH1(2.3,2,2) :: Matrix Complex Float

--       [0.2721 - 0.7398 %i  0.08902 - 1.412 %i]

nagHankelH1(2.12,-1,2) :: Matrix Complex Float

--       [- 0.7722 - 1.693 %i  2.601 + 6.527 %i]

-- nagHankelH2 : (DF, CDF, Int) -> List CDF ;

nagHankelH2(6,3.1-1.6*%i,2) :: Matrix Complex Float

--       [- 1.371 - 1.28 %i  - 1.491 - 5.993 %i]

-- nagScaledHankelH1 : (DF, CDF, Int) -> List CDF ;

nagScaledHankelH1(0,0.3+0.4*%i,2) :: Matrix Complex Float

--       [0.2477 - 0.9492 %i  - 1.488 - 0.8166 %i]

-- nagScaledHankelH2 : (DF, CDF, Int) -> List CDF ;

nagScaledHankelH2(6,3.1-1.6*%i,2) :: Matrix Complex Float

--       [7.05 + 6.052 %i  8.614 + 29.35 %i]

-- nagKelvinBer : DF -> DF ;

nagKelvinBer(0.1) :: Float

--       1.0

nagKelvinBer(1) :: Float

--       0.9844

nagKelvinBer(2.5) :: Float

--       0.4

nagKelvinBer(5) :: Float

--       - 6.23

nagKelvinBer(10) :: Float

--       138.8

nagKelvinBer(15) :: Float

--       - 2967.0

nagKelvinBer(60) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19AAF: IFAIL =     1
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19AAF. The error number (IFAIL value) is 1, please consult the
--      NAG manual via the Browser for diagnostic information.

nagKelvinBer(-1) :: Float

--       0.9844

-- nagKelvinBei : DF -> DF ;

nagKelvinBei(0.1) :: Float

--       0.0025

nagKelvinBei(1) :: Float

--       0.2496

nagKelvinBei(2.5) :: Float

--       1.457

nagKelvinBei(5) :: Float

--       0.116

nagKelvinBei(10) :: Float

--       56.37

nagKelvinBei(15) :: Float

--       - 2953.0

nagKelvinBei(60) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ABF: IFAIL =     1
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ABF. The error number (IFAIL value) is 1, please consult the
--      NAG manual via the Browser for diagnostic information.

nagKelvinBei(-1) :: Float

--       0.2496

-- nagKelvinKer : DF -> DF ;

nagKelvinKer(0) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ACF: IFAIL =     2
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ACF. The error number (IFAIL value) is 2, please consult the
--      NAG manual via the Browser for diagnostic information.

nagKelvinKer(0.1) :: Float

--       2.42

nagKelvinKer(1) :: Float

--       0.2867

nagKelvinKer(2.5) :: Float

--       - 0.06969

nagKelvinKer(5) :: Float

--       - 0.01151

nagKelvinKer(10) :: Float

--       0.0001295

nagKelvinKer(15) :: Float

--       - 0.1514 E -7

nagKelvinKer(1100) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ACF: IFAIL =     1
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ACF. The error number (IFAIL value) is 1, please consult the
--      NAG manual via the Browser for diagnostic information.

nagKelvinKer(-1) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ACF: IFAIL =     2
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ACF. The error number (IFAIL value) is 2, please consult the
--      NAG manual via the Browser for diagnostic information.

-- nagKelvinKei : DF -> DF ;

nagKelvinKei(0) :: Float

--       - 0.7854

nagKelvinKei(0.1) :: Float

--       - 0.7769

nagKelvinKei(1) :: Float

--       - 0.495

nagKelvinKei(2.5) :: Float

--       - 0.1107

nagKelvinKei(5) :: Float

--       0.01119

nagKelvinKei(10) :: Float

--       - 0.0003075

nagKelvinKei(15) :: Float

--       0.000007963

nagKelvinKei(1100) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ADF: IFAIL =     1
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ADF. The error number (IFAIL value) is 1, please consult the
--      NAG manual via the Browser for diagnostic information.

nagKelvinKei(-1) :: Float

-- ^ ABNORMAL EXIT from NAG Library routine S19ADF: IFAIL =     2
-- ^ NAG soft failure - control returned
--
--   Error signalled from user code:
--      An error was detected when calling the NAG Library routine
--      S19ADF. The error number (IFAIL value) is 2, please consult the
--      NAG manual via the Browser for diagnostic information.


-- nagFresnelS : DF -> DF ;

nagFresnelS(0) :: Float

--       0.0

nagFresnelS(0.5) :: Float

--       0.06473

nagFresnelS(1) :: Float

--       0.4383

nagFresnelS(2) :: Float

--       0.3434

nagFresnelS(4) :: Float

--       0.4205

nagFresnelS(5) :: Float

--       0.4992

nagFresnelS(6) :: Float

--       0.447

nagFresnelS(8) :: Float

--       0.4602

nagFresnelS(10) :: Float

--       0.4682

nagFresnelS(-1) :: Float

--       - 0.4383

nagFresnelS(1000) :: Float

--       0.4997

-- nagFresnelC : DF -> DF ;

nagFresnelC(0) :: Float

--       0.0

nagFresnelC(0.5) :: Float

--       0.4923

nagFresnelC(1) :: Float

--       0.7799

nagFresnelC(2) :: Float

--       0.4883

nagFresnelC(4) :: Float

--       0.4984

nagFresnelC(5) :: Float

--       0.5636

nagFresnelC(6) :: Float

--       0.4995

nagFresnelC(8) :: Float

--       0.4998

nagFresnelC(10) :: Float

--       0.4999

nagFresnelC(-1) :: Float

--       - 0.7799

nagFresnelC(1000) :: Float

--       0.5

-- nagEllipticIntegralRC : (DF, DF) -> DF ;

nagEllipticIntegralRC(0.5,1) :: Float

--       1.111

nagEllipticIntegralRC(1,1) :: Float

--       1.0

nagEllipticIntegralRC(1.5,1) :: Float

--       0.9312

-- nagEllipticIntegralRD : (DF, DF, DF) -> DF ;

nagEllipticIntegralRD(0.5,0.5,1) :: Float

--       1.479

nagEllipticIntegralRD(0.5,1,1) :: Float

--       1.211

nagEllipticIntegralRD(0.5,1.5,1) :: Float

--       1.061

nagEllipticIntegralRD(1,1,1) :: Float

--       1.0

nagEllipticIntegralRD(1,1.5,1) :: Float

--       0.8805

nagEllipticIntegralRD(1.5,1.5,1) :: Float

--       0.7775

-- nagEllipticIntegralRF : (DF, DF, DF) -> DF ;

nagEllipticIntegralRF(0.5,1,1.5) :: Float

--       1.028

nagEllipticIntegralRF(1,1.5,2) :: Float

--       0.826

nagEllipticIntegralRF(1.5,2,2.5) :: Float

--       0.7116

-- nagEllipticIntegralRJ : (DF, DF, DF, DF) -> DF ;

nagEllipticIntegralRJ(0.5,0.5,0.5,2) :: Float

--       1.118

nagEllipticIntegralRJ(0.5,0.5,1,2) :: Float

--       0.9221

nagEllipticIntegralRJ(0.5,0.5,1.5,2) :: Float

--       0.8115

nagEllipticIntegralRJ(0.5,1,1,2) :: Float

--       0.7671

nagEllipticIntegralRJ(0.5,1,1.5,2) :: Float

--       0.6784

nagEllipticIntegralRJ(0.5,1.5,1.5,2) :: Float

--       0.6017

nagEllipticIntegralRJ(1,1,1,2) :: Float

--       0.6438

nagEllipticIntegralRJ(1,1,1.5,2) :: Float

--       0.5722

nagEllipticIntegralRJ(1,1.5,1.5,2) :: Float

--       0.5101

nagEllipticIntegralRJ(1.5,1.5,1.5,2) :: Float

--       0.4561

outputGeneral()

output "End of tests"

#endif

@
\section{License}
<<license>>=
--Copyright (c) 1991-2002, The Numerical ALgorithms Group Ltd.
--All rights reserved.
--
--Redistribution and use in source and binary forms, with or without
--modification, are permitted provided that the following conditions are
--met:
--
--    - Redistributions of source code must retain the above copyright
--      notice, this list of conditions and the following disclaimer.
--
--    - Redistributions in binary form must reproduce the above copyright
--      notice, this list of conditions and the following disclaimer in
--      the documentation and/or other materials provided with the
--      distribution.
--
--    - Neither the name of The Numerical ALgorithms Group Ltd. nor the
--      names of its contributors may be used to endorse or promote products
--      derived from this software without specific prior written permission.
--
--THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
--IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
--TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
--PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
--OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
--EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
--PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
--PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
--LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
--NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
--SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
@
<<*>>=
<<license>>

-- NagSpecialFunctionsInterfacePackage

-- To test:
-- sed -ne '1,/^#if NeverAssertThis/d;/#endif/d;p' < nsfip.as > nsfip.input
-- axiom
-- )set nag host <some machine running nagd>
-- )r nsfip.input

#unassert saturn

#include "axiom.as"

DF   ==> DoubleFloat ;
CDF  ==> Complex DoubleFloat ;
MCDF ==> Matrix Complex DoubleFloat ;
INT  ==> Integer ;
RSLT ==> Result ;
SMBL ==> Symbol ;
STRG ==> String ;

<<NagSpecialFunctionsInterfacePackage>>
@
\eject
\begin{thebibliography}{99}
\bibitem{1} nothing
\end{thebibliography}
\end{document}
