/*    xfdiff.c   */

/*  xfdiff (a gtk frontend for diff+patch)
 *  Copyright (C)  Edscott Wilson Garcia under GNU GPL
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* Description:
 * 
 * Xfdiff is a module for Xftree (Xfce project by Olivier Fourdan, http://www.xfce.org)
 * This module may be run from Xftree or independently, with command line
 * options (use: xfdiff -h to get a summary).
 *
 * Xfdiff does no heavy processing. All it does is allow a fast and easy access
 * to the standard utilities diff and patch. Quoting Olivier: "the goal [for Xfce] 
 * is keep most system resources for the applications, and not to consume all memory 
 * and CPU usage with the desktop environment."
 * 
 * This program will open a window with two text widgets and display two files, 
 * highlighting the differences. Buttons permit rapid navigation through the 
 * diffences of the two files. Among the options, you may toggle line numbers, 
 * change highlight color, and font, toggle verbose output and other
 * stuff.
 * 
 * Xfdiff can also open patch files (created in unified format) and display in
 * the same manner the differences. Buttons and a drop-down list permit rapid 
 * navigation through every file that is modified by the patch.
 * Among the options, you can specify strip level, whether or not
 * a patch file is reversed, apply or undo a patch (and for GNUC only, create a 
 * patch file of your own).
 *
 * Xfdiff uses the standard libs from Xfce, GTK+, and modules from Xftree
 * and Xfclock.
 * 

 
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gmodule.h>

#include "utf_stuff.i"

#define __XFDIFF__
#include "xfdiff.h"
#include "xfdiff_misc.h"
#undef EXTERN
#include "globber.h"
#include "xfdiff_dlg.h"


static struct stat patch_st;

static char *cannot_read_patch_file, *identical_files, *patch_file_error, *empty_file;

G_MODULE_EXPORT
int extra_key_completion (gpointer user_data)
{
	return FALSE;
}


void
insert_void (GtkTextBuffer * buffer)
{
  GdkColor gray;
  GtkTextIter start,end;
  int i;
  char *space=" ";
  GtkTextTag *tag;
  
  tag=gtk_text_tag_table_lookup(gtk_text_buffer_get_tag_table(buffer),"bg_gray");
  if (!tag){
    gray.red = gray.green = gray.blue = 0x3fff;
    gtk_text_buffer_create_tag (buffer, "bg_gray",
                              "background_gdk", &gray,
                              NULL);
  }
  gtk_text_buffer_get_iter_at_offset (buffer, &end, 0);
  gtk_text_iter_forward_to_end (&end);
  for (i=0;i<255;i++) 
	  gtk_text_buffer_insert (buffer, &end, space, 1);
  space="\n"; 
  gtk_text_buffer_insert (buffer, &end, space, 1);
  
  start=end;
  gtk_text_iter_backward_chars (&start, 256);
  gtk_text_buffer_apply_tag_by_name (buffer, "bg_gray", &start, &end);
}

static void
_utf8_insert_tag (GtkTextBuffer * buffer, char *line, char *tag_id,gboolean filename)
{
  GtkTextIter end;
  GdkColor fore,back;
  GtkTextTag *tag;
  tag=gtk_text_tag_table_lookup(gtk_text_buffer_get_tag_table(buffer),tag_id);
  if (!tag){
	  /*printf("dbg:creating tag %s\n",tag_id);*/
    if (strcmp(tag_id,"bg_red")==0) {
      fore.red = fore.green = 0xffff; fore.blue = 0;
      back.red = 0xffff; back.green = back.blue = 0;
      tag=gtk_text_buffer_create_tag (buffer, tag_id,
                              "background_gdk", &colorbg,
                              "foreground_gdk", &colorfg,
                              NULL);
    } else if (strcmp(tag_id,"bg_blue")==0) {
      fore.red = fore.green = 0xffff; fore.blue = 0;
      back.blue = 0xffff; back.green = back.red = 0;
      tag=gtk_text_buffer_create_tag (buffer, tag_id,
                              "background_gdk", &back,
                              "foreground_gdk", &fore,
                              NULL);
    } else if (strcmp(tag_id,"bg_bluewhite")==0) {
      fore.red = fore.green = fore.blue = 0xffff; 
      back.blue = 0xffff; back.green = back.red = 0;
      tag=gtk_text_buffer_create_tag (buffer, tag_id,
                              "background_gdk", &back,
                              "foreground_gdk", &fore,
                              NULL);
    }

  }
  gtk_text_buffer_get_iter_at_offset (buffer, &end, 0);
  gtk_text_iter_forward_to_end (&end);
  

  gtk_text_buffer_insert_with_tags (buffer,&end,my_utf_string(line),
		  -1,tag,NULL);
  /*g_free (s);			*/
 
}

void
utf8_insert_tag (GtkTextBuffer * buffer, char *line, char *tag_id)
{
  _utf8_insert_tag (buffer,line,tag_id,FALSE);
}
void
utf8_file_insert (GtkTextBuffer * buffer, char *line)
{
  _utf8_insert_tag (buffer,line,"bg_blue",TRUE);
}



void
utf8_insert (GtkTextBuffer * buffer, char *line)
{
  gtk_text_buffer_insert_at_cursor (buffer, my_utf_string(line),-1);
}

static void
xfdiff_init (void)
{
/* convenience location of duplicate dialog texts  */
  /* non-static */

  no_left_path = N_("No left path has been selected.");
  no_right_path = N_("No right path has been selected.");
  /* static */

  identical_files = N_("Files are identical\n(or binary files that may or may not differ)");

  empty_file = N_("****Empty file****\n");
/* initial sizes for text files, sizes in line numbers and average char length */
  sizeR = 30, sizeL = 30, sizeH = 15;
  head = NULL, current = NULL;
  headF = NULL, currentF = NULL;
  reversed = strip = 0;
  style = NULL;
  xfce_style = NULL;
  the_font = NULL;
  diff = OKbutton = NULL;
  drawA = NULL;
  titleD = NULL;
  what_dir = text_buffer[1] = text_buffer[0] = NULL;
  titleR = titleL = NULL;
  titleP = NULL;
  diffbox = patchbox = NULL;
  patch_label_box = NULL;
  done_str = (char *) malloc (128);
  if (!done_str)
    xfdiff_abort (E_MALLOC);
  drawP = NULL;
  adj = NULL;
  lineW = lineH = rightC = leftC = 0;
  drawGC = NULL;
  fileO = fileRR = fileI = patchO = NULL;
  fileRD = fileLD = filename[1] = filename[0] = fileP = fileD = NULL;
  /* not initialized: colorfg,colorbg; */
  silent = 0;
  patching = 0;
  diff_is_done = 1;
  applying_patch = 0;
  autostrip = 1;
}

/* memcpy is necesary because patch file may contain binary data */
#ifdef __GNUC__
/* memcpy is a GNU extension.*/
#define MEMCPY memcpy
#else
/* memcpy is a GNU extension.*/
static void *
MEMCPY (void *dest, const void *src, size_t n)
{
  char *destC, *srcC;
  size_t i;

  destC = (char *) dest;
  srcC = (char *) src;
  for (i = 0; i < n; i++)
    destC[i] = srcC[i];
  return dest;
}
#endif
static void
assign_tmp_files (void)
{
  char pidT[32];
  char *tmp_dir = NULL;

  /* TMPDIR processing */

  if (!tmp_dir && getenv("TMPDIR") && strlen( getenv("TMPDIR")))
    tmp_dir = getenv ("TMPDIR");
  if (!tmp_dir && getenv("TMP") && strlen( getenv("TMP")))
    tmp_dir = getenv ("TMP");
  if (!tmp_dir && getenv("TEMP") && strlen( getenv("TEMP")))
    tmp_dir = getenv ("TEMP");
  if (!tmp_dir)
    tmp_dir = TMP_DIR;

  sprintf (pidT, "%x", (int) getpid ());
  if (fileO)
  {
    remove (fileO);
    free (fileO);
  }
  fileO = (char *) malloc (strlen (tmp_dir) + 2 + strlen ("xfdiffR.") + strlen (pidT));
  if (!fileO)
    xfdiff_abort (E_MALLOC);
  sprintf (fileO, "%s/%s%s", tmp_dir, "xfdiffR.", pidT);

  if (fileI)
  {
    remove (fileI);
    free (fileI);
  }
  fileI = (char *) malloc (strlen (tmp_dir) + 2 + strlen ("xfdiffL.") + strlen (pidT));
  if (!fileI)
    xfdiff_abort (E_MALLOC);
  sprintf (fileI, "%s/%s%s", tmp_dir, "xfdiffL.", pidT);

  if (fileRR)
  {
    remove (fileRR);
    free (fileRR);
  }
  fileRR = (char *) malloc (strlen (tmp_dir) + 2 + strlen ("xfdiff.rej.") + strlen (pidT));
  if (!fileRR)
    xfdiff_abort (E_MALLOC);
  sprintf (fileRR, "%s/%s%s", tmp_dir, "xfdiff.rej.", pidT);
}

static patched_file *
pushF (char *file, int offset, int length)
{
    patched_file *tmp;
  if (!headF)
  {
    headF = (patched_file *) malloc (sizeof (patched_file));
    if (!headF)
      xfdiff_abort (E_MALLOC);
    tmp = headF;
    tmp->previous = NULL;
  }
  else
  {
    for (tmp=headF; tmp->next; tmp=tmp->next);
    tmp->next = (patched_file *) malloc (sizeof (patched_file));
    if (!tmp->next) g_assert_not_reached();
    (tmp->next)->previous = tmp;
    tmp = tmp->next;

  }
  tmp->next = NULL;
  tmp->offset = offset;
  tmp->length = length;
  tmp->diffC = 0;
  tmp->file = tmp->newfile = NULL;

  tmp->file = assign (tmp->file, file);
  return tmp;
}

unsigned long blocksum(char *buffer, size_t buffer_size){
    int i;
    unsigned long block=0;
    
    for (i=0; i<buffer_size; i++){
	block += *(buffer+i);
    }
    return block;
}


static int
checkfile (char *file)
{
  FILE *check;
  if (!file)
    return FALSE;
  if ((check = fopen (file, "r")) == NULL)
    return FALSE;
  fclose (check);
  return TRUE;
}

static void
writebinary (GtkTextBuffer * text_location)
{
  utf8_insert (text_location, _("****Binary data****\n"));
}

static void
writeout (GtkTextBuffer * text_location, char *file)
{
  int fileH;
  fileH = open (file, O_RDONLY);
  if (fileH == -1)
  {
    utf8_insert (text_location, _(empty_file));
  }
  else
  {
    char *blok;
    ssize_t blok_s;
    int i;
    blok = (char *) malloc (BLOK_SIZE);
    if (!blok)
      xfdiff_abort (E_MALLOC);
    while ((blok_s = read (fileH, (void *) blok, BLOK_SIZE)) > 0)
    {
      for (i = 0; i < blok_s; i++)
      {
	if (((blok[i] < 9) && (blok[i] >= 0)) || ((blok[i] < 32) && (blok[i] > 13)))
	{
	  writebinary (text_location);
	  free (blok);
	  close (fileH);
	  return;
	}
      }
      utf8_insert (text_location, blok);

    }
    free (blok);
    close (fileH);
  }
}

#define BLOCK_SIZE 4096
unsigned long checksum(char *filename, size_t size){
    unsigned long sum=0;
    FILE *in;
    char buffer[BLOCK_SIZE];
    size_t count;
    in=fopen(filename,"r");
    if (!in) return sum;
    for (count=0; count < size/BLOCK_SIZE+1; count++){
	if (fread((void *)buffer, BLOCK_SIZE, 1, in)<1){
	    sum += blocksum(buffer,size%BLOCK_SIZE);
	    break;
	}
	sum += blocksum(buffer,BLOCK_SIZE);	
    }
    fclose(in);
    return sum;
}

int quick_diff(char **f){
    struct stat st[2];
    int i,st_result[2];
    for (i=0; i<2; i++){
	st_result[i] = (stat(f[i],st+i)==0)?1:0;
	/*g_message("stat result %s = %d",f[i],st_result[i]);*/
		
    }
    
    /* check for file availability */

    /* either file is binary... */
 
    
    /* neither file can be accessed */
    if (!st_result[1] && !st_result[0]) {
	/*g_message("--- neither file can be accessed: %s, %s",f[0],f[1]);*/
	return 0;
    }
    /* only one file can be accessed */
    if (!st_result[1] || !st_result[0]){
	/*g_message("+++ only one file can be accessed: %s, %s",f[0],f[1]);*/
	return 1;
    }
    /* check i-node */
    if (st[0].st_ino == st[1].st_ino) {
	/*g_message("--- i-node matches: %s, %s",f[0],f[1]);*/
	return 0;
    }
    /* check filesize */
    if (st[0].st_size != st[1].st_size) {
	/*g_message("+++ filesize fails: %s, %s",f[0],f[1]);*/
	return 1;
    }
    /* check sum */
    if (checksum(f[0], st[0].st_size) != checksum(f[1], st[1].st_size)) 
    {
	/*g_message("+++ checksum fails: %s, %s",f[0],f[1]);*/
	return 1;
    }
    /*g_message("--- checksum matches: %s, %s",f[0],f[1]);*/
    return 0;
}


/* functions for creating the file list on dir-dir diff: */
static 
int
pushemL (char *file)
{
  char *f[2];
  int i;
  char *base;

  if (strlen(file)<strlen(filename[0])+1) return 0;
  base = file+strlen(filename[0])+1;
  /*g_message("file=%s, base=%s",file,base);
  g_message("file=%s",file);*/

  for (i=0; i<2; i++) f[i]=g_build_filename(filename[i],base,NULL);
  /*g_free(base);
  g_message("f[0]=%s, f[1]=%s",f[0], f[1]);*/
  if (!quick_diff(f))
  {
      for (i=0; i<2; i++) g_free(f[i]);
      return 0;
  }
 
  
  g_free(f[0]);
  currentF=pushF (file, 0, 0);
  currentF->newfile=f[1];
  return 0;
}


static int
globber_run(){
      static void *glob_object=NULL;
      glob_object=globber_destroy(glob_object);
      glob_object = globber_create ();
      //if (xf_dlg_ask (diff,_("You have selected a directory.Should subdirectories be included?")))
	glob_set_options (glob_object, GLOBBER_RECURSIVE);
      glob_set_options (glob_object, GLOBBER_TYPE);
      glob_set_type (glob_object, S_IFREG);
      cleanF ();
      globber (glob_object, filename[0], pushemL, "*");
      /* now we no longer need the original filename[0] and filename[1] */
      currentF = headF;
      if (!currentF) return 0;
      filename[0] = assign (filename[0], currentF->file);
      filename[1] = assign (filename[1], currentF->newfile);
      return 1;
}

/* spot 3/3 where filename[1],filename[0] come in */
static int
pre_diff ()
{
  if (!filename[0])
  {
    filename[0] = prompt_path (_(no_left_path), filename[0]);
    if (!filename[0])
      return FALSE;
    fileLD = assign (fileLD, checkdir (filename[0]) ? filename[0] : NULL);
    update_titlesP ();
  }
  if (!filename[1])
  {
    filename[1] = prompt_path (_(no_right_path), filename[1]);
    if (!filename[1])
      return FALSE;
    fileRD = assign (fileRD, checkdir (filename[1]) ? filename[1] : NULL);
    update_titlesP ();
  }

  if (!patchO)
  {				/* not creating a patchfile */
    if (checkdir (filename[0]) && checkdir (filename[1]))
    {
      /* initialization for globber */
	globber_run();
    }

    if (checkdir (filename[0]) || checkdir (filename[1]))
    {
      static char *tmp_c = NULL;
      char *tmp_d;
      /* one of the selections is a directory. Prepare any-diff for the way gnu-diff would react. */
      if (checkdir (filename[0]))
      {
	tmp_d = strip_it (filename[1], max_strip (filename[1]));
	tmp_c = (char *) malloc (strlen (filename[0]) + strlen (tmp_d) + 2);
	if (!tmp_c)
	  xfdiff_abort (E_MALLOC);
	sprintf (tmp_c, "%s/%s", filename[0], tmp_d);
	if (filename[0])
	  free (filename[0]);
	filename[0] = tmp_c;
	tmp_c = NULL;
      }
      else
      {
	tmp_d = strip_it (filename[0], max_strip (filename[0]));
	tmp_c = (char *) malloc (strlen (filename[1]) + strlen (tmp_d) + 2);
	if (!tmp_c)
	  xfdiff_abort (E_MALLOC);
	sprintf (tmp_c, "%s/%s", filename[1], tmp_d);
	if (filename[1])
	  free (filename[1]);
	filename[1] = tmp_c;
	tmp_c = NULL;
      }
      /* now continue using filename[1] and filename[0] gnu-diff interpreted */

    }
    cleanP ();			/* clean difference polygons */
    cleanA ();
    cleanT ();			/* clean display area. */
    update_titlesP ();
    /* Case of void or unreadable files will be dealt with here. */
    if ((!checkfile (filename[0])) || (!checkfile (filename[1])))
    {
      writeout (text_buffer[0], filename[0]);
      writeout (text_buffer[1], filename[1]);
      return FALSE;
    }
  }				/* end if !creating patchfile */

  if (!silent)
  {
    if (currentF)
      show_diag (currentF->file);
    else
      show_diag (filename[0]);
    show_diag (_(" versus "));
    if (currentF)
      show_diag (currentF->newfile);
    else
      show_diag (filename[1]);
    show_diag (":\n");
  }
  return TRUE;
}

static void
post_diff (int diffC)
{
  if (patchO) return;
  if (!silent) {
    if (diffC) {
      gchar *g=g_strdup_printf(
	dngettext(GETTEXT_PACKAGE,"Only %d different section encountered","Total of %d different sections found",diffC),
	diffC
	      );
      show_diag ("xfdiff> *** "); 
      show_diag (g); 
      show_diag("\n");
      g_free (g);
    } else {
	show_diag ("xfdiff> *** "); 
	show_diag (_(identical_files)); 
	show_diag("\n");
    }
  }
}


#include <tubo.h>
void *tubo_object=NULL;
static int top_bufferline[2], 
	   bottom_bufferline[2],
	   current_line[2],
	   current_buffer_line;

static gboolean  diff_section=FALSE, diff_subsection=FALSE;
static FILE *skip_file[2];

static int
get_block_info(		char *line,
			int *file_line_number, 
			int *file_skiplines)
{
    gchar *p;
    int n1,n2,s1,s2;
    if (!line || !strlen(line)) return FALSE;
    if (line[0]!='@' || line[1]!='@')  return FALSE;
            
    p=strtok(line+2,","); if (!p) return FALSE;
    n1 = 0 - atoi(p); 
    n1--;
    p=strtok(NULL,"+"); if (!p) return FALSE;
    s1 = atoi(p);
    p=strtok(NULL,","); if (!p) return FALSE;
    n2 = atoi(p);
    n2--;
    p=strtok(NULL,"@"); if (!p) return FALSE;
    s2 = atoi(p);
    file_line_number[0]=n1;
    file_line_number[1]=n2;
    file_skiplines[0]=s1;
    file_skiplines[1]=s2;
    /*g_message("got_block_info: -%d,%d +%d,%d",n1,s1,n2,s2);*/
    return TRUE;
}

static void 
advance_buffers(	int *file_line_number, 
			int *file_skiplines)
{ 
    int i;
    char skip_line[256*256];
    for (i=0; i<2; i++) if (skip_file[i]) {
      int j;
      /* equal lines */
      for (; current_line[i]< file_line_number[i] && 
          !feof(skip_file[i]); current_line[i]++)
      {	  
	fgets(skip_line,256*256-1,skip_file[i]);
	utf8_insert (text_buffer[i], skip_line);
	if (i) current_buffer_line++;
      }
      /* skip diff sector */
      for (j=0; j<file_skiplines[i] && !feof(skip_file[i]); j++)
      {
	fgets(skip_line,256*256-1,skip_file[i]);
      }
    }
}
static void 
add_equal_line(char *line){
    int i;
    diff_subsection=FALSE;
    current_buffer_line++;
    for (i=0;i<2;i++){
	current_line[i]++;
	top_bufferline[i]=bottom_bufferline[i]=current_buffer_line;
	utf8_insert (text_buffer[i], line); 
    }
}
static void
add_different_line(char *line, int i) {
    diff_subsection=TRUE;
    utf8_insert_tag (text_buffer[i], line, "bg_red");
    current_line[i]++;
    bottom_bufferline[i]++;
}
static void
add_synchronization_lines(int *sync_count){
   int i;
   if (*sync_count==0) return;
   if (*sync_count < 0){
	for (i=0;i>*sync_count;i--) {
	    insert_void (text_buffer[1]);
	}
    } else {
	current_buffer_line += *sync_count;   
	for (i=0;i<*sync_count;i++){
	    insert_void (text_buffer[0]);
	}
    }
    *sync_count=0;
}
static void
add_previous_polygon(void){
    if (!current) {
	int value;
	gchar *g;
	current = pushP (top_bufferline, bottom_bufferline, current_line);
	g=g_strdup_printf(_("Showing difference number %d"),current->id);
	value = (current->topR < current->topL) ? 
	current->topR : current->topL;
	if (value >= lineH)  value -= lineH;
	else  value = 0;
	gtk_adjustment_set_value (GTK_ADJUSTMENT (adj), value);
	configure_event (drawA, NULL);
	show_diag("xfdiff> ");show_diag(g);show_diag("\n");
	g_free(g);	
    } else {
	pushP (top_bufferline, bottom_bufferline, current_line);
    }
}

static
void fork_function (void *data)
{
    char **argument = (char **)data;

    execvp(argument[0], argument);
    fprintf(stderr, "CHILD could not execvp: this should not happen");
    fprintf(stderr, "Do you have %s in your path?", argument[0]);
    fflush(NULL);
    usleep(500000);
    _exit(123);
}
    

static void fork_finished_function (pid_t pid, void *user_data){
    tubo_object=NULL;
}

static int
operate_stdout(int n, void *data, void *user_data)
{
    char *line;
    gboolean  binary_line = FALSE;
    static int  sync_count;
    static int file_line_number[2],file_skiplines[2];
    

    if(n)  binary_line = TRUE; /* this would mean binary data */
    line = (char *)data;

    if (binary_line) return TRUE;
    switch (line[0]){
      case '@':  
	if (!get_block_info(line, file_line_number,file_skiplines)) {
	  show_diag("error> get_block_info failed...");show_diag("\n");
	  return TRUE;
	} else {
	  gchar *g=g_strdup_printf(_("Different block: -%d,%d +%d,%d"),
			file_line_number[0],file_skiplines[0],
			file_line_number[1],file_skiplines[1]);
	  show_diag("xfdiff> "); show_diag(g); show_diag("\n");
	  g_free(g);
	}
	advance_buffers(file_line_number, file_skiplines);
	diff_section=TRUE;
	return TRUE;
      default:
        if (!diff_section) return TRUE;
        switch (line[0]){
          case ' ':
	    if (diff_subsection) {
		add_previous_polygon();
		add_synchronization_lines(&sync_count);
		configure_event (drawA, NULL);
		while (gtk_events_pending()) gtk_main_iteration();
	    }
	    add_equal_line(line+1);
	    break;
	  case '+':
	    sync_count++;
	    add_different_line(line+1,1);
	    break;
	  case '-':
	    current_buffer_line++;
	    sync_count--;
	    add_different_line(line+1,0);
	    break;
	}
    }
	
    return TRUE;
}

int
do_diff (void)
{
  int argc = 0;
  char *arguments[10];
  /*time_t t=time(NULL);*/
  int i;


  if (!pre_diff ()) return FALSE;

  if (checkdir (filename[1]) && checkdir (filename[0]))
  {

  }
  addedR = addedL = 0;
  show_diag (_("Working diff...\n"));

  /* prepare command line */
  arguments[argc++] = DIFF;

  {
	arguments[argc++] = "-u";
  }
  arguments[argc++] = filename[0];
  arguments[argc++] = filename[1];
  arguments[argc++] = (char *) 0;

  {
    PangoLayout *layout = NULL;
    PangoRectangle logical_rect;
    layout = gtk_widget_create_pango_layout(titleD,"W");
    pango_layout_get_pixel_extents(layout, NULL, &logical_rect);
    lineH= logical_rect.height;
    g_object_unref(layout);
  }


  for (i=0; i<2; i++) skip_file[i]=fopen(filename[i],"r");
  /*gdk_window_set_cursor (diff->window, cursor);*/
  gdk_flush ();
  /* execute... */
  current_buffer_line=0;
  for (i=0; i<2; i++){
      current_line[i]=top_bufferline[i]=bottom_bufferline[i]=0;
  }
  current=NULL;
  cleanP();
  diff_section=FALSE;
  diff_is_done=0;
  tubo_object= Tubo_full  (fork_function,arguments,
	    fork_finished_function,
	    NULL,
	    operate_stdout,
	    operate_stdout,
	    NULL,15);

  while (tubo_object){
      while (gtk_events_pending()) gtk_main_iteration();
      usleep(50);
  }
  for (i=0; i<2; i++) if (skip_file[i]){
    fclose(skip_file[i]);
    skip_file[i]=NULL;
  }
  configure_event (drawA, NULL);
    
  post_diff(polygon_ids);
  /*gdk_window_set_cursor (diff->window, NULL);*/
  gtk_widget_set_size_request (drawA, OKbutton->allocation.width, 2*lineH);
  diff_is_done=1;
  /*g_message("done in %d secs",(int)(time(NULL)-t));*/
  return TRUE;
}


/*#endif*/



static int
build_tmpL (char *infile)
{
  int fileH, fileHI;
  static char *file;

  if (!infile)
    return FALSE;

  if (file)
    free (file);
  file = (char *) malloc (strlen (fileD) + strlen (infile) + 2);
  if (!file)
    xfdiff_abort (E_MALLOC);
  sprintf (file, "%s/%s", fileD, infile);

  fileHI = creat (fileI, S_IRUSR | S_IWUSR);
  if (fileHI == -1)
    xfdiff_abort (E_FILE);

  fileH = open (file, O_RDONLY);

/*     printf("preparing %s as %s\n",file,fileI);*/

  if (fileH == -1)
  {				/* a new file! */
    write (fileHI, (void *) _(empty_file), strlen (_(empty_file)));
  }
  else
  {
    ssize_t blok_s;
    void *blok;
    blok = malloc (BLOK_SIZE);
    if (!blok)
      xfdiff_abort (E_MALLOC);
    while ((blok_s = read (fileH, blok, BLOK_SIZE)) > 0)
    {
      write (fileHI, blok, blok_s);
    }
    free (blok);
  }
  close (fileH);
  close (fileHI);
  return TRUE;
}



int
main (int argc, char *argv[])
{
  int i, leftA = 1, rightA = 2;

#ifdef ENABLE_NLS
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
#endif
    textdomain(GETTEXT_PACKAGE);
#endif

  xfdiff_init ();
  gtk_set_locale();
  gtk_init (&argc, &argv);

  if (argc >= 2)
    for (i = 1; i < argc; i++)
    {
 
      if ((strcmp (argv[i], "-h") == 0) || (strcmp (argv[i], "--help") == 0))
      {
	g_print ("%s %s", _("Xfdiff version"), PACKAGE_VERSION);
	g_print ("\n");
	g_print ("\n");
	g_print ("%s\n", _("Usage: xfdiff [ -h | --help | [Left File] [Right File] ]"));
	g_print ("\n");

	g_print ("%s\n", _("-h, --help  Print this message"));
	g_print ("\n");
	g_print ("%s\n", _("Xfdiff (c) 2001-2005 Edscott Wilson Garcia, under GNU-GPL"));
	g_print ("%s\n", _("Xfce modules (c) 1999-2005 Olivier Fourdan, under GNU-GPL"));
	g_print ("\n");
	exit (1);
      }
    }
  if (argc >= leftA + 1)
  {

      filename[0] = assign (filename[0], argv[leftA]);
  }
  if (argc >= rightA + 1)
  {

      filename[1] = assign (filename[1], argv[rightA]);
  }

  signal (SIGHUP, finish);
  signal (SIGSEGV, finish);
  signal (SIGKILL, finish);
  signal (SIGTERM, finish);
 
/*printf("1...\n");*/
 /* ICON_load_theme(NULL);*/
  /*add_pixmap_directory (PACKAGE_DATA_DIR G_DIR_SEPARATOR_S  "icons" G_DIR_SEPARATOR_S "Xfce" G_DIR_SEPARATOR_S "48x48" G_DIR_SEPARATOR_S "stock");*/
  diff = create_diff_window ();
  
  cursor = gdk_cursor_new (GDK_WATCH);
  /* process command line arguments, if any first dumping invalid values */

    if ((filename[1]) && (filename[0]))
      do_diff ();

  update_titlesP ();
  gtk_main ();
  return (0);
}
