/*
 * PSIP - A lightweight GTK GUI for pjsip
 * (C) James Budiono 2011, 2015
 * License: GNU GPL Version 3 or later, please see attached gpl-3.0.txt 
 * or http://www.gnu.org/copyleft/gpl.html
 * 
 * All the back-end specific functions (pjsua) are here.
 * Other backends can probably be fitted here too. 
 */
 
#include <unistd.h>
#include <pjsua-lib/pjsua.h>
#include "psip.h"

#define BACKEND_POOL_INITIAL_SIZE	1000
#define BACKEND_POOL_INCREMENT		1000

#define HIGHEST_PORT_NUMBER		65533		/* reserve one for TLS */
#define DEFAULT_SIP_PORT		5060

#define SIP_CODE_OK				200
#define SIP_CODE_OK_END			299
#define	SIP_CODE_BUSY_HERE		486
#define SIP_INTERNAL_ERROR		500

#define PJSUA_OWN_SPEAKER_PORT	0
#define PJSUA_OWN_MIC_PORT		0

#define REGISTRATION_CONTEXT	"registration"

//to enable the use of "compact" forms
extern pj_bool_t pjsip_use_compact_form;
extern pj_bool_t pjsip_include_allow_hdr_in_dlg;
extern pj_bool_t pjmedia_add_rtpmap_for_static_pt;

typedef struct _backend_state_struct backend_state_struct;
struct _backend_state_struct {
	pjsua_config 					cfg;
	pjsua_logging_config 			log_cfg;
	pjsua_acc_config 				account_cfg;
	pjsua_media_config				media_cfg;
	pj_pool_t						*pool;

	pjsua_acc_id 					account_id;
	pjsua_acc_id					local_account_id;
	pjsua_recorder_id				recorder_id;
	pjsua_player_id					player_id;	
	
	pjsua_transport_config 			udp_cfg, tcp_cfg, tls_cfg;
	pjsua_transport_id 				udp_id, tcp_id, tls_id;
	pjsua_acc_id					udp_acc_id, tcp_acc_id, tls_acc_id;

	gboolean						terminating;
	gint							incoming_call_timeout;
	pjrpid_element					online_status_element;
	
	//tonegen stuff
	pjmedia_port					*notification_tonegen;
	pjsua_conf_port_id				notification_port;
};

backend_state_struct *backend_state;




/* ============= Utility functions ============== */

void apply_audio_preference() {
	audio_settings as;
	get_audio_preference_settings (&as);
	
	pjmedia_aud_dev_info available_audio[MAX_AUDIO_DEVICES];
	unsigned count = MAX_AUDIO_DEVICES;
	int i, current_input, current_output;
	int chosen_input, chosen_output;

	//get current devices and list all available audio devices	
	pjsua_get_snd_dev (&current_input, &current_output);
	chosen_input = current_input; chosen_output = current_output;	
	
	pjsua_enum_aud_devs (available_audio, &count);	
	if (current_input >=0) g_print("Current input device %d:%s\n",	current_input, available_audio[current_input].name);
	if (current_output >= 0) g_print("Current output device %d:%s\n", current_output, available_audio[current_output].name);
	
	if (as.input) {
		for (i=0; i < count; i++) {
			if (!g_strcmp0 (available_audio[i].name, as.input) ) {
				chosen_input = i;
				break;
			}
		}
	}
	
	if (as.output) {
		for (i=0; i < count; i++) {
			if (!g_strcmp0 (available_audio[i].name, as.output) ) {
				chosen_output = i;
				break;
			}
		}		
	}

	if (chosen_input >= 0) g_print("Chosen input device %d:%s\n", chosen_input, available_audio[chosen_input].name);
	if (chosen_output >= 0) g_print("Chosen output device %d:%s\n", chosen_output, available_audio[chosen_output].name);
	if (chosen_input != current_input || chosen_output != current_output)
		if (pjsua_set_snd_dev (chosen_input, chosen_output) != PJ_SUCCESS) {
				g_print("Failed to set audio devices, reverting to previous one.\n");
				pjsua_set_snd_dev (current_input, current_output);
		}
}

/* check whether back-end is currently registered */
gboolean backend_is_registered() {
	return (backend_state->account_id != PJSUA_INVALID_ID);
}

//determine which account to use based on sip address & logged in state
//if address is local (no @), always use local account, otherwise use registered account if any
pjsua_acc_id which_account (gchararray address) {
	if (strchr (address, '@')) {
		if (backend_is_registered()) return backend_state->account_id;
		else return backend_state->local_account_id;
	}
	return backend_state->local_account_id;
}

/* make beep sound when im is received - no lock here, caller already lock_gdk */
void im_beep(im_data *p) {
	if (gtk_toggle_button_get_active(psip_state->beep_on_im_field))
		gdk_window_beep (gtk_widget_get_window (GTK_WIDGET(p->w)));
	start_notification_sound_command ((gchararray) gtk_entry_get_text(psip_state->im_command_field));
}

/* get all currently active conference bridge ports (or slots) */
pjsua_conf_port_id *get_active_conf_ports() {
	unsigned count = pjsua_conf_get_active_ports () + 1;
	pjsua_conf_port_id *conf_ports = g_malloc0 (count * sizeof (pjsua_conf_port_id));
	
	conf_ports[count-1] = PJSUA_INVALID_ID; //end of record marker
	if (pjsua_enum_conf_ports ( conf_ports, &count) != PJ_SUCCESS) {
		g_free (conf_ports);
		return NULL;
	}
	return conf_ports;
}

/* connect a conf_id (or conf_slots) to all active sources - except itself
 * This will make the conf_id hear all the sounds coming from all others
 */
void connect_to_all_sources (pjsua_conf_port_id conf_id) {
	pjsua_conf_port_id *conf_ports = get_active_conf_ports();
	int i;
	
	if (!conf_ports) return;
	for (i=0; conf_ports[i] != PJSUA_INVALID_ID; i++) {
		if (conf_ports[i] != conf_id) pjsua_conf_connect (conf_ports[i], conf_id);
	}
	g_free (conf_ports);
}

/* connect a conf_id (or conf_slots) to all active sinks - except itself
 * This will make any sound that the conf_id makes is heard by all others
 */
void connect_to_all_sinks (pjsua_conf_port_id conf_id) {
	pjsua_conf_port_id *conf_ports = get_active_conf_ports();
	int i;
	
	if (!conf_ports) return;
	for (i=0; conf_ports[i] != PJSUA_INVALID_ID; i++) {
		if (conf_ports[i] != conf_id) pjsua_conf_connect (conf_id, conf_ports[i]);
	}
	g_free (conf_ports);
}

/*
// disconnect a conf_id (or conf_slots) from all active sources - except itself 
void disconnect_from_all_sources (pjsua_conf_port_id conf_id) {
	pjsua_conf_port_id *conf_ports = get_active_conf_ports();
	int i;
	
	if (!conf_ports) return;
	for (i=0; conf_ports[i] != PJSUA_INVALID_ID; i++) {
		if (conf_ports[i] != conf_id) pjsua_conf_disconnect (conf_ports[i], conf_id);
	}
	g_free (conf_ports);
}

// disconnect a conf_id (or conf_slots) from all active sink - except itself
void disconnect_from_all_sinks (pjsua_conf_port_id conf_id) {
	pjsua_conf_port_id *conf_ports = get_active_conf_ports();
	int i;
	
	if (!conf_ports) return;
	for (i=0; conf_ports[i] != PJSUA_INVALID_ID; i++) {
		if (conf_ports[i] != conf_id) pjsua_conf_disconnect (conf_id, conf_ports[i]);
	}
	g_free (conf_ports);
}
*/

/* ============= Internal Callbacks ============== */

/* registration call back, required to see if login is successful */
void backend_on_registration_state_change(pjsua_acc_id acc_id) {
	if (backend_state->terminating) return;
	
	pjsua_acc_info info;
	pjsua_acc_get_info (acc_id, &info);
	g_print("Registration state change: %d, text: %.*s\n", info.status, (int)info.status_text.slen, info.status_text.ptr);
	
	if (info.status >= SIP_CODE_OK && info.status <= SIP_CODE_OK_END) {
		//success
		pjsua_acc_set_online_status2 (backend_state->account_id, TRUE, &backend_state->online_status_element);
		
		lock_gdk();	
		update_registration_status (TRUE);
		gchararray s, features;
		if (backend_state->account_cfg.use_srtp == PJMEDIA_SRTP_MANDATORY) features = "secure voice";
		else features = "";
		s = g_strdup_printf ("Logged in as %.*s (%s) ", (int)backend_state->account_cfg.id.slen, backend_state->account_cfg.id.ptr, features);
		push_status (REGISTRATION_CONTEXT, s);
		g_free (s);
		unlock_gdk();			
		
	} else if (info.status > SIP_CODE_OK_END) {
		//fail
		pjsua_acc_del (backend_state->account_id);
		backend_state->account_id = PJSUA_INVALID_ID;
		
		lock_gdk();
		gchararray s = g_strdup_printf ("Login error %d.\nMessages: %.*s", info.status, (int)info.status_text.slen, info.status_text.ptr);
		show_error(s);
		g_free (s);
		update_registration_status (FALSE);
		unlock_gdk();
	}
}

/* media state change, required to connect a call to the pjsua's internal bridge */
void backend_on_call_media_state_change(pjsua_call_id call_id) {
	if (backend_state->terminating) return;
		
	pjsua_call_info ci;
	pjsua_call_get_info(call_id, &ci);
	//g_print("Media status for call %d is %d\n", call_id, ci.media_status);
	
	switch (ci.media_status) {
		case PJSUA_CALL_MEDIA_ACTIVE:
		case PJSUA_CALL_MEDIA_REMOTE_HOLD:
			g_print("Media for call %d is active\n", call_id);
			connect_to_all_sources (ci.conf_slot);
			connect_to_all_sinks (ci.conf_slot);
			break; 
		
		//according pjsua, if we've got media error we've got to terminate
		case PJSUA_CALL_MEDIA_ERROR:
			g_print ("Media error on call %d\n", call_id);
			pjsua_call_hangup (call_id, SIP_INTERNAL_ERROR, NULL, NULL);
			break;
			
		//accoding to pjsua, don't do anything on these states
		case PJSUA_CALL_MEDIA_LOCAL_HOLD:
		//	disconnect_from_all_sources (ci.conf_slot);
		//	disconnect_from_all_sinks (ci.conf_slot);		
			g_print("Media for call %d is suspended\n", call_id);
			break;		
		case PJSUA_CALL_MEDIA_NONE:
			g_print("Media for call %d is inactive\n", call_id);
			break;
	}
}


/* incoming call, required to (obviously) if we want ability to answer call */
gboolean incoming_call_timeout (gpointer p) {
	gtk_dialog_response (psip_state->confirm_dialog, GTK_RESPONSE_NO);
	return FALSE;
}
void backend_on_incoming_call (pjsua_acc_id acc_id, pjsua_call_id call_id, pjsip_rx_data *rdata) {
	if (backend_state->terminating) return;
	
	gchararray address, state;
	pjsua_call_info ci;
	pjsua_call_get_info(call_id, &ci);

	address = g_strdup_printf ("%.*s", (int)ci.remote_info.slen, ci.remote_info.ptr);	
	state = g_strdup_printf ("%.*s", (int)ci.state_text.slen, ci.state_text.ptr);
	call_data *p = g_hash_table_lookup (psip_state->active_calls, address);
	g_print ("INCOMING CALL FROM %s %d=> %s -- %p\n", address, ci.state, state, p);

	//if we are already talking to this person, reject his/her other call ...	
	if (p) {
		pjsua_call_answer(call_id, SIP_CODE_BUSY_HERE, NULL, NULL); //486 is "busy here"
		lock_gdk();
		add_call_history_entry (address, "AUTO-REJECT");
		unlock_gdk();				
		g_free (state); g_free (address);
		return;
	}

	//we've got incoming call, sound the ring
	lock_gdk();
	gchararray cmd = (gchararray) gtk_entry_get_text(psip_state->ring_command_field);
	unlock_gdk();	
	GPid pid = start_notification_sound_command (cmd);
	
	pjsua_player_id ring_player_id = PJSUA_INVALID_ID;
	if (psip_state->ring_filename) {
		pj_str_t fn;
		if (pjsua_player_create ( pj_cstr(&fn, psip_state->ring_filename), 0, &ring_player_id) == PJ_SUCCESS)
			pjsua_conf_connect (pjsua_player_get_conf_port (ring_player_id), PJSUA_OWN_SPEAKER_PORT);
	}
	
	gchararray s = g_strdup_printf("Incoming call from %.*s.\nPick up call?", (int)ci.remote_info.slen, ci.remote_info.ptr);
	g_object_set (G_OBJECT (psip_state->confirm_dialog), "text", s, NULL);
	g_free(s);
	
	
	//prompt user
	lock_gdk();
	guint max_timeout = atoi(gtk_entry_get_text(psip_state->call_timeout_field));
	guint timeout_id = gdk_threads_add_timeout_seconds (max_timeout==0 ? DEFAULT_INCOMING_TIMEOUT : max_timeout, incoming_call_timeout, NULL);
	gint result = gtk_dialog_run(psip_state->confirm_dialog);
	g_source_remove (timeout_id);
	//gint result = GTK_RESPONSE_YES;
	unlock_gdk();

	//terminate the ring, if any
	if (ring_player_id != PJSUA_INVALID_ID) pjsua_player_destroy (ring_player_id); 
	stop_notification_sound_command (pid);
	
	if (result == GTK_RESPONSE_YES) {
		lock_gdk();
		add_call_history_entry (address, "INCOMING");
		p = add_active_call (address, state, call_id);
		unlock_gdk();
		pjsua_call_set_user_data (call_id, p);
		pjsua_call_answer(call_id, SIP_CODE_OK, NULL, NULL);
	} else {
		pjsua_call_answer (call_id, SIP_CODE_BUSY_HERE, NULL, NULL);
		lock_gdk();
		add_call_history_entry (address, "REJECTED");
		unlock_gdk();		
		g_free (state); g_free (address);
	}	
}

/* call state change - required to monitor call drops, successful connection */
void backend_on_call_state_change (pjsua_call_id call_id, pjsip_event *e) {
	if (backend_state->terminating) return;

	gchararray address, state;
	pjsua_call_info ci;
	pjsua_call_get_info(call_id, &ci);

	address = g_strdup_printf ("%.*s", (int)ci.remote_info.slen, ci.remote_info.ptr);	
	state = g_strdup_printf ("%.*s", (int)ci.state_text.slen, ci.state_text.ptr);
	//call_data *p = g_hash_table_lookup (psip_state->active_calls, address);
	call_data *p = (call_data *) pjsua_call_get_user_data (call_id);

	g_print ("%s XXXXXXXXXXXXXXXXXXX %d=> %s -- %p\n", address, ci.state, state, p);
	
	//only do stuff if we know this call
	if (p && p->call_id == call_id) {
		lock_gdk();
		update_call_state (p, state);
		unlock_gdk();

		if (ci.state == PJSIP_INV_STATE_DISCONNECTED) {	
			pjsua_call_set_user_data (call_id, NULL);
			lock_gdk();						
			remove_active_call (p);
			unlock_gdk();
		}		
	}
	g_free (state); g_free (address);
}

/* buddy state change, required to detect buddy's presence status */
void backend_on_buddy_state_change (pjsua_buddy_id buddy_id) {
	if (backend_state->terminating) return;
	
	pjsua_buddy_info info;
	GtkTreeIter *iter;
	gchararray s;
	
	pjsua_buddy_get_info (buddy_id, &info);
	iter = pjsua_buddy_get_user_data (buddy_id);
	s = g_strdup_printf("%.*s", (int)info.status_text.slen, info.status_text.ptr);
	g_print ("buddy: %.*s, status: %s, %d\n", (int)info.uri.slen, info.uri.ptr, s, info.status);		

	if (iter != NULL) {
		lock_gdk();
		buddyview_set_online_status (iter, info.status == PJSUA_BUDDY_STATUS_ONLINE, s);
		unlock_gdk();
	}
	g_free(s);
}

/* pager notification - received text message */
void backend_on_pager (pjsua_call_id call_id, const pj_str_t *from, const pj_str_t *to, 
					   const pj_str_t *contact, const pj_str_t *mime_type, const pj_str_t *body) {
	if (backend_state->terminating) return;
	
	im_data *p;
	gchararray address, mime, text;
	
	//g_print("im from: %.*s, mime: %.*s, message: %.*s",
	//        from->slen, from->ptr, mime_type->slen, mime_type->ptr, body->slen, body->ptr);
	address = g_strdup_printf("%.*s", (int)from->slen, from->ptr);
	mime = g_strdup_printf("%.*s", (int)mime_type->slen, mime_type->ptr);
	text = g_strdup_printf("%.*s", (int)body->slen, body->ptr);
	//g_print ("from: %s, message: %s\n", address, text);
	
	if ( g_strcmp0 (mime, "text/plain") && mime_type->slen) {
		g_free(text);
		text = g_strdup_printf ("Unsupported data %s is received.", mime);
	}
	g_free (mime); 
	
	//get our im window
	p = g_hash_table_lookup(psip_state->active_ims, address);
	if (p) {
		//im window already exist - just insert the text
		lock_gdk();
		im_window_insert_history_text(p, address, text, p->remote_prefix_tag);
		im_beep(p);
		unlock_gdk();
		g_free (address); 	
	} else {
		//otherwise we have to create a new window
		lock_gdk();
		p = create_new_im_window (address);
		im_window_insert_history_text(p, address, text, p->remote_prefix_tag);
		im_beep(p);
		unlock_gdk();
	}
	g_free (text);
}

/* === typing indication notification ===*/
void backend_on_typing (pjsua_call_id call_id, const pj_str_t *from, const pj_str_t *to, 
					const pj_str_t *contact, pj_bool_t is_typing) {
	if (backend_state->terminating) return;	
	
	im_data *p;
	gchararray address;
	//g_print("typing indication from: %.*s - %d\n", from->slen, from->ptr, is_typing);
	address = g_strdup_printf("%.*s", (int)from->slen, from->ptr);
		
	//get our im window
	p = g_hash_table_lookup(psip_state->active_ims, address);
	if (p) {
		lock_gdk();		
		if (is_typing) gtk_label_set_text (GTK_LABEL(p->typing), "Remote user is typing ...");
		else gtk_label_set_text (GTK_LABEL(p->typing), "");
		unlock_gdk();
	} 
	//otherwise ignore - window hasn't been opened yet for this buddy
	g_free (address);
	
}	

/* ========================= Exported backend functions ========================= */


/* ========= INIT / STOP ========== */

/* start the pjsua backend */
gboolean backend_start() {
	//0. init variables
	backend_state = g_malloc0 (sizeof(backend_state_struct));
	backend_state->terminating = FALSE;
	backend_state->local_account_id = backend_state->account_id = 
	backend_state->recorder_id = backend_state->player_id = 
	backend_state->udp_id = backend_state->tcp_id = backend_state->tls_id =
	backend_state->udp_acc_id = backend_state->tcp_acc_id = backend_state->tls_acc_id =
	backend_state->notification_port = PJSUA_INVALID_ID;
	backend_state->notification_tonegen = NULL;
	
	//according to pjsip doc, we need to:
	//1. create
	if (pjsua_create() != PJ_SUCCESS) {
		g_print("pjsua_create failed.");
		return FALSE;
	}
	backend_state->pool = pjsua_pool_create ("psip", BACKEND_POOL_INITIAL_SIZE, BACKEND_POOL_INCREMENT);
	if (!backend_state->pool) {
		g_print("pjsua_pool_create failed.");
		return FALSE;
	}
	
	//1.1 setup online status struct - can only do it here, it requires pj_strdup3 which requires memory pool
	pj_bzero(&backend_state->online_status_element, sizeof(backend_state->online_status_element));
	backend_state->online_status_element.type = PJRPID_ELEMENT_TYPE_PERSON;
	backend_state->online_status_element.activity = PJRPID_ACTIVITY_UNKNOWN;
	backend_state->online_status_element.note = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->online_status_field));	
	
	//2. setup configuration
	pjsua_config_default(&backend_state->cfg);
	pjsua_logging_config_default(&backend_state->log_cfg);	
	pjsua_media_config_default(&backend_state->media_cfg);
	
	//2.1 setup call backs
	backend_state->cfg.cb.on_reg_state = backend_on_registration_state_change;
	backend_state->cfg.cb.on_call_media_state = backend_on_call_media_state_change;
	backend_state->cfg.cb.on_incoming_call = backend_on_incoming_call;
	backend_state->cfg.cb.on_call_state = backend_on_call_state_change;
	backend_state->cfg.cb.on_buddy_state = backend_on_buddy_state_change;
	backend_state->cfg.cb.on_pager = backend_on_pager;
	backend_state->cfg.cb.on_typing = backend_on_typing;
	
	//2.2 apply settings
	pjsip_use_compact_form = PJ_TRUE;
	/* do not transmit Allow header */
	pjsip_include_allow_hdr_in_dlg = PJ_FALSE;
	/* Do not include rtpmap for static payload types (<96) */
	pjmedia_add_rtpmap_for_static_pt = PJ_FALSE;
	
	
	//backend_state->media_cfg.snd_auto_close_time = 1; //commented - this is the default setting
	GtkTreeIter iter;
	gint sampling_rate;
	if (gtk_combo_box_get_active_iter (psip_state->sampling_rate_field, &iter)) {
		gtk_tree_model_get (GTK_TREE_MODEL (psip_state->freqlist), &iter, 0, &sampling_rate, GTK_END_OF_LIST);
		g_print ("Sampling rate is %d Hz\n", sampling_rate);
		backend_state->media_cfg.clock_rate = backend_state->media_cfg.snd_clock_rate = sampling_rate;
	}
	backend_state->media_cfg.quality = gtk_range_get_value(GTK_RANGE(psip_state->quality_scaler));
	
	//SRTP setting
	//default is 1 (SSL mandatory), we change to 0 (SSL not mandatory) otherwise optional SRTP won't work
	backend_state->cfg.srtp_secure_signaling = 0;
	if (gtk_toggle_button_get_active(psip_state->disable_optional_srtp_field)) {
		backend_state->cfg.use_srtp = PJMEDIA_SRTP_DISABLED;
		g_print ("WARNING: Optional SRTP is disabled\n");
	}
	else backend_state->cfg.use_srtp = PJMEDIA_SRTP_OPTIONAL;
	
	//ICE setting
	backend_state->media_cfg.enable_ice = gtk_toggle_button_get_active(psip_state->enable_ice_field);
	if (backend_state->media_cfg.enable_ice) g_print ("WARNING: ICE is enabled\n");
	
	//TURN server
	pj_str_t turn = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->turn_server_field));
	if (turn.slen) {
		g_print ("WARNING --- Using TURN server: %.*s\n", (int)turn.slen, turn.ptr);
		pj_str_t turn_user = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->turn_user_field));
		pj_str_t turn_password = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->turn_password_field));
		pj_str_t turn_realm = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->turn_realm_field));
		
		backend_state->media_cfg.turn_server = turn;
		backend_state->media_cfg.turn_auth_cred.type = PJ_STUN_AUTH_CRED_STATIC;
		backend_state->media_cfg.turn_auth_cred.data.static_cred.data_type = PJ_STUN_PASSWD_PLAIN;		
		backend_state->media_cfg.turn_auth_cred.data.static_cred.username = turn_user;
		backend_state->media_cfg.turn_auth_cred.data.static_cred.data = turn_password;
		backend_state->media_cfg.turn_auth_cred.data.static_cred.realm = turn_realm;
		if (gtk_toggle_button_get_active(psip_state->turn_use_tcp_field)) backend_state->media_cfg.turn_conn_type = PJ_TURN_TP_TCP;
		backend_state->media_cfg.enable_turn = PJ_TRUE;
	}
	
	//Disable VAD
	backend_state->media_cfg.no_vad = gtk_toggle_button_get_active(psip_state->disable_vad_field);
	if (backend_state->media_cfg.no_vad) g_print ("WARNING: no_vad is turned on\n");
	
	//Log level and backend threads
	backend_state->log_cfg.console_level = gtk_range_get_value(GTK_RANGE(psip_state->loglevel_scaler));
	backend_state->cfg.thread_cnt = BACKEND_NUM_THREADS;
	
	//Max calls
	int max_calls = atoi(gtk_entry_get_text(psip_state->max_calls_field));
	if (max_calls <= 0) max_calls = DEFAULT_MAX_CALLS;
	if (max_calls > PJSUA_MAX_CALLS) max_calls = PJSUA_MAX_CALLS;
	backend_state->cfg.max_calls = max_calls;
	g_print ("max calls: %d\n", max_calls);

	//STUN server
	pj_str_t stun = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->stun_server_field));
	if (stun.slen) {
		g_print ("WARNING --- Using STUN server: %.*s\n", (int)stun.slen, stun.ptr);
		backend_state->cfg.stun_srv_cnt = 1;
		backend_state->cfg.stun_srv[0] = stun;
	}

	//3. init pjsua library
	if (pjsua_init(&backend_state->cfg, &backend_state->log_cfg, &backend_state->media_cfg) != PJ_SUCCESS) {
		g_print("pjsua_init failed.");
		return FALSE;
	}

	//4. create the transport config & apply settings
	pjsua_transport_config_default(&backend_state->udp_cfg);
	
	//Public IP
	pj_str_t public_ip = pj_strdup3 ( backend_state->pool, gtk_entry_get_text(psip_state->public_ip_field) );
	if (public_ip.slen) { 
		g_print ("WARNING --- Using public IP / hostname: %.*s\n", (int)public_ip.slen, public_ip.ptr);
		backend_state->udp_cfg.public_addr = public_ip;
	}

	//SIP port
	int sip_port = atoi(gtk_entry_get_text(psip_state->sip_port_field));
	if (sip_port == 0 || sip_port > HIGHEST_PORT_NUMBER) sip_port = DEFAULT_SIP_PORT;
	g_print("Using port %d\n", sip_port);
	backend_state->udp_cfg.port = sip_port;

	//duplicate udp settings for tcp and tls too
	pj_memcpy(&backend_state->tcp_cfg, &backend_state->udp_cfg, sizeof(pjsua_transport_config));
	pj_memcpy(&backend_state->tls_cfg, &backend_state->udp_cfg, sizeof(pjsua_transport_config));
	backend_state->tls_cfg.port++; //same port for udp/tcp, but one higher for tls

	//5. create UDP, TCP and TLS transports, register and set online state for all
	//	 UDP is required - it's the default for many
	if (pjsua_transport_create(PJSIP_TRANSPORT_UDP, &backend_state->udp_cfg, &backend_state->udp_id) == PJ_SUCCESS) {
		if (pjsua_acc_add_local (backend_state->udp_id, PJ_TRUE, &backend_state->udp_acc_id) == PJ_SUCCESS) {
			pjsua_acc_set_online_status (backend_state->udp_acc_id, TRUE);
			backend_state->local_account_id = backend_state->udp_acc_id;
			g_print ("Local account set to UDP\n");
		} else 	g_print("pjsua_acc_add_local UDP failed\n");
	} else g_print("pjsua_transport_create UDP failed.\n"); 

	// TCP is required - some tx/rx size is too large to fit into UDP and pjsip will refuse to work on it
	// For example call hold will not work if TCP is not enabled. TCP will take over as default if successful.
	// TCP can be disabled via preferences
	if (!gtk_toggle_button_get_active(psip_state->disable_tcp_field)) {
		if (pjsua_transport_create(PJSIP_TRANSPORT_TCP, &backend_state->tcp_cfg, &backend_state->tcp_id) == PJ_SUCCESS) {
			if (pjsua_acc_add_local (backend_state->tcp_id, PJ_TRUE, &backend_state->tcp_acc_id) == PJ_SUCCESS) {
				pjsua_acc_set_online_status (backend_state->tcp_acc_id, TRUE);
				backend_state->local_account_id = backend_state->tcp_acc_id;
				g_print ("Local account is now set to TCP\n");
			} else 	g_print("pjsua_acc_add_local TCP failed\n");
		} else g_print("pjsua_transport_create TCP failed.\n"); 
	} else g_print ("TCP transport is disabled\n");

	// TLS is experimental but is required for secure transports (sips: URI etc)	
	if (pjsua_transport_create(PJSIP_TRANSPORT_TLS, &backend_state->tls_cfg, &backend_state->tls_id) == PJ_SUCCESS) {
		if (pjsua_acc_add_local (backend_state->tls_id, PJ_FALSE, &backend_state->tls_acc_id) == PJ_SUCCESS) {
			pjsua_acc_set_online_status (backend_state->tls_acc_id, TRUE);
		} else 	g_print("pjsua_acc_add_local TLS failed\n");
	} else g_print("pjsua_transport_create TLS failed.\n"); 

	//6. And finally, start it
	if (pjsua_start() != PJ_SUCCESS) {
		g_print("pjsua_start failed\n");
		return FALSE;
	}
	
	//7. init audio from config
	apply_audio_preference();
	
	//8. create other stuff - e.g. creating tonegen
	if (pjmedia_tonegen_create (backend_state->pool, 8000, 1, 160, 16, 0, 
		&backend_state->notification_tonegen) == PJ_SUCCESS) {
		pjsua_conf_add_port (backend_state->pool, backend_state->notification_tonegen, &backend_state->notification_port);
		pjsua_conf_connect (backend_state->notification_port, PJSUA_OWN_SPEAKER_PORT);
	}
	return TRUE;
}

/* stop the pjsua backend */
void backend_stop() {
	backend_state->terminating = TRUE;
	
	//close the tonegen ports
    if (backend_state->notification_tonegen && backend_state->notification_port != PJSUA_INVALID_ID) {
		pjsua_conf_remove_port(backend_state->notification_port);
		pjmedia_port_destroy(backend_state->notification_tonegen);
	}
	
	//pjsua_acc_set_online_status	(backend_state->local_account_id, FALSE);	
	pjsua_destroy();
	//pj_pool_release(backend_state->pool); //pjsua_destroy already did this, avoid double-free
	g_free (backend_state);
}

/* reconfigure - set log level & apply audio preferences */
void backend_reconfigure() {
	apply_audio_preference();
	backend_state->log_cfg.console_level = gtk_range_get_value(GTK_RANGE(psip_state->loglevel_scaler));
	pjsua_reconfigure_logging(&backend_state->log_cfg);
}



/* ========= REGISTER / DE-REGISTER ========== */

/* Perform asynchronous de-registration */
void backend_deregister() {
	if (backend_is_registered()) {
		pjsua_acc_set_online_status	(backend_state->account_id, FALSE);
		pjsua_acc_del (backend_state->account_id);
		pop_status (REGISTRATION_CONTEXT);
		g_print("Deregister complete\n");
		backend_state->account_id = PJSUA_INVALID_ID;		
	}
}


/* Perform asynchronous registration */
void backend_register() {
	if (!active_account) return; // cannot register if there is no active account
	backend_deregister(); //always de-register first in case we have an active registration
	
	pjsua_acc_config_default (&backend_state->account_cfg);
	backend_state->account_cfg.id = pj_strdup3 (backend_state->pool, active_account->sip_url);
	backend_state->account_cfg.reg_uri = pj_strdup3 (backend_state->pool, active_account->registrar);
	backend_state->account_cfg.cred_count = 1;
	backend_state->account_cfg.cred_info[0].realm = pj_strdup3 (backend_state->pool, active_account->realm);
	backend_state->account_cfg.cred_info[0].scheme = pj_str ("digest");
	backend_state->account_cfg.cred_info[0].username = pj_strdup3 (backend_state->pool, active_account->user);
	backend_state->account_cfg.cred_info[0].data_type = PJSIP_CRED_DATA_PLAIN_PASSWD;
	backend_state->account_cfg.cred_info[0].data = pj_strdup3 (backend_state->pool, active_account->password);
	backend_state->account_cfg.srtp_secure_signaling = 0; 	
	
	if (gtk_toggle_button_get_active(psip_state->disable_optional_srtp_field)) 
		backend_state->cfg.use_srtp = PJMEDIA_SRTP_DISABLED;
	else backend_state->cfg.use_srtp = PJMEDIA_SRTP_OPTIONAL;
	if (active_account->enable_srtp)
		backend_state->account_cfg.use_srtp = PJMEDIA_SRTP_MANDATORY;
	pj_str_t proxy = pj_strdup3 (backend_state->pool, active_account->proxy);
	
	if (proxy.slen) {
		backend_state->account_cfg.proxy[0] = proxy;
		backend_state->account_cfg.proxy_cnt = 1;
	}
	
	pjsua_acc_add(&backend_state->account_cfg, PJ_FALSE, &backend_state->account_id);
}




/* ========= CALL FUNCTIONS ========== */

/* start a voice call */
gboolean backend_make_call(call_data *p) {	
	g_print("Calling %s...\n", p->address);
	pj_str_t uri;
	gboolean result;
	result = (pjsua_call_make_call(which_account (p->address), pj_cstr(&uri, p->address), 0, p, NULL, &p->call_id) == PJ_SUCCESS);
	if (!result && p->call_id != PJSUA_INVALID_ID) 
			pjsua_call_set_user_data (p->call_id, NULL);
	return result;
}

/* drop the voice call - hang up */
void backend_drop_call(call_data *p) {	
	g_print("Dropping call for %s\n", p->address);
	pjsua_call_hangup (p->call_id, SIP_CODE_OK, NULL, NULL);
}

/* adjust local mic volume, 0=mute */
void backend_adjust_local_mic_level (double level) {
	pjsua_conf_adjust_rx_level (0, level);
	//g_print("local volume set to: %f\n", level);
}

/* adjust remote speaker volume, 0=mute */
void backend_adjust_remote_speaker_level (call_data *p, double level) {
	pjsua_call_info ci;
	
	p->level = level;	
	pjsua_call_get_info (p->call_id, &ci);
	pjsua_conf_adjust_rx_level (ci.conf_slot, level);
	//g_print("remote speaker for %d set to: %f\n", p->call_id, level);
}


/* send dtmf digits, required for conference lines etc that requires digit sending */
void backend_send_dtmf(call_data *p, gchararray digit) {
	pj_str_t ddd;
	pjsua_call_dial_dtmf (p->call_id, pj_cstr(&ddd, digit));
}

/* hold call */
void backend_hold_call (call_data *p) {
	g_print ("Holding call %d\n", p->call_id);
	pjsua_call_set_hold (p->call_id, NULL);
}

/* resume call  */
void backend_unhold_call (call_data *p) {
	g_print ("Re-inviting call %d\n", p->call_id);
	pjsua_call_reinvite (p->call_id, PJSUA_CALL_UNHOLD, NULL);
}

/* get call statistic */
gboolean backend_get_call_statistic (call_data *p, gchararray buffer) {
	return (pjsua_call_dump (p->call_id, TRUE, buffer, STATISTIC_BUFFER_SIZE, "") == PJ_SUCCESS);
}



/* ========= PRESENCE FUNCTIONS ========== */

/* register an address to pjsua so that we can monitor its status */
gint backend_register_buddy (gchararray address, GtkTreeIter *iter) {
	pjsua_buddy_config cfg;
	gint buddy_id;

	pjsua_buddy_config_default (&cfg);
	cfg.uri = pj_strdup3 (backend_state->pool, address);
	cfg.subscribe = TRUE;
	cfg.user_data = iter;
	pjsua_buddy_add (&cfg, &buddy_id);
	return buddy_id;
}

/* deregister a buddy from pjsua */
GtkTreeIter *backend_deregister_buddy (gint buddy_id) {
	gpointer p=NULL;
	if (pjsua_buddy_is_valid(buddy_id)) {
		p = (GtkTreeIter *) pjsua_buddy_get_user_data (buddy_id);
		pjsua_buddy_set_user_data (buddy_id, NULL);
		pjsua_buddy_del (buddy_id);
	}
	return p;
}

/* refresh presence status of all buddies */
void backend_refresh_presence(gint buddy_id) {
	if (buddy_id != PJSUA_INVALID_ID)
		pjsua_buddy_update_pres (buddy_id);
}

/* set our online state */
void backend_state_set_online_status() {
	backend_state->online_status_element.note = pj_strdup3 (backend_state->pool, gtk_entry_get_text(psip_state->online_status_field));	
	if (backend_is_registered()) {
		pjsua_acc_set_online_status2 (backend_state->account_id, TRUE, &backend_state->online_status_element);
	}
}

/* ========= INSTANT MESSAGING FUNCTIONS ========== */

/* send message */
void backend_send_message(gchararray address, gchararray s) {
	pj_str_t to;
	pj_str_t content;
	pjsua_im_send (which_account (address), pj_cstr (&to, address), NULL, pj_cstr (&content, s), 
					NULL, NULL);
}
void backend_typing_indicator (gchararray address, gboolean is_typing) {
	pj_str_t to;	
	pjsua_im_typing (which_account (address), pj_cstr (&to, address), is_typing, NULL);
}


/* ========= RECORD / PLAY FUNCTIONS ========== */

gboolean backend_start_recording (gchararray filename) {
	if (backend_state->recorder_id != PJSUA_INVALID_ID) return FALSE; //can't record twice
	pj_str_t fn;
	gboolean result = (pjsua_recorder_create ( pj_cstr(&fn, filename), 0, NULL, 0, 0, &backend_state->recorder_id) == PJ_SUCCESS);
	//if (result) pjsua_conf_connect (PJSUA_OWN_MIC_PORT, pjsua_recorder_get_conf_port (backend_state->recorder_id));
	if (result) connect_to_all_sources (pjsua_recorder_get_conf_port (backend_state->recorder_id));
	return result;
}

void backend_stop_recording() {
	if (backend_state->recorder_id == PJSUA_INVALID_ID) return;
	pjsua_recorder_destroy (backend_state->recorder_id); 
	backend_state->recorder_id = PJSUA_INVALID_ID;
}

gboolean backend_start_playing (gchararray filename) {
	if (backend_state->player_id != PJSUA_INVALID_ID) return FALSE; //can't play twice
	pj_str_t fn;
	gboolean result = (pjsua_player_create ( pj_cstr(&fn, filename), 0, &backend_state->player_id) == PJ_SUCCESS);
	if (result) connect_to_all_sinks (pjsua_player_get_conf_port (backend_state->player_id));
	return result;
}

void backend_stop_playing() {
	if (backend_state->player_id == PJSUA_INVALID_ID) return;
	pjsua_player_destroy (backend_state->player_id); 
	backend_state->player_id = PJSUA_INVALID_ID;	
}

char *notification_types[BACKEND_NOTIFICATION_END] = {"abcabcabcabc", "cacacacaca" };
void backend_audible_notification (unsigned type) {
    if (backend_state->notification_tonegen && backend_state->notification_port != PJSUA_INVALID_ID &&
		type < BACKEND_NOTIFICATION_END) {
		char *digits = notification_types[type];
		int i, count = strlen (digits);
		pjmedia_tone_digit pj_digit;

		pj_digit.on_msec = 100;
		pj_digit.off_msec = 50;
		pj_digit.volume = 0;	
		for (i=0; i<count; i++) {
			pj_digit.digit = digits[i];
			pjmedia_tonegen_play_digits	(backend_state->notification_tonegen, 1, &pj_digit, 0);
		}
	}
}


/* ========= PREFERENCES ========== */
//currently will only set the preferences settings dialog
void backend_prepare_preferences_dialog() {
	unsigned count=MAX_AUDIO_DEVICES;
	int i, current_input, current_output;
	int input_index, output_index;
	pjmedia_aud_dev_info audio[MAX_AUDIO_DEVICES];

	//initially hide & unselect all devices
	for (i=0; i<MAX_AUDIO_DEVICES; i++) {
		gtk_widget_hide (GTK_WIDGET(psip_state->audio_inputs[i]));
		gtk_widget_hide (GTK_WIDGET(psip_state->audio_outputs[i]));
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(psip_state->audio_inputs[i]), FALSE);
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(psip_state->audio_outputs[i]), FALSE);
	}
	
	//get the currently used devices
	pjsua_get_snd_dev (&current_input, &current_output);
	
	//list all available audio devices and add that to preferences
	pjsua_enum_aud_devs (audio, &count);
	input_index = output_index = 0;
	for (i=0; i<count; i++) {
		//g_print("device: %d\nName: %s (driver: %s)\nI/O count: %d/%d\n",
		//		i,audio[i].name, audio[i].driver, audio[i].input_count, audio[i].output_count);
		if (audio[i].input_count > 0) {
			gtk_button_set_label (GTK_BUTTON(psip_state->audio_inputs[input_index]), audio[i].name);
			gtk_widget_show (GTK_WIDGET(psip_state->audio_inputs[input_index]));
			if (i == current_input)
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(psip_state->audio_inputs[input_index]), TRUE);
			input_index++;
		}
		if (audio[i].output_count > 0) {
			gtk_button_set_label (GTK_BUTTON(psip_state->audio_outputs[output_index]), audio[i].name);
			gtk_widget_show (GTK_WIDGET(psip_state->audio_outputs[output_index]));
			if (i == current_output)
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(psip_state->audio_outputs[output_index]), TRUE);
			output_index++;
		}
	}	
}

