/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* Generated by Together */

package ghidra.program.model.listing;

import java.util.ConcurrentModificationException;

import ghidra.program.model.address.Address;
import ghidra.program.model.lang.Register;
import ghidra.program.model.mem.MemBuffer;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.scalar.Scalar;
import ghidra.program.model.symbol.*;
import ghidra.program.model.util.PropertySet;

/**
 * Interface common to both instructions and data.
 */
public interface CodeUnit extends MemBuffer, PropertySet {

	/**
	 * Indicator for a mnemonic (versus an operand).
	 */
	public final static int MNEMONIC = -1;

	public static final int NO_COMMENT = -1;
	/**
	 * comment type for end of line
	 */
	public static final int EOL_COMMENT = 0;
	/**
	 * comment type that goes before a code unit
	 */
	public static final int PRE_COMMENT = 1;
	/**
	 * comment type that follows after a code unit
	 */
	public static final int POST_COMMENT = 2;
	/**
	 * Property name for plate comment type
	 */
	public static final int PLATE_COMMENT = 3;
	/**
	 * Property name for repeatable comment type
	 */
	public static final int REPEATABLE_COMMENT = 4;

//	/**
//	 * Property type for fall through property
//	 */
//	public static final int FALL_THROUGH = 4;

	/**
	 * Any comment property.
	 */
	public static final String COMMENT_PROPERTY = "COMMENT__GHIDRA_";
	/** 
	 * Property name for vertical space formatting
	 */
	public static final String SPACE_PROPERTY = "Space";
	/**
	 * Property name for code units that are instructions
	 */
	public static final String INSTRUCTION_PROPERTY = "INSTRUCTION__GHIDRA_";
	/**
	 * Property name for code units that are defined data
	 */
	public static final String DEFINED_DATA_PROPERTY = "DEFINED_DATA__GHIDRA_";

	/**
	 * Get the string representation of the starting address for
	 * this code unit.
	 * @param showBlockName true if the string should include the memory block name
	 * @param pad if true, the address will be padded with leading zeros.  Even if pad is
	 * false, the string will be padded to make the address string contain at least 4 digits.
	 * @return string representation of address
	 */
	public String getAddressString(boolean showBlockName, boolean pad);

	/**
	 * Get the label for this code unit.
	 */
	public String getLabel();

	/**
	 * Get the Symbols for this code unit.
	 * @throws ConcurrentModificationException if this object is no
	 * longer valid.
	 */
	public Symbol[] getSymbols();

	/**
	 * Get the Primary Symbol for this code unit.
	 * @throws ConcurrentModificationException if this object is no
	 * longer valid.
	 */
	public Symbol getPrimarySymbol();

	/**
	 * Get the starting address for this code unit.
	 */
	public Address getMinAddress();

	/**
	 * Get the ending address for this code unit.
	 */
	public Address getMaxAddress();

	/**
	 * Get the mnemonic for this code unit, e.g., MOV, JMP
	 */
	public String getMnemonicString();

	/**
	 * Get the comment for the given type
	 *
	 * @param commentType either EOL_COMMENT, PRE_COMMENT, 
	 * POST_COMMENT, or REPEATABLE_COMMENT
	 * @return the comment string of the appropriate type or null if no comment of
	 * that type exists for this codeunit
	 * @throws IllegalArgumentException if type is not one of the
	 * three types of comments supported
	 */
	public String getComment(int commentType);

	/**
	 * Get the comment for the given type and parse it into an array of strings
	 * such that each line is its own string.
	 *
	 * @param commentType either EOL_COMMENT, PRE_COMMENT, 
	 * POST_COMMENT, or REPEATABLE_COMMENT
	 * @return an array of strings where each item in the array is a line of text
	 * in the comment.  If there is no comment of the requested type, an empty array
	 * is returned.
	 * @throws IllegalArgumentException if type is not one of the
	 * three types of comments supported
	 */
	public String[] getCommentAsArray(int commentType);

	/**
	 * Set the comment for the given comment type.  Passing <code>null</code> clears the comment
	 *
	 * @param commentType either EOL_COMMENT, PRE_COMMENT, 
	 * POST_COMMENT, or REPEATABLE_COMMENT
	 * @param comment comment for code unit; null clears the comment
	 * 
	 * @throws IllegalArgumentException if type is not one of the
	 * three types of comments supported
	 */
	public void setComment(int commentType, String comment);

	/**
	 * Set the comment (with each line in its own string) for the given comment type
	 *
	 * @param commentType either EOL_COMMENT, PRE_COMMENT, 
	 * POST_COMMENT, or REPEATABLE_COMMENT
	 * @param comment an array of strings where each string is a single line of the comment.
	 * @throws IllegalArgumentException if type is not one of the
	 * three types of comments supported
	 */
	public void setCommentAsArray(int commentType, String[] comment);

	/**
	 * Return true if the given CodeUnit follows
	 * directly after this code unit.
	 * @param codeUnit the codeUnit being tested to see if it follows this codeUnit.
	 */
	public boolean isSuccessor(CodeUnit codeUnit);

	/**
	 * Get length of this code unit.
	 */
	public int getLength();

	/**
	 * Get the bytes that make up this code unit.
	 * @return an array of bytes that are in memory at the codeunits address.  The
	 * array length is the same as the codeUnits length
	 * @throws MemoryAccessException if the full number of bytes could not be read.
	 */
	public byte[] getBytes() throws MemoryAccessException;

	/**
	 * Copies max(buffer.length, code unit length) bytes into buffer starting at location offset in buffer.
	 * @param buffer byte array to copy into
	 * @param bufferOffset offset in byte array the copy will start
	 * @throws MemoryAccessException if the full number of bytes could not be read.
	 */
	public void getBytesInCodeUnit(byte[] buffer, int bufferOffset) throws MemoryAccessException;

	/**
	 * Returns true if address is contained in the range of this codeUnit
	 * @param testAddr the address to test.
	 */
	public boolean contains(Address testAddr);

	/**
	   * Compares the given address to the address range of this node.
	   *
	   * @param addr address to compare.
	   * @return a negative integer if this addr is greater than the maximum range address
	   *         zero if addr is in the range
	   *         a positive integer if addr is less than minimum range address
	   */
	public int compareTo(Address addr);

	/**
	 * Add a reference to the mnemonic for this code unit.
	 * @param refAddr address to add as a reference.
	 * @param refType the type of reference to add.
	 * @param sourceType the source of this reference
	*/
	public void addMnemonicReference(Address refAddr, RefType refType, SourceType sourceType);

	/**
	 * Remove a reference to the mnemonic for this code unit.
	 * @param refAddr the address to remove as a reference.
	 */
	public void removeMnemonicReference(Address refAddr);

	/**
	* Get references for the mnemonic for this code unit.
	* @return an array of memory references. A zero length array will be 
	* returned if there are no references for the mnemonic.
	*/
	public Reference[] getMnemonicReferences();

	/**
	 * Get the references for the operand index.
	 * @param index operand index (0 is the first operand)
	 */
	public Reference[] getOperandReferences(int index);

	/**
	 * Get the primary reference for the operand index.
	 * @param index operand index (0 is the first operand)
	 */
	public Reference getPrimaryReference(int index);

	/**
	 * Add a memory reference to the operand at the given index.
	 * @param index operand index
	 * @param refAddr reference address
	 * @param type the reference type to be added.
	 * @param sourceType the source of this reference
	 */
	public void addOperandReference(int index, Address refAddr, RefType type,
			SourceType sourceType);

	/**
	 * Remove a reference to the operand.
	 * @param index operand index
	 * @param refAddr address referencing the operand
	 */
	public void removeOperandReference(int index, Address refAddr);

	/**
	 * Get ALL memory references FROM this code unit.
	 * @return an array of memory references from this codeUnit or an empty array
	 * if there are no references.
	 */
	public Reference[] getReferencesFrom();

	/**
	 * Get an iterator over all references TO this code unit.
	 */
	public ReferenceIterator getReferenceIteratorTo();

	/**
	 * Returns the program that generated this CodeUnit.
	 */
	public Program getProgram();

	//////////////////////////////////////////////////////////////////////////
	/**
	 * Gets the external reference (if any) at the opIndex
	 * @param opIndex the operand index to look for external references
	 * @return the external reference at the operand or null if none exists.
	 */
	public ExternalReference getExternalReference(int opIndex);

	/**
	 * Remove external reference (if any) at the given opIndex
	 * opIndex the index of the operand from which to remove any external reference.
	 */
	public void removeExternalReference(int opIndex);

	/**
	 * Sets a memory reference to be the primary reference at its
	 * address/opIndex location. The primary reference is the one that
	 * is used in the getOperandRepresentation() method.
	 * @param ref the reference to be set as primary.
	 */
	public void setPrimaryMemoryReference(Reference ref);

	/**
	 * Sets a stack reference at the <code>offset</code> on the
	 * specified operand index, which effectively substitutes the previous
	 * operation interpretation
	 * <br>
	 * <i>NOTE: If another reference was previously set on the
	 * operand, then it will be replaced with this stack
	 * reference</i>
	 *
	 * @param opIndex      the index of the operand to set this stack reference
	 * @param offset       the (+/-) offset from stack base address
	 * @param sourceType   the source of this reference
	 * @param refType      type of reference, RefType.READ,WRITE,PTR...
	 */
	public void setStackReference(int opIndex, int offset, SourceType sourceType, RefType refType);

	/**
	 * Sets a register reference at the <code>offset</code> on the
	 * specified operand index, which effectively substitutes the previous
	 * operation interpretation
	 * <br>
	 * <i>NOTE: If another reference was previously set on the
	 * operand, then it will be replaced with this register
	 * reference</i>
	 *
	 * @param opIndex      the index of the operand to set this register reference
	 * @param reg		   a register
	 * @param sourceType   the source of this reference
	 * @param refType      type of reference, RefType.READ,WRITE,PTR...
	 */
	public void setRegisterReference(int opIndex, Register reg, SourceType sourceType,
			RefType refType);

	/**
	 * Get the number of operands for this code unit.
	 */
	public int getNumOperands();

	/**
	 * Get the Address for the given operand index if one exists.  Data
	 * objects have one operand (the value).
	 * @param opIndex index of the operand.
	 * @return An addres if the operand represents a fully qualified
	 * address (given the context), or if the operand is a Scalar treated
	 * as an address. Null is returned if no address or scalar exists on that 
	 * operand.
	 */
	public Address getAddress(int opIndex);

	/**
	 * Returns the scalar at the given operand index.  Data objects have
	 * one operand (the value).
	 * @param opIndex index of the operand.
	 * @return the scalar at the given operand index or null if no
	 * scalar exists at that index.
	 */
	public Scalar getScalar(int opIndex);

}
