import subprocess
from PySide6.QtGui import QIcon
from PySide6.QtWidgets import (
    QProgressDialog, QPushButton, QVBoxLayout, QDialog, QLabel, QComboBox, QHBoxLayout, QMessageBox
)
from PySide6.QtCore import Qt

from src.controller.switch_validate_controller import SwitchValidateController
from src.path_resolver import PathResolver
from src.controller.terminal_controller import TerminalController


class DockerController:
    """
    Provides static methods to manage Docker containers and images.
    """
    def __init__(self, context):
        self.tools_list = context.ui.tools_list
        self.tools = context.tools
        self.tool_cards = context.ui.tool_cards
        self.switch_validator = SwitchValidateController()
        self.docker_status_label = context.ui.docker_status_label
        self.docker_status_icon = context.ui.docker_status_icon

    @staticmethod
    def run_container(context):
        """
        Run the Docker container for the selected tool with specified flags.
        """
        current_item = context.ui.tools_list.currentItem()
        if current_item is not None and current_item.parent() is not None:
            parent = current_item.parent()
            current_index = parent.indexOfChild(current_item)
            current_tool = context.tools[current_index]
            current_card = context.ui.tool_cards[current_index]

            docker_image = current_tool["docker_image"]
            default_port = current_tool.get("default_port")
            flags = current_card.get_params()
            tool_name = current_tool["name"].lower()

            switch_validator = SwitchValidateController()
            if not switch_validator.validate_switch(flags, tool_name):
                msg = QMessageBox()
                msg.setIcon(QMessageBox.Icon.Critical)
                msg.setText("Invalid flags")
                msg.setWindowTitle("Error")
                msg.exec()
                return

            command = f'docker run --rm -it {default_port if default_port else ""} {docker_image} {flags}'
            TerminalController.run(command)
        else:
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Icon.Warning)
            msg.setText("Please select a valid tool.")
            msg.setInformativeText(
                "You need to select a tool from the list to run the Docker container.")
            msg.setWindowTitle("Warning")
            msg.setStandardButtons(QMessageBox.StandardButton.Ok)
            msg.exec()

    @staticmethod
    def check_container_status(tool, tool_item, download_status_label, run_button):
        """
        Check if Docker is running and if the Docker image is downloaded,
        then update the status label and run button.
        """
        docker_image = tool["docker_image"]

        try:
            subprocess.run(['docker', 'info'],
                           stdout=subprocess.PIPE,
                           stderr=subprocess.PIPE,
                           check=True)
        except subprocess.CalledProcessError:
            return

        try:
            result = subprocess.run(['docker', 'images', '-q', docker_image],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            if result.stdout.strip():
                tool_item.setIcon(0, QIcon(PathResolver.resource_path('resources/assets/check.png')))
            else:
                tool_item.setIcon(0, QIcon(PathResolver.resource_path('resources/assets/cross.png')))
        except subprocess.CalledProcessError as e:
            download_status_label.setText(
                f"Image download verification is not possible\n {e}")
            tool_item.setIcon(0, QIcon(PathResolver.resource_path('resources/assets/cross.png')))

        try:
            result = subprocess.run(['docker', 'ps', '-q', '-f', f'ancestor={docker_image}'],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            container_ids = result.stdout.strip().decode('utf-8').split('\n')
            if container_ids and container_ids[0]:
                if len(container_ids) > 1:
                    run_button.setText("Containers Running")
                else:
                    run_button.setText("Container Running")
            else:
                run_button.setText("Run Container")
        except subprocess.CalledProcessError as e:
            run_button.setText(
                f"Container status check is not possible\n {e}")

    @staticmethod
    def stop_container(tool, parent_widget):
        """
        Stop the Docker container associated with the specified tool by displaying a progress dialog.
        """
        docker_image = tool["docker_image"]
        progress_dialog = QProgressDialog("Stopping container...", None, 0, 100, parent_widget)
        progress_dialog.setWindowModality(Qt.WindowModality.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setAutoReset(False)
        progress_dialog.show()

        try:
            result = subprocess.run(['docker', 'ps', '-q', '-f', f'ancestor={docker_image}'],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            container_ids = result.stdout.strip().decode('utf-8').split('\n')
            progress_dialog.setValue(30)

            if container_ids and container_ids[0]:
                container_id = None
                if len(container_ids) > 1:
                    dialog = QDialog(parent_widget)
                    dialog.setWindowTitle("Select Container")

                    layout = QVBoxLayout()

                    label = QLabel("Choose container ID to stop:")
                    layout.addWidget(label)

                    combo_box = QComboBox()
                    combo_box.addItems([f"{cid}" for cid in container_ids])
                    layout.addWidget(combo_box)

                    button_layout = QHBoxLayout()

                    stop_all_button = QPushButton("Stop All Containers")
                    ok_button = QPushButton("Ok")
                    cancel_button = QPushButton("Cancel")

                    button_layout.addWidget(stop_all_button)
                    button_layout.addWidget(cancel_button)
                    button_layout.addWidget(ok_button)

                    layout.addLayout(button_layout)

                    def handle_stop_all():
                        dialog.accept()
                        DockerController.stop_all_containers(tool, parent_widget)

                    stop_all_button.clicked.connect(handle_stop_all)
                    ok_button.clicked.connect(dialog.accept)
                    cancel_button.clicked.connect(dialog.reject)

                    dialog.setLayout(layout)
                    dialog.exec()

                    if dialog.result() == QDialog.DialogCode.Accepted:
                        item = combo_box.currentText()
                        parts = item.split(': ')
                        if len(parts) > 0:
                            container_id = parts[0]
                        else:
                            print(f"Unexpected format for selected item: {item}")
                else:
                    container_id = container_ids[0]

                if container_id:
                    stop_process = subprocess.run(['docker', 'stop', container_id],
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.PIPE,
                                                  check=True)
                    progress_dialog.setValue(50)
                    if stop_process.returncode == 0:
                        print(f"Container {container_id} stopped successfully.")
                    else:
                        print(f"Failed to stop container {container_id}.")
                    progress_dialog.setValue(100)
                else:
                    print("No container selected to stop.")
            else:
                print("No running container found for this tool.")
        except subprocess.CalledProcessError as e:
            print(f"Failed to stop container: {e}")
        finally:
            progress_dialog.close()

    @staticmethod
    def stop_all_containers(tool, parent_widget):
        """
        Stop all Docker containers associated with the specified tool by displaying a progress dialog.
        """
        docker_image = tool["docker_image"]
        progress_dialog = QProgressDialog("Stopping containers...", None, 0, 100, parent_widget)
        progress_dialog.setWindowModality(Qt.WindowModality.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setAutoReset(False)
        progress_dialog.show()

        try:
            result = subprocess.run(['docker', 'ps', '-q', '-f', f'ancestor={docker_image}'],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            container_ids = result.stdout.strip().decode('utf-8').split('\n')
            progress_dialog.setValue(30)

            if container_ids and container_ids[0]:
                for i, container_id in enumerate(container_ids):
                    stop_process = subprocess.run(['docker', 'stop', container_id],
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.PIPE,
                                                  check=True)
                    progress_dialog.setValue(30 + int(70 * (i + 1) / len(container_ids)))
                    if stop_process.returncode == 0:
                        print(f"Container {container_id} stopped successfully.")
                    else:
                        print(f"Failed to stop container {container_id}.")
                progress_dialog.setValue(100)
            else:
                print("No running container found for this tool.")
        except subprocess.CalledProcessError as e:
            print(f"Failed to stop containers: {e}")
        finally:
            progress_dialog.close()
