/*
 * Copyright (c) 2018-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"
#include "../../bfloat.hpp"
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_bf16fp32_mmla_6VLx2(const bfloat16 *A, int lda, const bfloat16 *B, float *C, int ldc, int M, int N, int K, const float *bias, Activation act, bool accumulate) {
    const int K_stride = ((K + 3) / 4) * 4;
    const long loops_count = ((K + 8) / 16) - 1;
    K -= loops_count * 16;
    const long regs_count = (K / 8) - 1;
    K -= (regs_count + 1) * 8;
    const long leftovers = K;
    const long blocks_count = (K + 3) / 4;
    float nullbias[192];
    if (!accumulate && !bias) {
        memset(nullbias, 0, (3 * get_vector_length<float>() * sizeof(float)));
    }
    float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
    float maxval =   static_cast<float>(std::numeric_limits<float>::infinity());
    const float * const minptr = &minval;
    const float * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    int rows_to_compute;

    for (int y=0; y<M; y+=rows_to_compute) {
        const bfloat16 * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(bfloat16);

        float *c_ptr0 = C + (y * ldc);

        rows_to_compute = M-y;
        if (rows_to_compute > 4) {
            if (rows_to_compute % 4) {
                rows_to_compute = 4 - 1;
            } else {
                rows_to_compute = 4;
            }
        }

        for (int x0=0; x0<N; x0+=(3 * get_vector_length<float>())) {
            const long width = std::min((unsigned long)N-x0, (3 * get_vector_length<float>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = blocks_count;
            const bfloat16 *a_ptr0 = a_ptr0_base;
            const bfloat16 *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(float);
            const float *biasptr = bias ? bias+x0 : nullbias;

            switch(rows_to_compute) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z1.h, #0\n"
                        "ld1w z19.s, p0/z, [%[biasptr]]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "zip1 z20.s, z19.s, z19.s\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z21.s, z19.s, z19.s\n"
                        "ld1w z19.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "zip1 z22.s, z19.s, z19.s\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "zip2 z23.s, z19.s, z19.s\n"
                        "ld1w z19.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z24.s, z19.s, z19.s\n"
                        "zip2 z25.s, z19.s, z19.s\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z18.s, #0\n"
                        "ld1w z17.s, p0/z, [%[c_ptr0]]\n"
                        "mov z1.h, #0\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "zip1 z20.s, z17.s, z18.s\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z21.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z18.s, #0\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z22.s, z17.s, z18.s\n"
                        "zip2 z23.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "mov z18.s, #0\n"
                        "zip1 z24.s, z17.s, z18.s\n"
                        "zip2 z25.s, z17.s, z18.s\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z3.h, #0\n"
                        "subs %[loops], %[loops], #0x1\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "mov z1.h, #0\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z3.h, #0\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z1.h, #0\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #12\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z3.h, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "5:\n"
                        "ld1rw z18.s, p7/z, [%[minptr]]\n"
                        "ld1rw z19.s, p7/z, [%[maxptr]]\n"
                        "fmax z20.s, p7/m, z20.s, z18.s\n"
                        "fmax z21.s, p7/m, z21.s, z18.s\n"
                        "fmax z22.s, p7/m, z22.s, z18.s\n"
                        "fmax z23.s, p7/m, z23.s, z18.s\n"
                        "fmin z20.s, p7/m, z20.s, z19.s\n"
                        "fmin z21.s, p7/m, z21.s, z19.s\n"
                        "fmin z22.s, p7/m, z22.s, z19.s\n"
                        "fmin z23.s, p7/m, z23.s, z19.s\n"
                        "fmax z24.s, p7/m, z24.s, z18.s\n"
                        "uzp1 z0.s, z20.s, z21.s\n"
                        "fmax z25.s, p7/m, z25.s, z18.s\n"
                        "uzp1 z1.s, z22.s, z23.s\n"
                        "fmin z24.s, p7/m, z24.s, z19.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z25.s, p7/m, z25.s, z19.s\n"
                        "st1w z1.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "uzp1 z2.s, z24.s, z25.s\n"
                        "st1w z2.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z19.s, p0/z, [%[biasptr]]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z19.s, z19.s\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "zip2 z21.s, z19.s, z19.s\n"
                        "ld1w z19.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip1 z22.s, z19.s, z19.s\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "zip2 z23.s, z19.s, z19.s\n"
                        "ld1w z19.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z24.s, z19.s, z19.s\n"
                        "zip2 z25.s, z19.s, z19.s\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z17.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z18.s, p0/z, [c_ptr1]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z17.s, z18.s\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "zip2 z21.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1w z18.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "zip1 z22.s, z17.s, z18.s\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "zip2 z23.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z18.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z24.s, z17.s, z18.s\n"
                        "zip2 z25.s, z17.s, z18.s\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #12\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1rqh z2.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z3.h, p6/z, [a_ptr1]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "5:\n"
                        "ld1rw z18.s, p7/z, [%[minptr]]\n"
                        "ld1rw z19.s, p7/z, [%[maxptr]]\n"
                        "fmax z20.s, p7/m, z20.s, z18.s\n"
                        "fmax z21.s, p7/m, z21.s, z18.s\n"
                        "fmax z22.s, p7/m, z22.s, z18.s\n"
                        "fmax z23.s, p7/m, z23.s, z18.s\n"
                        "fmin z20.s, p7/m, z20.s, z19.s\n"
                        "fmin z21.s, p7/m, z21.s, z19.s\n"
                        "fmin z22.s, p7/m, z22.s, z19.s\n"
                        "fmin z23.s, p7/m, z23.s, z19.s\n"
                        "fmax z24.s, p7/m, z24.s, z18.s\n"
                        "uzp1 z0.s, z20.s, z21.s\n"
                        "uzp2 z1.s, z20.s, z21.s\n"
                        "uzp1 z2.s, z22.s, z23.s\n"
                        "uzp2 z3.s, z22.s, z23.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z24.s, p7/m, z24.s, z19.s\n"
                        "fmax z25.s, p7/m, z25.s, z18.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmin z25.s, p7/m, z25.s, z19.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "uzp1 z4.s, z24.s, z25.s\n"
                        "uzp2 z5.s, z24.s, z25.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z4.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #3\n"
                        "st1w z5.s, p2, [c_ptr1, #2, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "mov z3.h, #0\n"
                        "ld1w z19.s, p0/z, [%[biasptr]]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z19.s, z19.s\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "zip2 z21.s, z19.s, z19.s\n"
                        "ld1w z19.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z22.s, z19.s, z19.s\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "zip2 z23.s, z19.s, z19.s\n"
                        "ld1w z19.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z26.d, z20.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z27.d, z21.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "zip1 z24.s, z19.s, z19.s\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip2 z25.s, z19.s, z19.s\n"
                        "mov z28.d, z22.d\n"
                        "mov z29.d, z23.d\n"
                        "mov z30.d, z24.d\n"
                        "mov z31.d, z25.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "mov z3.h, #0\n"
                        "ld1w z17.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z18.s, p0/z, [c_ptr1]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z17.s, z18.s\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "zip2 z21.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1w z18.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip1 z22.s, z17.s, z18.s\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "zip2 z23.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z18.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z24.s, z17.s, z18.s\n"
                        "zip2 z25.s, z17.s, z18.s\n"
                        "ld1w z17.s, p0/z, [c_ptr2]\n"
                        "mov z18.s, #0\n"
                        "zip1 z26.s, z17.s, z18.s\n"
                        "zip2 z27.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "mov z18.s, #0\n"
                        "zip1 z28.s, z17.s, z18.s\n"
                        "zip2 z29.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "mov z18.s, #0\n"
                        "zip1 z30.s, z17.s, z18.s\n"
                        "zip2 z31.s, z17.s, z18.s\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p7/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z5.h, #0\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z3.h, #0\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p7/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z5.h, #0\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "mov z3.h, #0\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #12\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p6/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p6/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z5.h, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "5:\n"
                        "ld1rw z18.s, p7/z, [%[minptr]]\n"
                        "ld1rw z19.s, p7/z, [%[maxptr]]\n"
                        "fmax z20.s, p7/m, z20.s, z18.s\n"
                        "fmax z21.s, p7/m, z21.s, z18.s\n"
                        "fmax z22.s, p7/m, z22.s, z18.s\n"
                        "fmax z23.s, p7/m, z23.s, z18.s\n"
                        "fmin z20.s, p7/m, z20.s, z19.s\n"
                        "fmin z21.s, p7/m, z21.s, z19.s\n"
                        "fmin z22.s, p7/m, z22.s, z19.s\n"
                        "fmin z23.s, p7/m, z23.s, z19.s\n"
                        "fmax z24.s, p7/m, z24.s, z18.s\n"
                        "uzp1 z0.s, z20.s, z21.s\n"
                        "uzp2 z1.s, z20.s, z21.s\n"
                        "uzp1 z2.s, z22.s, z23.s\n"
                        "uzp2 z3.s, z22.s, z23.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z24.s, p7/m, z24.s, z19.s\n"
                        "fmax z25.s, p7/m, z25.s, z18.s\n"
                        "fmax z26.s, p7/m, z26.s, z18.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z27.s, p7/m, z27.s, z18.s\n"
                        "fmax z28.s, p7/m, z28.s, z18.s\n"
                        "fmin z25.s, p7/m, z25.s, z19.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z19.s\n"
                        "fmin z27.s, p7/m, z27.s, z19.s\n"
                        "fmin z28.s, p7/m, z28.s, z19.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z24.s, z25.s\n"
                        "uzp2 z5.s, z24.s, z25.s\n"
                        "uzp1 z6.s, z26.s, z27.s\n"
                        "fmax z29.s, p7/m, z29.s, z18.s\n"
                        "st1w z4.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmax z30.s, p7/m, z30.s, z18.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #3\n"
                        "fmax z31.s, p7/m, z31.s, z18.s\n"
                        "st1w z5.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "fmin z29.s, p7/m, z29.s, z19.s\n"
                        "fmin z30.s, p7/m, z30.s, z19.s\n"
                        "fmin z31.s, p7/m, z31.s, z19.s\n"
                        "st1w z6.s, p0, [c_ptr2]\n"
                        "uzp1 z7.s, z28.s, z29.s\n"
                        "uzp1 z8.s, z30.s, z31.s\n"
                        "st1w z7.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z8.s, p2, [c_ptr2, #2, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "cbnz %[accumulate], 1f\n"
                        "ld1w z19.s, p0/z, [%[biasptr]]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z19.s, z19.s\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "zip2 z21.s, z19.s, z19.s\n"
                        "ld1w z19.s, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "zip1 z22.s, z19.s, z19.s\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z23.s, z19.s, z19.s\n"
                        "ld1w z19.s, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z26.d, z20.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z27.d, z21.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "zip1 z24.s, z19.s, z19.s\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "zip2 z25.s, z19.s, z19.s\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z28.d, z22.d\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "mov z29.d, z23.d\n"
                        "mov z30.d, z24.d\n"
                        "mov z31.d, z25.d\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z17.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z18.s, p0/z, [c_ptr1]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "zip1 z20.s, z17.s, z18.s\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "zip2 z21.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        "ld1w z18.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "zip1 z22.s, z17.s, z18.s\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "zip2 z23.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "ld1w z18.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "zip1 z24.s, z17.s, z18.s\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "zip2 z25.s, z17.s, z18.s\n"
                        "ld1w z17.s, p0/z, [c_ptr2]\n"
                        "ld1w z18.s, p0/z, [c_ptr3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        "zip1 z26.s, z17.s, z18.s\n"
                        "zip2 z27.s, z17.s, z18.s\n"
                        "ld1w z17.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z18.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "zip1 z28.s, z17.s, z18.s\n"
                        "zip2 z29.s, z17.s, z18.s\n"
                        "ld1w z17.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z18.s, p2/z, [c_ptr3, #2, MUL VL]\n"
                        "zip1 z30.s, z17.s, z18.s\n"
                        "zip2 z31.s, z17.s, z18.s\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p7/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z5.h, p7/z, [a_ptr3]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3, #-0x10]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p7/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p7/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p7/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z5.h, p7/z, [a_ptr3]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1rqh z3.h, p6/z, [a_ptr3, #0x10]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #-4\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "trn1 z4.d, z0.d, z1.d\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "trn1 z5.d, z2.d, z3.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #12\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "b 5f\n"
                        "4:\n"
                        "trn2 z0.d, z0.d, z1.d\n"
                        "trn2 z1.d, z2.d, z3.d\n"
                        "ld1rqh z2.h, p6/z, [%[a_ptr0]]\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        "ld1rqh z3.h, p6/z, [a_ptr1]\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        "ld1rqh z4.h, p6/z, [a_ptr2]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1rqh z5.h, p6/z, [a_ptr3]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "addvl %[b_ptr0], %[b_ptr0], #6\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        "trn1 z0.d, z2.d, z3.d\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "trn1 z1.d, z4.d, z5.d\n"
                        "cbz %[blocks], 5f\n"
                        "trn2 z5.d, z4.d, z5.d\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0]]\n"
                        "trn2 z4.d, z2.d, z3.d\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        ".inst 0x6460e4d4 // bfmmla z20.s, z6.h, z0.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        ".inst 0x6460e4f5 // bfmmla z21.s, z7.h, z0.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        ".inst 0x6460e516 // bfmmla z22.s, z8.h, z0.h\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        ".inst 0x6461e4da // bfmmla z26.s, z6.h, z1.h\n"
                        ".inst 0x6460e537 // bfmmla z23.s, z9.h, z0.h\n"
                        ".inst 0x6460e558 // bfmmla z24.s, z10.h, z0.h\n"
                        ".inst 0x6460e579 // bfmmla z25.s, z11.h, z0.h\n"
                        ".inst 0x6461e4fb // bfmmla z27.s, z7.h, z1.h\n"
                        ".inst 0x6461e51c // bfmmla z28.s, z8.h, z1.h\n"
                        ".inst 0x6461e53d // bfmmla z29.s, z9.h, z1.h\n"
                        ".inst 0x6461e55e // bfmmla z30.s, z10.h, z1.h\n"
                        ".inst 0x6461e57f // bfmmla z31.s, z11.h, z1.h\n"
                        "b.eq 5f\n"
                        "ld1h z6.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "ld1h z7.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        ".inst 0x6464e4d4 // bfmmla z20.s, z6.h, z4.h\n"
                        ".inst 0x6464e4f5 // bfmmla z21.s, z7.h, z4.h\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        ".inst 0x6465e4da // bfmmla z26.s, z6.h, z5.h\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        ".inst 0x6465e4fb // bfmmla z27.s, z7.h, z5.h\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        ".inst 0x6464e516 // bfmmla z22.s, z8.h, z4.h\n"
                        ".inst 0x6464e537 // bfmmla z23.s, z9.h, z4.h\n"
                        ".inst 0x6464e558 // bfmmla z24.s, z10.h, z4.h\n"
                        ".inst 0x6464e579 // bfmmla z25.s, z11.h, z4.h\n"
                        ".inst 0x6465e51c // bfmmla z28.s, z8.h, z5.h\n"
                        ".inst 0x6465e53d // bfmmla z29.s, z9.h, z5.h\n"
                        ".inst 0x6465e55e // bfmmla z30.s, z10.h, z5.h\n"
                        ".inst 0x6465e57f // bfmmla z31.s, z11.h, z5.h\n"
                        "5:\n"
                        "ld1rw z18.s, p7/z, [%[minptr]]\n"
                        "ld1rw z19.s, p7/z, [%[maxptr]]\n"
                        "fmax z20.s, p7/m, z20.s, z18.s\n"
                        "fmax z21.s, p7/m, z21.s, z18.s\n"
                        "fmax z22.s, p7/m, z22.s, z18.s\n"
                        "fmax z23.s, p7/m, z23.s, z18.s\n"
                        "fmin z20.s, p7/m, z20.s, z19.s\n"
                        "fmin z21.s, p7/m, z21.s, z19.s\n"
                        "fmin z22.s, p7/m, z22.s, z19.s\n"
                        "fmin z23.s, p7/m, z23.s, z19.s\n"
                        "fmax z24.s, p7/m, z24.s, z18.s\n"
                        "uzp1 z0.s, z20.s, z21.s\n"
                        "uzp2 z1.s, z20.s, z21.s\n"
                        "uzp1 z2.s, z22.s, z23.s\n"
                        "uzp2 z3.s, z22.s, z23.s\n"
                        "st1w z0.s, p0, [%[c_ptr0]]\n"
                        "fmin z24.s, p7/m, z24.s, z19.s\n"
                        "fmax z25.s, p7/m, z25.s, z18.s\n"
                        "fmax z26.s, p7/m, z26.s, z18.s\n"
                        "st1w z1.s, p0, [c_ptr1]\n"
                        "fmax z27.s, p7/m, z27.s, z18.s\n"
                        "fmax z28.s, p7/m, z28.s, z18.s\n"
                        "fmin z25.s, p7/m, z25.s, z19.s\n"
                        "st1w z2.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmin z26.s, p7/m, z26.s, z19.s\n"
                        "fmin z27.s, p7/m, z27.s, z19.s\n"
                        "fmin z28.s, p7/m, z28.s, z19.s\n"
                        "st1w z3.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "uzp1 z4.s, z24.s, z25.s\n"
                        "uzp2 z5.s, z24.s, z25.s\n"
                        "uzp1 z6.s, z26.s, z27.s\n"
                        "uzp2 z7.s, z26.s, z27.s\n"
                        "st1w z4.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmax z29.s, p7/m, z29.s, z18.s\n"
                        "addvl %[c_ptr0], %[c_ptr0], #3\n"
                        "fmax z30.s, p7/m, z30.s, z18.s\n"
                        "st1w z5.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "fmax z31.s, p7/m, z31.s, z18.s\n"
                        "fmin z29.s, p7/m, z29.s, z19.s\n"
                        "fmin z30.s, p7/m, z30.s, z19.s\n"
                        "st1w z6.s, p0, [c_ptr2]\n"
                        "fmin z31.s, p7/m, z31.s, z19.s\n"
                        "uzp1 z8.s, z28.s, z29.s\n"
                        "uzp2 z9.s, z28.s, z29.s\n"
                        "st1w z7.s, p0, [c_ptr3]\n"
                        "uzp1 z10.s, z30.s, z31.s\n"
                        "uzp2 z11.s, z30.s, z31.s\n"
                        "st1w z8.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z9.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "st1w z10.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z11.s, p2, [c_ptr3, #2, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [accumulate] "r" (static_cast<uint64_t>(accumulate)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
