package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,``P`!````H"(``#0````XL0```````#0`(``-
M`"@`'``;``8````T````-````#0```"@`0``H`$```0````$`````P```"`"
M```@`@``(`(``!,````3````!`````$````!````````````````````\!$`
M`/`1```$`````!````$`````(````"`````@``#L-```[#0```4`````$```
M`0````!@````8````&```'1'``!T1P``!``````0```!````O*T``+R]``"\
MO0``7`(``'`"```&`````!````(```#$K0``Q+T``,2]```@`0``(`$```8`
M```$````!````-0!``#4`0``U`$``$P```!,````!`````0````$````5*<`
M`%2G``!4IP``(````"`````$````!````%/E=&3X`0``^`$``/@!```H````
M*`````0````$````4.5T9'"@``!PH```<*```!P!```<`0``!`````0```!1
MY71D```````````````````````````&````$````%+E=&2\K0``O+T``+R]
M``!$`@``1`(```0````!````!````!0````#````1TY5`(\*?)#[\I6_"M%\
M@-VW&/M2Q3>!!````!@````%````1TY5``$``<`$`````0````(``<`$````
M`0```"]L:6(O;&0M;&EN=7@N<V\N,@``*P```$4````(````"``````0$`0E
M0`2`*`$`!:"&`(2`@0`""R@$"#B`$D&*```(10``````````````1@``````
M````````1P````````!(````20````````!*`````````$P`````````````
M`$T```!.`````````%````!1````4P````````!4````50```%8`````````
M`````````````````````%<```!8````60```%H```!;``````````````!=
M`````````%X``````````````(D'<--K&IY0H^#[29E/PYOC1\8!9J_,9D-%
MU>R[XY)\LP&+;UCH[>QOQ=\?%8PES"[%F[9K"4/6(?_D#ONA;&##Y3\L#3J7
M"\_>1RW9<5@<?ZF<F^&G`@I"^M>':W^:?#-BV^W49-1\K4OCP```````````
M``````````#5`0`````````````2````:`$`````````````$@```,\!````
M`````````!(```"D`@`````````````2````"P(`````````````$@```(L"
M`````````````!(```#T`0`````````````2````Y@$`````````````$@``
M`$<"`````````````!(````!```````````````@````[P``````````````
M$@```'P!`````````````!(```!V```````````````2````U@(`````````
M````$@```/L!`````````````!(```"D```````````````2````;P$`````
M````````$@```)@!`````````````!(```##`0`````````````2````$```
M````````````(````+P"`````````````!(```"I`@`````````````2````
M_P``````````````$@```"P``````````````"````#X```````````````2
M````IP$`````````````$@````8!`````````````!(```"N`@``````````
M```2````D0``````````````%@```#P!`````````````!(```!:`@``````
M```````2````5P$`````````````$@```,<"`````````````!(````.`0``
M```````````2````H`$`````````````$0```.X"`````````````!(```#:
M```````````````2````)@$`````````````$@```*(#`````````````!(`
M``"Y```````````````2````R`$`````````````$@```,H`````````````
M`!$````M`0`````````````2````E@(`````````````$@```(0"````````
M`````!(````L`@`````````````2````]0(`````````````$@```#8!````
M`````````!(```"*`0`````````````2````]P``````````````$@```(@`
M`````````````!(````7`@`````````````2````>`(`````````````$@``
M`!@!`````````````!(```#(```````````````1````L@``````````````
M$@```-(``````````````!(```!K`@`````````````1````WP(`````````
M````$0```$P!`````````````!(```#A```````````````2````.`(`````
M````````$@```(,!`````````````!(```!5```````````````2````NP$`
M````````````$@```)$!`````````````!(```!G```````````````B````
MM@$`````````````$@```"0$```4P```!````!$`&`"&`P``8%(``!X"```2
M``X`-@0``,`]```T!```$@`.`"X$``!@0@``0@```!(`#@`L`P``\#L``#L`
M```2``X`>@,``#`\``")`0``$@`.`/X#```8P````````!``&`#L`P``+,``
M```````0`!D`5@,```!"``!9````$@`.`#D#````8```!````!$`$`"K`P``
M4$\``,4````2``X`XP,``"!0``!1````$@`.`!L#``"0.0``!0```!(`#@"W
M`P```,`````````0`!@`%P0``*`Y``!&`@``$@`.``4$``"P1```WP```!(`
M#@#/`P``<#@``#@````2``X`N0,```#`````````(``8`/$#``"010``@P<`
M`!(`#@#7`P``&,`````````0`!D`D`,``(!$```P````$@`.`&0#``"P.```
MV@```!(`#@#$`P``@%```-0!```2``X`8@```$`@``!4`@``$@`.`+T#``"@
M(@``+````!(`#@!``P``($T``"4"```2``X`1@````1@```$````$0`0``!?
M7V=M;VY?<W1A<G1?7P!?251-7V1E<F5G:7-T97)434-L;VYE5&%B;&4`7TE4
M35]R96=I<W1E<E1-0VQO;F5486)L90!?24]?<W1D:6Y?=7-E9`!?7VQI8F-?
M<W1A<G1?;6%I;@!?7V-X85]F:6YA;&EZ90!?7W)E9VES=&5R7V%T9F]R:P!R
M96%D;&EN:P!03%]C=7)R96YT7V-O;G1E>'0`4&5R;%]S=E]S971P=@!M96UC
M:'(`4&5R;%]S=E]S971P=FX`7U]E;G9I<F]N`'-T<FYC;7``<W1R;&5N`%]?
M;65M;6]V95]C:&L`<F5A;&QO8P!L<W1A=#8T`&%C8V5S<P!O<&5N9&ER`')E
M861D:7(V-`!?7W-P<FEN=&9?8VAK`'5N;&EN:P!C;&]S961I<@!R;61I<@!B
M;V]T7T1Y;F%,;V%D97(`4&5R;%]N97=84P!097)L7VYE=UA37V9L86=S`&=E
M='!I9`!?7VUE;6-P>5]C:&L`<W1R8VAR`'-T<F1U<`!S=')T;VL`<W1R8VUP
M`'-T<G)C:'(`<W1D97)R`%]?=F9P<FEN=&9?8VAK`&5X:70`;'-E96LV-`!R
M96%D`&=E='5I9`!M:V1I<@!?7V5R<FYO7VQO8V%T:6]N`%]?9G!R:6YT9E]C
M:&L`;W!E;C8T`%!E<FQ?9W9?9F5T8VAP=@!097)L7V=E=%]S=@!097)L7V-R
M;V%K7VYO8V]N=&5X=`!097)L7VUG7W-E=`!097)L7W-Y<U]I;FET,P!097)L
M7V%T9F]R:U]U;FQO8VL`4&5R;%]A=&9O<FM?;&]C:P!03%]D;U]U;F1U;7``
M4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?<&%R<V4`<&5R;%]D97-T<G5C=`!P
M97)L7V9R964`4&5R;%]S>7-?=&5R;0!P97)L7V%L;&]C`'!E<FQ?8V]N<W1R
M=6-T`'!E<FQ?<G5N`%!,7VUE;6]R>5]W<F%P`&UE;6-P>0!?7W-T86-K7V-H
M:U]F86EL`&QI8G!E<FPN<V\`;&EB8RYS;RXV`'!A<E]C=7)R96YT7V5X96,`
M<&%R7V)A<V5N86UE`%]F<%]H=P!84U]);G1E<FYA;'-?4$%27T)/3U0`<&%R
M7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R7V1I<FYA;64`
M<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A
M;G5P`%]?9&%T85]S=&%R=`!S:&%?=7!D871E`'AS7VEN:70`7U]B<W-?<W1A
M<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S971U
M<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7V1I90!P87)?
M:6YI=%]E;G8`1TQ)0D-?04))7T147U)%3%(`1TQ)0D-?,BXQ+C,`1TQ)0D-?
M,BXS-`!'3$E"0U\R+C0`1TQ)0D-?,BXR`$=,24)#7S(N,S,`1TQ)0D-?,BXS
M+C(`1TQ)0D-?,BXS+C0`1TQ)0D-?,BXQ`$=,24)#7S(N,``O=7-R+VQI8B]P
M97)L-2\U+C0R+V-O<F5?<&5R;"]#3U)%`````@`"``(``0`!``$``P`$``$`
M`0`"``(`!0`!``$``0`$``(``@`!``$``@`"``$`!@`$``(``0`!``$``0`!
M``$`!P`"``(``@`"``(``0`"``(``@`!``(``0`(``(``@`&``(``0`!``0`
M`@`"``(``0`!``$`!``!``(`"0`#``(`"@`"``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!`````0`*`!$#
M```0`````````$(._0````L`0P0``!````!S'VD)```*`%4$```0````M)&6
M!@``"0!A!```$````!1I:0T```@`;`0``!`````2:6D-```'`'8$```0````
MLY&6!@``!@"`!```$````'(9:0D```4`BP0``!````!T&6D)```$`)<$```0
M````$6EI#0```P"C!```$````!!I:0T```(`K00```````#TO@``!@$``/B^
M```&`@``_+X```8#````OP``!@0```2_```&!0``"+\```8&```,OP``!@<`
M`!"_```&"```%+\```8)```8OP``!@H``!R_```&"P``(+\```8,```DOP``
M!@T``"B_```&#@``++\```8/```POP``!A```#2_```&$0``.+\```82```\
MOP``!A,``$"_```&%```1+\```85``!(OP``!A8``$R_```&%P``4+\```88
M``!4OP``!AD``%B_```&&@``7+\```8;``!@OP``!AP``&2_```.'0``:+\`
M``8>``!LOP``!A\``'"_```&(```=+\```8A``!XOP``!B(``'R_```&(P``
M@+\```8D``"$OP``!B4``(B_```&)@``C+\```8G``"0OP``!B@``)2_```&
M*0``F+\```8J``"<OP``!BL``*"_```&+```I+\```8M``"HOP``!BX``*R_
M```&+P``L+\```8P``"TOP``!C$``+B_```&,@``O+\```8S``#`OP``!C0`
M`,2_```&-0``R+\```8V``#,OP``!C@``-"_```&.0``U+\```8Z``#8OP``
M!CL``-R_```&/```X+\```8]``#DOP``!CX``.B_```&/P``[+\```9!``#P
MOP``!D(``/2_```&0P``^+\```9$``#PO@``!T```+R]```#````_+\``$4`
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``!3@^P(Z,<"``"!P]N>``"+@S0```"%P'0"_]"#Q`A;P_^S!````/^C"```
M``````#_HPP```!H`````.G@____C4PD!(/D\/]Q_%6)Y5>)SU93Z'D"``"!
MPXV>``!1@^PLBT$$B474BT$(B4709:$4````B47DC4704(U%U%!1_Y,``0``
MBX,P````@\0,4%#_LX@```!GZ/0S``"+@_````"#Q!"`.``/A"4!``"+@X``
M``!EBP"+%X"(H@(```.-0@0]____/P^'BP$``(UR`X/L#,'@`HEUS%#_D^``
M``"+==2+/XG!B8-``0``BP:#Q!")`8V#!Z/__XE!!(N#,`$``(E!"(V#"J/_
M_XE!#(/_`0^.80$``(E-R(U&!(U1$(TTOHUV`"Z-M"8`````BPB#P`2#P@2)
M2OPYQG7QBTW(@\<#@^P,C8,L>?__QP2Y`````&H`4?]US%#_LS@!``#_DR0`
M``"#Q""%P`^$E@```(/L#/^S.`$``/^3O````(U5X(G'C8/#H?__Z!X1``"#
MQ!"%P`^$@P```(/L#/^S.`$``/^3'````(/$$/^3?````(M%Y&4K!10````/
MA<H```"-9?")^%E;7E]=C6'\P_^38````(F#.`$``(7`#X2=````@^P,4/^3
MD````(N#@````&6+$(/$$,:".@4```#IJ_[__X/L#/^S.`$``/^31````(/$
M$.E3____BX,\`0``A<!T$8/L#%#H'2T``(/$$.EB____C57@C8-XH?__Z&<0
M``")@SP!``"%P`^$1O___^O1BT7D92L%%````'4J4%"-@W.A____L_0```!0
M_Y/<````OP0```#IP?[__X/L#&H!_Y,4`0``Z"PR``!FD&:09I!FD&:09I`Q
M[5Z)X8/D\%!44N@8````@<,TG```:@!J`%%6_[,8`0``Z&G]___TBQPDPV:0
M9I"+'"3#9I!FD&:09I!FD&:0Z.````"!PO^;``"-BC0!``"-@C0!```YR'0=
MBX)<````A<!T$U6)Y8/L%%'_T(/$$,G#+HUT)@##C;0F`````.B@````@<*_
MFP``58GE4XV*-`$``(V"-`$``(/L!"G(B</!Z!_!^P(!V-'X=!2+DFP```"%
MTG0*@^P(4%'_TH/$$(M=_,G#D"Z-M"8`````58GE4^A7____@<-KFP``@^P$
M@+LT`0```'4HBX,0`0``A<!T$H/L#/^S(`$``/^3$`$``(/$$.@T____QH,T
M`0```8M=_,G#+HVT)@````#I6____XL4),-FD&:09I!FD&:09I!FD&:09I!F
MD&:09I!FD&:09I!FD&:09I!FD&:09I!FD&:09I!FD&:09I"058GE5U93C9WD
M_O__@>R,`0``B86(_O__98L-%````(E-Y(U('(V%I/[__RZ-="8`+HVT)@``
M```NC;0F`````(L1@\`$@\$$#\J)4/PYPW7OB[6H_O__BXW<_O__C96L_O__
MB[VD_O__BYW8_O__B;6`_O__C76LBX7@_O__B;6,_O__B[6`_O__B9UL_O__
MB;60_O__B?Z)C7#^__^)A73^__^)C93^__^)O83^___K%(VV`````"Z-M"8`
M````B8V0_O__BPJ)QXM"&(/"!#'(,=B+G93^__^)O93^__\Q\(NUD/[__]'`
MB4(T.Y6,_O__=<N+M8C^__^+O83^__^+7@2+3A"+!HM6#(F=E/[__XM>"(F5
MD/[__XV4#YEY@EJ)Q\'`!8F->/[__XN-D/[__P'"B=B+M93^__^)E8S^__\Q
MR(N5C/[__XF=A/[__R'PB;U\_O__,<@!PHGPP<@"B<:)V#'P(?@QV(G+BXV`
M_O__C8P9F7F"6HN=A/[__P'(B='!P04!R(GYB??!R0(QSXF-C/[__XGYB[VL
M_O__(='!R@(Q\8V\.YEY@EH!^8G'P<<%C1PYB[V,_O__B9V`_O__,=>)^8N]
ML/[__R'!,XV,_O__P<@"C;0^F7F"6@'QB=[!Q@6-/#&)UC'&B?&+M;3^__\A
MV8N=C/[__S'1C9PSF7F"6HG^`=G!Q@4!\8G&B8V,_O__BXV`_O__P<D",<Z)
M\XNUN/[__R'[P<\",<.-E#*9>8):`=J+G8S^___!PP6--!J+G;S^__^)RC'Z
M(Y6,_O__,<J-A!B9>8):`=")\L'"!0'"B96`_O__BX6,_O__B?J+G<#^___!
MR`(QPHV,&9EY@EHA\L'.`C'Z`<J+C8#^___!P06-'`J+C<3^__^)PC'R(Y6`
M_O__,<*-C`^9>8):B=\!RL''!0'ZB?>)E8S^__^+E8#^___!R@(QUXGYB[W(
M_O__(=DQ\8V$.)EY@EH!R(N-C/[__\'!!<'+`HT\"(N-S/[__XG0,=@CA8S^
M__\QT(V,#IEY@EJ)_@'(P<8%`?")WHF%@/[__XN%C/[__\'(`C'&B?&+M=#^
M__\A^<'/`C'9C90RF7F"6@'*BXV`_O__P<$%C30*BXW4_O__B<(Q^B.5@/[_
M_S'"C8P+F7F"6HGS`<K!PP6+C8#^__\!VHG[B96,_O__P<D"BY5L_O__,<LA
M\XV$`IEY@EJ+E8S^___!S@(Q^P'#B=#!P`4!PXG(,?`AT(N5</[__S'(C;PZ
MF7F"6HN5C/[__P'XB=_!QP7!R@(!^(GWB96,_O__,=>+E73^__\AW\'+`C'W
MC8P*F7F"6@'YB<?!QP6-%#F+O8S^__\QWXGYB[WD_O__(<$SC8S^___!R`*-
MM#Z9>8):B86`_O__`?&)UL'&!8T\,8G9B[7H_O__,<&)T"'1BY6,_O__,=F-
ME#*9>8):`<J)^<'!!8TT"HG"B;6,_O__BX6`_O__P<H"B<&)QC'1(?G!SP(Q
MP8N%[/[__XV<`YEY@EJ+A?#^__\!V8N=C/[__XV$!IEY@EJ+M8S^___!PP4!
MV8G3,?LCG8S^__\QTP'8B<O!PP7!S@(!V(GSB[7T_O__C;0RH>O9;HGZ,=HQ
MRL')`@'RB<;!Q@4!\HNU^/[__XV\-Z'KV6Z)WC'.,<;!R`(!_HG7P<<%`?Z)
MQXN%_/[__XV<`Z'KV6Z)R#'X,=#!R@(!V(GSP<,%`=B)TXN5`/___XV4$:'K
MV6Z)^3'9,?'!S@(!T8G"P<(%`=&+E03___^-O!>AZ]ENB=HQ\C'"`?J)S\''
M!0'ZP<@"B<>+A0C___^-A`.AZ]ENB?,Q^S'+P<D"`<.)T,'`!0'#BX4,____
MC;0&H>O9;HGX,<@QT,'*`@'PB=[!Q@4!\(G6BY40____C907H>O9;HG/,?<Q
MW\'+`@'7B<+!P@4!UXN5%/___XV,$:'KV6Z)\C':,<+!R`(!RHGYP<$%`<J)
MP8N%&/___XV$!J'KV6Z)WC'.,?[!SP(!QHG0P<`%`<:+A1S___^-G`.AZ]EN
MB<@Q^#'0P<H"`=B)\\'#!0'8B=.+E2#___^-E!&AZ]ENB?DQV3'QP<X"`=&)
MPL'"!0'1BY4D____C;P7H>O9;HG:,?(QPL'(`@'ZB<_!QP4!^HG'BX4H____
MC80#H>O9;HGS,?LQR\')`@'#B=#!P`4!PXN%+/___XVT!J'KV6Z)^#'(,=#!
MR@(!\(G>P<8%`?")UHN5,/___XV4%Z'KV6Z)SS'W,=\!UXG"P<(%P<L"`=>+
ME33___^-C!&AZ]ENB?(QVC'"P<@"`<J)^<'!!0'*B<&+A3C___^-A`:AZ]EN
MB=XQSC'^P<\"`<:)T,'`!0'&BX4\____C9P#H>O9;HG(,?@QT,'*`@'8B?/!
MPP4!V(N=0/___XV<&:'KV6Z)^3'1,?'!S@(!V8G#P<,%`=F)TXF-C/[__PGS
MB=$APR'Q"=F+G43___^-G!_<O!N/`=F+G8S^___!PP7!R`*-/!F)\XGQ"<,A
MP2.=C/[__PG9BYU(____C90:W+P;CXG[`<K!PP6)P0':B<.)E8#^__^+E8S^
M___!R@()TR'1(?O!SP()V8N=3/___XV<'MR\&X^+M8#^__\!V8N=@/[__\'#
M!0'9B8V,_O__B?F)UPG/(?Z)UR'/B?L)\XNU4/___XV$,-R\&X^)S@'8BYV,
M_O__P<,%C3P8BX6`_O__B<O!R`()QB'#([6,_O__"?.+M53___^-E#+<O!N/
MB?X!VL'&!8G#`?*)QHF5@/[__XN5C/[__\'*`@G6(=,A_L'/`@GSB[58____
MC8PQW+P;CXNU@/[__P'9BYV`_O__P<,%`=F)C8S^__^)^8G7"<\A_HG7(<^)
M^PGSB[5<____C80PW+P;CXG.`=B+G8S^___!PP6-/!B+A8#^__^)R\'(`@G&
M(<,CM8S^__\)\XNU8/___XV4,MR\&X^)_@':P<8%B<,!\HG&B96`_O__BY6,
M_O__P<H""=8ATR'^P<\""?.+M63___^-C#'<O!N/B[6`_O__`=F+G8#^___!
MPP4!V8F-C/[__XGYB=<)SR'^B=<ASXG["?.+M6C___^-A##<O!N/B<X!V(N=
MC/[__\'#!8T\&(N%@/[__XG+P<@""<8APR.UC/[__PGSB[5L____C90RW+P;
MCXG^`=K!Q@6)PP'RB<:)E8#^__^+E8S^___!R@()UB'3(?[!SP()\XNU</__
M_XV,,=R\&X^+M8#^__\!V8N=@/[__\'#!0'9B8V,_O__B?F)UPG/(?Z)UR'/
MB?L)\XNU=/___XV$,-R\&X^)S@'8BYV,_O__P<,%C3P8BX6`_O__B<O!R`()
MQB.UC/[__R'#"?.+M7C___^-E#+<O!N/B?X!VL'&!8G#`?*)QHF5@/[__XN5
MC/[__\'*`@G6(=,A_L'/`@GSB[5\____C8PQW+P;CXNU@/[__P'9BYV`_O__
MP<,%`=F)C8S^__^)^8G7"<\A_HG7(<^)^PGSBW6`C80PW+P;CXG.`=B+G8S^
M___!PP6-/!B+A8#^__^)R\'(`@G&(<,CM8S^__\)\XMUA(V4,MR\&X^)_@':
MP<8%B<,!\HG&B96`_O__BY6,_O__P<H""=8ATR'^"?.+=8C!SP*-C#'<O!N/
M`=F+G8#^___!PP6--!F)^8G7B=,)SXFUC/[__R'+B?Z+O8#^__\A_L'/`@GS
MBW6,C80PW+P;CXNUC/[__P'8B?/!PP4!V(G[B<\)WR'WB<XAW@G^BWV0C90Z
MW+P;CP'6B<+!P@4!UHN5C/[__\'*`HG7BU64C901UL%BRHG9,?DQP<'(`@'1
MB?+!P@4!T8M5F(V<$];!8LJ)^C'",?+!S@(!VHG+P<,%`=J+79R-O!_6P6+*
MB<,Q\S'+P<D"`?N)U\''!0'[BWV@C;PXUL%BRHGP,<@QT,'*`@'XB=_!QP4!
M^(G7BU6DC906UL%BRHG.,?XQWL'+`@'6B<+!P@4!UHM5J(V,$=;!8LJ)^C':
M,<+!R`(!RHGQP<$%`<J)P8M%K(V$!];!8LJ)WS'/,??!S@(!QXG0P<`%`<>+
M1;"-G`/6P6+*B<@Q\#'0`=B)^\'*`L'#!0'8B=.+5;2-E!'6P6+*B?$QV3'Y
MP<\"`=&)PL'"!0'1BU6XC;06UL%BRHG:,?HQPL'(`@'RB<[!Q@4!\HG&BT6\
MC80#UL%BRHG[,?,QR\')`@'#B=#!P`4!PXM%P(V\!];!8LJ)\#'(,=#!R@(!
M^(G?P<<%`?B)UXM5Q(V4%M;!8LJ)SC'^,=X!UHG"P<(%`=:+5<C!RP*-C!'6
MP6+*B?HQVC'"P<@"`<J)\<'!!0'*B<&+1<R-A`?6P6+*B=\QSS'WP<X"`<>)
MT,'`!0''BT70C9P#UL%BRHG(,?`QT,'*`@'8B?O!PP4!V(G3BU74C901UL%B
MRHGQ,=DQ^<'/`@'1B<+!P@4!T8M5V(VT%M;!8LJ)VC'Z,<+!R`(!\HG.P<8%
M`?*+==R-M#/6P6+*B?LQPS'+P<D"`?.)UHF-C/[__XN-?/[__\'&!0'SBW7@
MC;0QUL%BRHN-C/[__P'WB<8#A7C^__\QS@.-D/[__S'6P<H"`Y6$_O__`?Z)
MWP.=E/[__\''!0'^B[V(_O__B3>)7P2)5PB)3PR)1Q"+1>1E*P44````=0N!
MQ(P!``!;7E]=P^CR(@``9I!5B>575E.![``0``"##"0`@^PPZ)HB``"!QOR,
M``")E=3O__^)PXV]Y.___V6A%````(E%Y#'`C88DH?__:/\/``!74/^6V```
M`(/$$(7`?A.)PHU`_X"\%>/O__\`=4"%P'4PBX:`````@^P$_[74[___4V7_
M,/^63````(/$$(M%Y&4K!10```!U2XUE]%M>7UW#B<)FD"Z-M"8`````@^P$
M4HF5T.___VHO5_^6Z````(/$$(7`=*J+E=#O__^+AH````!25U-E_S#_EJP`
M``"#Q!#KJ>@4(@``C70F`.C)(0``@<$OC```58GE5U93@^P<B4W@BXFT````
MB47DBPF%R0^$D0````^V&(3;=`6`^SUU4C'VBQF%VW0\B57<B=^)RXE-V"Z-
MM"8`````@^P$5O]UY%>+1>#_D.P```"#Q!"%P'4&@#PW/70RBWL$@\,$A?]U
MV(G[C67TB=A;7E]=PXUT)@")Q@^V1@&#Q@&$P'0$/#UU\2MUY.N:9I"+1=R)
MVBM5V(U<-P'!^@*)$(UE](G86UY?7<,QV^N[D%6)Y5?H#2$``('':XL``%:)
MQE,QVX/L2(E%U&6A%````(E%Y#'`@#H]#Y3#`=-3_Y>@````C57@@\00B470
MB?#H^_[__X7`#X03`0``@^P,B<90_Y>@````@\00.T70#X/;````BT7@B[>T
M````P>`"B47,BTW4#[8!A,`/A+8!```\/0^$K@$``(G(C70F``^V4`&#P`&$
MTG0%@/H]=?`K1=2+3="+5<R#[`P#%HU$"`*)5<A0_Y?`````BU7(@\00B0*%
MP'17BP:+5<R+?=2+%!`/M@>(`CP]="B$P'0DC4<!ZQ"-=@`NC;0F`````(#Y
M/70/#[8(@\(!@\`!B`J$R77LQ@(],<"-M"8`````#[8,`XA,`@&#P`&$R77Q
MBT7D92L%%`````^%'@$``(UE]%M>7UW#D"Z-M"8`````,<"-M@`````NC;0F
M``````^V%`.(%`:#P`&$TG7RZ[^+M[0```"+#HL!A<`/A,T````QTHUT)@`N
MC;0F`````(G0@\(!@SR1`'7UC02%#````(E%Q(N'1`$``(E5S(7`=6:#[`R)
M3;S'AT0!```!````_W7$_Y?`````B47(@\00A<`/A%;___^+5<R-#)4`````
MB57`B4W,_W7$48M-O%%0_Y?\````BT7(BU7`B0:#Q!"+1<C'1)`$`````.EQ
M_O__+HUT)@"#[`C_=<11_Y<X````B47(B0:#Q!"%P`^$^O[__XM5S(T$E0``
M``")1<SKOXUT)@`QP.ED_O__QT7$"````#'2Z4C____H_AX``(VV`````"Z-
MM"8`````58GE5NBI'@``@<8+B0``4X/L>&6+'10```")7?2)PXU%E%!3_Y;4
M````@\00A<!U-XM5I('B`/```('Z`$```'0(@?H`H```=2"#[`AJ`E/_EF@`
M``"#Q!"%P`^4P`^VP.L'+HUT)@`QP(M5]&4K%10```!U!XUE^%M>7</H;AX`
M`(VV`````"Z-M"8`````58GE5U;H&!X``('&>H@``%.![(P```!EBQT4````
MB5WDB<.%VW0%@#L`=1R+1>1E*P44````#X5A`0``C67T6UY?7<.-="8`@^P,
M4_^6>````(G'@\00A<!TT8/L#%?_EI0```"#Q!"%P`^$X````(U0$X!X$RYU
M!X!Z`0!TW)"`>!,N#X3F````@^P,B95P____4_^6H````(N5</___XF%=/__
M_XD4)(F5;/____^6H````(N-=/___XU,`0*)#"2)C7#_____EL````")A73_
M__]8C88SH?__6HN5;/___U)34(N-</___U%J`O^U=/____^6Y````(U%A(/$
M&%#_M73_____EG````"#Q!"#^/]T#XM%E"4`\```/0!```!T;(/L#/^U=/__
M__^6I````(/$$(/L#/^U=/____^69````(/$$.D.____C78`@^P,5_^6N```
M`(D<)/^6S````(/$$.G!_O__+HUT)@"`>@$N#X40____@'H"``^$U_[__^D!
M____C;0F`````(N%=/___^AE_O__ZYGHOAP``(VV`````"Z-M"8`````58GE
M5E/HMNK__X'#RH8``(NS@````(V#@.'__U"-@SFA____LX0```!09?\V_Y/X
M````6(V#Y*+__UIJ`%!0C8,\CO__4(V#5:'__U!E_S;_DXP```"#Q""-9?A;
M7EW#Z$(<```%;X8``%6)Y8/L"(V0@.'__U*-D#FA____L(0```!2BY"`````
M9?\R_Y#X````@\00R<,NC;0F`````%6)Y5=64X'L`!```(,,)`"![``0``"#
M#"0`@^PLZ/[I__^!PQ*&``!EH10```")1>0QP(VUX^___XV]XM____^3%```
M`(/L"(V3+Z'__U)0C8-JH?__4&@!$```:@)6_Y/D````@\0<:``0``!75O^3
MV````#'2@\00A<!X08/L#(U(`8G&48F-U-____^3P````(G"@\00A<!T(XN-
MU-___U%65U")A=3?____DU````"+E=3?__^#Q!#&!#(`BT7D92L%%````'4*
MC67TB=!;7E]=P^@V&P``C;8`````Z1O___^-=@`NC;0F`````%6)Y5=64X'L
M`!```(,,)`"![)P```#H%>G__X'#*84``(M%"(F%=.___XV-A.___XG*C8-X
MH?__98L]%````(E]Y(M]#(F-:.___^C`^/__@^P(:B^)QO^U=.____^3/```
M`(/$$(7`=#:+M73O__^-@X&A__^)\NA3^?__B?"+5>1E*Q44````#X6R`0``
MC67T6UY?7<.0+HVT)@````"#[`Q7_Y,$`0``6EF-BXZA__^)C7#O__]14/^3
MT````(G'@\00A<!TH8V#D*'__XF%9.___XV#F:+__XF%8.___X`_``^$I```
M`(VV`````(7V=!*#[`A75O^3#`$``(/$$(7`=&&#[`Q7_Y.@````B<*-1`?_
M@\00.<=R#>L?D,8``(/H`3G'=`6`."]T\8/L#%?_DZ````")PH/$$(F5;.__
M_X/L#/^U=.____^3H````(N5;.___XU$`@*#Q!`]_P\``'9)@^P(_[5P[___
M:@#_D]````")QX/$$(7`#X3E_O__@#\`#X5B____A?9T!8`^+G01N@$```"-
MNW:A___KFHUT)@"`?@$`=+GKYX/L!/^U=.____^U9.___U>-O>3O____M6#O
M__]H`!```&H"5_^3Y````(/$&/^U:.___U?_DW````"#Q!"%P`^%<O___XN%
ME.___R4`\```/0"````/A5S___^#[`AJ`5?_DV@```"#Q!"%P`^%1?___XGZ
MC8.!H?__Z*'W__^#[`Q7_Y,$`0``@\00Z3[^___HVA@``&:0+HVT)@````!5
MB>56Z(D8``"!QNN"``!3BUT(@^P(:B]3_Y94````C5`!@\00A<")\`]%VHE=
M"(UE^%M>7?^@!`$``"Z-="8`58GE5U93@>P`$```@PPD`(/L+.B(YO__@<.<
M@@``98LU%````(EUY(MU"(7V#X2]````@#X`#X2T````@^P,5O^3H````(U(
M`8/$$(F%U.___X'Y`!````^'\@```(V5Y.___X/Y!`^#LP```(7)#X7K````
MBX74[___C40"_SG"<A?K&HUV`"Z-M"8`````Q@``@^@!.=!T!8`X+W3Q@^P(
M:B]2B974[____Y-4````BY74[___@\00A<!T,CG"<GZ#[`Q2_Y,$`0``@\00
MBU7D92L5%`````^%I0```(UE]%M>7UW#D"Z-M"8`````BT7D92L%%`````^%
MA````(V#=J'__XE%"(UE](G86UY?7?^@!`$``"Z-M"8`````BT0._(G7B80-
MX.___XN%U.___\'H`HG!\Z7I-O___Y#&``#I>O___RZ-M"8`````,<#I>/__
M_Y`NC;0F``````^V!HB%Y.____;!`@^$`____P^W=`[^9HFT#>+O___I\?[_
M_^@'%P``C;0F`````.BY%@``@<$?@0``58GE5U93@^P\B4W4B[FT````98LU
M%````(EUY(VQDJ'__XU5X(GPZ+GT__^%P'0OBU7@BP>-!)"+4`2)$(72=.&-
M="8`BU`(@\`$B1"%TG7TC57@B?#HBO3__X7`==&+1=2+N+0```"-L)NA__^-
M5>")\.AM]/__A<!T,XM5X(L'C020BU`$B1"%TG3A+HVT)@````"+4`B#P`2)
M$(72=?2-5>")\.@Z]/__A<!US8M%U(NXM````(VPHZ'__XU5X(GPZ!WT__^%
MP'0SBU7@BP>-!)"+4`2)$(72=.$NC;0F`````(M0"(/`!(D0A=)U](U5X(GP
MZ.KS__^%P'7-BT74B[BT````C;"LH?__C57@B?#HS?/__X7`=#.+5>"+!XT$
MD(M0!(D0A=)TX2Z-M"8`````BU`(@\`$B1"%TG7TC57@B?#HFO/__X7`=<V+
M1=2-B+.A__^+L+0```")3="+1="-5>#H>?/__X7`=#"+5>"+!HT$D(M0!(D0
MA=)TX(UT)@"+4`B#P`2)$(72=?2+1="-5>#H2?/__X7`=="+1=2+N+0```"-
ML,.A__^-5>")\.@L\___A<!T,HM5X(L'C020BU`$B1"%TG3AC;0F`````(M0
M"(/`!(D0A=)U](U5X(GPZ/KR__^%P'7.BT74C8AXH?__B["T````B4W,BT7,
MC57@Z-GR__^%P'0PBU7@BP:-!)"+4`2)$(72=."-="8`BU`(@\`$B1"%TG7T
MBT7,C57@Z*GR__^%P'70BT74C8C/H?__B["T````B4W$BT7$C57@Z(CR__^%
MP'0OBU7@BP:-!)"+4`2)$(72=."-=@"+4`B#P`2)$(72=?2+1<2-5>#H6?+_
M_X7`==&+1=2-B-FA__^+L+0```")3<B+1<B-5>#H./+__X7`="^+5>"+!HT$
MD(M0!(D0A=)TX(UV`(M0"(/`!(D0A=)U](M%R(U5X.@)\O__A<!UT8M%U(NX
MM````(VPXZ'__XU5X(GPZ.SQ__^%P'0RBU7@BP>-!)"+4`2)$(72=.&-M"8`
M````BU`(@\`$B1"%TG7TC57@B?#HNO'__X7`=<Z+1=2+N+0```"-L(&A__^-
M5>")\.B=\?__A<!T,XM5X(L'C020BU`$B1"%TG3A+HVT)@````"+4`B#P`2)
M$(72=?2-5>")\.AJ\?__A<!US8M%U(U5X(V`[:'__^A5\?__A<!T"HG"BT7(
MZ`?R__^+?=2-5>"-A_ZA___H-O'__X7`=`V)PHV'$*+__^CE\?__BT74C57@
MC8`;HO__Z!3Q__^%P'0NB<*+1<SHQO'__XM%Y&4K!10```!U.8M%U(V0/*+_
M_XM%T(/$/%M>7UWIHO'__XM%U(U5X(V`*Z+__^C1\/__A<!TQXG"BT7$Z(/Q
M___KN^C,$@``C70F`"Z-M"8`````Z+(2```%WWP``%6)Y8/L&&6+%10```")
M5?2-5?"-@,^A___HB/#__X7`="0/MA"`^C`/E,"$T@^4P@G0@_`!#[;`BU7T
M92L5%````'4&R<,QP.ONZ&<2``"-M"8`````58GE4^AGX/__@<-[?```@^P4
M9:$4````B47TC44,4(E%\(N#F````/]U"&H"_S#_DW0```#'!"3_````_Y,4
M`0``C;8`````+HVT)@````!5B>575E.-G"0``/__@>P`$```@PPD`#G<=?*#
M["SH_=___X'#$7P``(F%U/_^_XG&9:$4````B47D,<!J`FH`:@!6_Y,(`0``
MC8W<__[_@\#_@]+_B87(__[_B[7(__[_B=>)E<S__O_!_Q\Q^"GX,=(E__\`
M`#'Z,?@I^!GZB[W,__[_*<89UX/$$(7_#XCT````C70F`"Z-M"8`````B8W(
M__[_:@!75O^UU/_^__^3"`$``(N-R/_^_R'0@\00@_C_#X3B````@^P$:`@`
M`0!1B8W(__[__[74__[__Y-8````BXW(__[_@\00@_C_#X2:````@_@'=G"-
M1`'X.<AR:(FUR/_^_XT5)`$``(F]S/_^_^L/+HVT)@````"#Z`$YR'(YBSP:
M.3AU\HNS*`$``#EP!'7GB[7(__[_*<B+O<S__O^9`?`1^HM-Y&4K#10```!U
M;XUE]%M>7UW#B[7(__[_B[W,__[_@<8``/__@]?_B?")^C4``/__]]()T`^%
M&/___[C_____B<+KNXM%Y&4K!10```!U*H/L#(V#2Z+__U#H!/[__XM%Y&4K
M!10```!U#X/L#(V#/J+__U#HZ?W__^A$$```C70F`%6)Y5/H1][__X'#6WH`
M`(/L!/^3L````(/L#%#_DZ@```"#Q!"%P'0"BP"+7?S)PU6)Y5?HS0\``(''
M*WH``%:-5>!3@^PL98LU%````(EUY(MU"(V?5Z+__XG8Z,_M__^%P'0%@#@`
M=2:+1>1E*P44````#X60````C67TB?*)V%M>7UWI9N[__XVV`````(/L#(E%
MT%;_EZ````"+5=")1=2)%"2)5<S_EZ````"+3=2-3`$"B0PDB4W0_Y?`````
MBU7,@\0,B474C8>.H?__4E"-AYFB__]64(M-T%%J`O]UU/^7Y````(/$((M%
MY&4K!10```!U$8M5U(UE](G86UY?7>GF[?__Z#$/``"058GE5^CM#@``@<=+
M>0``5HV5(/___U.![`P!``"+10B)A03___]EH10```")1>2-AVZB___'19P`
M````QT6T`````,=%D`````")E13___^)192-AW:A__^)19B-APFB__^)1:2-
MAW.B__^)1:B-AR:B__^)1:R-AWNB__^)1;"-AW^B__^)18B-AX2B__^)18R-
MAWBA__^)A0#____H?>S__X7`=`N)PX`X``^%O@0``/^7L````(/L#%#_EZ@`
M``"#Q!"%P`^$0P0``(LPA?8/A#D$``"#[`Q6_Y>@````C5P``8/$$(/L#%/_
ME\````")P@^V!H/$$(F=#/___XF5^/[__X3`=&B-CXVB__^)C0C___]FD"Z-
MM"8`````@^P,#[;`@\8!4/^U"/___U-J`E*)E1#_____E^0```"+A0S___^+
ME1#___\YPXG!#T/+B<@IV(/``CG(#T+!*<N#P@*#Q""-7`/^#[8&A,!UKHU=
MI(V'$*+__^L,C78`BP.#PP2%P'0WBY44____Z(SK__^)QH7`=.:`.`!TX>B<
M[O__A<!TV(/L#%;_EP0!``")QH/$$(7`=3TNC70F`(U=E.L&C78`@\,$BS.%
M]G0G@#X`#X0F`P``B?#H7^[__X7`=..#[`Q6_Y<$`0``B<:#Q!"%P'30@^P,
M5O^7H````%K_M?C^__^)P_^7H````(V<`P0$``")G0S___^)'"3_E\````")
MA1#___^)PHV'DJ+___^U^/[__U"-AY"A__^)A0C___]0C8>7HO__5HG64%-J
M`E+_E^0```"#Q"AHP`$``%;_EQ@```"#Q!"#^/]U$?^7$````(L`@_@1#X7(
M`@``@^P(_[44_____[40_____Y?4````B<.#Q!"%P'1-@^P,_[40____BX4$
M_____S"-ATRC__]0,?:+AY@```!J`O\P_Y<L````@\0@BT7D92L%%`````^%
M@P0``(UE](GP6UY?7<.-M@````"+A3#___\E`/```#T`0```=:&+M3C_____
ME[`````YQG61BX4P____)?\!```]P`$```^%>____X/L#/^U#/____^7P```
M`(N5%/___XG&C8=BHO__Z-;I__^#Q!"%P`^$@P(``(/L"%"+A03_____,.BJ
M\/__@\00B87X_O__A<`/A&$"``"#[`S_M?C^____EZ````")PH/$$(/X`W90
M@^P(C8>@HO__4(N%^/[__XF5]/[__XU$$/Q0_Y<,`0``@\00A<!U*(N5]/[_
M_X/Z!`^$-P,``(N%^/[__X!\$/LO#X0F`P``+HVT)@````"+E13___^-A\^A
M__^)A03____H*>G__X7`#X3Q`0``#[8`A,`/A.8!```\,`^$W@$``(N%!/__
M_XV7/*+__^B]Z?___Y<4````C8_DHO__B8T4____45"-A\2B____M0C_____
MM1#___]0_[4,____:@)6_Y?D````C8?2HO__@\0@B84$____ZTF-M@````#_
MEQ````"#.!%U3_^7%````(/#`8/L#/^U%/___U-0_[4(_____[40_____[4$
M_____[4,____:@)6_Y?D````@\0P@^P(:,`!``!6_Y<8````@\00@_C_=*:#
M[`S_M1#_____EV0```"+A0#___^)\NCZZ/__B30DZ#+Z__^#Q!#IVOW__V:0
M+HVT)@`````Q]NGQ_/__D"Z-M"8`````C5V(ZP:-=@"#PP2+`X7`='V+E13_
M___H].?__X7`=.B`.`!TXX/L#%#_EP0!``"#Q!"%P'32B<;IBOO__X/L"%#_
MM1#___^+A03_____,(V'$*/__^E0_?__C;8`````@^P,4.BG^?__@\00BT7D
M92L%%`````^%UP$``(E="(UE](GX6UY?7?^@!`$``(/L#(VW9Z+__[L-````
M:@W_E\````")PH/$$(F%^/[__[A3````QX4,____#0```.DW^___+HUT)@"+
MA03___^+`(F%^/[__^F,_?__+HUT)@"#[`AJ`/^U^/[___^7*````(/$$(7`
M#X0&_O__B844____Z`GW__^)A?C^__^)E?S^__^%T@^(Y_W__X/`]FH`@]+_
M4E"+C13___]1_Y<(`0``C46"@\0,:@90BXT4____4?^76````(N-%/___X/$
M$(/X!@^%_@```(%]@@!#04,/A9O]__]F@7V&2$4/A8_]__]J`(N%^/[__XU=
MNXN5_/[__X/`SH/2_U)048F-%/____^7"`$``(/$#&HH4XN-%/___U'_EU@`
M``"#Q!"#^"@/A:````"-A^2B___&1>,`4(V'M:+__U/_M0C_____M1#___]0
M_[4,____:@)6_Y?D````@\0@Z<S]__^+A03___^+2`2%R0^$T?S__X/L#%&)
MC03_____EZ````"#Q!"#^`,/AK7\__^#[`B-EZ6B__]2BXT$____C40!_%#_
MEPP!``"+C03___^#Q!"%P`]%C?C^__^)C?C^___I??S__^C(!P``BT7D92L%
M%````'7O@^P,C8>JHO__4.A-]?__+HUT)@`NC;0F`````%6)Y5=6C57@4^BB
MU?__@<.V<0``@^PL98LU%````(EUY(NS@````&6+#@^V062)3=2(066-@[.A
M___H4>7__X7`=`R`.#&+3=0/A'$!``#H3/#__V6+#HV#XZ+__VH#:@%04?^3
M2````(/$$(7`=#"#[`2-D^6B__]2BU`,B474_S)E_S;_DTP```"+1=2+0`R#
MQ!"+`/9`"D`/A4@!``"-@^JB__]J`VH!4&7_-O^32````(/$$(7`=#2)1=2-
M5>"-@X&A___HQ.3__XM-U(G"BT$,BP"%T@^$\@```(/L!%)09?\V_Y-,````
M@\00@^P$C8/LHO__:@%09?\V_Y,@````C9,!H___@\0,4E!E_S;_DTP```!E
MBP:-5>#&0&4`C8-XH?__Z&/D__^#Q!")@SP!``")QH7`=$2#[`AHP`$``%;_
MDQ@```"-4`&#Q!"#^@%V"8/X$0^%G@```(M%Y&4K!10````/A;$```"-9?1;
M7E]=PY`NC;0F`````(/L#/^S0`$``.CB]O__@\00B8,\`0``B<:%P'6?BT7D
M92L%%````'5W@^P,C8.@H___4/^3W````(VT)@````"`>`$`#X2-_O__Z8#^
M__^0BY-``0``BQ+HT^+__^D-____C;8`````@^P(4&7_-O^3Q````(/$$.FC
M_O__BU7D92L5%````'474%:+@T`!``#_,(V#Q*/__U#_D]P```#H>P4``(UV
M`"Z-M"8`````58GE5E.-5?#H<]/__X'#AV\``(/L(&6+-10```")=?2+=0B-
M@\^A___H,N/__X7`=!L/MA"$T@^4P(#Z,`^4P@C0=0F%]G0%@#X`=1.+1?1E
M*P44````=6>-9?A;7EW#@^P,5NAW[/__B<)866HO4HE5Y/^35````(M5Y(7`
MC4@!#T71B10D_Y,$`0``@\0,C9.2HO__:@124/^3[````(/$$(7`=::+1?1E
M*P44````=0V-9?B)\%M>7>E0YO__Z*L$``"-=@`NC;0F`````.B2!```!;]N
M``!5B>6#[!1J8/^0P````,<``2-%9\=`!(FKS>_'0`C^W+J8QT`,=E0R$,=`
M$/#ATL/'0!0`````QT`8`````,=`7`````#)PXVT)@`````NC;0F`````%6)
MY5?H_0,``(''6VX``%93@^P<BU4(BTT0BUH8C03-``````-"%(UR'(E"%(G(
M@],`P>@=`=B)0AB+0ER%P`^%0P$``(/Y/P^.L@```(G/BUT,B54(P>\&B4T0
MC4?_P><&`WT,B47DB?")WHG#9I"+!H/&0(D#BT;$B4,$BT;(B4,(BT;,B4,,
MBT;0B4,0BT;4B4,4BT;8B4,8BT;<B4,<BT;@B4,@BT;DB4,DBT;HB4,HBT;L
MB4,LBT;PB4,PBT;TB4,TBT;XB4,XBT;\B4,\BT4(Z*?2__\Y_G63BT7DBWT,
MB=Z+31"+50B)P\'@!HU$!T#WVXE%#,'C!HU,&<"#^01R6XM%#(L`B4(<BT4,
MBT0(_(E$#OR-0B"+70R#X/PIQBGS`<Z#YOR#_@1R(8E5"(/F_#'_BQ0[B10X
M@\<$.?=R\XM5")`NC;0F`````(E*7(UE]%M>7UW#+HUT)@"%R73LBT4,#[8`
MB$(<]L$"=-Z+10P/MT0(_F:)1`[^Z\^-M"8`````NT`````IPSG+#T_9@^P$
M`?!3_W4,4/^7G````(M5"(G8BTT0`T)<@\00@_A`B4)<=9DIV8G0`5T,B4T0
MZ+?1__^+31"+50CI<?[__XUT)@`NC;0F`````.A2`@``!7]L``!5B>575E.#
M[!R+70R+=0B)1>2+0Q2+4QB-2QR)Q\'H`P_*B57<@^`_#\^)?>"-4`'&1`,<
M@(T$$8/Z.`^.!0$``+]`````*==T'C'2]\<!````#X6.`0``Q@00`,9$$`$`
M@\(".?IR\(G8B4W8Z"31___'0QP`````BTW8,<"->03'030`````@^?\*?F#
MP3C!Z0+SJXM%W(E#5(M%X(E#6(G8Z.[0__^+`\'H&(@&BP/!Z!"(1@&+`XAF
M`HL#B$8##[9#!XA&!`^W0P:(1@6+0P2(9@:+0P2(1@</MD,+B$8(#[=#"HA&
M"8M#"(AF"HM#"(A&"P^V0P^(1@P/MT,.B$8-BT,,B&8.BT,,B$8/#[9#$XA&
M$`^W0Q*(1A&+0Q"(9A*+0Q"(1A.+1>2)70B#Q!Q;7E]=_Z!D````9I`NC;0F
M`````+DX````*=&#^01R;(U0!,<``````(/B_,=$"/P`````*=`!R(/@_(/X
M!`^"'____XG'N00```"#Y_R)?=B->/_'`@````#![P(YP0^#_O[__X/G`75B
MQP0*`````,=$"@0`````@\$(.TW8<NGIW?[__XVV`````(7)#X3/_O__Q@``
M]L$"#X3#_O__,=)FB50(_NFW_O__+HVT)@````"Z`0```,8``#GZ#X)B_O__
MZ6W^__^-=@#'0@0`````N0@````[3=ARC>F!_O__BPPDPXLT),.+/"3#9I!F
MD&:0Z"(````%3VH``%6)Y8/L"/^P(`$``/]U$/]U#/]U"/^00````,G#BP0D
MPV:09I"0Z/+___\%'VH``%6)Y8/L"/^0R```````4X/L".COS?__@<,#:@``
M@\0(6\,`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````P````$``@`O<')O8R]S96QF+V5X90`E<R\E<P!$>6YA3&]A9&5R
M.CIB;V]T7T1Y;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`+W!R;V,O
M)6DO)7,`+@!005)?5$5-4`!005)?4%)/1TY!344`.@`O`%!%4DPU3$E"`%!%
M4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!
M5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!005)?1TQ/0D%,
M7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!
M3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`,0!L<V5E:R!F86EL960`<F5A9"!F
M86EL960`3$1?3$E"4D%265]0051(`%-94U1%30`O=&UP`%1%35!$25(`5$U0
M`%5315(`55-%4DY!344`)3`R>`!P87(M`"5S)7,E<R5S`'!A<FP`+G!A<@!S
M:&]R="!R96%D`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U)7,`)7,E<W1E
M;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D524TE/3@`Q+C`V
M-``M90`M+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O
M<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E
M8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R
M>2!Y;W5R(&]P97)A=&EO;BD*``!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I
M<F5C=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D
M:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````&5V86P@)V5X96,@
M+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R
M=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@
M*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S
M:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,
M15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?
M<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B
M(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I
M>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM
M>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[
M("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I
M>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A
M9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I
M.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P
M*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*
M?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP
M9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.
M?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA
M;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@
M9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H
M*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D[
M"G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@
M)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!
M5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N
M86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P*
M*3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@
M/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F
M("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB
M<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I
M;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE
M.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y
M<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S
M<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)
M3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N
M86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q
M=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P
M("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@
M("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE
M>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R
M+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;
M7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*
M;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D[
M"FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'
M24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H
M;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T
M(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D
M(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*
M;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA
M9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN
M9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!
M1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G
M:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T
M7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L
M(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@
M)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O
M=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B
M=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ
M93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S
M970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P
M.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U
M:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@
M("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F
M(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B
M=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO
M=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C
M<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q
M(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH
M/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@
M7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X
M="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F
M:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD
M4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64[
M"B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M
M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]
M/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R
M97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T
M.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]
M($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I
M;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I
M;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D
M=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?
M0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@
M/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@
M=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I
M;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE
M;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP
M;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64[
M"F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@
M(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D
M=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!
M4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E
M<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K
M97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA
M;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE
M(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92![
M"F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M
M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE
M?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z
M:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M
M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@
M+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?
M87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I
M.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D55
M4T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S
M(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2![
M"FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A
M;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R
M:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E
M)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T
M($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`
M861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;
M04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I
M('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@
M>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF
M="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]
M("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*
M?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I
M9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,
M)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P
M96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*
M)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD
M(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*
M:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ
M)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D
M7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O
M7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q
M=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@
M)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A
M<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N
M)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z
M.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E
M861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ
M05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E
M("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@
M;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T
M82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M
M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE
M9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q
M<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR
M96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@
M9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@[
M"GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H
M"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2
M*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#
M86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D[
M"G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A
M<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A
M<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F
M.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@
M86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2
M.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P
M<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF
M("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ
M7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L
M97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B
M=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R
M8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X
M<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM
M>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H
M87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N
M9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V
M<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@
M:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D
M*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@
M;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI
M)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II
M9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@
M)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H
M)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE
M+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M
M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D
M;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E
M<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*
M;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I
M;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY
M("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F
M:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ
M1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D
M/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^
M("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N
M="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H
M+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL
M"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S
M,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N
M="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D
M960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D
M;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N
M9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R
M;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P
M87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*
M:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z
M.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D
M9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD
M:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L
M("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A
M8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I
M;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F
M:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.
M)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q
M<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D
M(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S
M=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ
M2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I
M;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE
M+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z
M.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I
M;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A
M73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II
M<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N
M9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI
M"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA
M;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*
M<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R
M;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF
M("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A
M="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB
M97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M
M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE
M;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@
M/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.
M5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M
M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO
M=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A
M9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q
M6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN
M72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT
M<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U
M-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL
M97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@
M=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER
M(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P
M(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/
M1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B
M($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z
M.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER
M92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S
M<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R
M97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A
M9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E
M('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER
M92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE
M.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q
M=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I
M<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*
M979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T[
M"F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER
M92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE
M.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/
M.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T
M9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A
M<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA
M<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-
M4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM
M>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P
M=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R
M.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M
M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W
M=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D
M14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H
M)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK
M9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM0
M05)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92![
M"F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S
M(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA
M9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%
M(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#
M2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G
M:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B
M*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C
M:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B
M8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER
M*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`]
M(#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM
M:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I
M<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A
M;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R![
M"FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN
M86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE
M*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO
M<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C
M;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I
M;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I
M9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I
M(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*
M<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'
M3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R
M;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%2
M7U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^
M/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D
M<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%2
M7U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E
M='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5]
M(BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R
M96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.
M5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@
M)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L
M:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O
M9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L
M:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D
M<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD
M,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD
M97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@
M/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I
M;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D
M("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*
M8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?
M9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-
M17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%
M3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E
M('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%7
M3D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L
M971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,
M14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/
M0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II
M9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*
M?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@
M>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B([
M"GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#
M+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*
M87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*
M=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X[
M"G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*
M4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@
M;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R
M(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA
M;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K
M*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`
M.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)
M5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*
M,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,````!&P,[&`$``"(```"P
M?___8`$``-!___^(!0``,(+__S0!``"0@___A`$``&"1__^T`0``0)+__^0!
M````D___)`(``&"5__]8`@``\)7__X0"``"@E___M`(```"8___<`@``0)C_
M__P"```@F?__+`,``#"9__]``P``@)O__W`#``#`F___F`,``%"=___8`P``
MD*'__P@$``#PH?__+`0``$"B__](!```$*3__W@$``!`I/__G`0``""E___<
M!```L*S__R`%``#@KO__4`4``+"O___0!0``$+#___`%``#PL?__)`8```ZT
M__]4!@``$K3__V@&```6M/__?`8``""T__^0!@``1[3__[`&``!0M/__Q`8`
M`!0``````````7I2``%\"`$;#`0$B`$'"!`````<````](#__RP`````````
M%``````````!>E(``7P(`1L,!`2(`0``(````!P```!(?O__(``````."$8.
M#$H/"W0$>``_&CLJ,B0B+````$`````$@O__S@T```!!#@B%`D(-!4.'`X8$
M@P4#OPT*PT'&0<=!Q0P$!$$++````'````"DC___W`````!!#@B%`D(-!5"'
M`X8$@P4"?0K#0<9!QT'%#`0$00L`/````*````!4D/__OP````!,#@B%`D(-
M!4:'`X8$@P4";`K#0<9!QT'%#`0$10MO"L-!QD''0<4,!`1!"P```#````#@
M````U)#__U("````00X(A0)"#05!AP-,A@1#@P4#(`$*PT'&0<=!Q0P$!$H+
M```H````%`$```"3__^"`````$$."(4"0@T%088#3X,$`F<*PT'&0<4,!`1!
M"RP```!``0``9)/__Z(!````00X(A0)"#05"AP.&!%*#!6D*PT'&0<=!Q0P$
M!$4+`"0```!P`0``Y)3__V``````00X(A0)"#05"A@.#!`)8PT'&0<4,!`0<
M````F`$``!R5__\X`````$L."(4"0@T%:L4,!`0``"P```"X`0``/)7__]H`
M````00X(A0)"#05:AP.&!(,%`K0*PT'&0<=!Q0P$!$$+`!````#H`0``[)7_
M_P4`````````+````/P!``#HE?__1@(```!!#@B%`D(-!5.'`X8$@P4"?0K#
M0<9!QT'%#`0$2@L`)````"P"```(F/__.P````!!#@B%`D(-!4&&`TR#!&/#
M0<9!Q0P$!#P```!4`@``()C__XD!````00X(A0)"#050AP.&!(,%`M`*PT'&
M0<=!Q0P$!$H+7PK#0<9!QT'%#`0$3@L````L````E`(``'"9__\T!````$P.
M"(4"0@T%1H<#A@2#!0/R`PK#0<9!QT'%#`0$10L@````Q`(``("=__]9````
M`$L."(4"0@T%`D(*Q0P$!$$+```8````Z`(``+R=__]"`````$$."(4"0@T%
M08,#+`````0#``#PG?__S`$```!!#@B%`D(-!4.'`X8$@P4#5@$*PT'&0<=!
MQ0P$!$$+(````#0#``"0G___,`````!!#@B%`D(-!4&#`VO%PPP$!```/```
M`%@#``"<G___WP````!!#@B%`D(-!4&'`U.&!(,%>PK#0<9!QT'%#`0$2PL"
M<@K#0<9!QT'%#`0$10L``$````"8`P``/*#__X,'````00X(A0)"#05!AP-9
MA@2#!0/)`@K#0<9!QT'%#`0$1PL#I0(*PT'&0<=!Q0P$!$8+````+````-P#
M``"(I___)0(```!!#@B%`D(-!4:'`X8$@P4#:@$*PT'&0<=!Q0P$!$H+-```
M``P$``"(J?__Q0````!!#@B%`D(-!46&`X,$`E4*PT'&0<4,!`1!"P)9"L-!
MQD'%#`0$10M$````1`0``$!Z__]4`@```$0,`0!)$`4"=0!!$`<"=7Q$$`8"
M=7@0`P)U=$P/`W5P!@-M`0K!#`$`0<-!QD''0<5##`0$00L<````C`0``-BI
M__]1`````$L."(4"0@T%`D/%#`0$`#````"L!```&*K__]0!````00X(A0)"
M#05!AP-0A@2#!0-#`0K#0<9!QT'%#`0$1@L````L````X`0``,2K__\>`@``
M`$L."(4"0@T%1H<#A@2#!0,J`0K#0<9!QT'%#`0$4`L0````$`4``+*M__\$
M`````````!`````D!0``HJW__P0`````````$````#@%``"2K?__!```````
M```<````3`4``(BM__\G`````$L."(4"0@T%6<4,!`0``!````!L!0``CZW_
M_P0`````````&````(`%``"$K?__%@````!+#@B%`D(-!0`````````$````
M$`````$```!'3E4```````0````$````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````,`C``!P(P```0````8#```!````$0,``!T```"W
M!```#``````@```-````V%0``!D```"\O0``&P````0````:````P+T``!P`
M```$````]?[_;S0"```%````?`D```8```!\`P``"@```-H$```+````$```
M`!4``````````P```.2^```"````"````!0````1````%P```-@1```1````
MR`\``!(````0`@``$P````@````>````"````/O__V\!```(_O__;Q@/``#_
M__]O`0```/#__V]6#@``)````.`1```C````$````"4````$````````````
M````````````````````````````````````````````````````````Q+T`
M````````````-B``````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````$`@````````!,````I005(N<&T*`````.QB``!'0T,Z("A'3E4I
M(#$U+C(N,2`R,#(U,#@Q,P``+G-H<W1R=&%B`"YN;W1E+F=N=2YB=6EL9"UI
M9``N;F]T92YG;G4N<')O<&5R='D`+FEN=&5R<``N9VYU+FAA<V@`+F1Y;G-Y
M;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L+F1Y
M;@`N<F5L+G!L=``N<F5L<BYD>6X`+FEN:70`+G1E>'0`+F9I;FD`+G)O9&%T
M80`N96A?9G)A;65?:&1R`"YE:%]F<F%M90`N;F]T92Y!0DDM=&%G`"YI;FET
M7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N
M8V]M;65N=```````````````````````````````````````````````````
M````"P````<````"````U`$``-0!```D```````````````$`````````!X`
M```'`````@```/@!``#X`0``*```````````````!``````````Q`````0``
M``(````@`@``(`(``!,```````````````$`````````.0```/;__V\"````
M-`(``#0"``!(`0``!0`````````$````!````$,````+`````@```'P#``!\
M`P````8```8````!````!````!````!+`````P````(```!\"0``?`D``-H$
M``````````````$`````````4P```/___V\"````5@X``%8.``#`````!0``
M```````"`````@```&````#^__]O`@```!@/```8#P``L`````8````!````
M!`````````!O````"0````(```#(#P``R`\``!`"```%``````````0````(
M````>`````D```!"````V!$``-@1```(````!0```!<````$````"````($`
M```3`````@```.`1``#@$0``$```````````````!`````0```"+`````0``
M``8`````(````"```"````````````````0`````````?`````$````&````
M("```"`@```@```````````````0````!````)$````!````!@```$`@``!`
M(```EC0`````````````0`````````"7`````0````8```#85```V%0``!0`
M``````````````0`````````G0````$````"`````&````!@``!N0```````
M```````$`````````*4````!`````@```'"@``!PH```'`$`````````````
M!`````````"S`````0````(```",H0``C*$``,@%``````````````0`````
M````O0````<````"````5*<``%2G```@```````````````$`````````,L`
M```.`````P```+R]``"\K0``!```````````````!`````0```#7````#P``
M``,```#`O0``P*T```0```````````````0````$````XP````8````#````
MQ+T``,2M```@`0``!@`````````$````"````.P````!`````P```.2^``#D
MK@``'`$`````````````!`````0```#Q`````0````,`````P````+```!@`
M``````````````0`````````]P````@````#````&,```!BP```4````````
M```````$`````````/P````!````,``````````8L```&P``````````````
M`0````$````!`````P``````````````,[````4!``````````````$`````
#````
