/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.deployment.dev.filesystem.watch;

import io.quarkus.deployment.dev.filesystem.watch.FileChangeCallback;
import io.quarkus.deployment.dev.filesystem.watch.FileChangeEvent;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jboss.logging.Logger;

public class WatchServiceFileSystemWatcher
implements Runnable {
    private static final Logger log = Logger.getLogger(WatchServiceFileSystemWatcher.class);
    private static final AtomicInteger threadIdCounter = new AtomicInteger(0);
    private WatchService watchService;
    private final Map<Path, PathData> monitoredDirectories = Collections.synchronizedMap(new HashMap());
    private final Map<WatchKey, PathData> pathDataByKey = Collections.synchronizedMap(new IdentityHashMap());
    private volatile boolean stopped = false;
    private final Thread watchThread;

    public WatchServiceFileSystemWatcher(String name, boolean daemon) {
        try {
            this.watchService = FileSystems.getDefault().newWatchService();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        this.watchThread = new Thread((Runnable)this, name + " - " + String.valueOf(threadIdCounter));
        this.watchThread.setDaemon(daemon);
        this.watchThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (!this.stopped) {
            try {
                WatchKey key = this.watchService.take();
                if (key == null) continue;
                try {
                    PathData pathData = this.pathDataByKey.get(key);
                    if (pathData == null) continue;
                    ArrayList<FileChangeEvent> results = new ArrayList<FileChangeEvent>();
                    List<WatchEvent<?>> events = key.pollEvents();
                    HashSet<Path> addedFiles = new HashSet<Path>();
                    HashSet<Path> deletedFiles = new HashSet<Path>();
                    for (WatchEvent<?> watchEvent : events) {
                        FileChangeEvent.Type type;
                        Path eventPath = (Path)watchEvent.context();
                        Path targetFile = ((Path)key.watchable()).resolve(eventPath).toAbsolutePath();
                        if (watchEvent.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                            type = FileChangeEvent.Type.ADDED;
                            addedFiles.add(targetFile);
                            if (Files.isDirectory(targetFile, new LinkOption[0])) {
                                try {
                                    this.addWatchedDirectory(pathData, targetFile);
                                }
                                catch (IOException e) {
                                    log.debugf((Throwable)e, "Could not add watched directory %s", (Object)targetFile);
                                }
                            }
                        } else if (watchEvent.kind() == StandardWatchEventKinds.ENTRY_MODIFY) {
                            type = FileChangeEvent.Type.MODIFIED;
                        } else {
                            if (watchEvent.kind() != StandardWatchEventKinds.ENTRY_DELETE) continue;
                            type = FileChangeEvent.Type.REMOVED;
                            deletedFiles.add(targetFile);
                        }
                        results.add(new FileChangeEvent(targetFile, type));
                    }
                    key.pollEvents().clear();
                    Iterator it = results.iterator();
                    while (it.hasNext()) {
                        FileChangeEvent fileChangeEvent = (FileChangeEvent)it.next();
                        if (!pathData.isMonitored(fileChangeEvent.getFile())) {
                            it.remove();
                            continue;
                        }
                        if (fileChangeEvent.getType() == FileChangeEvent.Type.MODIFIED) {
                            if (addedFiles.contains(fileChangeEvent.getFile()) && deletedFiles.contains(fileChangeEvent.getFile()) || !addedFiles.contains(fileChangeEvent.getFile()) && !deletedFiles.contains(fileChangeEvent.getFile())) continue;
                            it.remove();
                            continue;
                        }
                        if (fileChangeEvent.getType() == FileChangeEvent.Type.ADDED) {
                            if (!deletedFiles.contains(fileChangeEvent.getFile())) continue;
                            it.remove();
                            continue;
                        }
                        if (fileChangeEvent.getType() != FileChangeEvent.Type.REMOVED || !addedFiles.contains(fileChangeEvent.getFile())) continue;
                        it.remove();
                    }
                    if (results.isEmpty()) continue;
                    for (FileChangeCallback callback : pathData.getCallbacks()) {
                        WatchServiceFileSystemWatcher.invokeCallback(callback, results);
                    }
                }
                finally {
                    if (key.reset()) continue;
                    this.monitoredDirectories.remove(key.watchable());
                }
            }
            catch (InterruptedException key) {
            }
            catch (ClosedWatchServiceException cwse) {
                break;
            }
        }
    }

    public synchronized void watchDirectoryRecursively(Path directory, FileChangeCallback callback) {
        try {
            Path absoluteDirectory = directory.toAbsolutePath();
            PathData data = this.monitoredDirectories.get(absoluteDirectory);
            if (data == null) {
                Set<Path> allDirectories = WatchServiceFileSystemWatcher.doScan(absoluteDirectory).keySet();
                data = new PathData(absoluteDirectory, List.of());
                for (Path dir : allDirectories) {
                    this.addWatchedDirectory(data, dir);
                }
                this.monitoredDirectories.put(absoluteDirectory, data);
            }
            data.addCallback(callback);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public synchronized void watchFiles(Path directory, List<Path> monitoredFiles, FileChangeCallback callback) {
        try {
            Path absoluteDirectory = directory.toAbsolutePath();
            PathData data = this.monitoredDirectories.get(absoluteDirectory);
            if (data == null) {
                data = new PathData(absoluteDirectory, monitoredFiles);
                this.addWatchedDirectory(data, absoluteDirectory);
                this.monitoredDirectories.put(absoluteDirectory, data);
            }
            data.addCallback(callback);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void addWatchedDirectory(PathData data, Path dir) throws IOException {
        WatchKey key = dir.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
        this.pathDataByKey.put(key, data);
        data.addWatchKey(key);
    }

    public synchronized void unwatchPath(Path directory, FileChangeCallback callback) {
        PathData data = this.monitoredDirectories.get(directory);
        if (data != null) {
            data.removeCallback(callback);
            if (data.getCallbacks().isEmpty()) {
                this.monitoredDirectories.remove(directory);
                for (WatchKey key : data.getWatchKeys()) {
                    key.cancel();
                    this.pathDataByKey.remove(key);
                }
            }
        }
    }

    public void close() throws IOException {
        this.stopped = true;
        this.watchThread.interrupt();
        if (this.watchService != null) {
            this.watchService.close();
        }
    }

    private static Map<Path, Long> doScan(Path directory) {
        HashMap<Path, Long> results = new HashMap<Path, Long>();
        ArrayDeque<Path> toScan = new ArrayDeque<Path>();
        toScan.add(directory);
        while (!toScan.isEmpty()) {
            Path next = (Path)toScan.pop();
            if (!Files.isDirectory(next, new LinkOption[0])) continue;
            try {
                results.put(next, Files.getLastModifiedTime(directory, new LinkOption[0]).toMillis());
                Stream<Path> list = Files.list(next);
                try {
                    list.forEach(p -> toScan.push(p.toAbsolutePath()));
                }
                finally {
                    if (list == null) continue;
                    list.close();
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException("Unable to scan: " + String.valueOf(next), e);
            }
        }
        return results;
    }

    private static void invokeCallback(FileChangeCallback callback, List<FileChangeEvent> results) {
        try {
            callback.handleChanges(results);
        }
        catch (Exception e) {
            log.error((Object)"Failed to invoke watch callback", (Throwable)e);
        }
    }

    private class PathData {
        private final Path path;
        private final List<FileChangeCallback> callbacks = new ArrayList<FileChangeCallback>();
        private final List<WatchKey> watchKeys = new ArrayList<WatchKey>();
        private final List<Path> monitoredFiles;

        private PathData(Path path, List<Path> monitoredFiles) {
            this.path = path;
            this.monitoredFiles = monitoredFiles.stream().map(p -> path.resolve((Path)p).toAbsolutePath()).collect(Collectors.toList());
        }

        private void addWatchKey(WatchKey key) {
            this.watchKeys.add(key);
        }

        private void addCallback(FileChangeCallback callback) {
            this.callbacks.add(callback);
        }

        private void removeCallback(FileChangeCallback callback) {
            this.callbacks.remove(callback);
        }

        private List<FileChangeCallback> getCallbacks() {
            return this.callbacks;
        }

        private List<WatchKey> getWatchKeys() {
            return this.watchKeys;
        }

        private boolean isMonitored(Path file) {
            if (this.monitoredFiles.isEmpty()) {
                return true;
            }
            Path absolutePath = file.isAbsolute() ? file : file.toAbsolutePath();
            for (Path monitoredFile : this.monitoredFiles) {
                if (!monitoredFile.equals(absolutePath)) continue;
                return true;
            }
            return false;
        }
    }
}

