/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.opinion;

import ghidra.app.util.MemoryBlockUtils;
import ghidra.app.util.Option;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.format.mz.MzExecutable;
import ghidra.app.util.bin.format.mz.MzRelocation;
import ghidra.app.util.bin.format.mz.OldDOSHeader;
import ghidra.app.util.importer.MessageLog;
import ghidra.app.util.opinion.AbstractLibrarySupportLoader;
import ghidra.app.util.opinion.LoadSpec;
import ghidra.app.util.opinion.Loader;
import ghidra.app.util.opinion.QueryOpinionService;
import ghidra.app.util.opinion.QueryResult;
import ghidra.program.database.mem.FileBytes;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressFactory;
import ghidra.program.model.address.AddressOutOfBoundsException;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.address.SegmentedAddress;
import ghidra.program.model.address.SegmentedAddressSpace;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataUtilities;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.ContextChangeException;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ProgramContext;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.reloc.Relocation;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.symbol.SymbolUtilities;
import ghidra.util.LittleEndianDataConverter;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

public class MzLoader
extends AbstractLibrarySupportLoader {
    public static final String MZ_NAME = "Old-style DOS Executable (MZ)";
    private static final String ENTRY_NAME = "entry";
    private static final int INITIAL_SEGMENT_VAL = 4096;
    private static final int FAR_RETURN_OPCODE = 203;
    private static final byte MOVW_DS_OPCODE = -70;
    private static final long MIN_BYTE_LENGTH = 4L;

    @Override
    public Collection<LoadSpec> findSupportedLoadSpecs(ByteProvider provider) throws IOException {
        ArrayList<LoadSpec> loadSpecs = new ArrayList<LoadSpec>();
        if (provider.length() < 4L) {
            return loadSpecs;
        }
        MzExecutable mz = new MzExecutable(provider);
        OldDOSHeader header = mz.getHeader();
        if (header.isDosSignature() && !header.hasNewExeHeader() && !header.hasPeHeader()) {
            List<QueryResult> results = QueryOpinionService.query(this.getName(), "" + header.e_magic(), null);
            for (QueryResult result : results) {
                loadSpecs.add(new LoadSpec((Loader)this, 0L, result));
            }
            if (loadSpecs.isEmpty()) {
                loadSpecs.add(new LoadSpec((Loader)this, 0L, true));
            }
        }
        return loadSpecs;
    }

    @Override
    public void load(ByteProvider provider, LoadSpec loadSpec, List<Option> options, Program program, TaskMonitor monitor, MessageLog log) throws IOException, CancelledException {
        FileBytes fileBytes = MemoryBlockUtils.createFileBytes(program, provider, monitor);
        AddressFactory af = program.getAddressFactory();
        if (!(af.getDefaultAddressSpace() instanceof SegmentedAddressSpace)) {
            throw new IOException("Selected Language must have a segmented address space.");
        }
        SegmentedAddressSpace space = (SegmentedAddressSpace)af.getDefaultAddressSpace();
        MzExecutable mz = new MzExecutable(provider);
        try {
            Set<RelocationFixup> relocationFixups = this.getRelocationFixups(space, mz, log, monitor);
            this.processMemoryBlocks(program, fileBytes, space, mz, relocationFixups, log, monitor);
            this.adjustSegmentStarts(program, monitor);
            this.processRelocations(program, space, mz, relocationFixups, log, monitor);
            this.processEntryPoint(program, space, mz, log, monitor);
            this.processRegisters(program, mz, log, monitor);
            this.markupHeaders(program, fileBytes, mz, log, monitor);
        }
        catch (CancelledException e) {
            return;
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    @Override
    public String getName() {
        return MZ_NAME;
    }

    @Override
    public int getTierPriority() {
        return 60;
    }

    private void markupHeaders(Program program, FileBytes fileBytes, MzExecutable mz, MessageLog log, TaskMonitor monitor) {
        monitor.setMessage("Marking up headers...");
        OldDOSHeader header = mz.getHeader();
        int blockSize = this.paragraphsToBytes(Short.toUnsignedInt(header.e_cparhdr()));
        try {
            Address headerSpaceAddr = AddressSpace.OTHER_SPACE.getAddress(0L);
            MemoryBlock headerBlock = MemoryBlockUtils.createInitializedBlock(program, true, "HEADER", headerSpaceAddr, fileBytes, 0L, blockSize, "", "", false, false, false, log);
            Address addr = headerBlock.getStart();
            DataUtilities.createData((Program)program, (Address)addr, (DataType)mz.getHeader().toDataType(), (int)-1, (DataUtilities.ClearDataMode)DataUtilities.ClearDataMode.CHECK_FOR_SPACE);
            List<MzRelocation> relocations = mz.getRelocations();
            if (!relocations.isEmpty()) {
                DataType relocationType = relocations.get(0).toDataType();
                int len = relocationType.getLength();
                addr = addr.add((long)Short.toUnsignedInt(header.e_lfarlc()));
                for (int i = 0; i < relocations.size(); ++i) {
                    monitor.checkCancelled();
                    DataUtilities.createData((Program)program, (Address)addr.add((long)(i * len)), (DataType)relocationType, (int)-1, (DataUtilities.ClearDataMode)DataUtilities.ClearDataMode.CHECK_FOR_SPACE);
                }
            }
        }
        catch (Exception e) {
            log.appendMsg("Failed to markup headers");
        }
    }

    private void processMemoryBlocks(Program program, FileBytes fileBytes, SegmentedAddressSpace space, MzExecutable mz, Set<RelocationFixup> relocationFixups, MessageLog log, TaskMonitor monitor) throws Exception {
        int extraAllocSize;
        int endOffset;
        monitor.setMessage("Processing memory blocks...");
        OldDOSHeader header = mz.getHeader();
        BinaryReader reader = mz.getBinaryReader();
        TreeSet<SegmentedAddress> knownSegments = new TreeSet<SegmentedAddress>();
        relocationFixups.stream().filter(RelocationFixup::valid).forEach(rf -> knownSegments.add(space.getAddress(rf.segment, 0)));
        knownSegments.add(space.getAddress(4096, 0));
        if (header.e_cs() > 0) {
            knownSegments.add(space.getAddress(4096 + header.e_cs() & 0xFFFF, 0));
        }
        if ((long)(endOffset = this.pagesToBytes(Short.toUnsignedInt(header.e_cp()) - 1) + Short.toUnsignedInt(header.e_cblp())) > reader.length()) {
            log.appendMsg("File is 0x%x bytes but header reports 0x%x".formatted(reader.length(), endOffset));
            endOffset = (int)reader.length();
        }
        MemoryBlock lastBlock = null;
        ArrayList orderedSegments = new ArrayList(knownSegments);
        for (int i = 0; i < orderedSegments.size(); ++i) {
            MemoryBlock block;
            MemoryBlock block2;
            SegmentedAddress segmentAddr = (SegmentedAddress)orderedSegments.get(i);
            int segmentFileOffset = this.addressToFileOffset(segmentAddr.getSegment() - 4096 & 0xFFFF, 0, header);
            if (segmentFileOffset < 0) {
                log.appendMsg("Invalid segment start file location: " + segmentFileOffset);
                continue;
            }
            int numBytes = 0;
            if (i + 1 < orderedSegments.size()) {
                SegmentedAddress end = (SegmentedAddress)orderedSegments.get(i + 1);
                int nextSegmentFileOffset = this.addressToFileOffset(end.getSegment() - 4096 & 0xFFFF, 0, header);
                numBytes = nextSegmentFileOffset - segmentFileOffset;
            } else {
                numBytes = endOffset - segmentFileOffset;
            }
            if (numBytes <= 0) {
                log.appendMsg("No file data available for defined segment at: " + String.valueOf(segmentAddr));
                continue;
            }
            int numUninitBytes = 0;
            if (segmentFileOffset + numBytes > endOffset) {
                int calcNumBytes = numBytes;
                if (segmentFileOffset > endOffset) {
                    numBytes = 0;
                    numUninitBytes = calcNumBytes;
                } else {
                    numBytes = endOffset - segmentFileOffset;
                    numUninitBytes = calcNumBytes - numBytes;
                }
            }
            if (numBytes > 0 && (block2 = MemoryBlockUtils.createInitializedBlock(program, false, "CODE_" + i, (Address)segmentAddr, fileBytes, (long)segmentFileOffset, numBytes, "", "mz", true, true, true, log)) != null) {
                lastBlock = block2;
            }
            if (numUninitBytes <= 0 || (block = MemoryBlockUtils.createUninitializedBlock(program, false, "CODE_" + i + "u", segmentAddr.add((long)numBytes), numUninitBytes, "", "mz", true, true, false, log)) == null) continue;
            lastBlock = block;
        }
        if ((long)endOffset < reader.length()) {
            int extraByteCount = (int)reader.length() - endOffset;
            log.appendMsg(String.format("File contains 0x%x extra bytes starting at file offset 0x%x", extraByteCount, endOffset));
        }
        if (lastBlock != null && (extraAllocSize = this.paragraphsToBytes(Short.toUnsignedInt(header.e_minalloc()))) > 0) {
            MemoryBlockUtils.createUninitializedBlock(program, false, "DATA", lastBlock.getEnd().add(1L), extraAllocSize, "", "mz", true, true, false, log);
        }
    }

    private void adjustSegmentStarts(Program program, TaskMonitor monitor) throws Exception {
        monitor.setMessage("Adjusting segments...");
        if (!program.hasExclusiveAccess()) {
            return;
        }
        Memory memory = program.getMemory();
        MemoryBlock[] blocks = memory.getBlocks();
        block0: for (int i = 1; i < blocks.length; ++i) {
            monitor.checkCancelled();
            MemoryBlock block = blocks[i];
            if (!block.isInitialized()) continue;
            int mIndex = 15;
            if (block.getSize() <= 16L) {
                mIndex = (int)block.getSize() - 2;
            }
            while (mIndex >= 0) {
                monitor.checkCancelled();
                Address offAddr = block.getStart().add((long)mIndex);
                int val = block.getByte(offAddr);
                if ((val &= 0xFF) == 203) {
                    Address splitAddr = offAddr.add(1L);
                    String oldName = block.getName();
                    String oldSourceName = block.getSourceName();
                    memory.split(block, splitAddr);
                    memory.join(blocks[i - 1], blocks[i]);
                    blocks = memory.getBlocks();
                    blocks[i].setName(oldName);
                    blocks[i].setSourceName(oldSourceName);
                    continue block0;
                }
                --mIndex;
            }
        }
        program.getListing().removeTree("Program Tree");
        program.getListing().createRootModule("Program Tree");
    }

    private void processRelocations(Program program, SegmentedAddressSpace space, MzExecutable mz, Set<RelocationFixup> relocationFixups, MessageLog log, TaskMonitor monitor) throws Exception {
        monitor.setMessage("Processing relocations...");
        Memory memory = program.getMemory();
        for (RelocationFixup relocationFixup : relocationFixups) {
            SegmentedAddress relocationAddress = relocationFixup.address();
            Relocation.Status status = Relocation.Status.FAILURE;
            try {
                if (relocationFixup.valid) {
                    memory.setShort((Address)relocationAddress, (short)relocationFixup.segment());
                    status = Relocation.Status.APPLIED;
                }
            }
            catch (MemoryAccessException e) {
                log.appendMsg(String.format("Failed to apply relocation: %s (%s)", relocationAddress, e.getMessage()));
            }
            program.getRelocationTable().add((Address)relocationAddress, status, 0, new long[]{relocationFixup.relocation.getSegment(), relocationFixup.relocation.getOffset(), relocationFixup.segment}, 2, null);
        }
    }

    private void processEntryPoint(Program program, SegmentedAddressSpace space, MzExecutable mz, MessageLog log, TaskMonitor monitor) {
        monitor.setMessage("Processing entry point...");
        OldDOSHeader header = mz.getHeader();
        int ipValue = Short.toUnsignedInt(header.e_ip());
        SegmentedAddress addr = space.getAddress(4096 + header.e_cs() & 0xFFFF, ipValue);
        SymbolTable symbolTable = program.getSymbolTable();
        try {
            symbolTable.createLabel((Address)addr, ENTRY_NAME, SourceType.IMPORTED);
            symbolTable.addExternalEntryPoint((Address)addr);
        }
        catch (InvalidInputException e) {
            log.appendMsg("Failed to process entry point");
        }
    }

    private void processRegisters(Program program, MzExecutable mz, MessageLog log, TaskMonitor monitor) {
        monitor.setMessage("Processing registers...");
        Symbol entry = SymbolUtilities.getLabelOrFunctionSymbol((Program)program, (String)ENTRY_NAME, err -> log.appendMsg(err));
        if (entry == null) {
            return;
        }
        LittleEndianDataConverter converter = LittleEndianDataConverter.INSTANCE;
        boolean shouldSetDS = false;
        long dsValue = 0L;
        try {
            for (MemoryBlock block : program.getMemory().getBlocks()) {
                if (!block.contains(entry.getAddress())) continue;
                byte instByte = block.getByte(entry.getAddress());
                if (instByte == -70) {
                    byte[] dsBytes = new byte[2];
                    block.getBytes(entry.getAddress().addWrap(1L), dsBytes);
                    dsValue = converter.getShort(dsBytes);
                    shouldSetDS = true;
                }
                break;
            }
        }
        catch (MemoryAccessException memoryAccessException) {
            // empty catch block
        }
        OldDOSHeader header = mz.getHeader();
        ProgramContext context = program.getProgramContext();
        Register ss = context.getRegister("ss");
        Register sp = context.getRegister("sp");
        Register ds = context.getRegister("ds");
        Register cs = context.getRegister("cs");
        try {
            context.setValue(sp, entry.getAddress(), entry.getAddress(), BigInteger.valueOf(Short.toUnsignedLong(header.e_sp())));
            context.setValue(ss, entry.getAddress(), entry.getAddress(), BigInteger.valueOf(Integer.toUnsignedLong(header.e_ss() + 4096 & 0xFFFF)));
            for (MemoryBlock block : program.getMemory().getBlocks()) {
                Address start = block.getStart();
                Address end = block.getEnd();
                if (!(start.getAddressSpace() instanceof SegmentedAddressSpace)) continue;
                BigInteger csValue = BigInteger.valueOf(Integer.toUnsignedLong(((SegmentedAddress)start).getSegment()));
                context.setValue(cs, start, end, csValue);
                if (!shouldSetDS) continue;
                context.setValue(ds, start, end, BigInteger.valueOf(dsValue));
            }
        }
        catch (ContextChangeException contextChangeException) {
            // empty catch block
        }
    }

    private Set<RelocationFixup> getRelocationFixups(SegmentedAddressSpace space, MzExecutable mz, MessageLog log, TaskMonitor monitor) throws CancelledException {
        HashSet<RelocationFixup> fixups = new HashSet<RelocationFixup>();
        OldDOSHeader header = mz.getHeader();
        BinaryReader reader = mz.getBinaryReader();
        for (MzRelocation relocation : mz.getRelocations()) {
            monitor.checkCancelled();
            int seg = relocation.getSegment();
            int off = relocation.getOffset();
            int relocationFileOffset = this.addressToFileOffset(seg, off, header);
            SegmentedAddress relocationAddress = space.getAddress(4096 + seg & 0xFFFF, off);
            try {
                int value = Short.toUnsignedInt(reader.readShort(relocationFileOffset));
                int relocatedSegment = 4096 + value & 0xFFFF;
                fixups.add(new RelocationFixup(relocation, relocationAddress, relocationFileOffset, relocatedSegment, true));
            }
            catch (AddressOutOfBoundsException | IOException e) {
                fixups.add(new RelocationFixup(relocation, relocationAddress, relocationFileOffset, 0, false));
                log.appendMsg(String.format("Failed to process relocation: %s (%s)", relocationAddress, e.getMessage()));
            }
        }
        return fixups;
    }

    private int addressToFileOffset(int segment, int offset, OldDOSHeader header) {
        return (segment << 4) + offset + this.paragraphsToBytes(Short.toUnsignedInt(header.e_cparhdr()));
    }

    private int paragraphsToBytes(int paragraphs) {
        return paragraphs << 4;
    }

    private int pagesToBytes(int pages) {
        return pages << 9;
    }

    private record RelocationFixup(MzRelocation relocation, SegmentedAddress address, int fileOffset, int segment, boolean valid) {
    }
}

