/* Copyright 1992-2023 Intel Corporation.                                     */
/*                                                                            */
/* This software and the related documents are Intel copyrighted              */
/* materials, and your use of them is governed by the express license         */
/* under which they were provided to you ("License"). Unless the License      */
/* provides otherwise, you may not use, modify, copy, publish,                */
/* distribute, disclose or transmit this software or the related              */
/* documents without Intel's prior written permission.                        */
/*                                                                            */
/* This software and the related documents are provided as is, with no        */
/* express or implied warranties, other than those that are expressly         */
/* stated in the License.                                                     */

#ifndef _INTEL_IHC_HLS_INTERNAL_HLS_FLOAT_INTER
#define _INTEL_IHC_HLS_INTERNAL_HLS_FLOAT_INTER

#include <stddef.h>

// uncomment to disable assert()
// #define NDEBUG
#include <cassert>

// Use (void) to silence unused warnings.
#define assertm(exp, msg) assert(((void)msg, exp))

// Emulation flow signatures need to be tagged tiwh SYCL_EXTERNAL when using
// this header with the SYCL compiler.
#ifdef __SYCL_COMPILER_VERSION
#define CONDITIONAL_SYCL_EXTERNAL SYCL_EXTERNAL
#else
#define CONDITIONAL_SYCL_EXTERNAL
#endif // __SYCL_COMPILER_VERSION

////////////////////////////////////////////////////////////////////////////////
// Intermediate Class Type
////////////////////////////////////////////////////////////////////////////////

/// Intermediate class to represent an arbitrary precision floating point type
/// between the HLD compiler and the low-level implementation for x86
// unsigned char is the underlying data type represented by
// hls_vpfp_intermediate_ty
struct hls_vpfp_intermediate_ty {
  size_t m_mantissa_bits;    /// Number of bits needed to represent the mantissa
  size_t m_exponent_bits;    /// Number of bits needed to represent the exponent
  unsigned char *m_mantissa; /// Stores the actual mantissa
  unsigned char *m_exponent; /// Stores the actual exponent
  bool m_sign_bit;           /// Contains the sign bit
};

/// Constructor that takes in a fixed size for the float type
/// Allocates the memory needed to store the data type. Note that the data
/// type cannot be changed after the fact.
extern "C" CONDITIONAL_SYCL_EXTERNAL void *
create_new_hls_vpfp_intermediate_ty(const size_t E, const size_t M);

/// Destructor - cleans up the memory used by the underlying data
extern "C" void CONDITIONAL_SYCL_EXTERNAL free_hls_vpfp_intermediate_ty(void *);

/// Return a handle to the underlying exponent array
// Functions that we define here do not need opaque pointers, changed for
// consistency
CONDITIONAL_SYCL_EXTERNAL unsigned char *get_exponent(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return input_casted->m_exponent;
}

/// Return a handle to the underlying mantissa array
CONDITIONAL_SYCL_EXTERNAL unsigned char *get_mantissa(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return input_casted->m_mantissa;
}

/// Returns the sign bit
CONDITIONAL_SYCL_EXTERNAL bool get_sign_bit(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return input_casted->m_sign_bit;
}

/// Returns the number of bits needed to store the exponent
CONDITIONAL_SYCL_EXTERNAL size_t get_exponent_bits(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return input_casted->m_exponent_bits;
}

/// Returns the number of words of the underlying data type needed to store
/// a given number of bits.
extern "C" CONDITIONAL_SYCL_EXTERNAL size_t getWordSize(size_t num_bits);

/// Returns the number of words needed to store the exponent
CONDITIONAL_SYCL_EXTERNAL size_t get_exponent_words(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return getWordSize(input_casted->m_exponent_bits);
}

/// Returns the number of bits needed to store the mantissa
CONDITIONAL_SYCL_EXTERNAL size_t get_mantissa_bits(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return input_casted->m_mantissa_bits;
}

/// Returns the number of words needed to store the mantissa
CONDITIONAL_SYCL_EXTERNAL size_t get_mantissa_words(const void *input_ty) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  return getWordSize(input_casted->m_mantissa_bits);
}

/// Sets the sign bit
CONDITIONAL_SYCL_EXTERNAL void set_sign_bit(void *input_ty, bool input) {
  assertm(input_ty, "NULL input");
  hls_vpfp_intermediate_ty *input_casted = (hls_vpfp_intermediate_ty *)input_ty;

  input_casted->m_sign_bit = input;
}

////////////////////////////////////////////////////////////////////////////////
// Library Function Signatures
////////////////////////////////////////////////////////////////////////////////

extern "C" CONDITIONAL_SYCL_EXTERNAL int
hls_inter_get_str(char *allocated_buffer, size_t buffer_size,
                  const void *input_ty, int base, int rounding);

/////////////////////// Basic Arithmetic Operations ////////////////////////////

extern "C" CONDITIONAL_SYCL_EXTERNAL void hls_inter_add(void *result,
                                                        const void *input_ty_A,
                                                        const void *input_ty_B,
                                                        const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL void hls_inter_sub(void *result,
                                                        const void *input_ty_A,
                                                        const void *input_ty_B,
                                                        const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL void hls_inter_div(void *result,
                                                        const void *input_ty_A,
                                                        const void *input_ty_B,
                                                        const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL void hls_inter_mul(void *result,
                                                        const void *input_ty_A,
                                                        const void *input_ty_B,
                                                        const bool subnormal);

////////////////////////// Comparison Operations ///////////////////////////////
extern "C" CONDITIONAL_SYCL_EXTERNAL bool hls_inter_gt(const void *input_ty_A,
                                                       const void *input_ty_B,
                                                       const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL bool hls_inter_ge(const void *input_ty_A,
                                                       const void *input_ty_B,
                                                       const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL bool hls_inter_lt(const void *input_ty_A,
                                                       const void *input_ty_B,
                                                       const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL bool hls_inter_le(const void *input_ty_A,
                                                       const void *input_ty_B,
                                                       const bool subnormal);

extern "C" CONDITIONAL_SYCL_EXTERNAL bool hls_inter_eq(const void *input_ty_A,
                                                       const void *input_ty_B,
                                                       const bool subnormal);

/////////////////////// Commonly Used Math Operations //////////////////////////
/// computes the square root of x -> x^(1/2)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_sqrt(void *result, const void *input_ty,
               const bool subnormal = false);

/// computes the cube root of x -> x^(1/3)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_cbrt(void *result, const void *input_ty,
               const bool subnormal = false);

/// computes the reciprocal of x -> 1/x
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_recip(void *result, const void *input_ty,
                const bool subnormal = false);

/// computes the reciprocal sqrt of x -> 1/sqrt(x)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_rsqrt(void *result, const void *input_ty,
                const bool subnormal = false);

/// computes the hypotenuse of x and y -> srqt(x^2 + y^2)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_hypot(void *result, const void *input_ty_A, const void *input_ty_B,
                const bool subnormal = false);

////////////////// Exponential and Logarithmic Functions /////////////////////
/// computes e to the power of x -> e^x
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_exp(void *result, const void *input_ty_A,
              const bool subnormal = false);

/// computes 2 to the power of x -> 2^x
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_exp2(void *result, const void *input_ty_A,
               const bool subnormal = false);

/// computes 10 to the power of x -> 10^x
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_exp10(void *result, const void *input_ty_A,
                const bool subnormal = false);

/// computes (e to the power of x) -1 -> e^x -1
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_expm1(void *result, const void *input_ty_A,
                const bool subnormal = false);

/// computes the natural log of x -> ln(x)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_log(void *result, const void *input_ty_A,
              const bool subnormal = false);

/// computes the base 2 log of x -> log2(x)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_log2(void *result, const void *input_ty_A,
               const bool subnormal = false);

/// computes the base 10 log of x -> log10(x)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_log10(void *result, const void *input_ty_A,
                const bool subnormal = false);

/// computes the natural log of (1+x) -> ln(1+x)
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_log1p(void *result, const void *input_ty_A,
                const bool subnormal = false);

///////////////////////// Power Functions ///////////////////////////////////
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_pow(void *result, const void *input_ty_A, const void *input_ty_B,
              const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_pown(void *result, const void *input_ty_A, const long int input_B,
               const bool subnormal = false);

/////////////////////// Trigonometric Functions /////////////////////////////
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_sin(void *result, const void *input_ty_A,
              const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_sinpi(void *result, const void *input_ty_A,
                const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_cos(void *result, const void *input_ty_A,
              const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_cospi(void *result, const void *input_ty_A,
                const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_asin(void *result, const void *input_ty_A,
               const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_asinpi(void *result, const void *input_ty_A,
                 const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_acos(void *result, const void *input_ty_A,
               const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_acospi(void *result, const void *input_ty_A,
                 const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_atan(void *result, const void *input_ty_A,
               const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_atanpi(void *result, const void *input_ty_A,
                 const bool subnormal = false);

extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_atan2(void *result, const void *input_ty_A, const void *input_ty_B,
                const bool subnormal = false);

/////////////////////// Conversion Functions ////////////////////////////////
extern "C" CONDITIONAL_SYCL_EXTERNAL void
hls_inter_convert(void *result, const void *input_ty, const int rounding = 0,
                  const bool subnormal = true);

extern "C" void CONDITIONAL_SYCL_EXTERNAL
hls_convert_int_to_hls_inter_signed(void *result, const long long int &input);

extern "C" void CONDITIONAL_SYCL_EXTERNAL hls_convert_int_to_hls_inter_unsigned(
    void *result, const unsigned long long int &input);

extern "C" void CONDITIONAL_SYCL_EXTERNAL hls_convert_hls_inter_to_int_signed(
    long long int &result, const void *input_ty, bool emit_warnings = false);

extern "C" void CONDITIONAL_SYCL_EXTERNAL hls_convert_hls_inter_to_int_unsigned(
    unsigned long long int &result, const void *input_ty,
    bool emit_warnings = false);

#endif // _INTEL_IHC_HLS_INTERNAL_HLS_FLOAT_INTER
