const std = @import("std");
const Allocator = std.mem.Allocator;
const mem = std.mem;
const path = std.fs.path;
const assert = std.debug.assert;

const Compilation = @import("Compilation.zig");
const build_options = @import("build_options");

pub const CRTFile = enum {
    crti_o,
    crtn_o,
    crt1_o,
    rcrt1_o,
    scrt1_o,
    libc_a,
    libc_so,
};

pub fn buildCRTFile(comp: *Compilation, crt_file: CRTFile, prog_node: *std.Progress.Node) !void {
    if (!build_options.have_llvm) {
        return error.ZigCompilerNotBuiltWithLLVMExtensions;
    }
    const gpa = comp.gpa;
    var arena_allocator = std.heap.ArenaAllocator.init(gpa);
    defer arena_allocator.deinit();
    const arena = arena_allocator.allocator();

    switch (crt_file) {
        .crti_o => {
            var args = std.ArrayList([]const u8).init(arena);
            try addCcArgs(comp, arena, &args, false);
            try args.appendSlice(&[_][]const u8{
                "-Qunused-arguments",
            });
            return comp.build_crt_file("crti", .Obj, .@"musl crti.o", prog_node, &.{
                .{
                    .src_path = try start_asm_path(comp, arena, "crti.s"),
                    .extra_flags = args.items,
                },
            });
        },
        .crtn_o => {
            var args = std.ArrayList([]const u8).init(arena);
            try addCcArgs(comp, arena, &args, false);
            try args.appendSlice(&[_][]const u8{
                "-Qunused-arguments",
            });
            return comp.build_crt_file("crtn", .Obj, .@"musl crtn.o", prog_node, &.{
                .{
                    .src_path = try start_asm_path(comp, arena, "crtn.s"),
                    .extra_flags = args.items,
                },
            });
        },
        .crt1_o => {
            var args = std.ArrayList([]const u8).init(arena);
            try addCcArgs(comp, arena, &args, false);
            try args.appendSlice(&[_][]const u8{
                "-fno-stack-protector",
                "-DCRT",
            });
            return comp.build_crt_file("crt1", .Obj, .@"musl crt1.o", prog_node, &.{
                .{
                    .src_path = try comp.zig_lib_directory.join(arena, &[_][]const u8{
                        "libc", "musl", "crt", "crt1.c",
                    }),
                    .extra_flags = args.items,
                },
            });
        },
        .rcrt1_o => {
            var args = std.ArrayList([]const u8).init(arena);
            try addCcArgs(comp, arena, &args, false);
            try args.appendSlice(&[_][]const u8{
                "-fPIC",
                "-fno-stack-protector",
                "-DCRT",
            });
            return comp.build_crt_file("rcrt1", .Obj, .@"musl rcrt1.o", prog_node, &.{
                .{
                    .src_path = try comp.zig_lib_directory.join(arena, &[_][]const u8{
                        "libc", "musl", "crt", "rcrt1.c",
                    }),
                    .extra_flags = args.items,
                },
            });
        },
        .scrt1_o => {
            var args = std.ArrayList([]const u8).init(arena);
            try addCcArgs(comp, arena, &args, false);
            try args.appendSlice(&[_][]const u8{
                "-fPIC",
                "-fno-stack-protector",
                "-DCRT",
            });
            return comp.build_crt_file("Scrt1", .Obj, .@"musl Scrt1.o", prog_node, &.{
                .{
                    .src_path = try comp.zig_lib_directory.join(arena, &[_][]const u8{
                        "libc", "musl", "crt", "Scrt1.c",
                    }),
                    .extra_flags = args.items,
                },
            });
        },
        .libc_a => {
            // When there is a src/<arch>/foo.* then it should substitute for src/foo.*
            // Even a .s file can substitute for a .c file.
            const target = comp.getTarget();
            const arch_name = archName(target.cpu.arch);
            var source_table = std.StringArrayHashMap(Ext).init(comp.gpa);
            defer source_table.deinit();

            try source_table.ensureTotalCapacity(compat_time32_files.len + src_files.len);

            for (src_files) |src_file| {
                try addSrcFile(arena, &source_table, src_file);
            }

            const time32_compat_arch_list = [_][]const u8{ "arm", "i386", "mips", "powerpc", "m68k" };
            for (time32_compat_arch_list) |time32_compat_arch| {
                if (mem.eql(u8, arch_name, time32_compat_arch)) {
                    for (compat_time32_files) |compat_time32_file| {
                        try addSrcFile(arena, &source_table, compat_time32_file);
                    }
                }
            }

            var c_source_files = std.ArrayList(Compilation.CSourceFile).init(comp.gpa);
            defer c_source_files.deinit();

            var override_path = std.ArrayList(u8).init(comp.gpa);
            defer override_path.deinit();

            const s = path.sep_str;

            var it = source_table.iterator();
            while (it.next()) |entry| {
                const src_file = entry.key_ptr.*;
                const ext = entry.value_ptr.*;

                const dirname = path.dirname(src_file).?;
                const basename = path.basename(src_file);
                const noextbasename = basename[0 .. basename.len - std.fs.path.extension(basename).len];
                const dirbasename = path.basename(dirname);

                var is_arch_specific = false;
                // Architecture-specific implementations are under a <arch>/ folder.
                if (isMuslArchName(dirbasename)) {
                    if (!mem.eql(u8, dirbasename, arch_name))
                        continue; // Not the architecture we're compiling for.
                    is_arch_specific = true;
                }
                if (!is_arch_specific) {
                    // Look for an arch specific override.
                    override_path.shrinkRetainingCapacity(0);
                    try override_path.writer().print("{s}" ++ s ++ "{s}" ++ s ++ "{s}.s", .{
                        dirname, arch_name, noextbasename,
                    });
                    if (source_table.contains(override_path.items))
                        continue;

                    override_path.shrinkRetainingCapacity(0);
                    try override_path.writer().print("{s}" ++ s ++ "{s}" ++ s ++ "{s}.S", .{
                        dirname, arch_name, noextbasename,
                    });
                    if (source_table.contains(override_path.items))
                        continue;

                    override_path.shrinkRetainingCapacity(0);
                    try override_path.writer().print("{s}" ++ s ++ "{s}" ++ s ++ "{s}.c", .{
                        dirname, arch_name, noextbasename,
                    });
                    if (source_table.contains(override_path.items))
                        continue;
                }

                var args = std.ArrayList([]const u8).init(arena);
                try addCcArgs(comp, arena, &args, ext == .o3);
                try args.appendSlice(&[_][]const u8{
                    "-Qunused-arguments",
                    "-w", // disable all warnings
                });
                const c_source_file = try c_source_files.addOne();
                c_source_file.* = .{
                    .src_path = try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", src_file }),
                    .extra_flags = args.items,
                };
            }
            return comp.build_crt_file("c", .Lib, .@"musl libc.a", prog_node, c_source_files.items);
        },
        .libc_so => {
            const target = comp.getTarget();
            const arch_define = try std.fmt.allocPrint(arena, "-DARCH_{s}", .{
                @tagName(target.cpu.arch),
            });
            const clang_argv: []const []const u8 = if (target.ptrBitWidth() == 64)
                &[_][]const u8{ "-DPTR64", arch_define }
            else
                &[_][]const u8{arch_define};

            const sub_compilation = try Compilation.create(comp.gpa, .{
                .local_cache_directory = comp.global_cache_directory,
                .global_cache_directory = comp.global_cache_directory,
                .cache_mode = .whole,
                .zig_lib_directory = comp.zig_lib_directory,
                .target = target,
                .root_name = "c",
                .main_pkg = null,
                .output_mode = .Lib,
                .link_mode = .Dynamic,
                .thread_pool = comp.thread_pool,
                .libc_installation = comp.bin_file.options.libc_installation,
                .emit_bin = Compilation.EmitLoc{ .directory = null, .basename = "libc.so" },
                .optimize_mode = comp.compilerRtOptMode(),
                .want_sanitize_c = false,
                .want_stack_check = false,
                .want_stack_protector = 0,
                .want_red_zone = comp.bin_file.options.red_zone,
                .omit_frame_pointer = comp.bin_file.options.omit_frame_pointer,
                .want_valgrind = false,
                .want_tsan = false,
                .emit_h = null,
                .strip = comp.compilerRtStrip(),
                .is_native_os = false,
                .is_native_abi = false,
                .self_exe_path = comp.self_exe_path,
                .verbose_cc = comp.verbose_cc,
                .verbose_link = comp.bin_file.options.verbose_link,
                .verbose_air = comp.verbose_air,
                .verbose_llvm_ir = comp.verbose_llvm_ir,
                .verbose_cimport = comp.verbose_cimport,
                .verbose_llvm_cpu_features = comp.verbose_llvm_cpu_features,
                .clang_passthrough_mode = comp.clang_passthrough_mode,
                .c_source_files = &[_]Compilation.CSourceFile{
                    .{ .src_path = try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "libc.S" }) },
                },
                .clang_argv = clang_argv,
                .skip_linker_dependencies = true,
                .soname = "libc.so",
            });
            defer sub_compilation.destroy();

            try comp.updateSubCompilation(sub_compilation, .@"musl libc.so", prog_node);

            try comp.crt_files.ensureUnusedCapacity(comp.gpa, 1);

            const basename = try comp.gpa.dupe(u8, "libc.so");
            errdefer comp.gpa.free(basename);

            comp.crt_files.putAssumeCapacityNoClobber(basename, .{
                .full_object_path = try sub_compilation.bin_file.options.emit.?.directory.join(comp.gpa, &[_][]const u8{
                    sub_compilation.bin_file.options.emit.?.sub_path,
                }),
                .lock = sub_compilation.bin_file.toOwnedLock(),
            });
        },
    }
}

fn archName(arch: std.Target.Cpu.Arch) [:0]const u8 {
    switch (arch) {
        .aarch64, .aarch64_be => return "aarch64",
        .arm, .armeb, .thumb, .thumbeb => return "arm",
        .x86 => return "i386",
        .mips, .mipsel => return "mips",
        .mips64el, .mips64 => return "mips64",
        .powerpc => return "powerpc",
        .powerpc64, .powerpc64le => return "powerpc64",
        .riscv64 => return "riscv64",
        .s390x => return "s390x",
        .wasm32, .wasm64 => return "wasm",
        .x86_64 => return "x86_64",
        else => unreachable,
    }
}

pub fn archNameHeaders(arch: std.Target.Cpu.Arch) [:0]const u8 {
    return switch (arch) {
        .x86 => return "x86",
        else => archName(arch),
    };
}

// Return true if musl has arch-specific crti/crtn sources.
// See lib/libc/musl/crt/ARCH/crt?.s .
pub fn needsCrtiCrtn(target: std.Target) bool {
    // zig fmt: off
    return switch (target.cpu.arch) {
        .riscv64,
        .wasm32, .wasm64 => return false,
        else => true,
    };
    // zig fmt: on
}

fn isMuslArchName(name: []const u8) bool {
    const musl_arch_names = [_][]const u8{
        "aarch64",
        "arm",
        "generic",
        "i386",
        "m68k",
        "microblaze",
        "mips",
        "mips64",
        "mipsn32",
        "or1k",
        "powerpc",
        "powerpc64",
        "riscv64",
        "s390x",
        "sh",
        "x32",
        "x86_64",
    };
    for (musl_arch_names) |musl_arch_name| {
        if (mem.eql(u8, musl_arch_name, name)) {
            return true;
        }
    }
    return false;
}

const Ext = enum {
    assembly,
    o3,
};

fn addSrcFile(arena: Allocator, source_table: *std.StringArrayHashMap(Ext), file_path: []const u8) !void {
    const ext: Ext = ext: {
        if (mem.endsWith(u8, file_path, ".c")) {
            if (mem.startsWith(u8, file_path, "musl/src/malloc/") or
                mem.startsWith(u8, file_path, "musl/src/string/") or
                mem.startsWith(u8, file_path, "musl/src/internal/"))
            {
                break :ext .o3;
            } else {
                break :ext .assembly;
            }
        } else if (mem.endsWith(u8, file_path, ".s") or mem.endsWith(u8, file_path, ".S")) {
            break :ext .assembly;
        } else {
            unreachable;
        }
    };
    // TODO do this at comptime on the comptime data rather than at runtime
    // probably best to wait until self-hosted is done and our comptime execution
    // is faster and uses less memory.
    const key = if (path.sep != '/') blk: {
        const mutable_file_path = try arena.dupe(u8, file_path);
        for (mutable_file_path) |*c| {
            if (c.* == '/') {
                c.* = path.sep;
            }
        }
        break :blk mutable_file_path;
    } else file_path;
    source_table.putAssumeCapacityNoClobber(key, ext);
}

fn addCcArgs(
    comp: *Compilation,
    arena: Allocator,
    args: *std.ArrayList([]const u8),
    want_O3: bool,
) error{OutOfMemory}!void {
    const target = comp.getTarget();
    const arch_name = archName(target.cpu.arch);
    const os_name = @tagName(target.os.tag);
    const triple = try std.fmt.allocPrint(arena, "{s}-{s}-musl", .{
        archNameHeaders(target.cpu.arch), os_name,
    });
    const o_arg = if (want_O3) "-O3" else "-Os";

    try args.appendSlice(&[_][]const u8{
        "-std=c99",
        "-ffreestanding",
        // Musl adds these args to builds with gcc but clang does not support them.
        //"-fexcess-precision=standard",
        //"-frounding-math",
        "-Wa,--noexecstack",
        "-D_XOPEN_SOURCE=700",

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "arch", arch_name }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "arch", "generic" }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "src", "include" }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "src", "internal" }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "musl", "include" }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "include", triple }),

        "-I",
        try comp.zig_lib_directory.join(arena, &[_][]const u8{ "libc", "include", "generic-musl" }),

        o_arg,

        "-fomit-frame-pointer",
        "-fno-unwind-tables",
        "-fno-asynchronous-unwind-tables",
        "-ffunction-sections",
        "-fdata-sections",
    });
}

fn start_asm_path(comp: *Compilation, arena: Allocator, basename: []const u8) ![]const u8 {
    const target = comp.getTarget();
    return comp.zig_lib_directory.join(arena, &[_][]const u8{
        "libc", "musl", "crt", archName(target.cpu.arch), basename,
    });
}

const src_files = [_][]const u8{
    "musl/src/aio/aio.c",
    "musl/src/aio/aio_suspend.c",
    "musl/src/aio/lio_listio.c",
    "musl/src/complex/__cexp.c",
    "musl/src/complex/__cexpf.c",
    "musl/src/complex/cabs.c",
    "musl/src/complex/cabsf.c",
    "musl/src/complex/cabsl.c",
    "musl/src/complex/cacos.c",
    "musl/src/complex/cacosf.c",
    "musl/src/complex/cacosh.c",
    "musl/src/complex/cacoshf.c",
    "musl/src/complex/cacoshl.c",
    "musl/src/complex/cacosl.c",
    "musl/src/complex/carg.c",
    "musl/src/complex/cargf.c",
    "musl/src/complex/cargl.c",
    "musl/src/complex/casin.c",
    "musl/src/complex/casinf.c",
    "musl/src/complex/casinh.c",
    "musl/src/complex/casinhf.c",
    "musl/src/complex/casinhl.c",
    "musl/src/complex/casinl.c",
    "musl/src/complex/catan.c",
    "musl/src/complex/catanf.c",
    "musl/src/complex/catanh.c",
    "musl/src/complex/catanhf.c",
    "musl/src/complex/catanhl.c",
    "musl/src/complex/catanl.c",
    "musl/src/complex/ccos.c",
    "musl/src/complex/ccosf.c",
    "musl/src/complex/ccosh.c",
    "musl/src/complex/ccoshf.c",
    "musl/src/complex/ccoshl.c",
    "musl/src/complex/ccosl.c",
    "musl/src/complex/cexp.c",
    "musl/src/complex/cexpf.c",
    "musl/src/complex/cexpl.c",
    "musl/src/complex/cimag.c",
    "musl/src/complex/cimagf.c",
    "musl/src/complex/cimagl.c",
    "musl/src/complex/clog.c",
    "musl/src/complex/clogf.c",
    "musl/src/complex/clogl.c",
    "musl/src/complex/conj.c",
    "musl/src/complex/conjf.c",
    "musl/src/complex/conjl.c",
    "musl/src/complex/cpow.c",
    "musl/src/complex/cpowf.c",
    "musl/src/complex/cpowl.c",
    "musl/src/complex/cproj.c",
    "musl/src/complex/cprojf.c",
    "musl/src/complex/cprojl.c",
    "musl/src/complex/creal.c",
    "musl/src/complex/crealf.c",
    "musl/src/complex/creall.c",
    "musl/src/complex/csin.c",
    "musl/src/complex/csinf.c",
    "musl/src/complex/csinh.c",
    "musl/src/complex/csinhf.c",
    "musl/src/complex/csinhl.c",
    "musl/src/complex/csinl.c",
    "musl/src/complex/csqrt.c",
    "musl/src/complex/csqrtf.c",
    "musl/src/complex/csqrtl.c",
    "musl/src/complex/ctan.c",
    "musl/src/complex/ctanf.c",
    "musl/src/complex/ctanh.c",
    "musl/src/complex/ctanhf.c",
    "musl/src/complex/ctanhl.c",
    "musl/src/complex/ctanl.c",
    "musl/src/conf/confstr.c",
    "musl/src/conf/fpathconf.c",
    "musl/src/conf/legacy.c",
    "musl/src/conf/pathconf.c",
    "musl/src/conf/sysconf.c",
    "musl/src/crypt/crypt.c",
    "musl/src/crypt/crypt_blowfish.c",
    "musl/src/crypt/crypt_des.c",
    "musl/src/crypt/crypt_md5.c",
    "musl/src/crypt/crypt_r.c",
    "musl/src/crypt/crypt_sha256.c",
    "musl/src/crypt/crypt_sha512.c",
    "musl/src/crypt/encrypt.c",
    "musl/src/ctype/__ctype_b_loc.c",
    "musl/src/ctype/__ctype_get_mb_cur_max.c",
    "musl/src/ctype/__ctype_tolower_loc.c",
    "musl/src/ctype/__ctype_toupper_loc.c",
    "musl/src/ctype/isalnum.c",
    "musl/src/ctype/isalpha.c",
    "musl/src/ctype/isascii.c",
    "musl/src/ctype/isblank.c",
    "musl/src/ctype/iscntrl.c",
    "musl/src/ctype/isdigit.c",
    "musl/src/ctype/isgraph.c",
    "musl/src/ctype/islower.c",
    "musl/src/ctype/isprint.c",
    "musl/src/ctype/ispunct.c",
    "musl/src/ctype/isspace.c",
    "musl/src/ctype/isupper.c",
    "musl/src/ctype/iswalnum.c",
    "musl/src/ctype/iswalpha.c",
    "musl/src/ctype/iswblank.c",
    "musl/src/ctype/iswcntrl.c",
    "musl/src/ctype/iswctype.c",
    "musl/src/ctype/iswdigit.c",
    "musl/src/ctype/iswgraph.c",
    "musl/src/ctype/iswlower.c",
    "musl/src/ctype/iswprint.c",
    "musl/src/ctype/iswpunct.c",
    "musl/src/ctype/iswspace.c",
    "musl/src/ctype/iswupper.c",
    "musl/src/ctype/iswxdigit.c",
    "musl/src/ctype/isxdigit.c",
    "musl/src/ctype/toascii.c",
    "musl/src/ctype/tolower.c",
    "musl/src/ctype/toupper.c",
    "musl/src/ctype/towctrans.c",
    "musl/src/ctype/wcswidth.c",
    "musl/src/ctype/wctrans.c",
    "musl/src/ctype/wcwidth.c",
    "musl/src/dirent/alphasort.c",
    "musl/src/dirent/closedir.c",
    "musl/src/dirent/dirfd.c",
    "musl/src/dirent/fdopendir.c",
    "musl/src/dirent/opendir.c",
    "musl/src/dirent/readdir.c",
    "musl/src/dirent/readdir_r.c",
    "musl/src/dirent/rewinddir.c",
    "musl/src/dirent/scandir.c",
    "musl/src/dirent/seekdir.c",
    "musl/src/dirent/telldir.c",
    "musl/src/dirent/versionsort.c",
    "musl/src/env/__environ.c",
    "musl/src/env/__init_tls.c",
    "musl/src/env/__libc_start_main.c",
    "musl/src/env/__reset_tls.c",
    "musl/src/env/__stack_chk_fail.c",
    "musl/src/env/clearenv.c",
    "musl/src/env/getenv.c",
    "musl/src/env/putenv.c",
    "musl/src/env/secure_getenv.c",
    "musl/src/env/setenv.c",
    "musl/src/env/unsetenv.c",
    "musl/src/errno/__errno_location.c",
    "musl/src/errno/strerror.c",
    "musl/src/exit/_Exit.c",
    "musl/src/exit/abort.c",
    "musl/src/exit/abort_lock.c",
    "musl/src/exit/arm/__aeabi_atexit.c",
    "musl/src/exit/assert.c",
    "musl/src/exit/at_quick_exit.c",
    "musl/src/exit/atexit.c",
    "musl/src/exit/exit.c",
    "musl/src/exit/quick_exit.c",
    "musl/src/fcntl/creat.c",
    "musl/src/fcntl/fcntl.c",
    "musl/src/fcntl/open.c",
    "musl/src/fcntl/openat.c",
    "musl/src/fcntl/posix_fadvise.c",
    "musl/src/fcntl/posix_fallocate.c",
    "musl/src/fenv/__flt_rounds.c",
    "musl/src/fenv/aarch64/fenv.s",
    "musl/src/fenv/arm/fenv-hf.S",
    "musl/src/fenv/arm/fenv.c",
    "musl/src/fenv/fegetexceptflag.c",
    "musl/src/fenv/feholdexcept.c",
    "musl/src/fenv/fenv.c",
    "musl/src/fenv/fesetexceptflag.c",
    "musl/src/fenv/fesetround.c",
    "musl/src/fenv/feupdateenv.c",
    "musl/src/fenv/i386/fenv.s",
    "musl/src/fenv/m68k/fenv.c",
    "musl/src/fenv/mips/fenv-sf.c",
    "musl/src/fenv/mips/fenv.S",
    "musl/src/fenv/mips64/fenv-sf.c",
    "musl/src/fenv/mips64/fenv.S",
    "musl/src/fenv/mipsn32/fenv-sf.c",
    "musl/src/fenv/mipsn32/fenv.S",
    "musl/src/fenv/powerpc/fenv-sf.c",
    "musl/src/fenv/powerpc/fenv.S",
    "musl/src/fenv/powerpc64/fenv.c",
    "musl/src/fenv/riscv64/fenv-sf.c",
    "musl/src/fenv/riscv64/fenv.S",
    "musl/src/fenv/s390x/fenv.c",
    "musl/src/fenv/sh/fenv-nofpu.c",
    "musl/src/fenv/sh/fenv.S",
    "musl/src/fenv/x32/fenv.s",
    "musl/src/fenv/x86_64/fenv.s",
    "musl/src/internal/defsysinfo.c",
    "musl/src/internal/floatscan.c",
    "musl/src/internal/i386/defsysinfo.s",
    "musl/src/internal/intscan.c",
    "musl/src/internal/libc.c",
    "musl/src/internal/procfdname.c",
    "musl/src/internal/sh/__shcall.c",
    "musl/src/internal/shgetc.c",
    "musl/src/internal/syscall_ret.c",
    "musl/src/internal/vdso.c",
    "musl/src/internal/version.c",
    "musl/src/ipc/ftok.c",
    "musl/src/ipc/msgctl.c",
    "musl/src/ipc/msgget.c",
    "musl/src/ipc/msgrcv.c",
    "musl/src/ipc/msgsnd.c",
    "musl/src/ipc/semctl.c",
    "musl/src/ipc/semget.c",
    "musl/src/ipc/semop.c",
    "musl/src/ipc/semtimedop.c",
    "musl/src/ipc/shmat.c",
    "musl/src/ipc/shmctl.c",
    "musl/src/ipc/shmdt.c",
    "musl/src/ipc/shmget.c",
    "musl/src/ldso/__dlsym.c",
    "musl/src/ldso/aarch64/dlsym.s",
    "musl/src/ldso/aarch64/tlsdesc.s",
    "musl/src/ldso/arm/dlsym.s",
    "musl/src/ldso/arm/dlsym_time64.S",
    "musl/src/ldso/arm/find_exidx.c",
    "musl/src/ldso/arm/tlsdesc.S",
    "musl/src/ldso/dl_iterate_phdr.c",
    "musl/src/ldso/dladdr.c",
    "musl/src/ldso/dlclose.c",
    "musl/src/ldso/dlerror.c",
    "musl/src/ldso/dlinfo.c",
    "musl/src/ldso/dlopen.c",
    "musl/src/ldso/dlsym.c",
    "musl/src/ldso/i386/dlsym.s",
    "musl/src/ldso/i386/dlsym_time64.S",
    "musl/src/ldso/i386/tlsdesc.s",
    "musl/src/ldso/m68k/dlsym.s",
    "musl/src/ldso/m68k/dlsym_time64.S",
    "musl/src/ldso/microblaze/dlsym.s",
    "musl/src/ldso/microblaze/dlsym_time64.S",
    "musl/src/ldso/mips/dlsym.s",
    "musl/src/ldso/mips/dlsym_time64.S",
    "musl/src/ldso/mips64/dlsym.s",
    "musl/src/ldso/mipsn32/dlsym.s",
    "musl/src/ldso/mipsn32/dlsym_time64.S",
    "musl/src/ldso/or1k/dlsym.s",
    "musl/src/ldso/or1k/dlsym_time64.S",
    "musl/src/ldso/powerpc/dlsym.s",
    "musl/src/ldso/powerpc/dlsym_time64.S",
    "musl/src/ldso/powerpc64/dlsym.s",
    "musl/src/ldso/riscv64/dlsym.s",
    "musl/src/ldso/s390x/dlsym.s",
    "musl/src/ldso/sh/dlsym.s",
    "musl/src/ldso/sh/dlsym_time64.S",
    "musl/src/ldso/tlsdesc.c",
    "musl/src/ldso/x32/dlsym.s",
    "musl/src/ldso/x86_64/dlsym.s",
    "musl/src/ldso/x86_64/tlsdesc.s",
    "musl/src/legacy/cuserid.c",
    "musl/src/legacy/daemon.c",
    "musl/src/legacy/err.c",
    "musl/src/legacy/euidaccess.c",
    "musl/src/legacy/ftw.c",
    "musl/src/legacy/futimes.c",
    "musl/src/legacy/getdtablesize.c",
    "musl/src/legacy/getloadavg.c",
    "musl/src/legacy/getpagesize.c",
    "musl/src/legacy/getpass.c",
    "musl/src/legacy/getusershell.c",
    "musl/src/legacy/isastream.c",
    "musl/src/legacy/lutimes.c",
    "musl/src/legacy/ulimit.c",
    "musl/src/legacy/utmpx.c",
    "musl/src/legacy/valloc.c",
    "musl/src/linux/adjtime.c",
    "musl/src/linux/adjtimex.c",
    "musl/src/linux/arch_prctl.c",
    "musl/src/linux/brk.c",
    "musl/src/linux/cache.c",
    "musl/src/linux/cap.c",
    "musl/src/linux/chroot.c",
    "musl/src/linux/clock_adjtime.c",
    "musl/src/linux/clone.c",
    "musl/src/linux/copy_file_range.c",
    "musl/src/linux/epoll.c",
    "musl/src/linux/eventfd.c",
    "musl/src/linux/fallocate.c",
    "musl/src/linux/fanotify.c",
    "musl/src/linux/flock.c",
    "musl/src/linux/getdents.c",
    "musl/src/linux/getrandom.c",
    "musl/src/linux/gettid.c",
    "musl/src/linux/inotify.c",
    "musl/src/linux/ioperm.c",
    "musl/src/linux/iopl.c",
    "musl/src/linux/klogctl.c",
    "musl/src/linux/membarrier.c",
    "musl/src/linux/memfd_create.c",
    "musl/src/linux/mlock2.c",
    "musl/src/linux/module.c",
    "musl/src/linux/mount.c",
    "musl/src/linux/name_to_handle_at.c",
    "musl/src/linux/open_by_handle_at.c",
    "musl/src/linux/personality.c",
    "musl/src/linux/pivot_root.c",
    "musl/src/linux/ppoll.c",
    "musl/src/linux/prctl.c",
    "musl/src/linux/prlimit.c",
    "musl/src/linux/process_vm.c",
    "musl/src/linux/ptrace.c",
    "musl/src/linux/quotactl.c",
    "musl/src/linux/readahead.c",
    "musl/src/linux/reboot.c",
    "musl/src/linux/remap_file_pages.c",
    "musl/src/linux/sbrk.c",
    "musl/src/linux/sendfile.c",
    "musl/src/linux/setfsgid.c",
    "musl/src/linux/setfsuid.c",
    "musl/src/linux/setgroups.c",
    "musl/src/linux/sethostname.c",
    "musl/src/linux/setns.c",
    "musl/src/linux/settimeofday.c",
    "musl/src/linux/signalfd.c",
    "musl/src/linux/splice.c",
    "musl/src/linux/stime.c",
    "musl/src/linux/swap.c",
    "musl/src/linux/sync_file_range.c",
    "musl/src/linux/syncfs.c",
    "musl/src/linux/sysinfo.c",
    "musl/src/linux/tee.c",
    "musl/src/linux/timerfd.c",
    "musl/src/linux/unshare.c",
    "musl/src/linux/utimes.c",
    "musl/src/linux/vhangup.c",
    "musl/src/linux/vmsplice.c",
    "musl/src/linux/wait3.c",
    "musl/src/linux/wait4.c",
    "musl/src/linux/x32/sysinfo.c",
    "musl/src/linux/xattr.c",
    "musl/src/locale/__lctrans.c",
    "musl/src/locale/__mo_lookup.c",
    "musl/src/locale/bind_textdomain_codeset.c",
    "musl/src/locale/c_locale.c",
    "musl/src/locale/catclose.c",
    "musl/src/locale/catgets.c",
    "musl/src/locale/catopen.c",
    "musl/src/locale/dcngettext.c",
    "musl/src/locale/duplocale.c",
    "musl/src/locale/freelocale.c",
    "musl/src/locale/iconv.c",
    "musl/src/locale/iconv_close.c",
    "musl/src/locale/langinfo.c",
    "musl/src/locale/locale_map.c",
    "musl/src/locale/localeconv.c",
    "musl/src/locale/newlocale.c",
    "musl/src/locale/pleval.c",
    "musl/src/locale/setlocale.c",
    "musl/src/locale/strcoll.c",
    "musl/src/locale/strfmon.c",
    "musl/src/locale/strtod_l.c",
    "musl/src/locale/strxfrm.c",
    "musl/src/locale/textdomain.c",
    "musl/src/locale/uselocale.c",
    "musl/src/locale/wcscoll.c",
    "musl/src/locale/wcsxfrm.c",
    "musl/src/malloc/calloc.c",
    "musl/src/malloc/free.c",
    "musl/src/malloc/libc_calloc.c",
    "musl/src/malloc/lite_malloc.c",
    "musl/src/malloc/mallocng/aligned_alloc.c",
    "musl/src/malloc/mallocng/donate.c",
    "musl/src/malloc/mallocng/free.c",
    "musl/src/malloc/mallocng/malloc.c",
    "musl/src/malloc/mallocng/malloc_usable_size.c",
    "musl/src/malloc/mallocng/realloc.c",
    "musl/src/malloc/memalign.c",
    "musl/src/malloc/oldmalloc/aligned_alloc.c",
    "musl/src/malloc/oldmalloc/malloc.c",
    "musl/src/malloc/oldmalloc/malloc_usable_size.c",
    "musl/src/malloc/posix_memalign.c",
    "musl/src/malloc/realloc.c",
    "musl/src/malloc/reallocarray.c",
    "musl/src/malloc/replaced.c",
    "musl/src/math/__cos.c",
    "musl/src/math/__cosdf.c",
    "musl/src/math/__cosl.c",
    "musl/src/math/__expo2.c",
    "musl/src/math/__expo2f.c",
    "musl/src/math/__fpclassify.c",
    "musl/src/math/__fpclassifyf.c",
    "musl/src/math/__fpclassifyl.c",
    "musl/src/math/__invtrigl.c",
    "musl/src/math/__math_divzero.c",
    "musl/src/math/__math_divzerof.c",
    "musl/src/math/__math_invalid.c",
    "musl/src/math/__math_invalidf.c",
    "musl/src/math/__math_invalidl.c",
    "musl/src/math/__math_oflow.c",
    "musl/src/math/__math_oflowf.c",
    "musl/src/math/__math_uflow.c",
    "musl/src/math/__math_uflowf.c",
    "musl/src/math/__math_xflow.c",
    "musl/src/math/__math_xflowf.c",
    "musl/src/math/__polevll.c",
    "musl/src/math/__rem_pio2.c",
    "musl/src/math/__rem_pio2_large.c",
    "musl/src/math/__rem_pio2f.c",
    "musl/src/math/__rem_pio2l.c",
    "musl/src/math/__signbit.c",
    "musl/src/math/__signbitf.c",
    "musl/src/math/__signbitl.c",
    "musl/src/math/__sin.c",
    "musl/src/math/__sindf.c",
    "musl/src/math/__sinl.c",
    "musl/src/math/__tan.c",
    "musl/src/math/__tandf.c",
    "musl/src/math/__tanl.c",
    "musl/src/math/aarch64/ceil.c",
    "musl/src/math/aarch64/ceilf.c",
    "musl/src/math/aarch64/fabs.c",
    "musl/src/math/aarch64/fabsf.c",
    "musl/src/math/aarch64/floor.c",
    "musl/src/math/aarch64/floorf.c",
    "musl/src/math/aarch64/fma.c",
    "musl/src/math/aarch64/fmaf.c",
    "musl/src/math/aarch64/fmax.c",
    "musl/src/math/aarch64/fmaxf.c",
    "musl/src/math/aarch64/fmin.c",
    "musl/src/math/aarch64/fminf.c",
    "musl/src/math/aarch64/llrint.c",
    "musl/src/math/aarch64/llrintf.c",
    "musl/src/math/aarch64/llround.c",
    "musl/src/math/aarch64/llroundf.c",
    "musl/src/math/aarch64/lrint.c",
    "musl/src/math/aarch64/lrintf.c",
    "musl/src/math/aarch64/lround.c",
    "musl/src/math/aarch64/lroundf.c",
    "musl/src/math/aarch64/nearbyint.c",
    "musl/src/math/aarch64/nearbyintf.c",
    "musl/src/math/aarch64/rint.c",
    "musl/src/math/aarch64/rintf.c",
    "musl/src/math/aarch64/round.c",
    "musl/src/math/aarch64/roundf.c",
    "musl/src/math/aarch64/sqrt.c",
    "musl/src/math/aarch64/sqrtf.c",
    "musl/src/math/aarch64/trunc.c",
    "musl/src/math/aarch64/truncf.c",
    "musl/src/math/acos.c",
    "musl/src/math/acosf.c",
    "musl/src/math/acosh.c",
    "musl/src/math/acoshf.c",
    "musl/src/math/acoshl.c",
    "musl/src/math/acosl.c",
    "musl/src/math/arm/fabs.c",
    "musl/src/math/arm/fabsf.c",
    "musl/src/math/arm/fma.c",
    "musl/src/math/arm/fmaf.c",
    "musl/src/math/arm/sqrt.c",
    "musl/src/math/arm/sqrtf.c",
    "musl/src/math/asin.c",
    "musl/src/math/asinf.c",
    "musl/src/math/asinh.c",
    "musl/src/math/asinhf.c",
    "musl/src/math/asinhl.c",
    "musl/src/math/asinl.c",
    "musl/src/math/atan.c",
    "musl/src/math/atan2.c",
    "musl/src/math/atan2f.c",
    "musl/src/math/atan2l.c",
    "musl/src/math/atanf.c",
    "musl/src/math/atanh.c",
    "musl/src/math/atanhf.c",
    "musl/src/math/atanhl.c",
    "musl/src/math/atanl.c",
    "musl/src/math/cbrt.c",
    "musl/src/math/cbrtf.c",
    "musl/src/math/cbrtl.c",
    "musl/src/math/ceil.c",
    "musl/src/math/ceilf.c",
    "musl/src/math/ceill.c",
    "musl/src/math/copysign.c",
    "musl/src/math/copysignf.c",
    "musl/src/math/copysignl.c",
    "musl/src/math/cos.c",
    "musl/src/math/cosf.c",
    "musl/src/math/cosh.c",
    "musl/src/math/coshf.c",
    "musl/src/math/coshl.c",
    "musl/src/math/cosl.c",
    "musl/src/math/erf.c",
    "musl/src/math/erff.c",
    "musl/src/math/erfl.c",
    "musl/src/math/exp.c",
    "musl/src/math/exp10.c",
    "musl/src/math/exp10f.c",
    "musl/src/math/exp10l.c",
    "musl/src/math/exp2.c",
    "musl/src/math/exp2f.c",
    "musl/src/math/exp2f_data.c",
    "musl/src/math/exp2l.c",
    "musl/src/math/exp_data.c",
    "musl/src/math/expf.c",
    "musl/src/math/expl.c",
    "musl/src/math/expm1.c",
    "musl/src/math/expm1f.c",
    "musl/src/math/expm1l.c",
    "musl/src/math/fabs.c",
    "musl/src/math/fabsf.c",
    "musl/src/math/fabsl.c",
    "musl/src/math/fdim.c",
    "musl/src/math/fdimf.c",
    "musl/src/math/fdiml.c",
    "musl/src/math/finite.c",
    "musl/src/math/finitef.c",
    "musl/src/math/floor.c",
    "musl/src/math/floorf.c",
    "musl/src/math/floorl.c",
    "musl/src/math/fma.c",
    "musl/src/math/fmaf.c",
    "musl/src/math/fmal.c",
    "musl/src/math/fmax.c",
    "musl/src/math/fmaxf.c",
    "musl/src/math/fmaxl.c",
    "musl/src/math/fmin.c",
    "musl/src/math/fminf.c",
    "musl/src/math/fminl.c",
    "musl/src/math/fmod.c",
    "musl/src/math/fmodf.c",
    "musl/src/math/fmodl.c",
    "musl/src/math/frexp.c",
    "musl/src/math/frexpf.c",
    "musl/src/math/frexpl.c",
    "musl/src/math/hypot.c",
    "musl/src/math/hypotf.c",
    "musl/src/math/hypotl.c",
    "musl/src/math/i386/__invtrigl.s",
    "musl/src/math/i386/acos.s",
    "musl/src/math/i386/acosf.s",
    "musl/src/math/i386/acosl.s",
    "musl/src/math/i386/asin.s",
    "musl/src/math/i386/asinf.s",
    "musl/src/math/i386/asinl.s",
    "musl/src/math/i386/atan.s",
    "musl/src/math/i386/atan2.s",
    "musl/src/math/i386/atan2f.s",
    "musl/src/math/i386/atan2l.s",
    "musl/src/math/i386/atanf.s",
    "musl/src/math/i386/atanl.s",
    "musl/src/math/i386/ceil.s",
    "musl/src/math/i386/ceilf.s",
    "musl/src/math/i386/ceill.s",
    "musl/src/math/i386/exp2l.s",
    "musl/src/math/i386/exp_ld.s",
    "musl/src/math/i386/expl.s",
    "musl/src/math/i386/expm1l.s",
    "musl/src/math/i386/fabs.c",
    "musl/src/math/i386/fabsf.c",
    "musl/src/math/i386/fabsl.c",
    "musl/src/math/i386/floor.s",
    "musl/src/math/i386/floorf.s",
    "musl/src/math/i386/floorl.s",
    "musl/src/math/i386/fmod.c",
    "musl/src/math/i386/fmodf.c",
    "musl/src/math/i386/fmodl.c",
    "musl/src/math/i386/hypot.s",
    "musl/src/math/i386/hypotf.s",
    "musl/src/math/i386/ldexp.s",
    "musl/src/math/i386/ldexpf.s",
    "musl/src/math/i386/ldexpl.s",
    "musl/src/math/i386/llrint.c",
    "musl/src/math/i386/llrintf.c",
    "musl/src/math/i386/llrintl.c",
    "musl/src/math/i386/log.s",
    "musl/src/math/i386/log10.s",
    "musl/src/math/i386/log10f.s",
    "musl/src/math/i386/log10l.s",
    "musl/src/math/i386/log1p.s",
    "musl/src/math/i386/log1pf.s",
    "musl/src/math/i386/log1pl.s",
    "musl/src/math/i386/log2.s",
    "musl/src/math/i386/log2f.s",
    "musl/src/math/i386/log2l.s",
    "musl/src/math/i386/logf.s",
    "musl/src/math/i386/logl.s",
    "musl/src/math/i386/lrint.c",
    "musl/src/math/i386/lrintf.c",
    "musl/src/math/i386/lrintl.c",
    "musl/src/math/i386/remainder.c",
    "musl/src/math/i386/remainderf.c",
    "musl/src/math/i386/remainderl.c",
    "musl/src/math/i386/remquo.s",
    "musl/src/math/i386/remquof.s",
    "musl/src/math/i386/remquol.s",
    "musl/src/math/i386/rint.c",
    "musl/src/math/i386/rintf.c",
    "musl/src/math/i386/rintl.c",
    "musl/src/math/i386/scalbln.s",
    "musl/src/math/i386/scalblnf.s",
    "musl/src/math/i386/scalblnl.s",
    "musl/src/math/i386/scalbn.s",
    "musl/src/math/i386/scalbnf.s",
    "musl/src/math/i386/scalbnl.s",
    "musl/src/math/i386/sqrt.c",
    "musl/src/math/i386/sqrtf.c",
    "musl/src/math/i386/sqrtl.c",
    "musl/src/math/i386/trunc.s",
    "musl/src/math/i386/truncf.s",
    "musl/src/math/i386/truncl.s",
    "musl/src/math/ilogb.c",
    "musl/src/math/ilogbf.c",
    "musl/src/math/ilogbl.c",
    "musl/src/math/j0.c",
    "musl/src/math/j0f.c",
    "musl/src/math/j1.c",
    "musl/src/math/j1f.c",
    "musl/src/math/jn.c",
    "musl/src/math/jnf.c",
    "musl/src/math/ldexp.c",
    "musl/src/math/ldexpf.c",
    "musl/src/math/ldexpl.c",
    "musl/src/math/lgamma.c",
    "musl/src/math/lgamma_r.c",
    "musl/src/math/lgammaf.c",
    "musl/src/math/lgammaf_r.c",
    "musl/src/math/lgammal.c",
    "musl/src/math/llrint.c",
    "musl/src/math/llrintf.c",
    "musl/src/math/llrintl.c",
    "musl/src/math/llround.c",
    "musl/src/math/llroundf.c",
    "musl/src/math/llroundl.c",
    "musl/src/math/log.c",
    "musl/src/math/log10.c",
    "musl/src/math/log10f.c",
    "musl/src/math/log10l.c",
    "musl/src/math/log1p.c",
    "musl/src/math/log1pf.c",
    "musl/src/math/log1pl.c",
    "musl/src/math/log2.c",
    "musl/src/math/log2_data.c",
    "musl/src/math/log2f.c",
    "musl/src/math/log2f_data.c",
    "musl/src/math/log2l.c",
    "musl/src/math/log_data.c",
    "musl/src/math/logb.c",
    "musl/src/math/logbf.c",
    "musl/src/math/logbl.c",
    "musl/src/math/logf.c",
    "musl/src/math/logf_data.c",
    "musl/src/math/logl.c",
    "musl/src/math/lrint.c",
    "musl/src/math/lrintf.c",
    "musl/src/math/lrintl.c",
    "musl/src/math/lround.c",
    "musl/src/math/lroundf.c",
    "musl/src/math/lroundl.c",
    "musl/src/math/m68k/sqrtl.c",
    "musl/src/math/mips/fabs.c",
    "musl/src/math/mips/fabsf.c",
    "musl/src/math/mips/sqrt.c",
    "musl/src/math/mips/sqrtf.c",
    "musl/src/math/modf.c",
    "musl/src/math/modff.c",
    "musl/src/math/modfl.c",
    "musl/src/math/nan.c",
    "musl/src/math/nanf.c",
    "musl/src/math/nanl.c",
    "musl/src/math/nearbyint.c",
    "musl/src/math/nearbyintf.c",
    "musl/src/math/nearbyintl.c",
    "musl/src/math/nextafter.c",
    "musl/src/math/nextafterf.c",
    "musl/src/math/nextafterl.c",
    "musl/src/math/nexttoward.c",
    "musl/src/math/nexttowardf.c",
    "musl/src/math/nexttowardl.c",
    "musl/src/math/pow.c",
    "musl/src/math/pow_data.c",
    "musl/src/math/powerpc/fabs.c",
    "musl/src/math/powerpc/fabsf.c",
    "musl/src/math/powerpc/fma.c",
    "musl/src/math/powerpc/fmaf.c",
    "musl/src/math/powerpc/sqrt.c",
    "musl/src/math/powerpc/sqrtf.c",
    "musl/src/math/powerpc64/ceil.c",
    "musl/src/math/powerpc64/ceilf.c",
    "musl/src/math/powerpc64/fabs.c",
    "musl/src/math/powerpc64/fabsf.c",
    "musl/src/math/powerpc64/floor.c",
    "musl/src/math/powerpc64/floorf.c",
    "musl/src/math/powerpc64/fma.c",
    "musl/src/math/powerpc64/fmaf.c",
    "musl/src/math/powerpc64/fmax.c",
    "musl/src/math/powerpc64/fmaxf.c",
    "musl/src/math/powerpc64/fmin.c",
    "musl/src/math/powerpc64/fminf.c",
    "musl/src/math/powerpc64/lrint.c",
    "musl/src/math/powerpc64/lrintf.c",
    "musl/src/math/powerpc64/lround.c",
    "musl/src/math/powerpc64/lroundf.c",
    "musl/src/math/powerpc64/round.c",
    "musl/src/math/powerpc64/roundf.c",
    "musl/src/math/powerpc64/sqrt.c",
    "musl/src/math/powerpc64/sqrtf.c",
    "musl/src/math/powerpc64/trunc.c",
    "musl/src/math/powerpc64/truncf.c",
    "musl/src/math/powf.c",
    "musl/src/math/powf_data.c",
    "musl/src/math/powl.c",
    "musl/src/math/remainder.c",
    "musl/src/math/remainderf.c",
    "musl/src/math/remainderl.c",
    "musl/src/math/remquo.c",
    "musl/src/math/remquof.c",
    "musl/src/math/remquol.c",
    "musl/src/math/rint.c",
    "musl/src/math/rintf.c",
    "musl/src/math/rintl.c",
    "musl/src/math/riscv64/copysign.c",
    "musl/src/math/riscv64/copysignf.c",
    "musl/src/math/riscv64/fabs.c",
    "musl/src/math/riscv64/fabsf.c",
    "musl/src/math/riscv64/fma.c",
    "musl/src/math/riscv64/fmaf.c",
    "musl/src/math/riscv64/fmax.c",
    "musl/src/math/riscv64/fmaxf.c",
    "musl/src/math/riscv64/fmin.c",
    "musl/src/math/riscv64/fminf.c",
    "musl/src/math/riscv64/sqrt.c",
    "musl/src/math/riscv64/sqrtf.c",
    "musl/src/math/round.c",
    "musl/src/math/roundf.c",
    "musl/src/math/roundl.c",
    "musl/src/math/s390x/ceil.c",
    "musl/src/math/s390x/ceilf.c",
    "musl/src/math/s390x/ceill.c",
    "musl/src/math/s390x/fabs.c",
    "musl/src/math/s390x/fabsf.c",
    "musl/src/math/s390x/fabsl.c",
    "musl/src/math/s390x/floor.c",
    "musl/src/math/s390x/floorf.c",
    "musl/src/math/s390x/floorl.c",
    "musl/src/math/s390x/fma.c",
    "musl/src/math/s390x/fmaf.c",
    "musl/src/math/s390x/nearbyint.c",
    "musl/src/math/s390x/nearbyintf.c",
    "musl/src/math/s390x/nearbyintl.c",
    "musl/src/math/s390x/rint.c",
    "musl/src/math/s390x/rintf.c",
    "musl/src/math/s390x/rintl.c",
    "musl/src/math/s390x/round.c",
    "musl/src/math/s390x/roundf.c",
    "musl/src/math/s390x/roundl.c",
    "musl/src/math/s390x/sqrt.c",
    "musl/src/math/s390x/sqrtf.c",
    "musl/src/math/s390x/sqrtl.c",
    "musl/src/math/s390x/trunc.c",
    "musl/src/math/s390x/truncf.c",
    "musl/src/math/s390x/truncl.c",
    "musl/src/math/scalb.c",
    "musl/src/math/scalbf.c",
    "musl/src/math/scalbln.c",
    "musl/src/math/scalblnf.c",
    "musl/src/math/scalblnl.c",
    "musl/src/math/scalbn.c",
    "musl/src/math/scalbnf.c",
    "musl/src/math/scalbnl.c",
    "musl/src/math/signgam.c",
    "musl/src/math/significand.c",
    "musl/src/math/significandf.c",
    "musl/src/math/sin.c",
    "musl/src/math/sincos.c",
    "musl/src/math/sincosf.c",
    "musl/src/math/sincosl.c",
    "musl/src/math/sinf.c",
    "musl/src/math/sinh.c",
    "musl/src/math/sinhf.c",
    "musl/src/math/sinhl.c",
    "musl/src/math/sinl.c",
    "musl/src/math/sqrt.c",
    "musl/src/math/sqrt_data.c",
    "musl/src/math/sqrtf.c",
    "musl/src/math/sqrtl.c",
    "musl/src/math/tan.c",
    "musl/src/math/tanf.c",
    "musl/src/math/tanh.c",
    "musl/src/math/tanhf.c",
    "musl/src/math/tanhl.c",
    "musl/src/math/tanl.c",
    "musl/src/math/tgamma.c",
    "musl/src/math/tgammaf.c",
    "musl/src/math/tgammal.c",
    "musl/src/math/trunc.c",
    "musl/src/math/truncf.c",
    "musl/src/math/truncl.c",
    "musl/src/math/x32/__invtrigl.s",
    "musl/src/math/x32/acosl.s",
    "musl/src/math/x32/asinl.s",
    "musl/src/math/x32/atan2l.s",
    "musl/src/math/x32/atanl.s",
    "musl/src/math/x32/ceill.s",
    "musl/src/math/x32/exp2l.s",
    "musl/src/math/x32/expl.s",
    "musl/src/math/x32/expm1l.s",
    "musl/src/math/x32/fabs.s",
    "musl/src/math/x32/fabsf.s",
    "musl/src/math/x32/fabsl.s",
    "musl/src/math/x32/floorl.s",
    "musl/src/math/x32/fma.c",
    "musl/src/math/x32/fmaf.c",
    "musl/src/math/x32/fmodl.s",
    "musl/src/math/x32/llrint.s",
    "musl/src/math/x32/llrintf.s",
    "musl/src/math/x32/llrintl.s",
    "musl/src/math/x32/log10l.s",
    "musl/src/math/x32/log1pl.s",
    "musl/src/math/x32/log2l.s",
    "musl/src/math/x32/logl.s",
    "musl/src/math/x32/lrint.s",
    "musl/src/math/x32/lrintf.s",
    "musl/src/math/x32/lrintl.s",
    "musl/src/math/x32/remainderl.s",
    "musl/src/math/x32/rintl.s",
    "musl/src/math/x32/sqrt.s",
    "musl/src/math/x32/sqrtf.s",
    "musl/src/math/x32/sqrtl.s",
    "musl/src/math/x32/truncl.s",
    "musl/src/math/x86_64/__invtrigl.s",
    "musl/src/math/x86_64/acosl.s",
    "musl/src/math/x86_64/asinl.s",
    "musl/src/math/x86_64/atan2l.s",
    "musl/src/math/x86_64/atanl.s",
    "musl/src/math/x86_64/ceill.s",
    "musl/src/math/x86_64/exp2l.s",
    "musl/src/math/x86_64/expl.s",
    "musl/src/math/x86_64/expm1l.s",
    "musl/src/math/x86_64/fabs.c",
    "musl/src/math/x86_64/fabsf.c",
    "musl/src/math/x86_64/fabsl.c",
    "musl/src/math/x86_64/floorl.s",
    "musl/src/math/x86_64/fma.c",
    "musl/src/math/x86_64/fmaf.c",
    "musl/src/math/x86_64/fmodl.c",
    "musl/src/math/x86_64/llrint.c",
    "musl/src/math/x86_64/llrintf.c",
    "musl/src/math/x86_64/llrintl.c",
    "musl/src/math/x86_64/log10l.s",
    "musl/src/math/x86_64/log1pl.s",
    "musl/src/math/x86_64/log2l.s",
    "musl/src/math/x86_64/logl.s",
    "musl/src/math/x86_64/lrint.c",
    "musl/src/math/x86_64/lrintf.c",
    "musl/src/math/x86_64/lrintl.c",
    "musl/src/math/x86_64/remainderl.c",
    "musl/src/math/x86_64/remquol.c",
    "musl/src/math/x86_64/rintl.c",
    "musl/src/math/x86_64/sqrt.c",
    "musl/src/math/x86_64/sqrtf.c",
    "musl/src/math/x86_64/sqrtl.c",
    "musl/src/math/x86_64/truncl.s",
    "musl/src/misc/a64l.c",
    "musl/src/misc/basename.c",
    "musl/src/misc/dirname.c",
    "musl/src/misc/ffs.c",
    "musl/src/misc/ffsl.c",
    "musl/src/misc/ffsll.c",
    "musl/src/misc/fmtmsg.c",
    "musl/src/misc/forkpty.c",
    "musl/src/misc/get_current_dir_name.c",
    "musl/src/misc/getauxval.c",
    "musl/src/misc/getdomainname.c",
    "musl/src/misc/getentropy.c",
    "musl/src/misc/gethostid.c",
    "musl/src/misc/getopt.c",
    "musl/src/misc/getopt_long.c",
    "musl/src/misc/getpriority.c",
    "musl/src/misc/getresgid.c",
    "musl/src/misc/getresuid.c",
    "musl/src/misc/getrlimit.c",
    "musl/src/misc/getrusage.c",
    "musl/src/misc/getsubopt.c",
    "musl/src/misc/initgroups.c",
    "musl/src/misc/ioctl.c",
    "musl/src/misc/issetugid.c",
    "musl/src/misc/lockf.c",
    "musl/src/misc/login_tty.c",
    "musl/src/misc/mntent.c",
    "musl/src/misc/nftw.c",
    "musl/src/misc/openpty.c",
    "musl/src/misc/ptsname.c",
    "musl/src/misc/pty.c",
    "musl/src/misc/realpath.c",
    "musl/src/misc/setdomainname.c",
    "musl/src/misc/setpriority.c",
    "musl/src/misc/setrlimit.c",
    "musl/src/misc/syscall.c",
    "musl/src/misc/syslog.c",
    "musl/src/misc/uname.c",
    "musl/src/misc/wordexp.c",
    "musl/src/mman/madvise.c",
    "musl/src/mman/mincore.c",
    "musl/src/mman/mlock.c",
    "musl/src/mman/mlockall.c",
    "musl/src/mman/mmap.c",
    "musl/src/mman/mprotect.c",
    "musl/src/mman/mremap.c",
    "musl/src/mman/msync.c",
    "musl/src/mman/munlock.c",
    "musl/src/mman/munlockall.c",
    "musl/src/mman/munmap.c",
    "musl/src/mman/posix_madvise.c",
    "musl/src/mman/shm_open.c",
    "musl/src/mq/mq_close.c",
    "musl/src/mq/mq_getattr.c",
    "musl/src/mq/mq_notify.c",
    "musl/src/mq/mq_open.c",
    "musl/src/mq/mq_receive.c",
    "musl/src/mq/mq_send.c",
    "musl/src/mq/mq_setattr.c",
    "musl/src/mq/mq_timedreceive.c",
    "musl/src/mq/mq_timedsend.c",
    "musl/src/mq/mq_unlink.c",
    "musl/src/multibyte/btowc.c",
    "musl/src/multibyte/c16rtomb.c",
    "musl/src/multibyte/c32rtomb.c",
    "musl/src/multibyte/internal.c",
    "musl/src/multibyte/mblen.c",
    "musl/src/multibyte/mbrlen.c",
    "musl/src/multibyte/mbrtoc16.c",
    "musl/src/multibyte/mbrtoc32.c",
    "musl/src/multibyte/mbrtowc.c",
    "musl/src/multibyte/mbsinit.c",
    "musl/src/multibyte/mbsnrtowcs.c",
    "musl/src/multibyte/mbsrtowcs.c",
    "musl/src/multibyte/mbstowcs.c",
    "musl/src/multibyte/mbtowc.c",
    "musl/src/multibyte/wcrtomb.c",
    "musl/src/multibyte/wcsnrtombs.c",
    "musl/src/multibyte/wcsrtombs.c",
    "musl/src/multibyte/wcstombs.c",
    "musl/src/multibyte/wctob.c",
    "musl/src/multibyte/wctomb.c",
    "musl/src/network/accept.c",
    "musl/src/network/accept4.c",
    "musl/src/network/bind.c",
    "musl/src/network/connect.c",
    "musl/src/network/dn_comp.c",
    "musl/src/network/dn_expand.c",
    "musl/src/network/dn_skipname.c",
    "musl/src/network/dns_parse.c",
    "musl/src/network/ent.c",
    "musl/src/network/ether.c",
    "musl/src/network/freeaddrinfo.c",
    "musl/src/network/gai_strerror.c",
    "musl/src/network/getaddrinfo.c",
    "musl/src/network/gethostbyaddr.c",
    "musl/src/network/gethostbyaddr_r.c",
    "musl/src/network/gethostbyname.c",
    "musl/src/network/gethostbyname2.c",
    "musl/src/network/gethostbyname2_r.c",
    "musl/src/network/gethostbyname_r.c",
    "musl/src/network/getifaddrs.c",
    "musl/src/network/getnameinfo.c",
    "musl/src/network/getpeername.c",
    "musl/src/network/getservbyname.c",
    "musl/src/network/getservbyname_r.c",
    "musl/src/network/getservbyport.c",
    "musl/src/network/getservbyport_r.c",
    "musl/src/network/getsockname.c",
    "musl/src/network/getsockopt.c",
    "musl/src/network/h_errno.c",
    "musl/src/network/herror.c",
    "musl/src/network/hstrerror.c",
    "musl/src/network/htonl.c",
    "musl/src/network/htons.c",
    "musl/src/network/if_freenameindex.c",
    "musl/src/network/if_indextoname.c",
    "musl/src/network/if_nameindex.c",
    "musl/src/network/if_nametoindex.c",
    "musl/src/network/in6addr_any.c",
    "musl/src/network/in6addr_loopback.c",
    "musl/src/network/inet_addr.c",
    "musl/src/network/inet_aton.c",
    "musl/src/network/inet_legacy.c",
    "musl/src/network/inet_ntoa.c",
    "musl/src/network/inet_ntop.c",
    "musl/src/network/inet_pton.c",
    "musl/src/network/listen.c",
    "musl/src/network/lookup_ipliteral.c",
    "musl/src/network/lookup_name.c",
    "musl/src/network/lookup_serv.c",
    "musl/src/network/netlink.c",
    "musl/src/network/netname.c",
    "musl/src/network/ns_parse.c",
    "musl/src/network/ntohl.c",
    "musl/src/network/ntohs.c",
    "musl/src/network/proto.c",
    "musl/src/network/recv.c",
    "musl/src/network/recvfrom.c",
    "musl/src/network/recvmmsg.c",
    "musl/src/network/recvmsg.c",
    "musl/src/network/res_init.c",
    "musl/src/network/res_mkquery.c",
    "musl/src/network/res_msend.c",
    "musl/src/network/res_query.c",
    "musl/src/network/res_querydomain.c",
    "musl/src/network/res_send.c",
    "musl/src/network/res_state.c",
    "musl/src/network/resolvconf.c",
    "musl/src/network/send.c",
    "musl/src/network/sendmmsg.c",
    "musl/src/network/sendmsg.c",
    "musl/src/network/sendto.c",
    "musl/src/network/serv.c",
    "musl/src/network/setsockopt.c",
    "musl/src/network/shutdown.c",
    "musl/src/network/sockatmark.c",
    "musl/src/network/socket.c",
    "musl/src/network/socketpair.c",
    "musl/src/passwd/fgetgrent.c",
    "musl/src/passwd/fgetpwent.c",
    "musl/src/passwd/fgetspent.c",
    "musl/src/passwd/getgr_a.c",
    "musl/src/passwd/getgr_r.c",
    "musl/src/passwd/getgrent.c",
    "musl/src/passwd/getgrent_a.c",
    "musl/src/passwd/getgrouplist.c",
    "musl/src/passwd/getpw_a.c",
    "musl/src/passwd/getpw_r.c",
    "musl/src/passwd/getpwent.c",
    "musl/src/passwd/getpwent_a.c",
    "musl/src/passwd/getspent.c",
    "musl/src/passwd/getspnam.c",
    "musl/src/passwd/getspnam_r.c",
    "musl/src/passwd/lckpwdf.c",
    "musl/src/passwd/nscd_query.c",
    "musl/src/passwd/putgrent.c",
    "musl/src/passwd/putpwent.c",
    "musl/src/passwd/putspent.c",
    "musl/src/prng/__rand48_step.c",
    "musl/src/prng/__seed48.c",
    "musl/src/prng/drand48.c",
    "musl/src/prng/lcong48.c",
    "musl/src/prng/lrand48.c",
    "musl/src/prng/mrand48.c",
    "musl/src/prng/rand.c",
    "musl/src/prng/rand_r.c",
    "musl/src/prng/random.c",
    "musl/src/prng/seed48.c",
    "musl/src/prng/srand48.c",
    "musl/src/process/_Fork.c",
    "musl/src/process/aarch64/vfork.s",
    "musl/src/process/arm/vfork.s",
    "musl/src/process/execl.c",
    "musl/src/process/execle.c",
    "musl/src/process/execlp.c",
    "musl/src/process/execv.c",
    "musl/src/process/execve.c",
    "musl/src/process/execvp.c",
    "musl/src/process/fexecve.c",
    "musl/src/process/fork.c",
    "musl/src/process/i386/vfork.s",
    "musl/src/process/posix_spawn.c",
    "musl/src/process/posix_spawn_file_actions_addchdir.c",
    "musl/src/process/posix_spawn_file_actions_addclose.c",
    "musl/src/process/posix_spawn_file_actions_adddup2.c",
    "musl/src/process/posix_spawn_file_actions_addfchdir.c",
    "musl/src/process/posix_spawn_file_actions_addopen.c",
    "musl/src/process/posix_spawn_file_actions_destroy.c",
    "musl/src/process/posix_spawn_file_actions_init.c",
    "musl/src/process/posix_spawnattr_destroy.c",
    "musl/src/process/posix_spawnattr_getflags.c",
    "musl/src/process/posix_spawnattr_getpgroup.c",
    "musl/src/process/posix_spawnattr_getsigdefault.c",
    "musl/src/process/posix_spawnattr_getsigmask.c",
    "musl/src/process/posix_spawnattr_init.c",
    "musl/src/process/posix_spawnattr_sched.c",
    "musl/src/process/posix_spawnattr_setflags.c",
    "musl/src/process/posix_spawnattr_setpgroup.c",
    "musl/src/process/posix_spawnattr_setsigdefault.c",
    "musl/src/process/posix_spawnattr_setsigmask.c",
    "musl/src/process/posix_spawnp.c",
    "musl/src/process/riscv64/vfork.s",
    "musl/src/process/s390x/vfork.s",
    "musl/src/process/sh/vfork.s",
    "musl/src/process/system.c",
    "musl/src/process/vfork.c",
    "musl/src/process/wait.c",
    "musl/src/process/waitid.c",
    "musl/src/process/waitpid.c",
    "musl/src/process/x32/vfork.s",
    "musl/src/process/x86_64/vfork.s",
    "musl/src/regex/fnmatch.c",
    "musl/src/regex/glob.c",
    "musl/src/regex/regcomp.c",
    "musl/src/regex/regerror.c",
    "musl/src/regex/regexec.c",
    "musl/src/regex/tre-mem.c",
    "musl/src/sched/affinity.c",
    "musl/src/sched/sched_cpucount.c",
    "musl/src/sched/sched_get_priority_max.c",
    "musl/src/sched/sched_getcpu.c",
    "musl/src/sched/sched_getparam.c",
    "musl/src/sched/sched_getscheduler.c",
    "musl/src/sched/sched_rr_get_interval.c",
    "musl/src/sched/sched_setparam.c",
    "musl/src/sched/sched_setscheduler.c",
    "musl/src/sched/sched_yield.c",
    "musl/src/search/hsearch.c",
    "musl/src/search/insque.c",
    "musl/src/search/lsearch.c",
    "musl/src/search/tdelete.c",
    "musl/src/search/tdestroy.c",
    "musl/src/search/tfind.c",
    "musl/src/search/tsearch.c",
    "musl/src/search/twalk.c",
    "musl/src/select/poll.c",
    "musl/src/select/pselect.c",
    "musl/src/select/select.c",
    "musl/src/setjmp/aarch64/longjmp.s",
    "musl/src/setjmp/aarch64/setjmp.s",
    "musl/src/setjmp/arm/longjmp.S",
    "musl/src/setjmp/arm/setjmp.S",
    "musl/src/setjmp/i386/longjmp.s",
    "musl/src/setjmp/i386/setjmp.s",
    "musl/src/setjmp/longjmp.c",
    "musl/src/setjmp/m68k/longjmp.s",
    "musl/src/setjmp/m68k/setjmp.s",
    "musl/src/setjmp/microblaze/longjmp.s",
    "musl/src/setjmp/microblaze/setjmp.s",
    "musl/src/setjmp/mips/longjmp.S",
    "musl/src/setjmp/mips/setjmp.S",
    "musl/src/setjmp/mips64/longjmp.S",
    "musl/src/setjmp/mips64/setjmp.S",
    "musl/src/setjmp/mipsn32/longjmp.S",
    "musl/src/setjmp/mipsn32/setjmp.S",
    "musl/src/setjmp/or1k/longjmp.s",
    "musl/src/setjmp/or1k/setjmp.s",
    "musl/src/setjmp/powerpc/longjmp.S",
    "musl/src/setjmp/powerpc/setjmp.S",
    "musl/src/setjmp/powerpc64/longjmp.s",
    "musl/src/setjmp/powerpc64/setjmp.s",
    "musl/src/setjmp/riscv64/longjmp.S",
    "musl/src/setjmp/riscv64/setjmp.S",
    "musl/src/setjmp/s390x/longjmp.s",
    "musl/src/setjmp/s390x/setjmp.s",
    "musl/src/setjmp/setjmp.c",
    "musl/src/setjmp/sh/longjmp.S",
    "musl/src/setjmp/sh/setjmp.S",
    "musl/src/setjmp/x32/longjmp.s",
    "musl/src/setjmp/x32/setjmp.s",
    "musl/src/setjmp/x86_64/longjmp.s",
    "musl/src/setjmp/x86_64/setjmp.s",
    "musl/src/signal/aarch64/restore.s",
    "musl/src/signal/aarch64/sigsetjmp.s",
    "musl/src/signal/arm/restore.s",
    "musl/src/signal/arm/sigsetjmp.s",
    "musl/src/signal/block.c",
    "musl/src/signal/getitimer.c",
    "musl/src/signal/i386/restore.s",
    "musl/src/signal/i386/sigsetjmp.s",
    "musl/src/signal/kill.c",
    "musl/src/signal/killpg.c",
    "musl/src/signal/m68k/sigsetjmp.s",
    "musl/src/signal/microblaze/restore.s",
    "musl/src/signal/microblaze/sigsetjmp.s",
    "musl/src/signal/mips/sigsetjmp.s",
    "musl/src/signal/mips64/sigsetjmp.s",
    "musl/src/signal/mipsn32/sigsetjmp.s",
    "musl/src/signal/or1k/sigsetjmp.s",
    "musl/src/signal/powerpc/restore.s",
    "musl/src/signal/powerpc/sigsetjmp.s",
    "musl/src/signal/powerpc64/restore.s",
    "musl/src/signal/powerpc64/sigsetjmp.s",
    "musl/src/signal/psiginfo.c",
    "musl/src/signal/psignal.c",
    "musl/src/signal/raise.c",
    "musl/src/signal/restore.c",
    "musl/src/signal/riscv64/restore.s",
    "musl/src/signal/riscv64/sigsetjmp.s",
    "musl/src/signal/s390x/restore.s",
    "musl/src/signal/s390x/sigsetjmp.s",
    "musl/src/signal/setitimer.c",
    "musl/src/signal/sh/restore.s",
    "musl/src/signal/sh/sigsetjmp.s",
    "musl/src/signal/sigaction.c",
    "musl/src/signal/sigaddset.c",
    "musl/src/signal/sigaltstack.c",
    "musl/src/signal/sigandset.c",
    "musl/src/signal/sigdelset.c",
    "musl/src/signal/sigemptyset.c",
    "musl/src/signal/sigfillset.c",
    "musl/src/signal/sighold.c",
    "musl/src/signal/sigignore.c",
    "musl/src/signal/siginterrupt.c",
    "musl/src/signal/sigisemptyset.c",
    "musl/src/signal/sigismember.c",
    "musl/src/signal/siglongjmp.c",
    "musl/src/signal/signal.c",
    "musl/src/signal/sigorset.c",
    "musl/src/signal/sigpause.c",
    "musl/src/signal/sigpending.c",
    "musl/src/signal/sigprocmask.c",
    "musl/src/signal/sigqueue.c",
    "musl/src/signal/sigrelse.c",
    "musl/src/signal/sigrtmax.c",
    "musl/src/signal/sigrtmin.c",
    "musl/src/signal/sigset.c",
    "musl/src/signal/sigsetjmp.c",
    "musl/src/signal/sigsetjmp_tail.c",
    "musl/src/signal/sigsuspend.c",
    "musl/src/signal/sigtimedwait.c",
    "musl/src/signal/sigwait.c",
    "musl/src/signal/sigwaitinfo.c",
    "musl/src/signal/x32/getitimer.c",
    "musl/src/signal/x32/restore.s",
    "musl/src/signal/x32/setitimer.c",
    "musl/src/signal/x32/sigsetjmp.s",
    "musl/src/signal/x86_64/restore.s",
    "musl/src/signal/x86_64/sigsetjmp.s",
    "musl/src/stat/__xstat.c",
    "musl/src/stat/chmod.c",
    "musl/src/stat/fchmod.c",
    "musl/src/stat/fchmodat.c",
    "musl/src/stat/fstat.c",
    "musl/src/stat/fstatat.c",
    "musl/src/stat/futimens.c",
    "musl/src/stat/futimesat.c",
    "musl/src/stat/lchmod.c",
    "musl/src/stat/lstat.c",
    "musl/src/stat/mkdir.c",
    "musl/src/stat/mkdirat.c",
    "musl/src/stat/mkfifo.c",
    "musl/src/stat/mkfifoat.c",
    "musl/src/stat/mknod.c",
    "musl/src/stat/mknodat.c",
    "musl/src/stat/stat.c",
    "musl/src/stat/statvfs.c",
    "musl/src/stat/umask.c",
    "musl/src/stat/utimensat.c",
    "musl/src/stdio/__fclose_ca.c",
    "musl/src/stdio/__fdopen.c",
    "musl/src/stdio/__fmodeflags.c",
    "musl/src/stdio/__fopen_rb_ca.c",
    "musl/src/stdio/__lockfile.c",
    "musl/src/stdio/__overflow.c",
    "musl/src/stdio/__stdio_close.c",
    "musl/src/stdio/__stdio_exit.c",
    "musl/src/stdio/__stdio_read.c",
    "musl/src/stdio/__stdio_seek.c",
    "musl/src/stdio/__stdio_write.c",
    "musl/src/stdio/__stdout_write.c",
    "musl/src/stdio/__toread.c",
    "musl/src/stdio/__towrite.c",
    "musl/src/stdio/__uflow.c",
    "musl/src/stdio/asprintf.c",
    "musl/src/stdio/clearerr.c",
    "musl/src/stdio/dprintf.c",
    "musl/src/stdio/ext.c",
    "musl/src/stdio/ext2.c",
    "musl/src/stdio/fclose.c",
    "musl/src/stdio/feof.c",
    "musl/src/stdio/ferror.c",
    "musl/src/stdio/fflush.c",
    "musl/src/stdio/fgetc.c",
    "musl/src/stdio/fgetln.c",
    "musl/src/stdio/fgetpos.c",
    "musl/src/stdio/fgets.c",
    "musl/src/stdio/fgetwc.c",
    "musl/src/stdio/fgetws.c",
    "musl/src/stdio/fileno.c",
    "musl/src/stdio/flockfile.c",
    "musl/src/stdio/fmemopen.c",
    "musl/src/stdio/fopen.c",
    "musl/src/stdio/fopencookie.c",
    "musl/src/stdio/fprintf.c",
    "musl/src/stdio/fputc.c",
    "musl/src/stdio/fputs.c",
    "musl/src/stdio/fputwc.c",
    "musl/src/stdio/fputws.c",
    "musl/src/stdio/fread.c",
    "musl/src/stdio/freopen.c",
    "musl/src/stdio/fscanf.c",
    "musl/src/stdio/fseek.c",
    "musl/src/stdio/fsetpos.c",
    "musl/src/stdio/ftell.c",
    "musl/src/stdio/ftrylockfile.c",
    "musl/src/stdio/funlockfile.c",
    "musl/src/stdio/fwide.c",
    "musl/src/stdio/fwprintf.c",
    "musl/src/stdio/fwrite.c",
    "musl/src/stdio/fwscanf.c",
    "musl/src/stdio/getc.c",
    "musl/src/stdio/getc_unlocked.c",
    "musl/src/stdio/getchar.c",
    "musl/src/stdio/getchar_unlocked.c",
    "musl/src/stdio/getdelim.c",
    "musl/src/stdio/getline.c",
    "musl/src/stdio/gets.c",
    "musl/src/stdio/getw.c",
    "musl/src/stdio/getwc.c",
    "musl/src/stdio/getwchar.c",
    "musl/src/stdio/ofl.c",
    "musl/src/stdio/ofl_add.c",
    "musl/src/stdio/open_memstream.c",
    "musl/src/stdio/open_wmemstream.c",
    "musl/src/stdio/pclose.c",
    "musl/src/stdio/perror.c",
    "musl/src/stdio/popen.c",
    "musl/src/stdio/printf.c",
    "musl/src/stdio/putc.c",
    "musl/src/stdio/putc_unlocked.c",
    "musl/src/stdio/putchar.c",
    "musl/src/stdio/putchar_unlocked.c",
    "musl/src/stdio/puts.c",
    "musl/src/stdio/putw.c",
    "musl/src/stdio/putwc.c",
    "musl/src/stdio/putwchar.c",
    "musl/src/stdio/remove.c",
    "musl/src/stdio/rename.c",
    "musl/src/stdio/rewind.c",
    "musl/src/stdio/scanf.c",
    "musl/src/stdio/setbuf.c",
    "musl/src/stdio/setbuffer.c",
    "musl/src/stdio/setlinebuf.c",
    "musl/src/stdio/setvbuf.c",
    "musl/src/stdio/snprintf.c",
    "musl/src/stdio/sprintf.c",
    "musl/src/stdio/sscanf.c",
    "musl/src/stdio/stderr.c",
    "musl/src/stdio/stdin.c",
    "musl/src/stdio/stdout.c",
    "musl/src/stdio/swprintf.c",
    "musl/src/stdio/swscanf.c",
    "musl/src/stdio/tempnam.c",
    "musl/src/stdio/tmpfile.c",
    "musl/src/stdio/tmpnam.c",
    "musl/src/stdio/ungetc.c",
    "musl/src/stdio/ungetwc.c",
    "musl/src/stdio/vasprintf.c",
    "musl/src/stdio/vdprintf.c",
    "musl/src/stdio/vfprintf.c",
    "musl/src/stdio/vfscanf.c",
    "musl/src/stdio/vfwprintf.c",
    "musl/src/stdio/vfwscanf.c",
    "musl/src/stdio/vprintf.c",
    "musl/src/stdio/vscanf.c",
    "musl/src/stdio/vsnprintf.c",
    "musl/src/stdio/vsprintf.c",
    "musl/src/stdio/vsscanf.c",
    "musl/src/stdio/vswprintf.c",
    "musl/src/stdio/vswscanf.c",
    "musl/src/stdio/vwprintf.c",
    "musl/src/stdio/vwscanf.c",
    "musl/src/stdio/wprintf.c",
    "musl/src/stdio/wscanf.c",
    "musl/src/stdlib/abs.c",
    "musl/src/stdlib/atof.c",
    "musl/src/stdlib/atoi.c",
    "musl/src/stdlib/atol.c",
    "musl/src/stdlib/atoll.c",
    "musl/src/stdlib/bsearch.c",
    "musl/src/stdlib/div.c",
    "musl/src/stdlib/ecvt.c",
    "musl/src/stdlib/fcvt.c",
    "musl/src/stdlib/gcvt.c",
    "musl/src/stdlib/imaxabs.c",
    "musl/src/stdlib/imaxdiv.c",
    "musl/src/stdlib/labs.c",
    "musl/src/stdlib/ldiv.c",
    "musl/src/stdlib/llabs.c",
    "musl/src/stdlib/lldiv.c",
    "musl/src/stdlib/qsort.c",
    "musl/src/stdlib/qsort_nr.c",
    "musl/src/stdlib/strtod.c",
    "musl/src/stdlib/strtol.c",
    "musl/src/stdlib/wcstod.c",
    "musl/src/stdlib/wcstol.c",
    "musl/src/string/aarch64/memcpy.S",
    "musl/src/string/aarch64/memset.S",
    "musl/src/string/arm/__aeabi_memcpy.s",
    "musl/src/string/arm/__aeabi_memset.s",
    "musl/src/string/arm/memcpy.S",
    "musl/src/string/bcmp.c",
    "musl/src/string/bcopy.c",
    "musl/src/string/bzero.c",
    "musl/src/string/explicit_bzero.c",
    "musl/src/string/i386/memcpy.s",
    "musl/src/string/i386/memmove.s",
    "musl/src/string/i386/memset.s",
    "musl/src/string/index.c",
    "musl/src/string/memccpy.c",
    "musl/src/string/memchr.c",
    "musl/src/string/memcmp.c",
    "musl/src/string/memcpy.c",
    "musl/src/string/memmem.c",
    "musl/src/string/memmove.c",
    "musl/src/string/mempcpy.c",
    "musl/src/string/memrchr.c",
    "musl/src/string/memset.c",
    "musl/src/string/rindex.c",
    "musl/src/string/stpcpy.c",
    "musl/src/string/stpncpy.c",
    "musl/src/string/strcasecmp.c",
    "musl/src/string/strcasestr.c",
    "musl/src/string/strcat.c",
    "musl/src/string/strchr.c",
    "musl/src/string/strchrnul.c",
    "musl/src/string/strcmp.c",
    "musl/src/string/strcpy.c",
    "musl/src/string/strcspn.c",
    "musl/src/string/strdup.c",
    "musl/src/string/strerror_r.c",
    "musl/src/string/strlcat.c",
    "musl/src/string/strlcpy.c",
    "musl/src/string/strlen.c",
    "musl/src/string/strncasecmp.c",
    "musl/src/string/strncat.c",
    "musl/src/string/strncmp.c",
    "musl/src/string/strncpy.c",
    "musl/src/string/strndup.c",
    "musl/src/string/strnlen.c",
    "musl/src/string/strpbrk.c",
    "musl/src/string/strrchr.c",
    "musl/src/string/strsep.c",
    "musl/src/string/strsignal.c",
    "musl/src/string/strspn.c",
    "musl/src/string/strstr.c",
    "musl/src/string/strtok.c",
    "musl/src/string/strtok_r.c",
    "musl/src/string/strverscmp.c",
    "musl/src/string/swab.c",
    "musl/src/string/wcpcpy.c",
    "musl/src/string/wcpncpy.c",
    "musl/src/string/wcscasecmp.c",
    "musl/src/string/wcscasecmp_l.c",
    "musl/src/string/wcscat.c",
    "musl/src/string/wcschr.c",
    "musl/src/string/wcscmp.c",
    "musl/src/string/wcscpy.c",
    "musl/src/string/wcscspn.c",
    "musl/src/string/wcsdup.c",
    "musl/src/string/wcslen.c",
    "musl/src/string/wcsncasecmp.c",
    "musl/src/string/wcsncasecmp_l.c",
    "musl/src/string/wcsncat.c",
    "musl/src/string/wcsncmp.c",
    "musl/src/string/wcsncpy.c",
    "musl/src/string/wcsnlen.c",
    "musl/src/string/wcspbrk.c",
    "musl/src/string/wcsrchr.c",
    "musl/src/string/wcsspn.c",
    "musl/src/string/wcsstr.c",
    "musl/src/string/wcstok.c",
    "musl/src/string/wcswcs.c",
    "musl/src/string/wmemchr.c",
    "musl/src/string/wmemcmp.c",
    "musl/src/string/wmemcpy.c",
    "musl/src/string/wmemmove.c",
    "musl/src/string/wmemset.c",
    "musl/src/string/x86_64/memcpy.s",
    "musl/src/string/x86_64/memmove.s",
    "musl/src/string/x86_64/memset.s",
    "musl/src/temp/__randname.c",
    "musl/src/temp/mkdtemp.c",
    "musl/src/temp/mkostemp.c",
    "musl/src/temp/mkostemps.c",
    "musl/src/temp/mkstemp.c",
    "musl/src/temp/mkstemps.c",
    "musl/src/temp/mktemp.c",
    "musl/src/termios/cfgetospeed.c",
    "musl/src/termios/cfmakeraw.c",
    "musl/src/termios/cfsetospeed.c",
    "musl/src/termios/tcdrain.c",
    "musl/src/termios/tcflow.c",
    "musl/src/termios/tcflush.c",
    "musl/src/termios/tcgetattr.c",
    "musl/src/termios/tcgetsid.c",
    "musl/src/termios/tcgetwinsize.c",
    "musl/src/termios/tcsendbreak.c",
    "musl/src/termios/tcsetattr.c",
    "musl/src/termios/tcsetwinsize.c",
    "musl/src/thread/__lock.c",
    "musl/src/thread/__set_thread_area.c",
    "musl/src/thread/__syscall_cp.c",
    "musl/src/thread/__timedwait.c",
    "musl/src/thread/__tls_get_addr.c",
    "musl/src/thread/__unmapself.c",
    "musl/src/thread/__wait.c",
    "musl/src/thread/aarch64/__set_thread_area.s",
    "musl/src/thread/aarch64/__unmapself.s",
    "musl/src/thread/aarch64/clone.s",
    "musl/src/thread/aarch64/syscall_cp.s",
    "musl/src/thread/arm/__aeabi_read_tp.s",
    "musl/src/thread/arm/__set_thread_area.c",
    "musl/src/thread/arm/__unmapself.s",
    "musl/src/thread/arm/atomics.s",
    "musl/src/thread/arm/clone.s",
    "musl/src/thread/arm/syscall_cp.s",
    "musl/src/thread/call_once.c",
    "musl/src/thread/clone.c",
    "musl/src/thread/cnd_broadcast.c",
    "musl/src/thread/cnd_destroy.c",
    "musl/src/thread/cnd_init.c",
    "musl/src/thread/cnd_signal.c",
    "musl/src/thread/cnd_timedwait.c",
    "musl/src/thread/cnd_wait.c",
    "musl/src/thread/default_attr.c",
    "musl/src/thread/i386/__set_thread_area.s",
    "musl/src/thread/i386/__unmapself.s",
    "musl/src/thread/i386/clone.s",
    "musl/src/thread/i386/syscall_cp.s",
    "musl/src/thread/i386/tls.s",
    "musl/src/thread/lock_ptc.c",
    "musl/src/thread/m68k/__m68k_read_tp.s",
    "musl/src/thread/m68k/clone.s",
    "musl/src/thread/m68k/syscall_cp.s",
    "musl/src/thread/microblaze/__set_thread_area.s",
    "musl/src/thread/microblaze/__unmapself.s",
    "musl/src/thread/microblaze/clone.s",
    "musl/src/thread/microblaze/syscall_cp.s",
    "musl/src/thread/mips/__unmapself.s",
    "musl/src/thread/mips/clone.s",
    "musl/src/thread/mips/syscall_cp.s",
    "musl/src/thread/mips64/__unmapself.s",
    "musl/src/thread/mips64/clone.s",
    "musl/src/thread/mips64/syscall_cp.s",
    "musl/src/thread/mipsn32/__unmapself.s",
    "musl/src/thread/mipsn32/clone.s",
    "musl/src/thread/mipsn32/syscall_cp.s",
    "musl/src/thread/mtx_destroy.c",
    "musl/src/thread/mtx_init.c",
    "musl/src/thread/mtx_lock.c",
    "musl/src/thread/mtx_timedlock.c",
    "musl/src/thread/mtx_trylock.c",
    "musl/src/thread/mtx_unlock.c",
    "musl/src/thread/or1k/__set_thread_area.s",
    "musl/src/thread/or1k/__unmapself.s",
    "musl/src/thread/or1k/clone.s",
    "musl/src/thread/or1k/syscall_cp.s",
    "musl/src/thread/powerpc/__set_thread_area.s",
    "musl/src/thread/powerpc/__unmapself.s",
    "musl/src/thread/powerpc/clone.s",
    "musl/src/thread/powerpc/syscall_cp.s",
    "musl/src/thread/powerpc64/__set_thread_area.s",
    "musl/src/thread/powerpc64/__unmapself.s",
    "musl/src/thread/powerpc64/clone.s",
    "musl/src/thread/powerpc64/syscall_cp.s",
    "musl/src/thread/pthread_atfork.c",
    "musl/src/thread/pthread_attr_destroy.c",
    "musl/src/thread/pthread_attr_get.c",
    "musl/src/thread/pthread_attr_init.c",
    "musl/src/thread/pthread_attr_setdetachstate.c",
    "musl/src/thread/pthread_attr_setguardsize.c",
    "musl/src/thread/pthread_attr_setinheritsched.c",
    "musl/src/thread/pthread_attr_setschedparam.c",
    "musl/src/thread/pthread_attr_setschedpolicy.c",
    "musl/src/thread/pthread_attr_setscope.c",
    "musl/src/thread/pthread_attr_setstack.c",
    "musl/src/thread/pthread_attr_setstacksize.c",
    "musl/src/thread/pthread_barrier_destroy.c",
    "musl/src/thread/pthread_barrier_init.c",
    "musl/src/thread/pthread_barrier_wait.c",
    "musl/src/thread/pthread_barrierattr_destroy.c",
    "musl/src/thread/pthread_barrierattr_init.c",
    "musl/src/thread/pthread_barrierattr_setpshared.c",
    "musl/src/thread/pthread_cancel.c",
    "musl/src/thread/pthread_cleanup_push.c",
    "musl/src/thread/pthread_cond_broadcast.c",
    "musl/src/thread/pthread_cond_destroy.c",
    "musl/src/thread/pthread_cond_init.c",
    "musl/src/thread/pthread_cond_signal.c",
    "musl/src/thread/pthread_cond_timedwait.c",
    "musl/src/thread/pthread_cond_wait.c",
    "musl/src/thread/pthread_condattr_destroy.c",
    "musl/src/thread/pthread_condattr_init.c",
    "musl/src/thread/pthread_condattr_setclock.c",
    "musl/src/thread/pthread_condattr_setpshared.c",
    "musl/src/thread/pthread_create.c",
    "musl/src/thread/pthread_detach.c",
    "musl/src/thread/pthread_equal.c",
    "musl/src/thread/pthread_getattr_np.c",
    "musl/src/thread/pthread_getconcurrency.c",
    "musl/src/thread/pthread_getcpuclockid.c",
    "musl/src/thread/pthread_getname_np.c",
    "musl/src/thread/pthread_getschedparam.c",
    "musl/src/thread/pthread_getspecific.c",
    "musl/src/thread/pthread_join.c",
    "musl/src/thread/pthread_key_create.c",
    "musl/src/thread/pthread_kill.c",
    "musl/src/thread/pthread_mutex_consistent.c",
    "musl/src/thread/pthread_mutex_destroy.c",
    "musl/src/thread/pthread_mutex_getprioceiling.c",
    "musl/src/thread/pthread_mutex_init.c",
    "musl/src/thread/pthread_mutex_lock.c",
    "musl/src/thread/pthread_mutex_setprioceiling.c",
    "musl/src/thread/pthread_mutex_timedlock.c",
    "musl/src/thread/pthread_mutex_trylock.c",
    "musl/src/thread/pthread_mutex_unlock.c",
    "musl/src/thread/pthread_mutexattr_destroy.c",
    "musl/src/thread/pthread_mutexattr_init.c",
    "musl/src/thread/pthread_mutexattr_setprotocol.c",
    "musl/src/thread/pthread_mutexattr_setpshared.c",
    "musl/src/thread/pthread_mutexattr_setrobust.c",
    "musl/src/thread/pthread_mutexattr_settype.c",
    "musl/src/thread/pthread_once.c",
    "musl/src/thread/pthread_rwlock_destroy.c",
    "musl/src/thread/pthread_rwlock_init.c",
    "musl/src/thread/pthread_rwlock_rdlock.c",
    "musl/src/thread/pthread_rwlock_timedrdlock.c",
    "musl/src/thread/pthread_rwlock_timedwrlock.c",
    "musl/src/thread/pthread_rwlock_tryrdlock.c",
    "musl/src/thread/pthread_rwlock_trywrlock.c",
    "musl/src/thread/pthread_rwlock_unlock.c",
    "musl/src/thread/pthread_rwlock_wrlock.c",
    "musl/src/thread/pthread_rwlockattr_destroy.c",
    "musl/src/thread/pthread_rwlockattr_init.c",
    "musl/src/thread/pthread_rwlockattr_setpshared.c",
    "musl/src/thread/pthread_self.c",
    "musl/src/thread/pthread_setattr_default_np.c",
    "musl/src/thread/pthread_setcancelstate.c",
    "musl/src/thread/pthread_setcanceltype.c",
    "musl/src/thread/pthread_setconcurrency.c",
    "musl/src/thread/pthread_setname_np.c",
    "musl/src/thread/pthread_setschedparam.c",
    "musl/src/thread/pthread_setschedprio.c",
    "musl/src/thread/pthread_setspecific.c",
    "musl/src/thread/pthread_sigmask.c",
    "musl/src/thread/pthread_spin_destroy.c",
    "musl/src/thread/pthread_spin_init.c",
    "musl/src/thread/pthread_spin_lock.c",
    "musl/src/thread/pthread_spin_trylock.c",
    "musl/src/thread/pthread_spin_unlock.c",
    "musl/src/thread/pthread_testcancel.c",
    "musl/src/thread/riscv64/__set_thread_area.s",
    "musl/src/thread/riscv64/__unmapself.s",
    "musl/src/thread/riscv64/clone.s",
    "musl/src/thread/riscv64/syscall_cp.s",
    "musl/src/thread/s390x/__set_thread_area.s",
    "musl/src/thread/s390x/__tls_get_offset.s",
    "musl/src/thread/s390x/__unmapself.s",
    "musl/src/thread/s390x/clone.s",
    "musl/src/thread/s390x/syscall_cp.s",
    "musl/src/thread/sem_destroy.c",
    "musl/src/thread/sem_getvalue.c",
    "musl/src/thread/sem_init.c",
    "musl/src/thread/sem_open.c",
    "musl/src/thread/sem_post.c",
    "musl/src/thread/sem_timedwait.c",
    "musl/src/thread/sem_trywait.c",
    "musl/src/thread/sem_unlink.c",
    "musl/src/thread/sem_wait.c",
    "musl/src/thread/sh/__set_thread_area.c",
    "musl/src/thread/sh/__unmapself.c",
    "musl/src/thread/sh/__unmapself_mmu.s",
    "musl/src/thread/sh/atomics.s",
    "musl/src/thread/sh/clone.s",
    "musl/src/thread/sh/syscall_cp.s",
    "musl/src/thread/synccall.c",
    "musl/src/thread/syscall_cp.c",
    "musl/src/thread/thrd_create.c",
    "musl/src/thread/thrd_exit.c",
    "musl/src/thread/thrd_join.c",
    "musl/src/thread/thrd_sleep.c",
    "musl/src/thread/thrd_yield.c",
    "musl/src/thread/tls.c",
    "musl/src/thread/tss_create.c",
    "musl/src/thread/tss_delete.c",
    "musl/src/thread/tss_set.c",
    "musl/src/thread/vmlock.c",
    "musl/src/thread/x32/__set_thread_area.s",
    "musl/src/thread/x32/__unmapself.s",
    "musl/src/thread/x32/clone.s",
    "musl/src/thread/x32/syscall_cp.s",
    "musl/src/thread/x86_64/__set_thread_area.s",
    "musl/src/thread/x86_64/__unmapself.s",
    "musl/src/thread/x86_64/clone.s",
    "musl/src/thread/x86_64/syscall_cp.s",
    "musl/src/time/__map_file.c",
    "musl/src/time/__month_to_secs.c",
    "musl/src/time/__secs_to_tm.c",
    "musl/src/time/__tm_to_secs.c",
    "musl/src/time/__tz.c",
    "musl/src/time/__year_to_secs.c",
    "musl/src/time/asctime.c",
    "musl/src/time/asctime_r.c",
    "musl/src/time/clock.c",
    "musl/src/time/clock_getcpuclockid.c",
    "musl/src/time/clock_getres.c",
    "musl/src/time/clock_gettime.c",
    "musl/src/time/clock_nanosleep.c",
    "musl/src/time/clock_settime.c",
    "musl/src/time/ctime.c",
    "musl/src/time/ctime_r.c",
    "musl/src/time/difftime.c",
    "musl/src/time/ftime.c",
    "musl/src/time/getdate.c",
    "musl/src/time/gettimeofday.c",
    "musl/src/time/gmtime.c",
    "musl/src/time/gmtime_r.c",
    "musl/src/time/localtime.c",
    "musl/src/time/localtime_r.c",
    "musl/src/time/mktime.c",
    "musl/src/time/nanosleep.c",
    "musl/src/time/strftime.c",
    "musl/src/time/strptime.c",
    "musl/src/time/time.c",
    "musl/src/time/timegm.c",
    "musl/src/time/timer_create.c",
    "musl/src/time/timer_delete.c",
    "musl/src/time/timer_getoverrun.c",
    "musl/src/time/timer_gettime.c",
    "musl/src/time/timer_settime.c",
    "musl/src/time/times.c",
    "musl/src/time/timespec_get.c",
    "musl/src/time/utime.c",
    "musl/src/time/wcsftime.c",
    "musl/src/unistd/_exit.c",
    "musl/src/unistd/access.c",
    "musl/src/unistd/acct.c",
    "musl/src/unistd/alarm.c",
    "musl/src/unistd/chdir.c",
    "musl/src/unistd/chown.c",
    "musl/src/unistd/close.c",
    "musl/src/unistd/ctermid.c",
    "musl/src/unistd/dup.c",
    "musl/src/unistd/dup2.c",
    "musl/src/unistd/dup3.c",
    "musl/src/unistd/faccessat.c",
    "musl/src/unistd/fchdir.c",
    "musl/src/unistd/fchown.c",
    "musl/src/unistd/fchownat.c",
    "musl/src/unistd/fdatasync.c",
    "musl/src/unistd/fsync.c",
    "musl/src/unistd/ftruncate.c",
    "musl/src/unistd/getcwd.c",
    "musl/src/unistd/getegid.c",
    "musl/src/unistd/geteuid.c",
    "musl/src/unistd/getgid.c",
    "musl/src/unistd/getgroups.c",
    "musl/src/unistd/gethostname.c",
    "musl/src/unistd/getlogin.c",
    "musl/src/unistd/getlogin_r.c",
    "musl/src/unistd/getpgid.c",
    "musl/src/unistd/getpgrp.c",
    "musl/src/unistd/getpid.c",
    "musl/src/unistd/getppid.c",
    "musl/src/unistd/getsid.c",
    "musl/src/unistd/getuid.c",
    "musl/src/unistd/isatty.c",
    "musl/src/unistd/lchown.c",
    "musl/src/unistd/link.c",
    "musl/src/unistd/linkat.c",
    "musl/src/unistd/lseek.c",
    "musl/src/unistd/mips/pipe.s",
    "musl/src/unistd/mips64/pipe.s",
    "musl/src/unistd/mipsn32/lseek.c",
    "musl/src/unistd/mipsn32/pipe.s",
    "musl/src/unistd/nice.c",
    "musl/src/unistd/pause.c",
    "musl/src/unistd/pipe.c",
    "musl/src/unistd/pipe2.c",
    "musl/src/unistd/posix_close.c",
    "musl/src/unistd/pread.c",
    "musl/src/unistd/preadv.c",
    "musl/src/unistd/pwrite.c",
    "musl/src/unistd/pwritev.c",
    "musl/src/unistd/read.c",
    "musl/src/unistd/readlink.c",
    "musl/src/unistd/readlinkat.c",
    "musl/src/unistd/readv.c",
    "musl/src/unistd/renameat.c",
    "musl/src/unistd/rmdir.c",
    "musl/src/unistd/setegid.c",
    "musl/src/unistd/seteuid.c",
    "musl/src/unistd/setgid.c",
    "musl/src/unistd/setpgid.c",
    "musl/src/unistd/setpgrp.c",
    "musl/src/unistd/setregid.c",
    "musl/src/unistd/setresgid.c",
    "musl/src/unistd/setresuid.c",
    "musl/src/unistd/setreuid.c",
    "musl/src/unistd/setsid.c",
    "musl/src/unistd/setuid.c",
    "musl/src/unistd/setxid.c",
    "musl/src/unistd/sh/pipe.s",
    "musl/src/unistd/sleep.c",
    "musl/src/unistd/symlink.c",
    "musl/src/unistd/symlinkat.c",
    "musl/src/unistd/sync.c",
    "musl/src/unistd/tcgetpgrp.c",
    "musl/src/unistd/tcsetpgrp.c",
    "musl/src/unistd/truncate.c",
    "musl/src/unistd/ttyname.c",
    "musl/src/unistd/ttyname_r.c",
    "musl/src/unistd/ualarm.c",
    "musl/src/unistd/unlink.c",
    "musl/src/unistd/unlinkat.c",
    "musl/src/unistd/usleep.c",
    "musl/src/unistd/write.c",
    "musl/src/unistd/writev.c",
    "musl/src/unistd/x32/lseek.c",
};

const compat_time32_files = [_][]const u8{
    "musl/compat/time32/__xstat.c",
    "musl/compat/time32/adjtime32.c",
    "musl/compat/time32/adjtimex_time32.c",
    "musl/compat/time32/aio_suspend_time32.c",
    "musl/compat/time32/clock_adjtime32.c",
    "musl/compat/time32/clock_getres_time32.c",
    "musl/compat/time32/clock_gettime32.c",
    "musl/compat/time32/clock_nanosleep_time32.c",
    "musl/compat/time32/clock_settime32.c",
    "musl/compat/time32/cnd_timedwait_time32.c",
    "musl/compat/time32/ctime32.c",
    "musl/compat/time32/ctime32_r.c",
    "musl/compat/time32/difftime32.c",
    "musl/compat/time32/fstat_time32.c",
    "musl/compat/time32/fstatat_time32.c",
    "musl/compat/time32/ftime32.c",
    "musl/compat/time32/futimens_time32.c",
    "musl/compat/time32/futimes_time32.c",
    "musl/compat/time32/futimesat_time32.c",
    "musl/compat/time32/getitimer_time32.c",
    "musl/compat/time32/getrusage_time32.c",
    "musl/compat/time32/gettimeofday_time32.c",
    "musl/compat/time32/gmtime32.c",
    "musl/compat/time32/gmtime32_r.c",
    "musl/compat/time32/localtime32.c",
    "musl/compat/time32/localtime32_r.c",
    "musl/compat/time32/lstat_time32.c",
    "musl/compat/time32/lutimes_time32.c",
    "musl/compat/time32/mktime32.c",
    "musl/compat/time32/mq_timedreceive_time32.c",
    "musl/compat/time32/mq_timedsend_time32.c",
    "musl/compat/time32/mtx_timedlock_time32.c",
    "musl/compat/time32/nanosleep_time32.c",
    "musl/compat/time32/ppoll_time32.c",
    "musl/compat/time32/pselect_time32.c",
    "musl/compat/time32/pthread_cond_timedwait_time32.c",
    "musl/compat/time32/pthread_mutex_timedlock_time32.c",
    "musl/compat/time32/pthread_rwlock_timedrdlock_time32.c",
    "musl/compat/time32/pthread_rwlock_timedwrlock_time32.c",
    "musl/compat/time32/pthread_timedjoin_np_time32.c",
    "musl/compat/time32/recvmmsg_time32.c",
    "musl/compat/time32/sched_rr_get_interval_time32.c",
    "musl/compat/time32/select_time32.c",
    "musl/compat/time32/sem_timedwait_time32.c",
    "musl/compat/time32/semtimedop_time32.c",
    "musl/compat/time32/setitimer_time32.c",
    "musl/compat/time32/settimeofday_time32.c",
    "musl/compat/time32/sigtimedwait_time32.c",
    "musl/compat/time32/stat_time32.c",
    "musl/compat/time32/stime32.c",
    "musl/compat/time32/thrd_sleep_time32.c",
    "musl/compat/time32/time32.c",
    "musl/compat/time32/time32gm.c",
    "musl/compat/time32/timer_gettime32.c",
    "musl/compat/time32/timer_settime32.c",
    "musl/compat/time32/timerfd_gettime32.c",
    "musl/compat/time32/timerfd_settime32.c",
    "musl/compat/time32/timespec_get_time32.c",
    "musl/compat/time32/utime_time32.c",
    "musl/compat/time32/utimensat_time32.c",
    "musl/compat/time32/utimes_time32.c",
    "musl/compat/time32/wait3_time32.c",
    "musl/compat/time32/wait4_time32.c",
};
